#!/usr/bin/perl
# batch.pl: Perl script to process a batch input file.
# Author: Doug DeJulio, ddj@hks.net, 1998/10/09
# $Id: batch.pl,v 1.3 1998/10/30 23:06:20 ddj Exp $

# This script is documented in "batch.txt".
# Sample input is available in "input.csv".

# Load the CCVS Perl module.
use CCVS;

# Check the arguments.
if ($#ARGV > 2) {
    die "too many arguments";
}

if ($#ARGV < 0) {
    die "too few arguments";
}

# Output file.
if ($#ARGV == 2) {
    open(OUT, ">" . $ARGV[2]);
} else {
    open(OUT, ">-");
}

# Input file.
if ($#ARGV >= 1) {
    open(IN, $ARGV[1]);
} else {
    open(IN, "-");
}

# Configuration to use.
$config = $ARGV[0];

$s = CCVS::init($config);

foreach (<IN>) {
    # Parse a line from the CSV file.  Perl makes this easy.

    # strip out newline and all quotation marks
    chop; s/\"//g;
    # split on commas
    ($command, $data) = split / *, */,$_,2;
    @data = split / *, */,$data;

    # Do the command.
    &$command;
}

# Check to see if last invoice is done yet.  Poll once per second.
do {
    sleep 1;
    $status = CCVS::status($s, $invoice);
} until ($status == CV_AUTH
	 || $status == CV_DENIED
	 || $status == CV_REVIEW);

# All but the postauths have been processed.  Do the postauths.
foreach (@postauths) {
    (CCVS::sale($s, $_) == CV_OK)
	|| die "Problem issuing postauth.";
}

# Now do a "close batch" if it's been requested.
if ($do_settlement) {
    system("/usr/local/ccvs/sbin/cvupload", $config);
}

# Now all the transactions have been processed.  Report their status!
foreach $invoice (@report) {
    (CCVS::status($s, $invoice));
    %r = split / *{|} */,CCVS::textvalue($s);
    # %status is now an associative array of all the key/value pairs
    # in the transaction's status.
    print $r{'invoice'}, " ", $r{'status'}, "\n";
    if ($r{'status'} == 'denied'
	|| $r{'status'} == 'review'
	|| $r{'status'} == 'bad') {
	$output = "N";
    } elsif ($r{'status'} == 'auth'
	     || $r{'status'} == 'done'
	     || $r{'status'} == 'ready') {
	$output = "Y";
    } else {
	$output = "?";
    }

    # When we start to return the authcode, it should go here.
    $output .= "000000";

    $output .= $r{'invoice'};

    print OUT $output, "\n";
}

# Do a whole transaction (auth + sale, but no settlement)
sub C1 {
    $invoice = &C4(@data);
    push @postauths, ($invoice);
    return $invoice;
}

# Do a reversal
sub C2 {
    local ($comment, $inv, $other_junk) = @data;
    (CCVS::reverse($s, $inv) == CV_OK)
	|| die "Problem issuing reversal";
    $invoice = $inv;
    push @report, ($inv);
    return $inv;
}

# Do a return.
sub C3 {
    local ($comment, $inv, $cardnum, $expdate, $amount) = @data;

    # Create the transaction.
    (CCVS::new($s, $inv) == CV_OK
     && CCVS::add($s, $inv, CV_ARG_CARDNUM, $cardnum) == CV_OK
     && CCVS::add($s, $inv, CV_ARG_EXPDATE, $expdate) == CV_OK
     && CCVS::add($s, $inv, CV_ARG_AMOUNT, $amount) == CV_OK)
	|| die "Problem creating transaction.";
    # Add comment if requested.
    (CCVS::add($s, $inv, CV_ARG_COMMENT, $comment) == CV_OK)
	|| die "Problem adding comment to transaction.";
    # Do the return.
    (CCVS::return($s, $inv) == CV_OK)
	|| die "Problem issuing return.";

    push @report, ($inv);
    return $inv;
}

# Do a preauth
sub C4 {
    local ($comment, $inv, $cardnum, $expdate, $amount, $zip, $addr) = @data;

    # Create the transaction.
    (CCVS::new($s, $inv) == CV_OK
     && CCVS::add($s, $inv, CV_ARG_CARDNUM, $cardnum) == CV_OK
     && CCVS::add($s, $inv, CV_ARG_EXPDATE, $expdate) == CV_OK
     && CCVS::add($s, $inv, CV_ARG_AMOUNT, $amount) == CV_OK)
	|| die "Problem creating transaction.";
    # Add AVS data if requested.
    if ($zip) {
	(CCVS::add($s, $inv, CV_ARG_AVS_ZIPCODE, $zip) == CV_OK)
	    || die "Problem adding AVS zipcode to transaction.";
    }
    if ($addr) {
	(CCVS::add($s, $inv, CV_ARG_AVS_ADDRESS, $addr) == CV_OK)
	    || die "Problem adding AVS zipcode to transaction.";
    }
    # Add comment if requested.
    if ($comment) {
	(CCVS::add($s, $inv, CV_ARG_COMMENT, $comment) == CV_OK)
	    || die "Problem adding comment to transaction.";
    }
    # Do the authorization.
    (CCVS::auth($s, $inv) == CV_OK)
	|| die "Problem issuing authorization request.";

    push @report, ($inv);
    $invoice = $inv;
    return $inv;
}

# Do a postauth.
sub C5 {
    local ($comment, $inv, $other_crud) = @data;
    push @postauths, ($inv);
    push @report, ($inv);
    return $inv;
}

# Do a settlement
sub ST {
    # No arguments.
    # We need to force cvupload to run, after everything else.
    $do_settlement = 1;
}
