/* cv_cli_auth.c: CLI utility program to perform an authorization. */
/* Author: Doug DeJulio, ddj@hks.net, 19970917 */

static const char rcsid[] = "$Id: cv_cli_auth.c,v 1.1 1997/09/17 19:58:27 ddj Exp $";

#include <stdio.h>		/* printf() */
#include <stdlib.h>		/* getenv() */
#include <unistd.h>		/* getopt() */

#include "cv_api.h"		/* CCVS C API. */

int main(int argc, char *argv[])
{
  extern char *optarg;
  extern int optind, opterr, optopt;
  int opt;
  void *session;
  int r;
  char *p;
  int state = 0;

  char *config = NULL;
  char *invoice = NULL;
  char *cardnum = NULL;
  char *edate = NULL;
  char *amount = NULL;

  /* Process arguments. */
  while ((opt = getopt(argc, argv, "C:p:n:d:")) != EOF) {
    switch (opt) {
    case 'C':			/* Configuration. */
      config = optarg;
      break;
    case 'p':			/* Price. */
      amount = optarg;
      break;
    case 'n':			/* Card number. */
      cardnum = optarg;
      break;
    case 'd':
      edate = optarg;		/* Expiration date. */
      break;
    case '?':
      fprintf(stderr, "%s: unknown option '%c', ignoring...\n",
	      argv[0], optopt);
    }
  }

  if ((argc - optind) != 1
      || amount == NULL
      || cardnum == NULL
      || edate == NULL) {
    fprintf(stderr,
	    "usage: %s [-C configname] -p amount -n cardnum -d expdate"
	    " invoice\n",
	    argv[0]);
    return 1;
  }

  /* First non-switch argument is the invoice to use. */
  invoice = argv[optind];
  /* All arguments are handled. */

  /* Initialize with configuration name. */
  if (config == NULL) {		/* No config argument? */
    config = getenv("CCVS_CONFIG_NAME");
    if (config == NULL) {	/* No environment variable either? */
      fprintf(stderr, "%s: You must specify a configuration.\n", argv[0]);
      return 1;
    }
  }

  /* Config name, invoice, price, cardnum and edate are all set. */
  session = cv_init(config);
  if (session == CV_SESS_BAD) {
    fprintf(stderr,
	    "%s: Unable to initialize with configuration '%s'.\n",
	    argv[0], config);
    return 1;
  }

  /* Call cv_create() with required arguments, then optional arguments
     in argtype/argval pairs, then a special arg to end the arg list. */
  r = cv_create(session, invoice,
		CV_ARG_AMOUNT, amount, /* Amount of transaction. */
		CV_ARG_CARDNUM, cardnum, /* Credit card number. */
		CV_ARG_EXPDATE, edate, /* Expiration date. */
		CV_ARG_END);

  if (r == CV_E_DUPINVOICE) {	/* Duplicate invoice? */
    fprintf(stderr, "%s: Transaction '%s' already exists.\n",
	    argv[0], invoice);
    return 1;
  }

  if (r != CV_OK) {
    fprintf(stderr, "%s: Some problem with transaction '%s'.\n",
	    argv[0], invoice);
    return 1;
  }
  
  /* The create has succeeded.  Now do the auth. */
  r = cv_auth(session, invoice);

  if (r != CV_OK) {
    fprintf(stderr, "%s: Some problem authorizing transaction '%s'.\n",
	    argv[0], invoice);
    return 1;
  }

  /* The auth has succeeded.  Poll its status until it's not "unauth". */
  while ((r = cv_status(session, invoice)) == CV_UNAUTH) {
    sleep(1);
  }

  /* Now we display the status.  See the "status" CLI source for a
     more detailed explanation of this. */
  p = cv_textvalue(session);
  while (p != NULL && *p != '\0') {
    switch (*p) {
    case '{':			/* End of 'key', start of 'value'. */
      printf(": ");
      state = 1;
      break;
    case '}':			/* End of 'value', start of next 'key'. */
      printf("\n");
      state = 0;
      break;
    case ' ':
      if (state == 1)
	printf(" ");
      break;
    default:
      printf("%c", *p);
    }
    p++;
  }

  /* Clean up. */
  cv_done(session);

  return 0;
}
