/* ccvsmodule.c: Python module for CCVS. */
/* Author: Doug DeJulio, ddj@hks.net, 1998/09/11 */

static char const rcsid[] = "$Id: ccvsmodule.c,v 1.7 1999/03/08 17:59:48 zarf Exp $";

#include <Python.h>

#include "cv_api.h"

/* Declare functions not in cv_api.h but in libccvs.a. */
char *hks_ptr_ptrtostring(void *vptr);
void *hks_ptr_stringtoptr(char *str);

/* Initializer. */
static PyObject *
ccvs_init(PyObject *self,
	   PyObject *args)
{
  char *name;
  void *vsess;
  char *p;
  PyObject *retval;

  if (!PyArg_ParseTuple(args, "s", &name))
    return Py_None;

  /* Initialize. */
  vsess = cv_init(name);

  if (!vsess)
    return Py_None;

  /* Convert the (void*) into a string. */
  p = hks_ptr_ptrtostring(vsess);

  /* Build a return value out of the string and free it. */
  retval = Py_BuildValue("s", p);
  free(p);

  return retval;
}

static PyObject *
ccvs_done(PyObject *self,
	  PyObject *args)
{
  char *sess;
  void *vsess;

  if (!PyArg_ParseTuple(args, "s", &sess))
    return Py_None;

  /* Recover pointer from string. */
  vsess = hks_ptr_stringtoptr(sess);
  /* Do the "done" call. */
  cv_done(vsess);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
ccvs_new(PyObject *self,
	 PyObject *args)
{
  char *psess;
  char *invoice;
  void *sess;
  register int r;

  if (!PyArg_ParseTuple(args, "ss", &psess, &invoice))
    return NULL;

  /* Recover pointer from string. */
  sess = hks_ptr_stringtoptr(psess);

  r = cv_new(sess, invoice);

  return Py_BuildValue("s", cv_ret2str(r));  
}

static PyObject *
ccvs_add(PyObject *self,
	 PyObject *args)
{
  char *psess;
  void *sess;
  char *invoice;
  char *pargtype;
  int argtype;
  char *argval;
  register int r;

  if (!PyArg_ParseTuple(args, "ssss", &psess, &invoice, &pargtype, &argval))
    return NULL;

  /* Recover pointer from string. */
  sess = hks_ptr_stringtoptr(psess);
  /* Convert argtype string to argtype integer. */
  argtype = cv_str2arg(pargtype);

  r = cv_add(sess, invoice, argtype, argval);

  return Py_BuildValue("s", cv_ret2str(r));
}

static PyObject *
ccvs_delete(PyObject *self,
	    PyObject *args)
{
  char *psess;
  void *sess;
  char *invoice;
  register int r;

  if (!PyArg_ParseTuple(args, "ss", &psess, &invoice))
    return NULL;

  /* Recover pointer from string. */
  sess = hks_ptr_stringtoptr(psess);

  r = cv_delete(sess, invoice);

  return Py_BuildValue("s", cv_ret2str(r));
}

static PyObject *
ccvs_auth(PyObject *self,
	  PyObject *args)
{
  char *psess;
  void *sess;
  char *invoice;
  register int r;

  if (!PyArg_ParseTuple(args, "ss", &psess, &invoice))
    return NULL;

  /* Recover pointer from string. */
  sess = hks_ptr_stringtoptr(psess);

  r = cv_auth(sess, invoice);

  return Py_BuildValue("s", cv_ret2str(r));
}

static PyObject *
ccvs_return_(PyObject *self,
	    PyObject *args)
{
  char *psess;
  void *sess;
  char *invoice;
  register int r;

  if (!PyArg_ParseTuple(args, "ss", &psess, &invoice))
    return NULL;

  /* Recover pointer from string. */
  sess = hks_ptr_stringtoptr(psess);

  r = cv_return(sess, invoice);

  return Py_BuildValue("s", cv_ret2str(r));
}

static PyObject *
ccvs_reverse(PyObject *self,
	     PyObject *args)
{
  char *psess;
  void *sess;
  char *invoice;
  register int r;

  if (!PyArg_ParseTuple(args, "ss", &psess, &invoice))
    return NULL;

  /* Recover pointer from string. */
  sess = hks_ptr_stringtoptr(psess);

  r = cv_reverse(sess, invoice);

  return Py_BuildValue("s", cv_ret2str(r));
}

static PyObject *
ccvs_sale(PyObject *self,
	  PyObject *args)
{
  char *psess;
  void *sess;
  char *invoice;
  register int r;

  if (!PyArg_ParseTuple(args, "ss", &psess, &invoice))
    return NULL;

  /* Recover pointer from string. */
  sess = hks_ptr_stringtoptr(psess);

  r = cv_sale(sess, invoice);

  return Py_BuildValue("s", cv_ret2str(r));
}

static PyObject *
ccvs_void(PyObject *self,
	  PyObject *args)
{
  char *psess;
  void *sess;
  char *invoice;
  register int r;

  if (!PyArg_ParseTuple(args, "ss", &psess, &invoice))
    return NULL;

  /* Recover pointer from string. */
  sess = hks_ptr_stringtoptr(psess);

  r = cv_void(sess, invoice);

  return Py_BuildValue("s", cv_ret2str(r));
}

static PyObject *
ccvs_status(PyObject *self,
	    PyObject *args)
{
  char *psess;
  void *sess;
  char *invoice;
  register int r;

  if (!PyArg_ParseTuple(args, "ss", &psess, &invoice))
    return NULL;

  /* Recover pointer from string. */
  sess = hks_ptr_stringtoptr(psess);

  r = cv_status(sess, invoice);

  return Py_BuildValue("s", cv_stat2str(r));
}

static PyObject *
ccvs_count(PyObject *self,
	   PyObject *args)
{
  char *psess;
  void *sess;
  char *ptype;
  int type;
  register int r;

  if (!PyArg_ParseTuple(args, "ss", &psess, &ptype))
    return NULL;

  /* Recover pointer from string. */
  sess = hks_ptr_stringtoptr(psess);
  /* Convert type string into type integer. */
  type = cv_str2stat(ptype);

  r = cv_count(sess, type);

  return Py_BuildValue("i", r);
}

static PyObject *
ccvs_lookup(PyObject *self,
	    PyObject *args)
{
  char *psess;
  void *sess;
  char *ptype;
  int type;
  int inum;
  

  if (!PyArg_ParseTuple(args, "ssi", &psess, &ptype, &inum))
    return NULL;

  /* Recover pointer from string. */
  sess = hks_ptr_stringtoptr(psess);
  /* Convert type string into type integer. */
  type = cv_str2stat(ptype);

  cv_lookup(sess, type, inum);

  return Py_BuildValue("s", cv_textvalue(sess));
}

static PyObject *
ccvs_report(PyObject *self,
	    PyObject *args)
{
  char *psess;
  void *sess;
  char *ptype;
  int type;
  register int r;

  if (!PyArg_ParseTuple(args, "ss", &psess, &ptype))
    return NULL;

  /* Recover pointer from string. */
  sess = hks_ptr_stringtoptr(psess);
  /* Convert type string into type integer. */
  type = cv_str2rep(ptype);

  r = cv_report(sess, type);

  return Py_BuildValue("s", cv_stat2str(r));
}

static PyObject *
ccvs_command(PyObject *self,
	    PyObject *args)
{
  char *psess;
  void *sess;
  char *ptype;
  char *argval;
  int type;
  register int r;

  if (!PyArg_ParseTuple(args, "sss", &psess, &ptype, &argval))
    return NULL;

  /* Recover pointer from string. */
  sess = hks_ptr_stringtoptr(psess);
  /* Convert type string into type integer. */
  type = cv_str2cmd(ptype);

  r = cv_command(sess, type, argval);

  return Py_BuildValue("s", cv_stat2str(r));
}

static PyObject *
ccvs_textvalue(PyObject *self,
	       PyObject *args)
{
  char *psess;
  void *sess;

  if (!PyArg_ParseTuple(args, "s", &psess))
    return NULL;

  /* Recover pointer from string. */
  sess = hks_ptr_stringtoptr(psess);

  return Py_BuildValue("s", cv_textvalue(sess));
}

static PyMethodDef CcvsMethods[] = {
  {"init", ccvs_init, METH_VARARGS},
  {"done", ccvs_done, METH_VARARGS},
  {"new", ccvs_new, METH_VARARGS},
  {"add", ccvs_add, METH_VARARGS},
  {"delete", ccvs_delete, METH_VARARGS},
  {"auth", ccvs_auth, METH_VARARGS},
  {"return_", ccvs_return_, METH_VARARGS},
  {"reverse", ccvs_reverse, METH_VARARGS},
  {"sale", ccvs_sale, METH_VARARGS},
  {"void", ccvs_void, METH_VARARGS},
  {"status", ccvs_status, METH_VARARGS},
  {"count", ccvs_count, METH_VARARGS},
  {"lookup", ccvs_lookup, METH_VARARGS},
  {"report", ccvs_report, METH_VARARGS},
  {"command", ccvs_command, METH_VARARGS},
  {"textvalue", ccvs_textvalue, METH_VARARGS},
  {NULL, NULL}
};

void
initccvs()
{
  (void) Py_InitModule("ccvs", CcvsMethods);
}
