/* cv_cli_status.c: CLI utility program to check transaction status. */
/* Author: Doug DeJulio, ddj@hks.net, 19970917 */

static const char rcsid[] = "$Id: cv_cli_status.c,v 1.5 1999/02/22 17:06:36 ddj Exp $";

#include <stdio.h>		/* printf() */
#include <stdlib.h>		/* getenv() */
#include <unistd.h>		/* getopt() */

#include "cv_api.h"		/* CCVS C API. */

int main(int argc, char *argv[])
{
  char *config = NULL;		/* Configuration name. */
  char *invoice = NULL;		/* Invoice to check status of. */
  extern char *optarg;
  extern int optind, opterr, optopt;
  int opt;
  void *session;
  int r;
  char *p;			/* Text pointer. */
  int state = 0;

  int versp = 0;

  /* Process arguments. */
  /* Configuration name is specified by -C or an environment variable. */
  while ((opt = getopt(argc, argv, "c:C:V")) != EOF) {
    switch (opt) {
    case 'c':
    case 'C':			/* Configuration name. */
      config = optarg;		/* Store it. */
      break;
    case 'V':
      versp = 1;
      break;
    case '?':
      fprintf(stderr, "%s: unknown option '%c', ignoring...\n",
	      argv[0], optopt);
    }
  }

  if (!versp && (argc - optind) != 1) {
    fprintf(stderr, "usage: %s [-C configname] [-V | invoice]\n", argv[0]);
    return 1;
  }
  /* First non-switch argument is the invoice to use. */
  invoice = argv[optind];
  /* Arguments handled. */
  
  /* Initialize with configuration name. */
  if (config == NULL) {	/* No config argument? */
    config = getenv("CCVS_CONFIG_NAME");
    if (config == NULL) {	/* No argument variable either? */
      fprintf(stderr,
	      "%s: You must specify a configuration.\n",
	      argv[0]);
      return 1;
    }
  }
  
  /* Config name and invoice are both set now. */
  session = cv_init(config);
  if (session == CV_SESS_BAD) {
    fprintf(stderr,
	    "%s: Unable to initialize with configuration '%s'.\n",
	    argv[0], config);
    return 1;
  }

  /* Do we just report CCVS version info and exit now? */
  if (versp) {
    puts(cv_textvalue(session)); /* Print the version. */
    cv_done(session);		/* Close the session. */
    return 0;			/* Terminate. */
  }
  
  r = cv_status(session, invoice);
  
  /* The data contained in the return value is also encoded in the
     supplementary text, so for this program we can really just ignore
     the return value if we're going to parse the text. */

  /* If you only need the "state" of the transaction
     (eg. "authorized", "done", ...) it's easier to check the return
     value.  It'll be CV_NEW, CV_AUTH, CV_DONE, or one of the other
     states defined in the cv_api.h file. */

  /* Now we have to display the supplementary text. */
  p = cv_textvalue(session);
  
  /* The text returned by cv_textvalue is a bunch of key/value pairs
     in the form "key {value} key {value}...", and we want to display
     it as lines of the form "key: value".  This should do it. */
  
  while (p != NULL && *p != '\0') {
    switch (*p) {
    case '}':			/* End of value. */
      printf("\n");		/* Show newline. */
      state = 0;		/* Next is "key". */
      break;
    case '{':			/* Start of value. */
      printf(": ");
      state = 1;
      break;
    case ' ':
      if (state == 1)
	printf(" ");
      break;
    default:
      printf("%c", *p);
      break;
    }
    p++;
  }

  /* Clean up. */
  cv_done(session);

  return 0;			/* This command never "fails". */
}
