/* cv_cli_list.c: CLI utility program to list transactions in a given state. */
/* Author: Doug DeJulio, ddj@hks.net, 19970917 */

static const char rcsid[] = "$Id: cv_cli_list.c,v 1.6 1999/02/22 17:06:03 ddj Exp $";

#include <stdio.h>		/* printf() */
#include <stdlib.h>		/* getenv() */
#include <unistd.h>		/* getopt() */

#include "cv_api.h"

int main(int argc, char *argv[])
{
  char *config = NULL;		/* Configuration name. */
  char *statename = NULL;	/* Name of state to list. */
  int state = CV_NONE;
  extern char *optarg;
  extern int optind, opterr, optopt;
  int opt;
  void *session;
  int r;
  register i;

  int versp = 0;

  /* Process arguments. */
  /* Configuration name is specified by -C or an environment variable. */
  while ((opt = getopt(argc, argv, "c:C:V")) != EOF) {
    switch (opt) {
    case 'c':
    case 'C':			/* Configuration name. */
      config = optarg;		/* Store it. */
      break;
    case 'V':
      versp = 1;
      break;
    case '?':			/* Unknown option, give warning. */
      fprintf(stderr, "%s: unknown option '%c', ignoring...\n",
	      argv[0], optopt);
    }
  }

  if (!versp && (argc - optind) != 1) {
    fprintf(stderr, "usage: %s [-C configname] [-V | state]\n", argv[0]);
    return 1;
  }

  /* First non-switch argument is the invoice to use. */
  statename = argv[optind];
  /* Arguments handled. */
  /* Initialize with configuration name. */
  if (config == NULL) {		/* No config argument? */
    config = getenv("CCVS_CONFIG_NAME");
    if (config == NULL) {	/* No config argument & no env var. */
      fprintf(stderr,
	      "%s: You must specify a configuration.\n",
	      argv[0]);
      return 1;
    }
  }

  /* Config name and invoice are both set now. */
  session = cv_init(config);
  if (session == CV_SESS_BAD) {
    fprintf(stderr,
	    "%s: Unable to initialize with configuration '%s'.\n",
	    argv[0], config);
    return 1;
  }

  /* Do we just report CCVS version info and exit now? */
  if (versp) {
    puts(cv_textvalue(session)); /* Print the version. */
    cv_done(session);		/* Close the session. */
    return 0;			/* Terminate. */
  }

  /*** DO THE LISTING ***/
  if (strcmp("new", statename) == 0) {
    state = CV_NEW;
  } else if (strcmp("auth", statename) == 0) {
    state = CV_AUTH;
  } else if (strcmp("ready", statename) == 0) {
    state = CV_READY;
  } else if (strcmp("done", statename) == 0) {
    state = CV_DONE;
  } else if (strcmp("unauth", statename) == 0) {
    state = CV_UNAUTH;
  } else if (strcmp("denied", statename) == 0) {
    state = CV_DENIED;
  } else if (strcmp("review", statename) == 0) {
    state = CV_REVIEW;
  } else if (strcmp("bad", statename) == 0) {
    state = CV_BAD;
  } else {
    fprintf(stderr,
	    "%s: State must be one of new, auth, ready, done, unauth, "
	    "denied, review or bad.\n",
	    argv[0]);
    return 1;
  }

  for (i=1; i <= cv_count(session, state); i++) {
    r = cv_lookup(session, state, i);
    if (r == CV_OK) {		/* Did the lookup succeed? */
      printf("%s\n", cv_textvalue(session)); /* Show the invoice. */
    }
  }

  /* Clean up. */
  cv_done(session);

  return 0;
}
