/*
 * This class implements a Java GUI application which demonstrates
 * simple CCVS processing. It can also be run as an applet, although
 * since it has to load a native library, it's not useful for a public
 * web site. You might adapt it for an in-house web site, though.
 *
 * Before you run this (either as an application or an applet), you
 * must set your CLASSPATH and LD_LIBRARY_PATH environment variables
 * as documented in the CCVS Java Programmer's Manual. If you have
 * used the default installation locations, this means
 *
 * add /usr/local/ccvs/class/ccvs.jar to CLASSPATH
 * add /usr/local/ccvs/class to LD_LIBRARY_PATH
 *
 * Then type
 *
 * java AppDemo (to run as an application)
 * appletviewer AppDemo.html (to run as an applet)
 *
 * The window provides fields for you to enter a invoice number,
 * credit card number, expiration date, and amount. Hit "auth" to
 * authenticate a credit-card transaction. The applet uses the "demo"
 * configuration, so this will not actually contact any clearing
 * house; it will only test to see if the credit-card number has a
 * valid checksum.
 * 
 * A few seconds after you press the "auth" button, if the ccvsd
 * daemon is running, a window will pop up displaying the result of
 * the auth.  You can then use the "sale" button to mark the
 * transaction for settlement (which will occur when you run
 * cvupload.) Note that the card number, expiration date, and amount
 * fields are ignored for "sale"; CCVS uses the information that you
 * put in the original "auth" for the given invoice.
 *
 * The "return" button creates a transaction with the given invoice,
 * card number, expiration date, and amount, and marks it for
 * settlement (during cvupload.)
 *
 * The "delete" button deletes a transaction with the given invoice
 * number. Remember, you cannot delete a transaction which is waiting
 * for authentication or settlement.
 *
 * This demo code requires CCVS 3.0 or later.
 * 
 * $Id: AppDemo.java,v 1.1 1998/11/06 17:25:34 zarf Exp $ */

import java.awt.*;
import java.awt.event.*;
import java.util.*;

import net.hks.Ccvs;
import net.hks.CcvsException;

public class AppDemo 
  extends java.applet.Applet 
  implements ActionListener {
  
  private boolean active = false;
  private Ccvs session = null;

  private TextField invoicefield, cardnumfield, expdatefield, amountfield;
  private TextArea resultarea;

  public static void main(String[] args) {
    Frame fram = new Frame("CCVS Java Demo");
    java.applet.Applet applet = new AppDemo();
    fram.add(applet, "Center");
    applet.init();
    fram.setSize(300, 300);
    fram.pack();
    fram.show();
    applet.start();
    applet.stop();
    applet.destroy();
  }

  public void init() {
    String configname = "demo";

    try {
      /* Create a Ccvs session object. */
      session = new Ccvs(configname); 
      active = true;
    }
    catch (CcvsException ex) {
      System.out.println(ex.toString());
    }

    /* Lay out all the GUI elements -- tedious but also uninteresting. */

    GridBagLayout lm = new GridBagLayout();
    GridBagConstraints c = new GridBagConstraints();
    Component comp;
    Button but;
    TextField tf;
    TextArea ta;
    setLayout(lm);

    Font fieldfont = new Font("Helvetica", Font.PLAIN, 10);

    /* Create the input textfields */

    c.fill = GridBagConstraints.NONE;
    c.gridx = 0;
    c.gridy = 0;
    c.gridwidth = 1;
    c.weightx = 0;
    comp = new Label("Invoice");
    lm.setConstraints(comp, c);
    add(comp);

    c.fill = GridBagConstraints.HORIZONTAL;
    c.gridx = 1;
    c.gridy = 0;
    c.gridwidth = 4;
    c.weightx = 1.0;
    tf = new TextField("TEST0001", 8);
    tf.setFont(fieldfont);
    lm.setConstraints(tf, c);
    add(tf);
    invoicefield = tf;

    c.fill = GridBagConstraints.NONE;
    c.gridx = 0;
    c.gridy = 2;
    c.gridwidth = 1;
    c.weightx = 0;
    comp = new Label("Card No.");
    lm.setConstraints(comp, c);
    add(comp);

    c.fill = GridBagConstraints.HORIZONTAL;
    c.gridx = 1;
    c.gridy = 2;
    c.gridwidth = 4;
    c.weightx = 1.0;
    tf = new TextField("0000-0000-0000-0000");
    tf.setFont(fieldfont);
    lm.setConstraints(tf, c);
    add(tf);
    cardnumfield = tf;

    c.fill = GridBagConstraints.NONE;
    c.gridx = 0;
    c.gridy = 3;
    c.gridwidth = 1;
    c.weightx = 0;
    comp = new Label("Exp. Date");
    lm.setConstraints(comp, c);
    add(comp);

    c.fill = GridBagConstraints.HORIZONTAL;
    c.gridx = 1;
    c.gridy = 3;
    c.gridwidth = 4;
    c.weightx = 1.0;
    tf = new TextField("12/99");
    tf.setFont(fieldfont);
    lm.setConstraints(tf, c);
    add(tf);
    expdatefield = tf;

    c.fill = GridBagConstraints.NONE;
    c.gridx = 0;
    c.gridy = 4;
    c.gridwidth = 1;
    c.weightx = 0;
    comp = new Label("Amount");
    lm.setConstraints(comp, c);
    add(comp);

    c.fill = GridBagConstraints.HORIZONTAL;
    c.gridx = 1;
    c.gridy = 4;
    c.gridwidth = 4;
    c.weightx = 1.0;
    tf = new TextField("1.00");
    tf.setFont(fieldfont);
    lm.setConstraints(tf, c);
    add(tf);
    amountfield = tf;

    /* Create the action buttons */

    c.fill = GridBagConstraints.NONE;
    c.gridx = 0;
    c.gridy = 5;
    c.gridwidth = 1;
    c.weightx = 0;
    comp = new Label("Perform");
    lm.setConstraints(comp, c);
    add(comp);

    c.fill = GridBagConstraints.HORIZONTAL;
    c.gridx = 1;
    c.gridy = 5;
    c.gridwidth = 1;
    c.weightx = 1.0;
    but = new Button("Auth");
    lm.setConstraints(but, c);
    add(but);
    but.setActionCommand("AUTH");
    but.addActionListener(this);

    c.fill = GridBagConstraints.HORIZONTAL;
    c.gridx = 2;
    c.gridy = 5;
    c.gridwidth = 1;
    c.weightx = 1.0;
    but = new Button("Sale");
    lm.setConstraints(but, c);
    add(but);
    but.setActionCommand("SALE");
    but.addActionListener(this);

    c.fill = GridBagConstraints.HORIZONTAL;
    c.gridx = 3;
    c.gridy = 5;
    c.gridwidth = 1;
    c.weightx = 1.0;
    but = new Button("Return");
    lm.setConstraints(but, c);
    add(but);
    but.setActionCommand("RETURN");
    but.addActionListener(this);

    c.fill = GridBagConstraints.HORIZONTAL;
    c.gridx = 4;
    c.gridy = 5;
    c.gridwidth = 1;
    c.weightx = 1.0;
    but = new Button("Delete");
    lm.setConstraints(but, c);
    add(but);
    but.setActionCommand("DELETE");
    but.addActionListener(this);

    /* Create the result area. */

    c.fill = GridBagConstraints.BOTH;
    c.gridx = 0;
    c.gridy = 6;
    c.gridwidth = 5;
    c.weighty = 1.0;
    ta = new TextArea();
    ta.setFont(fieldfont);
    ta.setEditable(false);
    lm.setConstraints(ta, c);  
    add(ta);
    resultarea = ta;
  }

  /* This method is called when any button is hit (because we have
     set the AppDemo object to be an actionListener for the buttons.) */
  public void actionPerformed(ActionEvent ev) {
    String cmd = ev.getActionCommand();

    if (!active) {
      /* We don't have a session object. Don't do anything. */
      resultarea.setText("The CCVS library could not be initialized.\n");
      return;
    }

    /* See what the command name is, and execute it. */

    if (cmd.equals("AUTH") || cmd.equals("RETURN")) {
      /* auth and return both involve creating a new transaction,
	 with all attendant attributes. */

      boolean doingauth = (cmd.equals("AUTH"));
      int res;
      String invoice = invoicefield.getText();

      /* Create it. */
      res = session.cv_new(invoice);

      /* Add the data from the fields. */
      if (res == net.hks.Ccvs.CV_OK) {
	res = session.cv_add(invoice, net.hks.Ccvs.CV_ARG_CARDNUM,
	  cardnumfield.getText());
      }
      if (res == net.hks.Ccvs.CV_OK) {
	res = session.cv_add(invoice, net.hks.Ccvs.CV_ARG_EXPDATE,
	  expdatefield.getText());
      }
      if (res == net.hks.Ccvs.CV_OK) {
	res = session.cv_add(invoice, net.hks.Ccvs.CV_ARG_AMOUNT,
	  amountfield.getText());
      }

      /* Perform the auth or return. */
      if (res == net.hks.Ccvs.CV_OK) {
	if (doingauth)
	  res = session.cv_auth(invoice);
	else
	  res = session.cv_return(invoice);
      }
      /* Display the result. */
      showResult(res, session.errorString(res));

      /* If we did an auth, start a new thread which watches that
	 invoice to see when it gets processed by ccvsd. */
      if (doingauth) {
	if (res == net.hks.Ccvs.CV_OK) {
	  StateAwaiter awaiter = new StateAwaiter(session, invoice, 
	    net.hks.Ccvs.CV_UNAUTH);
	  awaiter.start();
	}
      }
    }

    else if (cmd.equals("SALE")) {
      /* Mark the invoice for settlement. */
      int res;
      String invoice = invoicefield.getText();
      res = session.cv_sale(invoice);
      showResult(res, session.errorString(res));
    }

    else if (cmd.equals("DELETE")) {
      /* Delete the invoice. */
      int res;
      String invoice = invoicefield.getText();
      res = session.cv_delete(invoice);
      showResult(res, session.errorString(res));
    }

  }

  /* Display a numeric code, its string equivalent, and the most
     recent textvalue() result text. */
  private void showResult(int res, String resstr) {
    String str = "Result: " + res + " (" + resstr + ")\n";
    str = str + session.cv_textvalue() + "\n";
    resultarea.setText(str);
  }

  /* This inner class implements a thread which waits for a particular
     invoice to get out of a particular state. */
  class StateAwaiter extends Thread 
    implements ActionListener {

    private Ccvs session = null;
    private String invoice = null; /* What invoice are we waiting for? */
    private int waitstatus; /* What status are we waiting to get out of? */
    private Frame selfframe = null; /* The AWT window we display when we
				       finish waiting. */

    public StateAwaiter(Ccvs session_v, String invoice_v,
      int waitstatus_v) {
      session = session_v;
      invoice = invoice_v;
      waitstatus = waitstatus_v;
    }

    public void run() {
      int res;

      try {

	/* While the invoice is still in the given state, wait more. */
	while ((res = session.cv_status(invoice)) == waitstatus) {
	  try {
	    sleep(1000);
	  } catch (InterruptedException ex) { };
	}

	/* Create a window and store the result text in it. */

	String str = "Result: " + res + " (" 
	  + session.statusString(res) + ")\n";
	str = str + session.cv_textvalue() + "\n";  

	TextArea ta = new TextArea(str);
        Font fieldfont = new Font("Helvetica", Font.PLAIN, 10);
	ta.setFont(fieldfont);
	ta.setEditable(false);

	Frame fram = new Frame("Status: " + invoice);
	fram.add(ta, "Center");

	Button but = new Button("Ok");
	fram.add(but, "South");
	but.setActionCommand("CLOSE");
	but.addActionListener(this);

	fram.pack();
	fram.show();

	selfframe = fram;

	/* The thread exits here. The window will stick around until
	   the "ok" button is pressed. */
      } catch (Exception ex) {
	System.out.println(ex + ex.getMessage());
	ex.printStackTrace(System.out);
      }
    }

    /* This method is called when any button is hit (because we have
       set this object to be an actionListener for the buttons.) */
    public void actionPerformed(ActionEvent ev) {
      if (selfframe != null) {
	/* Close the window. */
	selfframe.dispose();
      }
    }
  }
}


