#!/usr/bin/perl
# cvdemo.pl: Demonstration of the Perl API for CCVS.
# Author: Doug DeJulio, ddj@hks.net, 19970923
# $Id: cvdemo.pl,v 1.8 1999/03/05 15:49:59 ddj Exp $

# First, make sure the CCVS:: functions are available.
use CCVS;

# To use CCVS, first you need to open a session.
# This is how you do that:

$sess = CCVS::init("demo");
if ($sess == CV_SESS_BAD) {	# Couldn't initialize?
    die "couldn't initialize";
}

# Let's say little billy wants to buy a new 10-speed bike for $99.95.
# Here's how you'd do that:

# To do an authorization, you have to first create the transaction.
# Note that all the arguments (including the amount) are strings.

# Each transaction must have a unique invoice.
$invoice = "DEMO0001";

# The standard Perl "do or die" idiom works well here.
CCVS::new($sess, $invoice) == CV_OK
    || die "can't create";
CCVS::add($sess, $invoice, CV_ARG_AMOUNT, "99.95") == CV_OK
    || die "can't add amount.";
CCVS::add($sess, $invoice, CV_ARG_CARDNUM, "0000 0000 00000") == CV_OK
    || die "can't add cardnum";
CCVS::add($sess, $invoice, CV_ARG_EXPDATE, "02/03") == CV_OK
    || die "can't add expdate";

# Nothing has happened yet.  Let's do the authorization.
CCVS::auth($sess, $invoice) == CV_OK
    || die "couldn't issue auth request";

# In theory, the background processor is working on this right now.
# Let's loop until it's done.
do {
    sleep 1;
    $status = CCVS::status($sess, $invoice);
} until ($status == CV_AUTH
	 || $status == CV_DENIED
	 || $status == CV_REVIEW);

# At this point, the status is either "CV_AUTH", "CV_DENIED" or "CV_REVIEW".
# Let's get the full extended status and unpack it into an associative array.
%authresult = split / *{|} */,CCVS::textvalue($sess);
# Discussion: Since the extended status is returned in key/value
# pairs, and the keys are alphanumerics with underlines but no
# whitespace, and the values are enclosed in {} brackets, this isn't
# hard.  The given "split" takes a string of the form "foo {bar baz}
# quux {fnord}" and turn it into a list with elements "foo", "bar
# baz", "quux" and "fnord".  Assigning this to an associative array
# will match up the key/value pairs correctly.

# If everything was succesful, push through the sale.
if ($status == CV_AUTH) {
    (CCVS::sale($sess, $invoice) == CV_OK)
	|| die "couldn't issue sale";
}

# If the auth was denied, report it.
if ($status == CV_DENIED) {
    print "Transaction $invoice was denied; the response was:\n",
    $authresult{'result_text'},
    "\n";
}

# If the auth was in the review state, try a more complex
# decision-making process.
if ($status == CV_REVIEW) {
    # If it's less than $20, let it through.
    if ($authresult{'amount'} < 20) {
	(CCVS::sale($sess, $invoice) == CV_OK)
	    || die "couldn't issue sale";
    } else {
	print "Transaction $invoice is up for review and was for >$20; response was:\n",
	$authresult{'result_text'},
	"\n";
    }
}

# When you're finished, you need to clean up, like this:
CCVS::done($sess);
