%{
/* cv_cli_batch.l: CLI utility program to do batch processing. */
/* Author: Doug DeJulio, ddj@hks.net, 1998/12/03 */

/* Note: This is a "flex" program.  You need to build it with "flex".
   Many versions of "lex" won't do the trick.  The source code for
   "flex" is available from prep.ai.mit.edu in /pub/gnu, among other
   places. */

static const char * const rcsid = "$Id: cv_cli_batch.l,v 1.7 2000/04/19 15:42:08 ddj Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "cv_api.h"

/* I don't normally like global variables, but when using lex it does
   make things much easier. */
int datum = 0;
int di = 0;
/* Right now, we don't permit more than 64 columns or individual
   fields longer than 256 characters.  With the sofware as it
   currently stands, these limits are huge enough to be overkill.
   Still, if you want to change this, here's where to do it. */
enum _commands {
  PREAUTH,
  POSTAUTH,
  BOTH,
  RETURN,
  STATUS,
  UNKNOWN
} command = UNKNOWN;
char data[64][256];
char *infields[64] = { "command", "invoice", "amount", "cardnum", "expdate" };
int numinfields = 5;
char *outfields[64] = { "invoice", "status" };
int numoutfields = 2;
/* We need a queue of invoices to report on when we terminate. */
struct repdata {
  char *invoice[9];		/* Invoices are never >8 characters. */
  struct repdata *next;
};
struct repdata *reports = NULL;
struct repdata *postauths = NULL;
char lastauth[9] = "";		/* Last invoice that's authed. */
void *sess;			/* CCVS session object. */

int invpush(struct repdata **fifo, char *invoice) {
  struct repdata *data;
  if (!*fifo) {
    /* If FIFO is empty, start it. */
    *fifo = calloc(1, sizeof(struct repdata));
    if (!*fifo) {
      /* return 1 on failure to calloc() */
      return 1;
    }
    data = *fifo;
  } else {
    /* If FIFO is not empty, jump to the tail. */
    data = *fifo;
    while (data->next) {
      data = data->next;
    }
    data->next = calloc(1, sizeof(struct repdata));
    if (!data->next) {
      return 1;
    }
    data = data->next;
  }
  
  strncpy((char *)data->invoice, (char *)invoice, 8);

  return 0;
}

char *invpop(struct repdata **fifo) {
  struct repdata *tmp;
  static char invoice[9];

  if (!*fifo) {
    return NULL;
  }

  strcpy((char *)invoice, (char *)(*fifo)->invoice);
  tmp = *fifo;
  *fifo = tmp->next;
  free(tmp);

  return invoice;
}

%}

FS		{WHITESPACE},{WHITESPACE}
RS		\n
WHITESPACE	[ \t]*

%%

^auth/{FS}	{ command = PREAUTH; }
^sale/{FS}	{ command = POSTAUTH; }
^both/{FS}	{ command = BOTH; }
^return/{FS}	{ command = RETURN; }
^status/{FS}	{ command = STATUS; }
{FS}		{ data[datum++][di] = '\0'; di = 0; }
{RS}		{ data[datum++][di] = '\0'; di = 0; return 1; }
\"  		{ /* strip out quotes */ ; }
.		{ data[datum][di++]=*yytext; }

%%

/* user code */

static void doHelp(char *prognam)
{
  printf("usage: $s [options]\n", prognam);
  printf("Options are:\n"
	 " -c name (CCVS configuration name)\n"
	 " -i filename (input file name)\n"
	 " -o filename (output file name)\n"
	 " -I formatstring (input data format)\n"
	 " -O formatstring (output data format)\n"
	 " -h (display this help)\n");
  return;
}

int doCreate(void)
{
  register int i;
  int r;

  r = cv_new(sess, data[1]);
  if (r != CV_OK) {
    return r;
  }
  for (i=2; i<datum; i++) {
    r = cv_add(sess, data[1], cv_str2arg(infields[i]), data[i]);
    if (r != CV_OK) {
      /* It didn't work, nuke it to prevent confusion. */
      cv_delete(sess, data[1]);
      return r;
    }
  }

  return CV_OK;
}

char *getValue(char *string, char *key) {
  static char buf[256];
  int state = 0;		/* 1 means inside {} */
  register int i,j;
  int done = 0;
  int len = strlen(string);

  for (i=0; i<len && !done; i++) {
    switch (state) {
    case 0:
      if (string[i] == '{') {
	state = 1;
	break;
      }
      while (string[i] == ' ') {
	i++;
      }
      if (!strncmp(string+i, key, strlen(key))) {
	i += strlen(key);
	if (string[i] == ' ') {
	  while (string[i] != '{') {
	    i++;
	  }
	  done = 1;
	}
      }
      break;
    case 1:
      if (string[i] == '}') {
	state = 0;
	
      }	
      break;
    }
  }

  if (i == len) {
    return " ";
  }

  for (j=0; string[i] != '}'; i++, j++) {
    buf[j] = string[i];
  }
  buf[j] = '\0';

  return buf;
}

int main(int argc, char *argv[])
{
  extern char *optarg;
  extern int optind, opterr, optopt;
  int opt;
  int line = 0;
  register int i;
  char *infile, *outfile, *infmt, *outfmt;
  FILE *inp;
  char *p;
  char *config;

  int versp = 0;

  /* Initialize file name pointers. */
  infile = outfile = NULL;

  /* Default format strings. */
  infmt = outfmt = NULL;

  /* Read some environment variables. */
  /* Getenv returns either the data, or NULL if the variable isn't set. */
  /* configuration name */
  config = getenv("CCVS_CONFIG_NAME");
  /* format strings */
  p = getenv("CCVS_FORMAT_IN");
  if (p != NULL)
    infmt = p;
  p = getenv("CCVS_FORMAT_OUT");
  if (p != NULL)
    outfmt = p;

  while ((opt = getopt(argc, argv, "i:o:c:I:O:hv")) != EOF) {
    switch (opt) {
    case 'c':			/* config name */
      config = optarg;
      break;
    case 'i':			/* input file name follows */
      infile = optarg;
      break;
    case 'o':			/* output file name follows */
      outfile = optarg;
      break;
    case 'I':			/* input file format specifier */
      infmt = optarg;
      break;
    case 'O':			/* output file format specifier */
      outfmt = optarg;
      break;
    case 'h':
      doHelp(argv[0]);
      return 0;
    case 'v':			/* display version/release info */
      versp = 1;
      break;
    case '?':
      fprintf(stderr, "%s: unknown option '%c', ignoring...\n",
	      argv[0], optopt);
      break;
    default:
      fprintf(stderr, "%s: problem parsing arguments, exiting...\n",
	      argv[0]);
      return 1;
    }
  }

  /* Options have been processed.  We're good to go. */
  /* A null infile means "use stdin".  A null outfile means "use stdout". */
  /* The infmt and outfmt specifiers are set. */
  
  /* Are we using non-default infmt and/or outfmt strings? */
  /* Parse format strings into arrays.  Function strtok() is evil, but
     apropriate here. */
  /* The first two infields are *always* command and invoice. */
  if (infmt) {
    infields[2] = strtok(infmt, ", ");
    numinfields = 3;
    while (infields[numinfields] = strtok(NULL, ", ")) {
      numinfields++;
    }
  }

  /* The first outfield is *always* invoice. */
  if (outfmt) {
    outfields[1] = strtok(outfmt, ", ");
    numoutfields = 2;
    while (outfields[numoutfields] = strtok(NULL, ", ")) {
      numoutfields++;
    }
  }

  /* Let's do the apropriate freopen()s. */
  if (infile) {
    inp = fopen(infile, "r");
    if (!inp) {
      fprintf(stderr, "%s: can't open %s for reading.\n",
	      argv[0], infile);
    }
  } else {
    inp = stdin;
  }

  if (outfile) {
    /* Change the "w" to an "a" to make it append to the outfile
       instead of replacing it. */
    if (!freopen(outfile, "w", stdout)) {
      fprintf(stderr, "%s: can't open %s for writing.\n",
	      argv[0], outfile);
    }
  }

  /* Let's initialize CCVS now. */
  if (!config) {
    fprintf(stderr, "%s: You must specify a configuration.\n",
	    argv[0]);
    return 1;
  }
  sess = cv_init(config);
  if (sess == CV_SESS_BAD) {
    fprintf(stderr, "%s: can't use configuration %s.\n",
	    argv[0], config);
    return 1;
  }

  /* Do we just report CCVS version info and exit now? */
  if (versp) {
    puts(cv_textvalue(sess));	/* Print the version. */
    cv_done(sess);		/* Close the session. */
    return 0;			/* Terminate. */
  }

  /* Now lets parse the input file into the "data" array, over and over. */
  /* Hurrah for {,f}lex! */

  /* The yylex() function returns 0 on EOF, and we've made it return 1
     when we've got the data from one line of the input file.  So,
     when yylex() returns nonzero, process a line of input, and when
     it returns 0, terminate. */

  yyin = inp;

  /* This loop uses the classic only-reasonable-use-for-goto, which is
     to break out of a deeply nested loop.  See K&R p. 66 for a
     description of why *this* use of goto isn't inherently evil. */
  /* Besides, people expect code in a lex ".l" file to be ugly... */
  while (yylex()) {
    line++;
    /* Here's where we do the input-related work. */

    /* If there are fewer than two arguments, that's a problem... */
    if (datum < 2) {
      fprintf(stderr, "%s: line %d doesn't have an invoice!\n",
	      argv[0], line);
    }

    if (invpush(&reports, data[1])) {
      fprintf(stderr,
	      "%s: can't get memory to process transaction %s (line %d), stopping.\n",
	      argv[0], data[1], line);
      break;
    }

    /* What's the command? */
    switch (command) {
      /* If it's a recognized command, data[1] is the invoice,
	 and data[2 to (datum - 1)] are the arguments. */
    case PREAUTH:
      if (doCreate()) {
	fprintf(stderr, "%s: can't create transaction %s (line %d), skipping.\n",
		argv[0], data[1], line);
      } else {
	if (cv_auth(sess, data[1])) {
	  fprintf(stderr, "%s: problem issuing auth on %s (line %d).\n",
		  argv[0], data[1], line);
	} else {
	  strncpy(lastauth, data[1], 8);
	}
      }
      break;
    case POSTAUTH:
      /* The "doCreate()" will probably fail because the transaction will
	 probably already exist, but when the transaction does *not*
	 already exist this is essential. */
      doCreate();
      if (invpush(&postauths, data[1])) {
	fprintf(stderr,
		"%s: can't get memory to process transaction %s, stopping.\n",
		argv[0], data[1]);
	goto postloop;
      }
      break;
    case BOTH:
      if (doCreate()) {
	fprintf(stderr, "%s: can't create transaction %s (line %d), skipping.\n",
		argv[0], data[1], line);
      } else {
	if (cv_auth(sess, data[1])) {
	  fprintf(stderr, "%s: problem issuing auth on %s (line %d).\n",
		  argv[0], data[1], line);
	} else {
	  strncpy(lastauth, data[1], 8);
	}
      }
      if (invpush(&postauths, data[1])) {
	fprintf(stderr,
		"%s: can't get memory to process transaction %s, stopping.\n",
		argv[0], data[1]);
	goto postloop;
      }
      break;
    case RETURN:
      if (doCreate()) {
	fprintf(stderr, "%s: can't create transaction %s (line %d), skipping.\n",
		argv[0], data[1], line);
      } else {
	if (cv_return(sess, data[1])) {
	  fprintf(stderr, "%s, problem issuing return on %s (line %d).\n",
		  argv[0], data[1], line);
	}
      }
      break;
    case STATUS:
      break;
    case UNKNOWN:
      /* If it's not a recognized command, data[0] will have the
	 command text. */
      fprintf(stderr, "%s: unknown command \"%s\" (line %d)\n", argv[0], data[0], line);
      break;
    default:
      /* This should *never* be reached. */
      fprintf(stderr, "%s: bug!\n", argv[0]);
    }

    datum = 0; command = UNKNOWN;
  }

  /* Here's the label we use for the rare non-evil goto. */
postloop:
  
  /* Here's where we do the post-authorization work. */
  /* Loop until the last auth is processed. */
  if (strlen(lastauth)) {
    do {
      sleep(1);
      i = cv_status(sess, lastauth);
    } while (i != CV_AUTH
	     && i != CV_REVIEW
	     && i != CV_DENIED
	     && i != CV_NEW);
  }

  /* Now do all the postauthorizatoins. */
  while (p = invpop(&postauths)) {
    /* Problems here will be discovered during status reporting,
       so just do it. */
    cv_sale(sess, p);
  }

  /* Here's where we do the output-related work. */
  while (p = invpop(&reports)) {
    printf("%s", p);
    cv_status(sess, p);
    for (i=1; i<numoutfields; i++) {
      printf(", %s", getValue(cv_textvalue(sess), outfields[i]));
    }
    printf("\n");
  }

  cv_done(sess);
  return 0;
}
