/* cv_tcl.c: Functions to add the CCVS API to TCL. */
/* Author: Doug DeJulio, ddj@hks.net, 970314 */

static const char rcsid[] = "$Id: cv_tcl.c,v 2.29 2000/03/27 20:41:43 aplotkin Exp $";

#include <stdlib.h>
#include <string.h>
#include <tcl.h>

#include "hks_util.h"

#include "cv_tcl.h"
#include "cv_api.h"
#include "cv_debug.h"
#include "cv_argconv.h"

#define BUFLEN (256)
#define TBUFLEN (256)

#define TCLMODVERS "3.3"

typedef struct _cv_tclData {
  cv_session sess;
} *cv_tclData;

/* Init command. */
int cv_tcl_init(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  extern char *hks_log_ident;
  static char buf[TBUFLEN] = "";
  cv_tclData data;

  data = (cv_tclData) clientData;
  
  if (argc != 2) {
    Tcl_AppendResult(interp,
		     "usage: ",
		     argv[0],
		     " configname",
		     NULL);
    return TCL_ERROR;
  }
  
  data->sess = cv_init(argv[1]);
  if (data->sess == CV_SESS_BAD) {
    Tcl_AppendResult(interp,
		     "Unable to use configuration named '",
		     argv[1],
		     "'.",
		     NULL);
    return TCL_ERROR;
  }
  
  /* Set the logging name for config "foo" to "TCL CCVS(foo)" if possible. */
  /* Note: this is the only non-thread-safe operation, and it only
     affects logging output.  All logging output in a process will be
     marked with the last string this was set to. */
  
  sprintf(buf, "TCL CCVS(%s)", argv[1]);
  hks_log_ident = buf;

  Tcl_AppendResult(interp,
		   cv_textvalue(data->sess),
		   NULL);

  return TCL_OK;
}

/* Close a session. */
int cv_tcl_done(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  cv_tclData data = (cv_tclData) clientData;

  if (argc != 1) {
    Tcl_AppendResult(interp,
		     "usage: ",
		     argv[0],
		     NULL);
    return TCL_ERROR;
  }

  if (data->sess != NULL) {
    cv_done(data->sess);
    data->sess = NULL;
  }

  return TCL_OK;
}

/* After we're done, clean up. */
void cv_tcl_cleanup(ClientData clientData)
{
  cv_tclData data = clientData;
  void *sess = data->sess;

  if (sess != NULL)
    cv_done(sess);

  return;
}

/* New command. */
int cv_tcl_new(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  cv_tclData data;
  cv_session sess;
  int retval;

  data = clientData;
  sess = data->sess;

  if (argc != 2) {
    Tcl_AppendResult(interp,
		     "usage: ",
		     argv[0],
		     " invoice",
		     NULL);
    return TCL_ERROR;
  }

  retval = cv_new(sess, argv[1]);

  switch (retval) {
  case CV_OK:
    Tcl_AppendResult(interp,
		     "status {ok} invoice {",
		     argv[1],
		     "} ",
		     cv_textvalue(sess),
		     NULL);
    return TCL_OK;
  case CV_E_DUPINVOICE:
  case CV_E_BADINVOICE:
    retval = TCL_OK;
    break;
  default:
    retval = TCL_ERROR;
    break;
  }

  Tcl_AppendResult(interp,
		   "status {error} invoice {",
		   argv[1],
		   "} ",
		   cv_textvalue(sess),
		   NULL);

  return retval;
}

/* Add command. */
int cv_tcl_add(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  cv_tclData data;
  cv_session sess;
  char *invoice, *argval;
  int argtype;
  int retval;

  data = clientData;
  sess = data->sess;

  if (argc != 4) {
    Tcl_AppendResult(interp,
		     "usage: ",
		     argv[0],
		     " invoice argtype argval",
		     NULL);
    return TCL_ERROR;
  }

  invoice = argv[1];
  argtype = cv_str2arg(argv[2]);
  argval = argv[3];

  retval = cv_add(sess, invoice, argtype, argval);
  
  switch (retval) {
  case CV_OK:
    Tcl_AppendResult(interp,
		     "status {ok} invoice {",
		     invoice,
		     "} ",
		     cv_textvalue(sess),
		     NULL);
    return TCL_OK;
  case CV_E_DUPINVOICE:
    retval = TCL_OK;
    break ;
  case CV_E_UNKNOWN:
  case CV_E_DATA:
  default:
    retval = TCL_ERROR;
  }
  
  Tcl_AppendResult(interp,
		   "status {error} invoice {",
		   invoice,
		   "} ",
		   cv_textvalue(sess),
		   NULL);
  
  return retval;
}

/* Delete command. */
int cv_tcl_delete(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  cv_tclData data;
  cv_session sess;
  int retval;

  data = clientData;
  sess = data->sess;

  if (argc != 2) {
    Tcl_AppendResult(interp,
		     "usage: ",
		     argv[0],
		     " invoice",
		     NULL);
    return TCL_ERROR;
  }

  retval = cv_delete(sess, argv[1]);

  switch (retval) {
  case CV_OK:
    Tcl_AppendResult(interp,
		     "status {ok} invoice {",
		     argv[1],
		     "} ",
		     cv_textvalue(sess),
		     NULL);
    return TCL_OK;
  case CV_E_BADINVOICE:
    retval = TCL_OK;
    break;
  default:
    retval = TCL_ERROR;
    break;
  }

  Tcl_AppendResult(interp,
		   "status {error} invoice {",
		   argv[1],
		   "} ",
		   cv_textvalue(sess),
		   NULL);

  return retval;
}

/* Deferred auth command. */
int cv_tcl_dauth(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  int retval;
  cv_tclData data;
  cv_session sess;
  char *invoice;

  data = clientData;
  sess = data->sess;

  if (argc != 2) {
    Tcl_AppendResult(interp,
		     "usage: ",
		     argv[0],
		     " invoice",
		     NULL);
    return TCL_ERROR;
  } else {
    invoice = argv[1];
  }
  
  retval = cv_auth(sess, invoice);

  switch (retval) {
  case CV_OK:
    Tcl_AppendResult(interp,
		     "type {auth} status {ok}",
		     NULL);
    return TCL_OK;
  default:
    Tcl_AppendResult(interp,
		     "invoice {", invoice, "} ",
		     "type {auth} status {error} ",
		     cv_textvalue(sess), NULL);
    return TCL_ERROR;
  }
}

/* Reverse command. */
int cv_tcl_reverse(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  int retval;
  char buf[BUFLEN];
  cv_tclData data;
  cv_session sess;
  char *invoice = NULL;

  data = (cv_tclData) clientData;
  sess = data->sess;
  if (sess == CV_SESS_BAD) {
    Tcl_SetResult(interp,
		  "CCVS must be initialized before use.",
		  TCL_STATIC);
    return TCL_ERROR;
  }

  if (argc == 2) {
    invoice = argv[1];
    retval = cv_reverse(sess, invoice);
  } else {
    Tcl_AppendResult(interp,
		     "usage: ",
		     argv[0],
		     " invoice",
		     NULL);
    return TCL_ERROR;
  }
  
  /* Must check return value. */
  switch (retval) {
  case CV_OK:
    sprintf(buf, "type {reverse} status {ok} invoice {%s}", invoice);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
  case CV_E_SYNTAXERROR:
  case CV_E_DUPINVOICE:
  case CV_E_BADINVOICE:
    Tcl_AppendResult(interp,
		     "invoice {",
		     invoice,
		     "} type {reverse} status {error} ",
		     cv_textvalue(sess),
		     NULL);
    return TCL_OK;
  default:
    sprintf(buf, "text {unknown error %d}", retval);
    Tcl_AppendResult(interp,
		     "invoice {",
		     invoice,
		     "} type {reverse} status {error} ",
		     buf,
		     NULL);
    return TCL_ERROR;
  }
}

/* Sale command. */
int cv_tcl_sale(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  int retval;
  char buf[BUFLEN];
  cv_tclData data;
  cv_session sess;
  char *invoice = NULL;

  data = (cv_tclData) clientData;
  sess = data->sess;
  if (sess == CV_SESS_BAD) {
    Tcl_SetResult(interp,
		  "CCVS must be initialized before use.",
		  TCL_STATIC);
    return TCL_ERROR;
  }


  if (argc == 2) {
    invoice = argv[1];
    retval = cv_sale(sess, invoice);
  } else {
    Tcl_AppendResult(interp,
		     "usage: ",
		     argv[0],
		     " invoice",
		     NULL);
    return TCL_ERROR;
  }

  /* Must check return value. */
  switch (retval) {
  case CV_OK:
    sprintf(buf, "type {sale} status {ok} invoice {%s}", invoice);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
  case CV_E_SYNTAXERROR:
  case CV_E_DUPINVOICE:
  case CV_E_BADINVOICE:
    Tcl_AppendResult(interp,
		     "invoice {",
		     invoice,
		     "} type {sale} status {error} ",
		     cv_textvalue(sess),
		     NULL);
    return TCL_OK;
  default:
    sprintf(buf, "text {unknown error %d}", retval);
    Tcl_AppendResult(interp,
		     "invoice {",
		     invoice,
		     "} type {sale} status {error} ",
		     buf,
		     NULL);
    return TCL_ERROR;
  }
}

/* Return command. */
int cv_tcl_return(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  int retval;
  char buf[BUFLEN];
  cv_tclData data;
  cv_session sess;
  char *invoice = NULL;

  data = (cv_tclData) clientData;
  sess = data->sess;
  if (sess == CV_SESS_BAD) {
    Tcl_SetResult(interp,
		  "CCVS must be initialized before use.",
		  TCL_STATIC);
    return TCL_ERROR;
  }

  if (argc != 2) {
    Tcl_AppendResult(interp,
		     "usage: ",
		     argv[0],
		     " invoice",
		     NULL);
    return TCL_ERROR;
  } else {
    invoice = argv[1];
  }

  retval = cv_return(sess, invoice);

  switch (retval) {
  case CV_OK:
    Tcl_AppendResult(interp,
		     "type {return} status {ok} invoice {",
		     invoice, "}",
		     NULL);
    return TCL_OK;
  case CV_E_SYNTAXERROR:
  case CV_E_DUPINVOICE:
  case CV_E_BADINVOICE:
    Tcl_AppendResult(interp,
		     "invoice {", invoice, "} ",
		     "type {return} status {error} ",
		     cv_textvalue(sess),
		     NULL);
    return TCL_OK;
  default:
    sprintf(buf, "text {unknown error %d}", retval);
    Tcl_AppendResult(interp,
		     "invoice {", invoice, "} ",
		     "type {return} status {error} ",
		     buf,
		     NULL);
    return TCL_ERROR;
  }
}

/* Void command. */
int cv_tcl_void(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  int retval;
  char buf[BUFLEN];
  cv_tclData data;
  cv_session sess;
  char *invoice = NULL;

  data = (cv_tclData) clientData;
  sess = data->sess;
  if (sess == CV_SESS_BAD) {
    Tcl_SetResult(interp,
		  "CCVS must be initialized before use.",
		  TCL_STATIC);
    return TCL_ERROR;
  }


  if (argc == 2) {
    invoice = argv[1];
    retval = cv_void(sess, invoice);
  } else {
    Tcl_AppendResult(interp,
		     "usage: ",
		     argv[0],
		     " invoice",
		     NULL);
    return TCL_ERROR;
  }

  /* Must check return value. */
  switch (retval) {
  case CV_OK:
    sprintf(buf, "type {void} status {ok} invoice {%s}", invoice);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
  case CV_E_SYNTAXERROR:
  case CV_E_DUPINVOICE:
  case CV_E_BADINVOICE:
    Tcl_AppendResult(interp,
		     "invoice {",
		     invoice,
		     "} type {void} status {error} ",
		     cv_textvalue(sess),
		     NULL);
    return TCL_OK;
  default:
    sprintf(buf, "text {unknown error %d}", retval);
    Tcl_AppendResult(interp,
		     "invoice {",
		     invoice,
		     "} type {void} status {error} ",
		     buf,
		     NULL);
    return TCL_ERROR;
  }
}

/* Get status of an invoice */
int cv_tcl_status(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  cv_tclData data;
  cv_session sess;
  int retval;

  if (argc != 2) {
    Tcl_SetResult(interp,
		  "usage: cv_status invoice",
		  TCL_STATIC);
    return TCL_ERROR;
  }

  data = (cv_tclData) clientData;
  sess = data->sess;
  if (sess == CV_SESS_BAD) {
    Tcl_SetResult(interp,
		  "CCVS must be initialized before use.",
		  TCL_STATIC);
    return TCL_ERROR;
  }
  retval = cv_status(sess, argv[1]);

  Tcl_AppendResult(interp,
		   "type {status} ",
		   cv_textvalue(sess),
		   NULL);

  return TCL_OK;
}

int cv_tcl_count(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  int r = 0;
  register int i;
  cv_tclData data;
  int cv_count_type = CV_NONE;
  cv_session sess = NULL;
  char buf[BUFLEN];

  data = (cv_tclData) clientData;
  sess = data->sess;
  if (sess == CV_SESS_BAD) {
    Tcl_SetResult(interp,
		  "CCVS must be initialized before use.",
		  TCL_STATIC);
    return TCL_ERROR;
  }

  /* If you count everything with no args,
     you get back an associative array. */
  if (argc == 1) {		/* Default to counting everything. */
    r = cv_count(sess, CV_ALL);
    sprintf(buf, "total {%d} ", r);
    Tcl_AppendResult(interp,
		     "type {count} ",
		     buf,
		     cv_textvalue(sess),
		     NULL);
    return TCL_OK;
  }

  /* If you count a subset, return an integer.  More useful in some ways. */

  for (i = 1; i < argc; i++) {
    if (hks_util_strcasecmp("new", argv[i]) == 0) {
      cv_count_type |= CV_NEW;
    } else if (hks_util_strcasecmp("auth", argv[i]) == 0) {
      cv_count_type |= CV_AUTH;
    } else if (hks_util_strcasecmp("ready", argv[i]) == 0) {
      cv_count_type |= CV_READY;
    } else if (hks_util_strcasecmp("done", argv[i]) == 0) {
      cv_count_type |= CV_DONE;
    } else if (hks_util_strcasecmp("unauth", argv[i]) == 0) {
      cv_count_type |= CV_UNAUTH;
    } else if (hks_util_strcasecmp("denied", argv[i]) == 0) {
      cv_count_type |= CV_DENIED;
    } else if (hks_util_strcasecmp("review", argv[i]) == 0) {
      cv_count_type |= CV_REVIEW;
    } else if (hks_util_strcasecmp("bad", argv[i]) == 0) {
      cv_count_type |= CV_BAD;
    } else if (hks_util_strcasecmp("all", argv[i]) == 0) {
      cv_count_type |= CV_ALL;
    } else {
      Tcl_AppendResult(interp,
		       "usage: ",
		       argv[0],
		       " [["
		       "new|auth|ready|done|unauth|denied|review|bad|all"
		       "]...]",
		       NULL);
      return TCL_ERROR;
    }
  }

  r = cv_count(sess, cv_count_type);

  sprintf(buf, "%d", r);

  Tcl_AppendResult(interp,
		   buf,
		   NULL);

  return TCL_OK;
}

int cv_tcl_lookup(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  int r;
  cv_tclData data;
  cv_session sess = NULL;
  int type = CV_NONE;
  int num = 0;

  data = (cv_tclData) clientData;
  sess = data->sess;
  if (sess == CV_SESS_BAD) {
    Tcl_SetResult(interp,
		  "CCVS must be initialized before use.",
		  TCL_STATIC);
    return TCL_ERROR;
  }

  if (argc != 3) {
    Tcl_SetResult(interp,
		  "usage: cv_lookup type number",
		  TCL_STATIC);
    return TCL_ERROR;
  }

  if (hks_util_strcasecmp("new", argv[1]) == 0) {
    type = CV_NEW;
  } else   if (hks_util_strcasecmp("auth", argv[1]) == 0) {
    type = CV_AUTH;
  } else   if (hks_util_strcasecmp("ready", argv[1]) == 0) {
    type = CV_READY;
  } else   if (hks_util_strcasecmp("done", argv[1]) == 0) {
    type = CV_DONE;
  } else   if (hks_util_strcasecmp("unauth", argv[1]) == 0) {
    type = CV_UNAUTH;
  } else   if (hks_util_strcasecmp("denied", argv[1]) == 0) {
    type = CV_DENIED;
  } else   if (hks_util_strcasecmp("review", argv[1]) == 0) {
    type = CV_REVIEW;
  } else   if (hks_util_strcasecmp("bad", argv[1]) == 0) {
    type = CV_BAD;
  } else {
    Tcl_AppendResult(interp,
		     "usage: ",
		     argv[0],
		     " auth|ready|done|unauth|denied|review|bad n",
		     NULL);
    return TCL_ERROR;
  }
  
  num = atoi(argv[2]);

  r = cv_lookup(sess, type, num);

  Tcl_AppendResult(interp,
		   cv_textvalue(sess),
		   NULL);

  if (r == CV_OK)
    return TCL_OK;
  else
    return TCL_ERROR;
}

int cv_tcl_report(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  cv_session sess;
  int retval;

  sess = ((cv_tclData)clientData)->sess;

  if (argc != 2) {
    Tcl_AppendResult(interp,
		     "usage: ",
		     argv[0],
		     "reptype",
		     NULL);
    return TCL_ERROR;
  }

  retval = cv_report(sess, cv_str2rep(argv[1]));

  switch (retval) {
  case CV_OK:
    Tcl_AppendResult(interp,
		     "status {ok} ",
		     cv_textvalue(sess),
		     NULL);
    return TCL_OK;
  case CV_E_SYNTAXERROR:
    Tcl_AppendResult(interp,
		     "status {error} text {invalid report type}",
		     NULL);
    return TCL_OK;
  case CV_E_UNKNOWN:
    /* When cv_report returns CV_E_UNKNOWN, use cv_debug_error() to
       get more info. */
    Tcl_AppendResult(interp,
		     "status {error} text {",
		     cv_debug_strerror(cv_debug_error()),
		     "}",
		     NULL);
    return TCL_OK;
  default:
    retval = TCL_ERROR;
  }

  Tcl_AppendResult(interp,
		   "status {error} ",
		   cv_textvalue(sess),
		   NULL);

  return retval;
}

int cv_tcl_command(void *clientData, Tcl_Interp *interp, int argc, char **argv)
{
  cv_session sess;
  int retval;

  sess = ((cv_tclData)clientData)->sess;

  if (argc != 3) {
    Tcl_AppendResult(interp,
		     "usage: ",
		     argv[0],
		     "cmd argval",
		     NULL);
    return TCL_ERROR;
  }

  retval = cv_command(sess, cv_str2cmd(argv[1]), argv[2]);

  switch (retval) {
  case CV_OK:
    Tcl_AppendResult(interp,
		     cv_textvalue(sess),
		     NULL);
    return TCL_OK;
  case CV_E_UNKNOWN:
    /* When cv_command returns CV_E_UNKNOWN, use cv_debug_error() to
       get more info. */
    Tcl_AppendResult(interp,
		     "status {error} text {",
		     cv_debug_strerror(cv_debug_error()),
		     "}",
		     NULL);
    return TCL_OK;
  default:
    Tcl_AppendResult(interp,
		   "status {error} ",
		   cv_textvalue(sess),
		   NULL);
    return TCL_ERROR;
  }
}

/* Initialize and register everything with TCL. */
int Ccvs_Init(Tcl_Interp *interp)
{
  cv_tclData data;
  ClientData clientData;

  /*  data = &dataBuf; */
  data = calloc(1, sizeof(struct _cv_tclData));
  if (data == NULL) {
    interp->result = "CCVS " TCLMODVERS " failed to initialize.";
    return TCL_ERROR;
  }
  data->sess = CV_SESS_BAD;
  clientData = (ClientData) data;

  /* Add the commands. */
  Tcl_CreateCommand(interp,     /* Interpreter. */
                    "cv_init",	/* Command name. */
                    cv_tcl_init, /* Implementation. */
                    clientData, /* Data to pass in. */
                    cv_tcl_cleanup); /* Procedure to clean up. */
  Tcl_CreateCommand(interp,	/* Interpreter. */
		    "cv_done",	/* Command name. */
		    cv_tcl_done, /* Implementation. */
		    clientData,	/* Data to pass in. */
		    (void (*)()) NULL);
#if 0
  Tcl_CreateCommand(interp,     /* Interpreter. */
                    "cv_create", /* Command name. */
                    cv_tcl_create, /* Implementation. */
                    clientData, /* Data to pass in. */
                    (void (*)()) NULL); /* Procedure to clean up. */
#endif /* 0 */
  Tcl_CreateCommand(interp,     /* Interpreter. */
                    "cv_new", /* Command name. */
                    cv_tcl_new, /* Implementation. */
                    clientData, /* Data to pass in. */
                    (void (*)()) NULL); /* Procedure to clean up. */
  Tcl_CreateCommand(interp,     /* Interpreter. */
                    "cv_add", /* Command name. */
                    cv_tcl_add, /* Implementation. */
                    clientData, /* Data to pass in. */
                    (void (*)()) NULL); /* Procedure to clean up. */
  Tcl_CreateCommand(interp,     /* Interpreter. */
                    "cv_delete", /* Command name. */
                    cv_tcl_delete, /* Implementation. */
                    clientData, /* Data to pass in. */
                    (void (*)()) NULL); /* Procedure to clean up. */
  Tcl_CreateCommand(interp,     /* Interpreter. */
                    "cv_auth",	/* Command name. */
                    cv_tcl_dauth, /* Implementation. */
                    clientData, /* Data to pass in. */
                    (void (*)()) NULL); /* Procedure to clean up. */
  Tcl_CreateCommand(interp,     /* Interpreter. */
                    "cv_reverse", /* Command name. */
                    cv_tcl_reverse, /* Implementation. */
                    clientData, /* Data to pass in. */
                    (void (*)()) NULL); /* Procedure to clean up. */
  Tcl_CreateCommand(interp,     /* Interpreter. */
                    "cv_sale",	/* Command name. */
                    cv_tcl_sale, /* Implementation. */
                    clientData, /* Data to pass in. */
                    (void (*)()) NULL); /* Procedure to clean up. */
  Tcl_CreateCommand(interp,     /* Interpreter. */
                    "cv_return", /* Command name. */
                    cv_tcl_return, /* Implementation. */
                    clientData, /* Data to pass in. */
                    (void (*)()) NULL); /* Procedure to clean up. */
  Tcl_CreateCommand(interp,     /* Interpreter. */
                    "cv_void",	/* Command name. */
                    cv_tcl_void, /* Implementation. */
                    clientData, /* Data to pass in. */
                    (void (*)()) NULL); /* Procedure to clean up. */
  Tcl_CreateCommand(interp,	/* Interpreter. */
		    "cv_status", /* Command name. */
		    cv_tcl_status, /* Implementation. */
		    clientData,	/* Data to pass in. */
		    (void (*)()) NULL);	/* Procedure to clean up. */
  Tcl_CreateCommand(interp,	/* Interpreter. */
		    "cv_count", /* Command name. */
		    cv_tcl_count, /* Implementation. */
		    clientData,	/* Data to pass in. */
		    (void (*)()) NULL);	/* Procedure to clean up. */
  Tcl_CreateCommand(interp,	/* Interpreter. */
		    "cv_lookup", /* Command name. */
		    cv_tcl_lookup, /* Implementation. */
		    clientData,	/* Data to pass in. */
		    (void (*)()) NULL);	/* Procedure to clean up. */
  Tcl_CreateCommand(interp,	/* Interpreter. */
		    "cv_report", /* Command name. */
		    cv_tcl_report, /* Implementation. */
		    clientData,	/* Data to pass in. */
		    (void (*)()) NULL);	/* Procedure to clean up. */
  Tcl_CreateCommand(interp,	/* Interpreter. */
		    "cv_command", /* Command name. */
		    cv_tcl_command, /* Implementation. */
		    clientData,	/* Data to pass in. */
		    (void (*)()) NULL);	/* Procedure to clean up. */


  /* Set the result. */
  interp->result = "CCVS " TCLMODVERS ", Copyright 2000 Red Hat, Inc.  All rights reserved.";

  return TCL_OK;
}
