/* cv_cli_return.c: CLI utility program to perform returns. */
/* Author: Doug DeJulio, ddj@hks.net, 19970917 */

static const char rcsid[] = "$Id: cv_cli_return.c,v 1.4 1999/11/23 20:07:07 cactus Exp $";

#include <stdio.h>		/* printf() */
#include <stdlib.h>		/* getenv() */
#include <unistd.h>		/* getopt() */

#include "cv_api.h"		/* CCVS C API. */

int main(int argc, char *argv[])
{
  extern char *optarg;
  extern int optind, opterr, optopt;
  int opt;
  void *session;
  int r;

  char *config = NULL;
  char *invoice = NULL;
  char *cardnum = NULL;
  char *edate = NULL;
  char *amount = NULL;

  /* Process arguments. */
  while ((opt = getopt(argc, argv, "C:p:n:d:")) != EOF) {
    switch (opt) {
    case 'C':			/* Configuration. */
      config = optarg;
      break;
    case 'p':			/* Price. */
      amount = optarg;
      break;
    case 'n':			/* Card number. */
      cardnum = optarg;
      break;
    case 'd':
      edate = optarg;		/* Expiration date. */
      break;
    case '?':
      fprintf(stderr, "%s: unknown option '%c', ignoring...\n",
	      argv[0], optopt);
    }
  }

  if ((argc - optind) != 1
      || amount == NULL
      || cardnum == NULL
      || edate == NULL) {
    fprintf(stderr,
	    "usage: %s [-C configname] -p amount -n cardnum -d expdate"
	    " invoice\n",
	    argv[0]);
    return 1;
  }

  /* First non-switch argument is the invoice to use. */
  invoice = argv[optind];
  /* All arguments are handled. */

  /* Initialize with configuration name. */
  if (config == NULL) {		/* No config argument? */
    config = getenv("CCVS_CONFIG_NAME");
    if (config == NULL) {	/* No environment variable either? */
      fprintf(stderr, "%s: You must specify a configuration.\n", argv[0]);
      return 1;
    }
  }

  /* Config name, invoice, price, cardnum and edate are all set. */
  session = cv_init(config);
  if (session == CV_SESS_BAD) {
    fprintf(stderr,
	    "%s: Unable to initialize with configuration '%s'.\n",
	    argv[0], config);
    return 1;
  }

  /* Call cv_new() to create a new invoice. */

  r = cv_new(session, invoice);

  if (r != CV_OK) {
    fprintf(stderr, "%s: Unable to create invoice '%s'.\n",
	    argv[0], invoice);
  }

  /* Call cv_create() with required arguments, then optional arguments
     in argtype/argval pairs, then a special arg to end the arg list. */

  /* Now build the transaction with successive calls to cv_add() */

  /* Amount of transaction. */

  r = cv_add(session, invoice, CV_ARG_AMOUNT, amount);

  if (r == CV_E_DUPINVOICE) {	/* Duplicate invoice? */
    fprintf(stderr, "%s: Transaction '%s' already exists.\n",
	    argv[0], invoice);
    fprintf(stderr,"%s\n",cv_textvalue(session));
    return 1;
  }

  if (r != CV_OK) {
    fprintf(stderr, "%s: Some problem with transaction '%s'.\n",
	    argv[0], invoice);
    fprintf(stderr,"%s\n",cv_textvalue(session));
    return 1;
  }

  /* Credit card number */  

  r = cv_add(session, invoice, CV_ARG_CARDNUM, cardnum);

  if (r != CV_OK) {
    fprintf(stderr, "%s: Some problem with transaction '%s'.\n",
	    argv[0], invoice);
    fprintf(stderr,"%s\n",cv_textvalue(session));
    return 1;
  }

  /* Credit card Expiration date */  

  r = cv_add(session, invoice, CV_ARG_EXPDATE, edate);

  if (r != CV_OK) {
    fprintf(stderr, "%s: Some problem with transaction '%s'.\n",
	    argv[0], invoice);
    fprintf(stderr,"%s\n",cv_textvalue(session));
    return 1;
  }
  
  /* The transaction has been built.  Now do the return. */

  r = cv_return(session, invoice);

  switch (r) {
  case CV_OK:
    printf("%s: Transaction '%s' processed.\n",
	   argv[0], invoice);
    break;
  case CV_E_DUPINVOICE:
  case CV_E_BADINVOICE:
    printf("%s: Invalid operation on transaction '%s'.\n",
	   argv[0], invoice);
    break;
  default:
    printf("%s: Unknown problem processing transaction '%s'.\n",
	   argv[0], invoice);
  }

  /* Clean up. */
  cv_done(session);

  return r;
}
  
	    
