# ACCESS.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

package ACCESS;

use strict;

require DBLIB;
require SEC;

# 
# AVAILABLE FUNCTIONS AND VARIABLES DEFINED IN THIS PACKAGE:
#
#
#
#  &check_auth
#  &get_username
#
#
#  &access_user_list_all
#  &access_user_create
#  &access_user_delete
#  &acesss_user_get_attr
#  &access_user_set_attr
#  &access_user_verify_id
#  &access_user_find_by_name
#
#  &access_user_list_all_groups
#  &access_user_add_to_group
#  &access_user_remove_from_group
#  &access_user_remove_from_all_groups
#
#  &access_user_list_all_perms
#  &access_user_grant_perm
#  &access_user_revoke_perm
#  &access_user_clear_perms
#
#
#
#  &access_group_list_all
#  &access_group_create
#  &access_group_delete
#  &acesss_group_get_attr
#  &access_group_set_attr
#  &access_group_verify_id
#  &access_group_find_by_name
#
#  &access_group_list_all_perms
#  &access_group_grant_perm
#  &access_group_revoke_perm
#  &access_group_clear_perms
#
#
#
#  &access_perm_list_all
#  &access_perm_create
#  &access_perm_delete
#  &acesss_perm_get_attr
#  &access_perm_set_attr
#  &access_perm_find_by_name
#  &access_perm_verify_id
#
# =============================================================================
#

sub access_user_list_all {
  return generic_list_all('user');
}

sub access_group_list_all {
  return generic_list_all('group');
}

sub access_perm_list_all {
  return generic_list_all('perm');
}

sub generic_list_all {
  my($type) = shift;

  my($str) = "select access_${type}_id, name ";

  if ($type eq 'user') {
    $str .= ", username ";
  }

  $str .=    "from access_${type}
              where access_${type}_id <> 0
              order by name";

  return DBLIB::db_fetchall_arrayref($str);
}

#
# -----------------------------------------------------------------------------
#

sub access_user_create {
  return generic_create('user', @_);
}

sub access_group_create {
  return generic_create('group', @_);
}

sub access_perm_create {
  return generic_create('perm', @_);
}

sub generic_create {
  my($type, %vals) = @_;
  my(@vkeys, $str, $key, $val);  

  @vkeys = keys(%vals);
  if ($#vkeys < 0) {
    return;
  }

  #
  # the first user created is always a super-user
  #

  if ($type eq 'user') {
    $str = access_user_list_all();
    if ($#{$str} < 0) {
      $vals{'superuser'} = 'Y';
    }
  }

  # get the next type id
  
  my($nextid) = DBLIB::seq_next_get("access_${type}_sq");

  # create the object

  $str = "insert into access_${type} (access_${type}_id, ";

  foreach $key (@vkeys) {
    $str .= "${key}, ";
  }
  chop $str;
  chop $str;

  $str .= ") values ('$nextid', ";

  foreach $key (@vkeys) {
    $val = $vals{$key};
    $str .= "'$val', ";
  }
  chop $str;
  chop $str;

  $str .= ")";

  DBLIB::db_do($str);

  return($nextid);  
}

#
# -----------------------------------------------------------------------------
#

sub access_user_clear_perms {
  my($str);
  my($id) = shift;

  $str = "delete from access_user_perm
          where access_user_id = '$id'";

  DBLIB::db_do($str);
}


sub access_group_clear_perms {
  my($str);
  my($id) = shift;

  $str = "delete from access_group_perm
          where access_group_id = '$id'";

  DBLIB::db_do($str);
}


#
# -----------------------------------------------------------------------------
#

sub access_user_delete {
  my($str);
  my($id) = $_[0];

  $str = "delete from access_user_perm
          where access_user_id = '$id'";

  DBLIB::db_do($str);

  $str = "delete from access_user_group
          where access_user_id = '$id'";

  DBLIB::db_do($str);

  return generic_delete('user', @_);
}

sub access_group_delete {
  my($str);
  my($id) = $_[0];

  $str = "delete from access_group_perm
          where access_group_id = '$id'";

  DBLIB::db_do($str);

  return generic_delete('group', @_);
}

sub access_perm_delete {
  my($str);
  my($id) = $_[0];

  $str = "delete from access_user_perm
          where access_perm_id = '$id'";

  DBLIB::db_do($str);

  $str = "delete from access_group_perm
          where access_perm_id = '$id'";

  DBLIB::db_do($str);

  return generic_delete('perm', @_);
}

sub generic_delete {
  my($type, $id) = @_;
  DBLIB::db_do("delete from access_${type} where access_${type}_id = '$id'");
}


#
# -----------------------------------------------------------------------------
#

sub access_user_get_attr {
  return generic_get_attr('user', @_);
}

sub access_group_get_attr {
  return generic_get_attr('group', @_);
}

sub access_perm_get_attr {
  return generic_get_attr('perm', @_);
}

sub generic_get_attr {
  my($type, $id, @vals) = @_;
  my($val, $str);

  if ($#vals < 0) {
    return ();
  }

  $str = "select ";
  foreach $val (@vals) {
    $str .= "$val, ";
  }
  chop $str;
  chop $str;

  $str .= " from access_${type} where access_${type}_id = '$id'";

  return DBLIB::db_fetchrow_array($str);
}

#
# -----------------------------------------------------------------------------
#

sub access_user_set_attr {
  generic_set_attr('user', @_);
}

sub access_group_set_attr {
  generic_set_attr('group', @_);
}

sub access_perm_set_attr {
  generic_set_attr('perm', @_);
}

sub generic_set_attr {
  my($type, $id, %vals) = @_;
  my($str, $key, @vkeys, $val);

  @vkeys = keys(%vals);
  if ($#vkeys < 0) {
    return;
  }

  $str = "update access_${type} set ";
  foreach $key (@vkeys) {
    $val = $vals{$key};
    $str .= "$key='$val', ";
  }
  chop $str;
  chop $str;

  $str .= " where access_${type}_id = '$id'";

  DBLIB::db_do($str);
}

#
# -----------------------------------------------------------------------------
#

sub access_user_verify_id {
  return DBLIB::verify_id('access_user_id', 'access_user', @_);
}

sub access_group_verify_id {
  return DBLIB::verify_id('access_group_id', 'access_group', @_);
}

sub access_perm_verify_id {
  return DBLIB::verify_id('access_perm_id', 'access_perm', @_);
}

#
# -----------------------------------------------------------------------------
#

sub access_user_grant_perm {
  my($id1, $id2) = @_;
  my($oldid2) = $id2;

  $id2 = access_perm_find_by_name($id2);
  if (!defined($id2) || $id2 eq "") {
    MILTON::fatal_error("No such permission: $oldid2");
  }

  generic_connect('user', 'perm', $id1, $id2);
}

sub access_user_add_to_group {
  generic_connect('user', 'group', @_);
}

sub access_group_grant_perm {
  my($id1, $id2) = @_;
  my($oldid2) = $id2;

  $id2 = access_perm_find_by_name($id2);
  if (!defined($id2) || $id2 eq "") {
    MILTON::fatal_error("No such permission: $oldid2");
  }

  generic_connect('group', 'perm', $id1, $id2);
}

sub generic_connect {
  my($t1, $t2, $id1, $id2) = @_;
  my($str);

  $str = "insert into access_${t1}_${t2} (access_${t1}_id, access_${t2}_id)
          values ('$id1', '$id2')";

  DBLIB::db_do($str);
}

#
# -----------------------------------------------------------------------------
#

sub access_user_revoke_perm {
  generic_disconnect('user', 'perm', @_);
}

sub access_user_remove_from_group {
  generic_disconnect('user', 'group', @_);
}

sub access_group_revoke_perm {
  generic_grant_perm('group', 'perm', @_);
}

sub generic_disconnect {
  my($t1, $t2, $id1, $id2) = @_;
  my($str);

  if ($t2 eq 'perm') {
    $id2 = access_perm_find_by_name($id2);
  }

  $str = "delete from access_${t1}_${t2}
          where access_${t1}_id = '$id1'
          and access_${t2}_id = '$id2'";

  DBLIB::db_do($str);
}

#
# -----------------------------------------------------------------------------
#

sub access_user_find_by_name {
  return generic_find_by_name('user', 'username', @_);
}

sub access_group_find_by_name {
  return generic_find_by_name('group', 'name', @_);
}

sub access_perm_find_by_name {
  return generic_find_by_name('perm', 'name', @_);
}

sub generic_find_by_name {
  my($type, $field, $name) = @_;
  my($str);

  $str = "select access_${type}_id from access_${type} where ${field}='$name'";

  my($id) = DBLIB::db_fetchrow_array($str);
  return $id;
}

#
# -----------------------------------------------------------------------------
#

sub access_user_remove_from_all_groups {
  my($id) = shift;
  DBLIB::db_do("delete from access_user_group where access_user_id = '$id'");
}

#
# -----------------------------------------------------------------------------
#

sub access_user_list_all_perms {
  return generic_list_all_x('user','perm', @_);
}

sub access_user_list_all_groups {
  return generic_list_all_x('user','group', @_);
}

sub access_group_list_all_perms {
  return generic_list_all_x('group','perm', @_);
}

sub generic_list_all_x {
  my($t1, $t2, $id) = @_;

  my($str) = "select access_${t2}.access_${t2}_id, access_${t2}.name
              from access_${t1}_${t2}, access_${t2}
              where access_${t1}_${t2}.access_${t1}_id = '$id'
              and access_${t1}_${t2}.access_${t2}_id = 
                  access_${t2}.access_${t2}_id";

  return DBLIB::db_fetchall_arrayref($str);
}

#
# -----------------------------------------------------------------------------
#
# There are four ways a user can be validated to perform a function:
#   1: if there are no users
#   2: if he is a super-user
#   3: if he, as a user, has permission to perform the function
#   4: if any of the groups to which he belongs can perform the function
#

sub check_auth {
  my($perm) = shift;
  my($str, $uid, $pid);

  my($uname) = get_username();

  if (!defined($uname) || ($uname eq "")) {
    MILTON::auth_error("No username found!");
  }

  $uname = DBLIB::db_string_clean($uname);

  #
  # are there no users?
  #

  $str = access_user_list_all();
  if ($#{$str} < 0) {
    return;
  }

  #
  # look up user id and permission id
  #

  $uid = access_user_find_by_name($uname);

  if (!defined($uid) || ($uid == 0)) {
    MILTON::auth_error("No such user ($uname)!");
  }

  $pid = access_perm_find_by_name($perm);

  if (!defined($pid) || ($pid eq "")) {
    MILTON::auth_error("No such permission!");
  }

  #
  # is he a super-user?
  #

  ($str) = access_user_get_attr($uid, 'superuser');
  if ($str eq 'Y') {
    return;
  }

  #
  # check user-level permissions
  #

  $str = "select access_perm_id from access_user_perm
          where access_user_id = '$uid'
          and access_perm_id = '$pid'";

  ($str) = DBLIB::db_fetchrow_array($str);
  if ($str eq $pid) {
    return;
  }

  #
  # now check group-level permissions
  #

  my(@col, $row, $gids);

  $gids = access_user_list_all_groups($uid);

  foreach $row (@$gids) {
    push @col, $row->[0];
  }

  if ($#col > -1) {
    $str = "select access_perm_id from access_group_perm
            where access_perm_id = '$pid'
            and access_group_id in (" . join(",", @col) . ")";

    ($str) = DBLIB::db_fetchrow_array($str);
    if ($str eq $pid) {
      return;
    }
  }

  MILTON::auth_error($perm);
}

#
# -----------------------------------------------------------------------------
#

sub get_username {
  return $ENV{'REMOTE_USER'};
}

#
# -----------------------------------------------------------------------------
#


1;
