# BASKET.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

#  Some conventions:
#    In order to easily find idea placeholders, I've commented them
#    with specific strings.  The following exist:
#   
#    XXX -- potential (or actual) problems; non-optimal solutions.
#    DDD -- discounting engine hooks
#    SSS -- inventory allocation hooks
#
# The new basket philosophy: %udat should only occur in the load_basket
# and store_basket routines.  Everything else that the basket needs to know
# should be stored in global variables (global??).
#
# That way, should a new implementation be needed for the state engine,
# it will be as localized as possible.
#
# Of course, on the flip side of that, it means that we need to have 
# double copies of all of the information: one copy in udat, and one copy
# in the global variable.  Hmmmmm.
#
# In accordance with the new philosophy, I've localized all references to
# the HTML::Embperl::fdat hash, setting it equal to a local hash.  That
# way, if we ever have to rip it out, it shouldn't be too hard.
#
# -----------------------------------------------------------------------------
#
# AVAILABLE FUNCTIONS AND VARIABLES DEFINED IN THIS PACKAGE:
#
#
#  ===== Public =====
#
#  &basket_init
#  &neatly_dispose_of_basket
#
#  &line
#  &grand_total
#  &sub_total
#  &tax_total
#  &ship_total
#
#  &payment_type_str
#  &payment_type_num
#  &payment_type_widgettype
#  &ship_type_str
#  &ship_type_num
#  &tax_type_str
#  &tax_type_num
#  &order_num
#  &order_date
#  &gencomments
#  &exists_discount
#
#  &show_basket_every_time_button
#  &show_shipping_address_button
#  &show_use_security_button
#  &need_billing_address
#  &need_widget
#  &end_line
#
#  &total_items
#  &total_weight
#  &total_volume
#  &total_quantity
#
#  &shippingtype_list_all_calc
#  &paymenttype_list_all
#
#
#  ===== Private =====
#
#  &set_up_form_data
#
#  &process_add
#  &process_view
#
#  &process_clear2
#  &process_recalc2
#  &process_continue2
#  &process_next2
#
#  &process_back3
#  &process_continue3
#  &process_recalc3
#  &process_next3
#
#  &process_direct4
#  &process_recalc4
#  &process_next4
#
#  &update_screen_2
#  &update_screen_3
#  &update_screen_4
#
#  &update_s2_quantities
#  &update_s2_widgets
#  &update_s2_show_always_button
#  &update_s2_gen_comments
#
#  &inventory_allocate_item
#  &inventory_reallocate_item
#  &inventory_deallocate_item
#  &inventory_deallocate_all_items
#  &inventory_final_allocation
#
#  &update_s3_shipping_address
#  &update_s3_payment_type
#  &update_s3_tax_type
#  &update_s3_coupon
#  &update_s3_same_billing_button
#  &update_s3_use_security_button
#
#  &update_s4_shipping_type
#  &update_s4_billing_info
#  &update_s4_payment_info
#
#  &update_line_totals
#  &update_totals
#
#  &does_basket_exist
#  &create_new_basket
#  &type_get_first_all
#  &type_get_first_shippingtype
#  &type_get_first_paymenttype
#  &type_get_first_taxtype
#  &load_basket
#  &store_basket
#  &is_basket_empty
#  &should_we_show_basket_every_time
#  &add_item_to_basket
#  &return_to_shopping
#
#  &add_error
#  &check_em
#  &ensure_appropriate_environment
#  &ensure_basket_exists
#  &ensure_user_is_authorized_to_shop
#
#  &determine_referer
#
#  &discount_calculate_for_new_item
#  &discount_calculate_for_new_qty
#
#  &analyze_widgets
#  &analyze_simple_options
#  &precheck_widget_text
#  &translate_widget_to_text
#  &dispatch_order
#  &email_receipt
#
#  %fdata
#
#  @error_messages
#
#  @name
#  @price
#  @weight
#  @volume
#  @qty
#  @discount
#  @total
#  @ids
#  @uniqueids
#  @skus
#  @stockids
#  @mixids
#  @cellids
#  @widget_text
#  @widget_type
#  @widget_choice
#
#  $gencomments
#  $referer
#
#  $curr_unique_seq
#  $yesdiscount
#
#  $use_security
#  $always_show_basket
#  $same_billing
#
#  $payment_type
#  $tax_type
#  $shipping_type
#  $coupon_code
#
#  $sub_total
#  $tax_total
#  $ship_total
#  $discount_total
#  $grand_total
#
#  $s_fname
#  $s_lname
#  $s_address1
#  $s_address2
#  $s_city
#  $s_state
#  $s_zip
#  $s_country
#
#  $b_fname
#  $b_lname
#  $b_address1
#  $b_address2
#  $b_city
#  $b_state
#  $b_zip
#  $b_country
#  $b_homephone
#  $b_workphone
#  $b_fax
#  $b_email
#  $b_company
#
#  $pf1
#  $pf2
#  $pf3
#  $pf4
#
#  $ordernum
#  $orderdate
#
#
#  ... *whew*
#

#
# -----------------------------------------------------------------------------
#

package BASKET;

use strict;
require ITEM;
require KNAR;
require DBLIB;
require ORDER;
require ORDERLINE;
require MATRIX;
require MILTON;
require SNIPPET;
require SENDMAIL;
require SHIPPING;
require SHIPMENT;
require PAYMENT;
require TAX;
require ORDERSTATS;
require PAYMENT_TYPE;
require TAX_TYPE;
require SEC;
require CUSTOMER;
require TABLE;

## decalare all of these variable so that they belong to the package

use vars qw(@error_messages @name @price @weight @volume @qty @discount @total @ids @uniqueids @skus @stockids @mixids @cellids @widget_text @widget_type @widget_choice);

use vars qw($gencomments $referer $curr_unique_seq $yesdiscount $use_security $always_show_basket $same_billing $payment_type $tax_type $shipping_type $coupon_code $sub_total $tax_total $ship_total $discount_total $grand_total);

use vars qw($s_fname $s_lname $s_address1 $s_address2 $s_city $s_state $s_zip $s_country $b_fname $b_lname $b_address1 $b_address2 $b_company $b_city $b_state $b_zip $b_country $b_homephone $b_workphone $b_fax $b_email $pf1 $pf2 $pf3 $pf4 $ordernum $orderdate);

use vars qw(%fdata $session_id);

use vars qw($stockid $mixid $cellid $sub_total_txt $tax_total_txt $ship_total_txt $grand_total_txt $ordernum_txt $orderdate_txt $call_for_shipping_txt);

@error_messages=();

@name=();
@price=();
@weight=();
@volume=();
@qty=();
@discount=();
@total=();
@ids=();
@uniqueids=();
@skus=();
@stockids=();
@mixids=();
@cellids=();
@widget_text=();
@widget_type=();
@widget_choice=();

$gencomments="";
$referer="";

$curr_unique_seq="";
$yesdiscount="";

$use_security="";
$always_show_basket="";
$same_billing="";

$payment_type="";
$tax_type="";
$shipping_type="";
$coupon_code="";

$sub_total="";
$tax_total="";
$ship_total="";
$discount_total="";
$grand_total="";

$s_fname="";
$s_lname="";
$s_address1="";
$s_address2="";
$s_city="";
$s_state="";
$s_zip="";
$s_country="";

$b_fname="";
$b_lname="";
$b_address1="";
$b_address2="";
$b_company="";
$b_city="";
$b_state="";
$b_zip="";
$b_country="";
$b_homephone="";
$b_workphone="";
$b_fax="";
$b_email="";

$pf1="";
$pf2="";
$pf3="";
$pf4="";

$ordernum="";
$orderdate="";


# -----------------------------------------------------------------------------
# For use with STATE.pm by load_basket and store_basket
$session_id = "";
sub set_session_id { $session_id = shift; };

#
# =============================================================================
#

sub set_up_form_data {
  %fdata = %HTML::Embperl::fdat;
}

#
# -----------------------------------------------------------------------------
#
# token and token.x are both checked, so that all pages in the 
# site can use text order buttons or image order buttons.
#
# screen 1: show_empty_basket
#        2: show_basket
#        3: show_shipping
#        4: show_payment
#        5: show_thankyou
#
#

sub basket_init {
  @error_messages=();

  $HTML::Embperl::optRawInput=1;
  $HTML::Embperl::escmode=0;

  set_up_form_data();

  #
  # entry points: add and view
  #

  if (defined($fdata{'add'}) || defined($fdata{'add.x'})) {

    return(process_add());

  } elsif (defined($fdata{'view'}) || defined($fdata{'view.x'})) {

    return(process_view());

  #
  # screen 1: continue shopping (from "your basket is empty" screen)
  # unused - was causing more trouble than it solved

#  } elsif (defined($fdata{'continue1'}) || defined($fdata{'continue1.x'})) {
#    determine_referer();
#    return_to_shopping();

  #
  # screen 2: clear, continue shopping, recalculate, next
  #

  } elsif (defined($fdata{'clear2'}) || defined($fdata{'clear2.x'})) {

    return(process_clear2());

  } elsif (defined($fdata{'continue2'}) || defined($fdata{'continue2.x'})) {

    return(process_continue2());

  } elsif (defined($fdata{'recalc2'}) || defined($fdata{'recalc2.x'})) {

    return(process_recalc2());

  } elsif (defined($fdata{'next2'}) || defined($fdata{'next2.x'})) {

    return(process_next2());

  #
  # screen 3: back, continue shopping, next
  #

  } elsif (defined($fdata{'back3'}) || defined($fdata{'back3.x'})) {

    return(process_back3());

  } elsif (defined($fdata{'continue3'}) || defined($fdata{'continue3.x'})) {

    return(process_continue3());

  } elsif (defined($fdata{'recalc3'}) || defined($fdata{'recalc3.x'})) {

    return(process_recalc3());

  } elsif (defined($fdata{'next3'}) || defined($fdata{'next3.x'})) {

    return(process_next3());

  #
  # this is a special entry point for when the browser is
  # redirected to a secure mode.
  #
  } elsif (defined($fdata{'direct4'})) {

    return(process_direct4());

  #
  # screen 4: back, continue shopping, recalculate, finalize
  #

#
# XXX This button has been removed because it generates an unavoidable
#     error message.  The system attempts to redirect from secure to 
#     unsecure mode, which produces a handy (and un-turn-off-able) message.
#     On the user's browser, that is.  Doh.
#
#  } elsif (defined($fdata{'back4'}) || defined($fdata{'back4.x'})) {
#
#    return(process_back4());
#
#  } elsif (defined($fdata{'continue4'}) || defined($fdata{'continue4.x'})) {
#
#    return(process_continue4());
#

  } elsif (defined($fdata{'recalc4'}) || defined($fdata{'recalc4.x'})) {

    return(process_recalc4());

  } elsif (defined($fdata{'next4'}) || defined($fdata{'next4.x'})) {

    return(process_next4());

  #
  #  misc. functions
  #

  } elsif (defined($fdata{'showlist'}) || defined($fdata{'showlist.x'})) {

    return(process_showlist());

  } elsif (defined($fdata{'choose'}) || defined($fdata{'choose.x'})) {

    return(process_choose());

  } else {

    add_error('Unknown function!');
    return 0;
  }
}


#
# =============================================================================
#
# Point of entry buttons
#
# =============================================================================
#

sub process_add {
  # determine whether or not they are authorized to use the shopping basket
  if (!ensure_user_is_authorized_to_shop()) { return 0; }

  load_basket();

  # do they have a shopping basket?
  if (!does_basket_exist()) {
    create_new_basket();
  }

  determine_referer();

  if (!add_item_to_basket()) { return 0; }

  # update tax_total, sub_total, grand_total, ship_total
  if (!update_totals()) { return 0; }

  store_basket();

  if (should_we_show_basket_every_time()) {
    if (is_basket_empty()) {
      return 1;
    } else {
      return 2;
    }
  } else {
    return_to_shopping();
  }
}

#
# -----------------------------------------------------------------------------
#

sub process_view {
  # determine whether or not they are authorized to use the shopping basket
  if (!ensure_user_is_authorized_to_shop()) { return 0; }

  load_basket();

  # do they have a shopping basket?
  if (!does_basket_exist()) {
    create_new_basket();
  }

  determine_referer();

  store_basket();

  if (is_basket_empty()) {
    return 1;
  } else {
    return 2;
  }
}

#
# =============================================================================
#
# Screen two buttons
#
# =============================================================================
#

sub process_clear2() {

  if (!ensure_appropriate_environment()) { return 0; }

  # first, deallocate all of the items
  if (!inventory_deallocate_all_items()) { return 0; }

  # update the "Show me my shopping basket every time" button
  if (!update_s2_show_always_button()) { return 0; }

  neatly_dispose_of_basket();

  store_basket(); # can't fail
  return_to_shopping();
}

#
# -----------------------------------------------------------------------------
#

sub process_recalc2() {
  if (!ensure_appropriate_environment()) { return 0; }
  if (!update_screen_2()) { return 0; }

  store_basket(); # can't fail

  # after updating their quantites, they may have gotten rid of all of the
  # items in their shopping basket...

  if (is_basket_empty()) {
    return 1;
  } else {
    return 2;
  }
}

#
# -----------------------------------------------------------------------------
#

sub process_continue2() {
  if (!ensure_appropriate_environment()) { return 0; }
  if (!update_screen_2()) { return 0; }

  store_basket();
  return_to_shopping();
}

#
# -----------------------------------------------------------------------------
#

sub process_next2() {
  if (!ensure_appropriate_environment()) { return 0; }

  my($errflag) = 0;
  if(length($fdata{gencomments}) > TABLE::width('cat_order', 'general_ins')) {
      add_error('"Customer Comments" too long. Max length: 4000 characters.');
      $errflag = 1;
  }

  if (!update_screen_2()) { return 0; }

  return 0 if $errflag;

  store_basket();

  # he might have emptied the basket when updating quantities; check and see.

  if (is_basket_empty()) {
    return 1;
  } else {
    return 3;
  }
}

#
# =============================================================================
#
# Screen three buttons
#
# =============================================================================
#

sub process_back3() {
  if (!ensure_appropriate_environment()) { return 0; }
  if (!update_screen_3(0)) { return 0; }
  store_basket();
  return 2;
}

#
# -----------------------------------------------------------------------------
#

sub process_continue3() {
  if (!ensure_appropriate_environment()) { return 0; }
  if (!update_screen_3(0)) { return 0; }
  store_basket();
  return_to_shopping();
}

#
# -----------------------------------------------------------------------------
#

sub process_recalc3() {
  if (!ensure_appropriate_environment()) { return 0; }
  if (!update_screen_3(0)) { return 0; }
  store_basket();
  return 3;
}

#
# -----------------------------------------------------------------------------
#

sub process_next3() {
  if (!ensure_appropriate_environment()) { return 0; }
  if (!update_screen_3(1)) { return 0; }
  store_basket();

  #
  # here, we would ordinarily just return 4, which is the next screen
  # in line, but we need to be secure about the thing, so we have to 
  # redirect their browser to a secure URL.
  #

  if ($use_security eq "off") {
    return 4;
  } else {
    # do a secure redirect
    my($surl) = KNAR::knar_entry_get('TALLYMAN_SECURE_URL');
    MILTON::redirect("$surl/cart.epl?direct4=ok");
  }
}


#
# =============================================================================
#
# Screen four buttons
#
# =============================================================================
#

#
# XXX this button has been disabled!!
#     see note in basket_init() code
#
#sub process_back4() {
#  if (!ensure_appropriate_environment()) { return 0; }
#  if (!update_screen_4(0)) { return 0; }
#  store_basket(); # can't fail
#  return 3;
#}

#
# -----------------------------------------------------------------------------
#
# same as above
#
#sub process_continue4() {
#  if (!ensure_appropriate_environment()) { return 0; }
#  if (!update_screen_4(0)) { return 0; }
#  store_basket(); # can't fail
#  return_to_shopping();
#}


#
# -----------------------------------------------------------------------------
#
# this function exists to allow the redirection system to enter the
# shopping basket without having to submit any values.
#
sub process_direct4() {
  if (!ensure_appropriate_environment()) { return 0; }
  return 4;
}

#
# -----------------------------------------------------------------------------
#

sub process_recalc4(0) {
  if (!ensure_appropriate_environment()) { return 0; }
  if (!update_screen_4()) { return 0; }
  store_basket();
  return 4;
}

#
# -----------------------------------------------------------------------------
#

sub process_next4() {
  if (!ensure_appropriate_environment()) { return 0; }
  if (!update_screen_4(1)) { return 0; }

  # do the actual order dispatching
  if (!dispatch_order()) { return 0; }

  store_basket();

  # go to screen 5 (thank you screen)
  return 5;
}

#
# =============================================================================
#
# Misc. functions
#
# =============================================================================
#

sub process_showlist {
  if (!ensure_appropriate_environment()) { return 0; }  

  if (!defined($fdata{'itemnum'}) || $fdata{'itemnum'} =~ /\D/ ||
      $fdata{'itemnum'} < 0 || $fdata{'itemnum'} > $#name) {
    add_error('Bad itemnum.');
    return 0;
  }

  store_basket();

  # go to screen 6 (list of all combinations)
  return 6;
}

sub process_choose {
  if (!ensure_appropriate_environment()) { return 0; }  

  if (!defined($fdata{'itemnum'}) || $fdata{'itemnum'} =~ /\D/ ||
      $fdata{'itemnum'} < 0 || $fdata{'itemnum'} > $#name) {
    add_error('Bad itemnum.');
    return 0;
  }

  my($i) = $fdata{'itemnum'};
  SEC::untaint_ref(\$i);

  if (!defined($fdata{'stockid'}) || $fdata{'stockid'} =~ /\D/) {
    add_error('Bad stockid.');
    return 0;
  }

  my($newstockid) = $fdata{'stockid'};
  SEC::untaint_ref(\$newstockid);

  my($newcellid) = ITEM::mix_lookup_cell_given_item($newstockid, $mixids[$i]);
  my($sku, $price, $weight, $volume, $atp, $rt, $drq)
     = STOCK::get_info($newstockid);

  if (!defined($price) || ($price eq "")) {
    add_error("Bad stock_id: $newstockid'");
    return 0;
  }

  $price[$i] = $price;
  $weight[$i] = $weight;
  $volume[$i] = $volume;
  $skus[$i] = $sku;
  $stockids[$i] = $newstockid;
  $cellids[$i] = $newcellid;

  if (!update_line_totals()) { return 0; }
  if (!update_totals()) { return 0; }

  store_basket();

  # go to screen 2 (basket quantities)
  return 2;
}

#
#
# =============================================================================
#
# These functions just handle updating the HTML forms
#
# =============================================================================
#

sub update_screen_2 {
  # update all of those quantity boxes
  if (!update_s2_quantities()) { return 0; }

  # update all of the product variation widgets
  if (!update_s2_widgets()) { return 0; }

  # update the "Show me my shopping basket every time" button
  if (!update_s2_show_always_button()) { return 0; }

  # update the general comments box
  if (!update_s2_gen_comments()) { return 0; }

  # now that the quantities have changed, update the line totals.
  # note that this should come _after_ the update_s2_widgets call,
  # since it's possible that it could change the price of the item.
  if (!update_line_totals()) { return 0; }

  # update tax_total, sub_total, grand_total, ship_total
  if (!update_totals()) { return 0; }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_screen_3 {
  my($required) = shift;

  # update the shipping address
  if (!update_s3_shipping_address($required)) { return 0; }

  # update the payment type
  if (!update_s3_payment_type()) { return 0; }

  # update the tax type
  if (!update_s3_tax_type()) { return 0; }

  # update the coupon / gift certificate
  if (!update_s3_coupon()) { return 0; }

  # update the "same billing address" button
  if (!update_s3_same_billing_button()) { return 0; }

  # update the "use security" button
  if (!update_s3_use_security_button()) { return 0; }

  if (!update_totals()) { return 0; }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_screen_4 {
  my($required) = shift;

  # update their shipping method selection
  if (!update_s4_shipping_type()) { return 0; }

  # update their billing info (if needed)
  if (!update_s4_billing_info($required)) { return 0; }

  # update their payment info 
  if (!update_s4_payment_info($required)) { return 0; }

  if (!update_totals()) { return 0; }

  return 1;
}

#
# =============================================================================
#
# =============================================================================
#

sub update_s2_quantities {
  my($i, $total, $realindex, $text, $newqty);

  # if any of the quantities are 0, are nonexistant, or consist of non-digit
  # characters, remove the item from the basket

  $total= $#name + 1;
  for ($i=0, $realindex=0; $i < $total; $i++, $realindex++) {

    $text = $fdata{"${i}qty"};
    $text =~ s/[^0-9]//g;
    SEC::untaint_ref(\$text);

    if (!defined($text) ||
        ($text =~ /\D/) ||
        ($text==0)) {

      inventory_deallocate_item($ids[$realindex]);

      splice(@name,          $realindex, 1);
      splice(@price,         $realindex, 1);
      splice(@weight,        $realindex, 1);
      splice(@volume,        $realindex, 1);
      splice(@qty,           $realindex, 1);
      splice(@discount,      $realindex, 1);
      splice(@total,         $realindex, 1);
      splice(@ids,           $realindex, 1);
      splice(@uniqueids,     $realindex, 1);
      splice(@skus,          $realindex, 1);
      splice(@stockids,      $realindex, 1);
      splice(@mixids,        $realindex, 1);
      splice(@cellids,       $realindex, 1);
      splice(@widget_text,   $realindex, 1);
      splice(@widget_type,   $realindex, 1);
      splice(@widget_choice, $realindex, 1);
      $realindex--;

    } else {

      $newqty = $qty[$realindex] - $text;
      $qty[$realindex] = $text;

      inventory_reallocate_item($ids[$realindex],$newqty);
    }
  }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_line_totals {
  my($i);

  for ($i=0; $i < $#name+1; $i++) {
    $total[$i]=($price[$i] * $qty[$i]) - $discount[$i];
  }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_s2_widgets {
  my($i, $j, @nodes, $done, $hohum);
  my($sku, $price, $weight, $volume, $atp, $rt, $drq, $newcellid, $newstockid);

  # check all of the mix'ed products, so's we can update
  # their little widgets

  HOHUMDIDDLYTHING: for ($i=0; $i < $#name + 1; $i++) {
    if ($widget_type[$i] == 0) { next HOHUMDIDDLYTHING; }

    #
    # Update simple widget
    #

    if ($widget_type[$i] == 1) {
      $newstockid = $fdata{"widget_simple_$uniqueids[$i]"};

      if (!defined($newstockid) || ($newstockid =~ /\D/)) { # Bad form data!
        add_error("Cannot find value for 'widget_simple_$uniqueids[$i]'");
        return 0;
      }
      SEC::untaint_ref(\$newstockid);

      $newcellid = ITEM::mix_lookup_cell_given_item($newstockid, $mixids[$i]);
      ($sku, $price, $weight, $volume, $atp, $rt, $drq)
         = STOCK::get_info($newstockid);

      if (!defined($price) || ($price eq "")) {
        add_error("Bad stock_id: $newstockid'");
        return 0;
      }

      $price[$i] = $price;
      $weight[$i] = $weight;
      $volume[$i] = $volume;
      $skus[$i] = $sku;
      $stockids[$i] = $newstockid;
      $cellids[$i] = $newcellid;

    #
    # Update complex widget
    #

    } elsif ($widget_type[$i]==2) { 
      $j = 0;
      $done = 0;
      while ($done == 0) {
        $hohum = $fdata{"widget_$uniqueids[$i]_${j}"};
        if (!defined($hohum) || $hohum eq "") {
          $done = 1;
        } elsif ($hohum =~ /\d+/) {
          SEC::untaint_ref(\$hohum);
          $nodes[$j] = $hohum;
        } else { 
          add_error("Bad form data: widget_$uniqueids[$i]_${j} is corrupt.");
          return 0;
        }
        $j++;
      }

      ($newcellid, $newstockid) 
        = ITEM::cell_find_given_ranges($mixids[$i], @nodes);

      if (!defined($newcellid) || $newcellid == 0) { 
        # PANIC 
        add_error("The combination of items you selected (for the item named '$name[$i]') does not exist.<br>
If you would like, we can show you a <a href=\"cart.epl?showlist=ok&itemnum=$i\">list of all possible combinations</a>.");
        return 0;
      }

      ($sku, $price, $weight, $volume, $atp, $rt, $drq)
        = STOCK::get_info($newstockid);

      $price[$i] = $price;
      $weight[$i] = $weight;
      $volume[$i] = $volume;
      $skus[$i] = $sku;
      $stockids[$i] = $newstockid;
      $cellids[$i] = $newcellid;      

    #
    # XXX widget type 3 is the wizard-based widgets.  Not yet implemented.
    #

    #
    # This is the non-mix based complex widget
    #
    } elsif ($widget_type[$i] == 4) {
      $widget_choice[$i] = "";

      for ($j=0; 
           defined($fdata{"widget_$uniqueids[$i]_$j"}); 
           $j++) {
        $widget_choice[$i] .= $fdata{"widget_$uniqueids[$i]_$j"};
        $widget_choice[$i] .= " ";
      }

      chop($widget_choice[$i]);

    }
  }  

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_s2_show_always_button {
  if ($fdata{'always_show_basket'} eq "on") {
    $always_show_basket = "on";
  } else {
    $always_show_basket = "off";
  }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_s2_gen_comments {
  $gencomments = $fdata{'gencomments'};
  return 1;
}

#
# -----------------------------------------------------------------------------
#
# SSS -- inventory allocation stubs
#

#
# initial allocation of an item
# takes: an item id representing the item to be allocated, and an initial 
#        quantity to allocate
#
sub inventory_allocate_item {
  my($itemid,$qty) = @_;
  return 1;
}

#
# the user may request a change of the number of items to order, which
# can be either more or less than what he originally specified.
# if the new quantity is 0, inventory_deallocate_item will be called instead.
# takes: an item id, and a quantity difference (can be positive or
#        negative.
#
sub inventory_reallocate_item {
  my($itemid,$qty_diff)=@_;
  return 1;
}

#
# deallocates an item
# takes: an item id
#
sub inventory_deallocate_item {
  my($itemid) = shift;
  return 1;
}

#
# deallocates all items (for use when the shopping basket is cleared)
# takes: nothing
#        I wrote this one for you... :)
#
sub inventory_deallocate_all_items {
  my($i);

  for ($i=0; $i < $#ids+1; $i++) {
    if (!inventory_deallocate_item($ids[$i])) { return 0; }
  }
  return 1;
}

#
# although unlikely, it is possible that the actual amount of inventory
# allocated is different from the quantities specified in the basket.
# This could happen if, for example, the quantities of items were updated
# in the shopping basket, and the correct amount of inventory was allocated,
# but an error occured before the shopping basket could be saved.  The next
# time the basket was loaded (and the order possibly completed), the user
# will have finalized the order with the old quantities in the basket, but
# with the new quantities actually being allocated.
#
# This function is called just before the order has been logged to the order
# capture system.  This is a chance to reconcile all such discrepancies.
#
# Or, if you don't care about allocating an item when it's placed in the
# basket, and would rather just allocate them all when the transaction
# is finalized, you could do it here.
#
sub inventory_final_allocation {
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_s3_shipping_address {
  my($required) = shift;

  # verify that they've put something into the form elements

  my(@formitems)=("Shipping first name:Ecom_ShipTo_Postal_Name_First",
                  "Shipping last name:Ecom_ShipTo_Postal_Name_Last",
                  "Shipping street:Ecom_ShipTo_Postal_Street_Line1",
                  "Shipping city:Ecom_ShipTo_Postal_Street_City",
                  "Shipping state:Ecom_ShipTo_Postal_Street_StateProv",
                  "Shipping zip code:Ecom_ShipTo_Postal_Street_PostalCode",
                  "Shipping country:Ecom_ShipTo_Postal_Street_CountryCode");

  if ($required) {
    if (!check_em(@formitems)) { return 0; }
#
# this is no good, since they could live in canada.
#    if ($fdata{'s_zip'} =~ /\D/) {
#      add_error('Your shipping zip code must consist of the digits 0-9');
#      return(0);
#    }
  }

  # and update the values
  $s_fname     = $fdata{'Ecom_ShipTo_Postal_Name_First'};
  $s_lname     = $fdata{'Ecom_ShipTo_Postal_Name_Last'};
  $s_address1  = $fdata{'Ecom_ShipTo_Postal_Street_Line1'};
  $s_address2  = $fdata{'Ecom_ShipTo_Postal_Street_Line2'};
  $s_city      = $fdata{'Ecom_ShipTo_Postal_Street_City'};
  $s_state     = $fdata{'Ecom_ShipTo_Postal_Street_StateProv'};
  $s_zip       = $fdata{'Ecom_ShipTo_Postal_Street_PostalCode'};
  $s_country   = $fdata{'Ecom_ShipTo_Postal_Street_CountryCode'};

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_s3_payment_type {
  if ($payment_type != -1) {
    $payment_type = $fdata{'payment_type'};
    PAYMENT_TYPE::verify_id(\$payment_type, 1);
  }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_s3_tax_type {
  if ($tax_type != -1) {
    $tax_type = $fdata{'tax_type'};
    TAX_TYPE::verify_id(\$tax_type, 1);
  }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_s3_coupon {
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_s3_same_billing_button {
  if ($fdata{'same_billing'} eq "on") {
    $same_billing = "on";
  } else {
    $same_billing = "off";
  }
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_s3_use_security_button {
  if ($fdata{'use_security'} eq "on") {
    $use_security = "on";
  } else {
    $use_security = "off";
  }
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_s4_shipping_type {
  if ($shipping_type != -1) {
    $shipping_type = $fdata{'shipping_type'};
    SHIPPING::shippingtype_verify_id(\$shipping_type, 1);
  }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_s4_billing_info {
  my($required) = shift;

  my(@formitems) = ("Billing first name:Ecom_BillTo_Postal_Name_First",
                    "Billing last name:Ecom_BillTo_Postal_Name_Last",
                    "Billing street:Ecom_BillTo_Postal_Street_Line1",
                    "Billing city:Ecom_BillTo_Postal_Street_City",
                    "Billing state:Ecom_BillTo_Postal_Street_StateProv",
                    "Billing zip code:Ecom_BillTo_Postal_Street_PostalCode");

  my(@formitems2) = ("Billing home phone:b_homephone",
                     "Billing email:Ecom_BillTo_Online_Email");

  # only needed if the shipping address and billing addresses are 
  # different...

  if ($same_billing eq "on") {
    if ($required) {
      if (!check_em(@formitems)) { return 0; }
#
# this is no good, since they could live in canada.
#      if ($fdata{'b_zip'} =~ /\D/) {
#        add_error('Your billing zip code must consist of the digits 0-9');
#        return 0;
#      }
    }
    $b_fname     = $fdata{'Ecom_BillTo_Postal_Name_First'};
    $b_lname     = $fdata{'Ecom_BillTo_Postal_Name_Last'};
    $b_address1  = $fdata{'Ecom_BillTo_Postal_Street_Line1'};
    $b_address2  = $fdata{'Ecom_BillTo_Postal_Street_Line2'};
    $b_city      = $fdata{'Ecom_BillTo_Postal_Street_City'};
    $b_state     = $fdata{'Ecom_BillTo_Postal_Street_StateProv'};
    $b_zip       = $fdata{'Ecom_BillTo_Postal_Street_PostalCode'};
    $b_country   = $fdata{'Ecom_BillTo_Postal_Street_CountryCode'};
  }

  # ...but these ones are required no matter what

  if ($required) {
    if (!check_em(@formitems2)) { return 0; }
  }

  $b_homephone = $fdata{'b_homephone'};
  $b_workphone = $fdata{'b_workphone'};
  $b_fax       = $fdata{'b_fax'};
  $b_email     = $fdata{'Ecom_BillTo_Online_Email'};
  $b_company   = $fdata{'b_company'};

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_s4_payment_info {
  my($required) = shift;

  my(@formitems);

  #
  # is it a credit card?
  #   if so, some of the fields have been renamed for ECML 1.0 compliance:
  #   pf_1 = name = Ecom_Payment_Card_Name
  #   pf_2 = card number = Ecom_Payment_Card_Number
  #   pf_3 = company = pf_3 (the same)
  #   pf_41 = expiration date month = Ecom_Payment_Card_ExpDate_Month
  #   pf_42 = expiration date year = Ecom_Payment_Card_ExpDate_Year
  #

  my($type) = payment_type_widgettype();

  if ($type eq 'CC') {
    @formitems = ('Credit card name:Ecom_Payment_Card_Name',
                  'Credit card number:Ecom_Payment_Card_Number',
                  'Expiration date month:Ecom_Payment_Card_ExpDate_Month',
                  'Expiration date year:Ecom_Payment_Card_ExpDate_Year');

    if ($required) {
      if (!check_em(@formitems)) { return 0; }
    }

    $pf1 = $fdata{'Ecom_Payment_Card_Name'};
    $pf2 = $fdata{'Ecom_Payment_Card_Number'};
    if (!defined($fdata{'pf_3'}) || $fdata{'pf_3'} eq "") {
      $pf3 = ' ';
    } else {
      $pf3 = $fdata{'pf_3'};
    }

    # check expiration month

    if ($fdata{'Ecom_Payment_Card_ExpDate_Month'} =~ /\D/) {
      add_error('Invalid expiration month: non-digit characters');
      return(0);
    }
    if ($fdata{'Ecom_Payment_Card_ExpDate_Month'} > 12 || 
        $fdata{'Ecom_Payment_Card_ExpDate_Month'} < 1) {
      add_error('Invalid expiration month: out of range');
      return(0);
    }

    # check expiration year

    if ($fdata{'Ecom_Payment_Card_ExpDate_Year'} =~ /\D/) {
      add_error('Invalid expiration year: non-digit characters');
      return(0);
    }

    if ($fdata{'Ecom_Payment_Card_ExpDate_Year'} < 1999) {
      add_error('Invalid expiration year: out of range');
      return(0);
    }

    $pf4 = DBLIB::date_string_make(0,0,0,1,
             $fdata{'Ecom_Payment_Card_ExpDate_Month'},
             $fdata{'Ecom_Payment_Card_ExpDate_Year'});

  #
  # is it a check?
  #  pf_1 = account number
  #  pf_2 = routing number
  #

  } elsif ($type eq 'CHECK') {
    @formitems = ('Account number:pf_1','Routing number:pf_2');
    if ($required) {
      if (!check_em(@formitems)) { return 0; }
    }

    $pf1 = $fdata{'pf_1'};
    $pf2 = $fdata{'pf_2'};
    $pf3 = ' ';

    $pf4 = DBLIB::date_string_make(0,0,0,1,1,1969);

  #
  # is it a purchase order?
  #  pf_1 = purchase order number
  #

  } elsif ($type eq 'PO') {
    @formitems = ('Purchase order number:pf_1');
    if ($required) {
      if (!check_em(@formitems)) { return 0; }
    }

    $pf1 = $fdata{'pf_1'};
    $pf2 = ' ';
    $pf3 = ' ';

    $pf4 = DBLIB::date_string_make(0,0,0,1,1,1969);


  #
  # Nope, it's a COD (or something else); no info required
  #

  } else {
    $pf1 = ' ';
    $pf2 = ' ';
    $pf3 = ' ';
    $pf4 = DBLIB::date_string_make(0,0,0,1,1,1969);
  }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub update_totals {
  my($i,$flat,$label,$percent);

  # check to see if any of the discounts are non-zero.
  # if so, set the variable

  $sub_total = 0;
  $yesdiscount = 0;

  for ($i=0; $i < $#name+1; $i++) {
    $sub_total += $total[$i];
    if ($discount[$i] != 0) {
      $yesdiscount=1;
    }
  }  

  if ($tax_type != -1) {
      ## calculate the tax--if tax is defined at this point
      ##  $tax_total = ORDER::tax_calculate_charge($tax_type);  

      my ($tax_label, $percentage) = TAX_TYPE::get_info($tax_type);
      $tax_total = $sub_total * ($percentage / 100);
  } else {
      $tax_total = 0;
  }

  ## calculate shipping total
  if ($shipping_type != -1) {
    $ship_total = SHIPPING::shipping_calculate_charge($shipping_type);
  } else {
    $ship_total = 0;
  }

  $grand_total = $sub_total + $tax_total;

  if ($ship_total != -1) {
    $grand_total += $ship_total;
  }

  return 1;
}

#
# =============================================================================
#
# RANDOM LOGICAL DIVISION
#
# =============================================================================
#

sub does_basket_exist {
  # this is an arbitrary token; it could have been anything.
  # whether it is on or off, it ought to be defined.
  if (($always_show_basket ne "on") && ($always_show_basket ne "off")) {
    return 0;
  } else {
    return 1;
  }
}

#
# -----------------------------------------------------------------------------
#

sub create_new_basket {
    
    # Tell me again why we're using globals...
    # Clear everything out so we don't get somebody else's data
    # when using mod_perl

    @name = @price = @weight = @volume = @qty = @discount = @total =
	@ids = @uniqueids = @skus = @stockids = @mixids = @cellids =
	    @widget_text = @widget_type = @widget_choice = ();

    $curr_unique_seq = $referer = $yesdiscount = $always_show_basket =
	$gencomments = $use_security = $s_fname = $s_lname =
	$s_address1 = $s_address2 = $s_city = $s_state = $s_zip =
	$s_country = $payment_type = $tax_type = $coupon_code =
	$shipping_type = $same_billing = $b_fname = $b_lname =
	$b_address1 = $b_address2 = $b_city = $b_state = $b_zip =
	$b_country = $b_homephone = $b_workphone = $b_fax = $b_email =
	$b_company = $sub_total = $tax_total = $ship_total =
	$discount_total = $grand_total = '';


  # prime the values

  $curr_unique_seq = 0;
  $always_show_basket = 'on';
  $referer = "none.html";
  $gencomments = "";

  $use_security = "on";
  $same_billing = "on";

  $coupon_code = "";

  $sub_total = 0;
  $tax_total = 0;
  $ship_total = 0;
  $discount_total = 0;
  $grand_total = 0;
  $yesdiscount = 0;

  type_get_first_all();

  # and store the basket

  store_basket();
}

#
# -----------------------------------------------------------------------------
#

sub type_verify_all {
  #
  # make sure that the values selected are ok.
  #

  if (!SHIPPING::shippingtype_verify_id(\$shipping_type)) {
    type_get_first_shippingtype();
  }

  if (!PAYMENT_TYPE::verify_id(\$payment_type)) {
    type_get_first_paymenttype();
  }

  if (!TAX_TYPE::verify_id(\$tax_type)) {
    type_get_first_taxtype();
  }
}

#
# -----------------------------------------------------------------------------
#

sub type_get_first_all {
  type_get_first_shippingtype();
  type_get_first_paymenttype();
  type_get_first_taxtype();
}

sub type_get_first_shippingtype {
  # 0=id, 1=label, 2=avail, 3=calculated, 4=algorithm, 5=min, 6=extra
  my($ref) = SHIPPING::shippingtype_list_all_available();
  $shipping_type = $ref->[0][0];

  # it is possible that the merchant hasn't defined any shipping methods!
  if (!defined($shipping_type) || $shipping_type eq "0") {
    $shipping_type = -1;
  }
}

sub type_get_first_paymenttype {
  # 0=id, 1=label, 2=avail
  my($ref) = PAYMENT_TYPE::list_all_available();
  $payment_type = $ref->[0][0];

  if (!defined($payment_type) || $payment_type eq "0") {
    $payment_type = -1;
  }
}

sub type_get_first_taxtype {
  # 0=id, 1=label, 2=percent
  my($ref) = TAX_TYPE::list_all();
  $tax_type = $ref->[0][0];

  if (!defined($tax_type) || $tax_type eq "0") {
    $tax_type = -1;
  }
}

#
# -----------------------------------------------------------------------------
#

sub load_basket {

  my(%state);

  my($hash_ref) = STATE::state_get($session_id);
  %state = ((ref($hash_ref) eq "HASH") ? %$hash_ref : ());

  # load the items in the basket


  @name=split(/\0/,$state{'name'});
  @price=split(/\0/,$state{'price'});
  @weight=split(/\0/,$state{'weight'});
  @volume=split(/\0/,$state{'volume'});
  @qty=split(/\0/,$state{'qty'});
  @discount=split(/\0/,$state{'discount'});
  @total=split(/\0/,$state{'total'});
  @ids=split(/\0/,$state{'ids'});
  @uniqueids=split(/\0/,$state{'uniqueids'});
  @skus=split(/\0/,$state{'skus'});
  @stockids=split(/\0/,$state{'stockids'});
  @mixids=split(/\0/,$state{'mixids'});
  @cellids=split(/\0/,$state{'cellids'});
  @widget_text=split(/\0/,$state{'widget_text'});
  @widget_type=split(/\0/,$state{'widget_type'});
  @widget_choice=split(/\0/,$state{'widget_choice'});


  # load miscellaneous

  $curr_unique_seq    = $state{'curr_unique_seq'};
  $referer            = $state{'referer'};
  $yesdiscount        = $state{'yesdiscount'};

  # load basket options

  $always_show_basket = $state{'always_show_basket'};
  $gencomments        = $state{'gencomments'};
  $use_security       = $state{'use_security'};

  # load shipping address

  $s_fname            = $state{'s_fname'};
  $s_lname            = $state{'s_lname'};
  $s_address1         = $state{'s_address1'};
  $s_address2         = $state{'s_address2'};
  $s_city             = $state{'s_city'};
  $s_state            = $state{'s_state'};
  $s_zip              = $state{'s_zip'};
  $s_country          = $state{'s_country'};

  # load payment type selected
  $payment_type       = $state{'payment_type'};

  # load tax type selected
  $tax_type           = $state{'tax_type'};

  # load coupon or certificate code
  $coupon_code        = $state{'coupon_code'};

  # load shipping type selected
  $shipping_type      = $state{'shipping_type'};

  # load billing info 
  $same_billing       = $state{'same_billing'};
  $b_fname            = $state{'b_fname'};
  $b_lname            = $state{'b_lname'};
  $b_address1         = $state{'b_address1'};
  $b_address2         = $state{'b_address2'};
  $b_city             = $state{'b_city'};
  $b_state            = $state{'b_state'};
  $b_zip              = $state{'b_zip'};
  $b_country          = $state{'b_country'};
  $b_homephone        = $state{'b_homephone'};
  $b_workphone        = $state{'b_workphone'};
  $b_fax              = $state{'b_fax'};
  $b_email            = $state{'b_email'};
  $b_company          = $state{'b_company'};

  # load the totals

  $sub_total          = $state{'sub_total'};
  $tax_total          = $state{'tax_total'};
  $ship_total         = $state{'ship_total'};
  $discount_total     = $state{'discount_total'};
  $grand_total        = $state{'grand_total'};

  # if you ever wanted to load payment info, here's where you'd do it.
  # just store $pf1, $pf2, $pf3, $pf4...

}

#
# -----------------------------------------------------------------------------
#

sub store_basket {
  my(%state);

  my($hash_ref) = STATE::state_get($session_id);
  %state = ((ref($hash_ref) eq "HASH") ? %$hash_ref : ());

  # store the items in the basket

  $state{'name'}=join("\0",@name);
  $state{'price'}=join("\0",@price);
  $state{'weight'}=join("\0",@weight);
  $state{'volume'}=join("\0",@volume);
  $state{'qty'}=join("\0",@qty);
  $state{'discount'}=join("\0",@discount);
  $state{'total'}=join("\0",@total);
  $state{'ids'}=join("\0",@ids);
  $state{'uniqueids'}=join("\0",@uniqueids);
  $state{'skus'}=join("\0",@skus);
  $state{'stockids'}=join("\0",@stockids);
  $state{'mixids'}=join("\0",@mixids);
  $state{'cellids'}=join("\0",@cellids);
  $state{'widget_text'}=join("\0",@widget_text);
  $state{'widget_type'}=join("\0",@widget_type);
  $state{'widget_choice'}=join("\0",@widget_choice);

  # store miscellaneous

  $state{'curr_unique_seq'}    = $curr_unique_seq;
  $state{'referer'}            = $referer;
  $state{'yesdiscount'}        = $yesdiscount;

  # store basket options

  $state{'always_show_basket'} = $always_show_basket;
  $state{'gencomments'}        = $gencomments;
  $state{'use_security'}       = $use_security;

  # store shipping address

  $state{'s_fname'}            = $s_fname;
  $state{'s_lname'}            = $s_lname;
  $state{'s_address1'}         = $s_address1;
  $state{'s_address2'}         = $s_address2;
  $state{'s_city'}             = $s_city;
  $state{'s_state'}            = $s_state;
  $state{'s_zip'}              = $s_zip;
  $state{'s_country'}          = $s_country;

  # store payment type selected

  $state{'payment_type'}       = $payment_type;

  # store tax type selected

  $state{'tax_type'}           = $tax_type;

  # store coupon or certificate code

  $state{'coupon_code'}        = $coupon_code;

  # store shipping type selected

  $state{'shipping_type'}      = $shipping_type;

  # store billing demographic info 

  $state{'same_billing'}       = $same_billing;
  $state{'b_fname'}            = $b_fname;
  $state{'b_lname'}            = $b_lname;
  $state{'b_address1'}         = $b_address1;
  $state{'b_address2'}         = $b_address2;
  $state{'b_city'}             = $b_city;
  $state{'b_state'}            = $b_state;
  $state{'b_zip'}              = $b_zip;
  $state{'b_country'}          = $b_country;
  $state{'b_homephone'}        = $b_homephone;
  $state{'b_workphone'}        = $b_workphone;
  $state{'b_fax'}              = $b_fax;
  $state{'b_email'}            = $b_email;
  $state{'b_company'}          = $b_company;

  # store the totals

  $state{'sub_total'}          = $sub_total;
  $state{'tax_total'}          = $tax_total;
  $state{'ship_total'}         = $ship_total;
  $state{'discount_total'}     = $discount_total;
  $state{'grand_total'}        = $grand_total;

  # store payment info here

  # Save the state, either via the STATE module, or Embperl's tied %udat
  STATE::state_put($session_id, \%state);

}

#
# -----------------------------------------------------------------------------
#

sub is_basket_empty {
  # just check to see if the arrays contain anything.
  if ($#ids < 0) {
    return 1;
  } else {
    return 0;
  }
}

#
# -----------------------------------------------------------------------------
#

sub should_we_show_basket_every_time {
  if ($always_show_basket eq "on") {
    return 1;
  } else {
    return 0;
  }
}

#
# -----------------------------------------------------------------------------
#

sub neatly_dispose_of_basket {

  # erase arrays

  @name          = ();
  @price         = ();
  @weight        = ();
  @volume        = ();
  @qty           = ();
  @discount      = ();
  @total         = ();
  @ids           = ();
  @uniqueids     = ();
  @skus          = ();
  @stockids      = ();
  @mixids        = ();
  @cellids       = ();
  @widget_text   = ();
  @widget_type   = ();
  @widget_choice = ();

  # verify all of the types they've selected; fix if needed.  This will
  # provide at least one way that user can 'reset' their basket if they
  # need to.  This could be a problem in the case that an administrator
  # is editing the available options at the same time the person is
  # shopping (specifically, if he's deleting all the options).

  type_verify_all();

  # note that we _don't_ reset the curr_unique_seq to 0.  That's just
  # to avoid potential problems if the user decides to use his back
  # button.     ...call me paranoid, I guess.
  # $curr_unique_seq=0;

  store_basket();
}

#
# -----------------------------------------------------------------------------
#

sub add_item_to_basket {
  my($label, $sku, $price, $weight, $volume, $atp, $rt, $drq);
  my($stockid, $mixid, $cellid);
  my(@pnums, $pnum, $discount, $new_qty, $all_qty);

  # check to see that product id exists
  if (!defined($fdata{'pnum'})) {
    add_error('You must specify a product to add to your shopping basket!');
    return 0;
  }

  # what is the global quantity specifier?
  $all_qty = $fdata{'all_qty'};
  $all_qty =~ s/[^0-9]//g;
  if (!defined($all_qty) || ($all_qty == 0) || ($all_qty eq "")) {
    $all_qty = 1;
  }


  # split on multiple items
  @pnums = split(/\t/,$fdata{'pnum'});

  # look up each item
  foreach $pnum (@pnums) {

    # you can specify an optional number of each item to add,
    # either on a per-item basis (by defining the $id_qty field),
    # or on a group basis (by defining the all_qty field).

    $new_qty = $fdata{"${pnum}_qty"};
    $new_qty =~ s/[^0-9]//g;

    if (!defined($new_qty) || ($new_qty == 0) || ($new_qty eq "")) {
      $new_qty = $all_qty;
    }

    # clean up pnum
    if (!ITEM::item_verify_id(\$pnum)) {
      add_error('That item does not exist!');
      return 0;
    }

    ($label, $sku, $price, $weight, $volume, $atp, $rt, $drq,
     $stockid,$mixid,$cellid) = ITEM::item_get_info($pnum);

    # XXX here, you might want to check to see if the item is already
    # in their basket.  If it is, you could just increment the quantity.
    # However, that doesn't quite work for situations where a person
    # really wants two separate entries in the shopping basket.  For example,
    # they may want two different variations on a product -- they order
    # the same golf club twice, and they want one to be men's right handed,
    # and the other to be ladies left handed.  They _don't_ want both
    # to have the same options, which is what would happen if you didn't
    # create a separate entry.
    #
    # So, if you need to, you could do something like this:
    #
    # $found=0;
    # for ($i=0; $i<$#name+1; $i++) {
    #   if ($ids[$i]==$pnum) {
    #     $qty[$i]++;
    #     # DDD discounting hook: update discount here, based on quantity
    #     $found=1;
    #   }
    # }
    # if (!$found) { # modify arrays... }
    #

    push(@ids,$pnum);
    push(@name,$label);
    push(@price,$price);
    push(@weight,$weight);
    push(@volume,$volume);
    push(@qty,$new_qty);

    # DDD calculate discount for this product
    $discount = discount_calculate_for_new_item();
    push(@discount,$discount);
    push(@total,($price*$new_qty)-$discount);

    push(@uniqueids,$curr_unique_seq);
    $curr_unique_seq++;

    push(@skus,$sku);
    push(@stockids,$stockid);
    push(@mixids,$mixid);
    push(@cellids,$cellid);

    my($type,$text) = analyze_widgets($#name);
    push(@widget_text,$text);
    push(@widget_type,$type);
    push(@widget_choice,"");

    # initial inventory allocation
    inventory_allocate_item($pnum,$new_qty);  
  }

  return 1;
}

#
# -----------------------------------------------------------------------------
#
# XXX um, is this totally broken, or what?  It dumps me to my system home page.
sub return_to_shopping {
  my $jed;
  my $url=$referer;

  ($jed,$url) = split(/\/\//,$url,2);
  ($jed,$url) = split(/\//,$url,2);
  $url = "/" . $url;

  MILTON::redirect($url);
}

#
# -----------------------------------------------------------------------------
#

sub add_error {
  $error_messages[$#error_messages + 1] = shift;
}

#
# -----------------------------------------------------------------------------
#

sub check_em {
  my(@items) = @_;
  my($i, $str, $key, $err_occured);

  $err_occured = 0;

  foreach $i (@items) {
    ($str,$key) = split(/:/,$i,2);
    if (!defined($fdata{$key})) {
      add_error("You must enter a value for $str!");    
      $err_occured=1;
    }
  }

  if ($err_occured == 1) {
    return(0);
  } else {
    return(1);
  }
}

#
# -----------------------------------------------------------------------------
#

sub ensure_appropriate_environment {
  # determine whether or not they are authorized to use the shopping basket
  if (!ensure_user_is_authorized_to_shop()) { return 0; }

  # do they have a shopping basket?
  if (!ensure_basket_exists()) { return 0; }

  return 1; 
}

#
# -----------------------------------------------------------------------------
#

sub ensure_basket_exists {
  # do they have a shopping basket?
  # if not, it's a bogus request, or something is wrong with the state engine

  load_basket();
  if (!does_basket_exist()) {
    add_error('You must have a shopping basket to use this function!');
    return 0;
  }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub ensure_user_is_authorized_to_shop {
  # a placeholder function for later expansion
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub determine_referer {
  $referer = undef;

  if (defined($ENV{'HTTP_REFERER'})) {
    $referer = $ENV{'HTTP_REFERER'};
  } else {
      if (!MILTON::offline_mode()) {
	  require Apache;
	  my $r = Apache->request;
	  if(defined($r->header_in("Referer"))) {
	      $referer = $r->header_in("Referer");
	  }
      }
  }

  unless($referer) {
      $referer = KNAR::knar_entry_get('REGEN_OUTPUT_URL');
      $referer .= "/" unless($referer =~ m|/$|);
  }
	  
}

#
# -----------------------------------------------------------------------------
#
# DDD here are the discounting stubs
#

#
# called when a new item is added to the shopping basket
# returns the discount for this item, _not_ a success value
#

sub discount_calculate_for_new_item {
  return 0;
}

#
# called when the quantities in a shopping basket are updated
# (they may or may not have actually changed)
# returns the discount for this item, _not_ a success value
#

sub discount_calculate_for_new_qty {
#
# we might want to do something like the following:
#  (a 10% discount based on quantity)
#
#  for ($i=0; $i<$#name+1; $i++) {
#    if ($qty[$i] >= 100) {
#      $discount[$i]=$price[$i]* 0.1;
#    } else {
#      $discount[$i]=0;
#    }
#  }
  return 0;
}

#
# =============================================================================
#
# SHOPPING-BASKET ACCESSIBLE FUNCTIONS
#
# =============================================================================
#


#
# -----------------------------------------------------------------------------
#
# prints out various useful information for a basket
#

sub line {
  my($var,$i)=@_;

  if ($i > $#name) { return 'ARRAY OUT OF RANGE'; }

  if ($var eq "name")        { return $name[$i]; }
  elsif ($var eq "price")    { return sprintf '%.2f',$price[$i]; }
  elsif ($var eq "weight")   { return sprintf '%.2f',$weight[$i]; }
  elsif ($var eq "volume")   { return sprintf '%.2f',$volume[$i]; }
  elsif ($var eq "qty")      { return $qty[$i]; }
  elsif ($var eq "discount") { return sprintf '%.2f',$discount[$i]; }
  elsif ($var eq "sku")      { return $skus[$i]; }
  elsif ($var eq "total")    { return sprintf '%.2f',$total[$i]; }
  elsif ($var eq "widget")   { return (precheck_widget_text($i)); }
  elsif ($var eq "qtybox")   { 
return "<input type=text size=4 maxlength=4 name=${i}qty value=\"$qty[$i]\">"; 
  }
}

sub grand_total { return(sprintf '%.2f',$grand_total); }

sub sub_total { return(sprintf '%.2f',$sub_total); }

sub tax_total { return(sprintf '%.2f',$tax_total); }

sub ship_total { return(sprintf '%.2f',$ship_total); }

sub payment_type_str {
  if ($payment_type == -1) {
    return '(none defined)';
  } else {
    return(PAYMENT_TYPE::get_label($payment_type));
  }
}

sub payment_type_num {
  return($payment_type);
}

sub payment_type_widgettype {
  my($id) = shift || $payment_type;

  if ($id == -1) {
    return '';
  } else {
    my @ref = PAYMENT_TYPE::get_info($id);
    return $ref[2];
  }
}

sub ship_type_str {
  if ($shipping_type == -1) {
    return '(none defined)';
  } else {
    return(SHIPPING::shippingtype_get_label($shipping_type));
  }
}

sub ship_type_num {
  return($shipping_type);
}

sub tax_type_str {
  if ($tax_type == -1) {
    return '(none defined)';
  } else {
    return(TAX_TYPE::get_label($tax_type));
  }
}

sub tax_type_num {
  return($tax_type);
}

sub order_num {
  return($ordernum);
}

sub order_date {
  return($orderdate);
}

sub gencomments {
  return($gencomments);
}

sub exists_discount {
  return($yesdiscount);
}

sub show_basket_every_time_button {
  if ($always_show_basket eq "on") {
    return "<input type=checkbox checked name=\"always_show_basket\">";
  } else {
    return "<input type=checkbox name=\"always_show_basket\">";
  }
}

sub show_shipping_address_button {
  if ($same_billing eq "on") {
    return "<input type=checkbox checked name=\"same_billing\">";
  } else {
    return "<input type=checkbox name=\"same_billing\">";
  }
}

sub show_use_security_button {
  if ($use_security eq "on") {
    return "<input type=checkbox checked name=\"use_security\">";
  } else {
    return "<input type=checkbox name=\"use_security\">";
  }
}

sub need_billing_address {
  if ($same_billing eq "on") {
    return(1);
  } else {
    return(0);
  }
}

sub need_widget {
  my($item) = shift;

  if ($widget_type[$item]==0) { return 0; }
  else { return 1; }
}

sub end_line {
  return "</tr><tr>";
}

sub total_items {
  return($#name + 1);
}

sub total_weight {
  my($i, $hohum);

  for ($i=0, $hohum = 0; $i < $#ids + 1; $i++) {
      $hohum += ($weight[$i] * $qty[$i]);
  }

  ## make sure this value is clean
  SEC::require_and_clean('total_weight', 'FLOAT', \$hohum);

  return($hohum);
}

sub total_volume {
  my($i, $hohum);

  for ($i=0, $hohum = 0; $i < $#ids + 1; $i++) {
    $hohum += ($volume[$i] * $qty[$i]);
  }

  ## make sure this value is clean
  SEC::require_and_clean('total_volume', 'FLOAT', \$hohum);

  return($hohum);
}

sub total_quantity {
  my($i, $hohum);

  for ($i=0, $hohum = 0; $i < $#ids + 1; $i++) {
    $hohum += $qty[$i];
  }

  ## make sure this value is clean
  SEC::require_and_clean('total_quantity', 'INT', \$hohum);

  return($hohum);
}

#
# -----------------------------------------------------------------------------
#

sub shippingtype_list_all_calc {
  my $i;

  # 0=id, 1=label, 2=avail, 3=calculated, 4=algorithm, 5=min, 6=extra
  my($ref) = SHIPPING::shippingtype_list_all_available();

  for ($i = 0; $i < $#{$ref} + 1; $i++) {
    $ref->[$i][7] = SHIPPING::shipping_calculate_charge($ref->[$i][0]);
  }
  return($ref);
}

#
# -----------------------------------------------------------------------------
#

sub paymenttype_list_all {
  return(PAYMENT_TYPE::list_all_available());
}

#
# -----------------------------------------------------------------------------
#
# Type 0: no widgets are needed
#
# Type 1: the item uses a mix, and there are less than $WIDGET_THRESHOLD
#         total variations.  Just return a single pull-down menu
# 
# Type 2: the item uses a mix, and there is a fully populated grid of
#         multiple options.  Return a set of pull down menus, one for
#         each range.
#
# Type 3: Complicated / special product.  Return a wizard button.
#         NOT YET IMPLEMENTED
#
# Type 4: The product has no mix, but has a simple pull-down interface
#         snippet associated with it.  Call the analyzer and return.
#

sub analyze_widgets {
  my($item) = shift;
  my($tmpstr, $typenum);

  # first of all, this item may have a simple options select box
  # associated with it (snippet type 'Basket Options').  If so, and the
  # mixid is 0, use that.

  $tmpstr = SNIPPET::snip_get_byname($THING::item, $ids[$item],
                                     'Basket Options');

  if (defined($tmpstr) && $tmpstr ne "") {
    return (4, analyze_simple_options($item,$tmpstr));
  }
  $tmpstr = "";

  # um.  no mix? no widgets.
  if ($mixids[$item] == 0) { return (0,"NO WIDGETS NEEDED"); }

  ($typenum, $tmpstr) = ITEM::item_widget_analysis_get($ids[$item]);

  $tmpstr =~ s/%%/$uniqueids[$item]/g;

  return($typenum, $tmpstr);
}

#
# -----------------------------------------------------------------------------
#
# each menu item is on a new line
# multiple menus are made by inserting two newlines
# ex: menu item 1.1
#     menu item 1.2
#     menu item 1.3
#
#     menu item 2.1
#     menu item 2.2
#     menu item 2.3
# 

sub analyze_simple_options {
  my($item,$text) = @_;
  my(@menus,@values,$result,$i,$j);

  $text =~ s/\r//g;
  @menus = split(/\n\n/,$text);
  for ($i=0; $i<$#menus+1; $i++) {
    $result .= "<select name=widget_$uniqueids[$item]_$i>\n";

    @values = split(/\n/,$menus[$i]);
    for ($j=0; $j<$#values+1;$j++) {
      $result .= "<option value=\"${i}_${j}\"> $values[$j]</option>\n";
    }

    $result .= "</select>\n";
  }

  return($result);
}

#
# -----------------------------------------------------------------------------
#
# For each item, we store a string representing its selection boxes.  The
# string is stored in such a way that it is guaranteed _not_ to conflict
# with any other set of selection boxes, even if the item has exactly the
# same options.  That's why we keep a "unique id sequence" -- so that we
# can identify each shopping basket item uniquely.  The string is stored
# in a state where _no_ item is selected; this function checks the appropriate
# values in the string, in preparation for displaying the options to the user.
#

sub precheck_widget_text {
  my($anum) = shift;  # this is the item number
  my($nodes, $i, $tmptext, @tmpa);

  if ($widget_type[$anum] == 1) {
    $tmptext = $widget_text[$anum];
    $i = $stockids[$anum];
    $tmptext =~ s/value="$i"/value="$i" selected/g;
    return($tmptext);


  } elsif ($widget_type[$anum] == 2) {
    $nodes = ITEM::node_list_cell($cellids[$anum]);
    $tmptext = $widget_text[$anum];

    foreach $i (@$nodes) {
      $tmptext =~ s/value="$i"/value="$i" selected/g;
    }

    return($tmptext);

  } elsif ($widget_type[$anum] == 4) {
    $tmptext = $widget_text[$anum];
    @tmpa = split(/ /,$widget_choice[$anum]);

    foreach $i (@tmpa) {
      $tmptext =~ s/value="$i"/value="$i" selected/g;
    }

    return($tmptext);

  } else {
    return($widget_text[$anum]);
  }
}

#
# -----------------------------------------------------------------------------
# This function creates a string representation of the product options
# selected.  It's used to log the information with the order, so that the
# merchant knows what variation was requested.
#

sub translate_widget_to_text {
  my($item) = shift;

  my(@tmpa,@grid,@menus,@values);
  my($result,$text,$i,$j,$x,$y);

  if ($widget_type[$item]==0) { 
    return ' '; 

  } elsif ($widget_type[$item]==1) {
    return ITEM::cell_get_range_labels($cellids[$item]);

  } elsif ($widget_type[$item]==2) {
    return ITEM::cell_get_range_labels($cellids[$item]);

  } elsif ($widget_type[$item]==4) {

    # first, array-ify the menus

    $text = SNIPPET::snip_get_byname($THING::item, $ids[$item],
                                     'Basket Options');

    $text =~ s/\r//g;
    @menus = split(/\n\n/,$text);
    for ($i=0; $i<$#menus+1; $i++) {
      @values = split(/\n/,$menus[$i]);
      for ($j=0; $j<$#values+1;$j++) {
        $grid[$i][$j] = $values[$j];
      }
    }

    @tmpa=split(/ /,$widget_choice[$item]);

    foreach $j (@tmpa) {
      ($x,$y) = split(/_/,$j);
      $result .= $grid[$x][$y];
      $result .= " -- ";
    }
    $result =~ s/ -- $//;
    return($result);
  }
  return ' '; 
}

#
# -----------------------------------------------------------------------------
#
# This function actually dispatches an order.
#

sub dispatch_order {
  my($i, $ol_id, $tmptext);
  my($ship_id, $bill_id, $c_id, $order_id);
  my($affiliate_id, $aentrytime);
  my($foo, $dirty_email, $newname_ref, $newskus_ref, $s_newname);

  # enclose the whole thing in a transaction, so that we don't get loose
  # data floating around if something fails

  DBLIB::db_transact_begin();

  # inventory hook -- final allocation / allocation reconciliation

  inventory_final_allocation();


  # start by escaping all text
  # note that this ruins the data...

  $dirty_email = $b_email;  # this is for the receipt.
  ($newname_ref, $newskus_ref) = clean_all_text();


  #
  # create shipping and billing addresses and customer
  #

  if (need_billing_address()) {
    $s_newname = $s_fname . " " . $s_lname;
    $ship_id = ADDRESS::create($s_newname, $s_address1, $s_address2, $s_city,
			       $s_state, $s_zip, $s_country);
    $s_newname = $b_fname . " " . $b_lname;
    $bill_id = ADDRESS::create($s_newname, $b_address1, $b_address2, $b_city,
			       $b_state, $b_zip, $b_country);
    $c_id = CUSTOMER::create($s_newname, $b_company, $b_email, $b_homephone, 
                             $b_workphone);
    
  } else {
    $s_newname = $s_fname . " " . $s_lname;
    $ship_id = ADDRESS::create($s_newname, $s_address1, $s_address2, $s_city,
			       $s_state, $s_zip, $s_country);
    $bill_id = $ship_id;
    $c_id = CUSTOMER::create($s_newname, $b_company, $b_email, $b_homephone, 
                             $b_workphone);

  }

  CUSTOMER::address_assoc($c_id, $ship_id, "shipping");
  CUSTOMER::address_assoc($c_id, $bill_id, "billing");


  #
  # implement the affiliates thing
  #

  my($hash_ref) = STATE::state_get($session_id);
  my(%state) = ((ref($hash_ref) eq "HASH") ? %$hash_ref : ());
  my($affiliate_cookie) = $state{'affiliate'};

  if ($affiliate_cookie) {
    ($affiliate_id,$aentrytime) = split(/_/,$affiliate_cookie);

    if (SEC::is_int($affiliate_id)) {
      my($alabel,$durl,$aurl,$ahits,$asales,$adollars,$aexpire,$areset)
         = AFFILIATES::affiliates_get_info($affiliate_id);

      if ($aexpire != 0 && (time - $aentrytime > $aexpire)) { 
        # expired affiliate; set it to 0.
        $affiliate_id = 0; 
      } else {
        AFFILIATES::affiliates_increment_sales($affiliate_id, $grand_total);
      }
    }

  } else {
    $affiliate_id = 0;
  }

  #
  #  create the actual order
  #

  my($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime(time);
  $year += 1900;
  $mon++;

  my($order_log_date_str)
     = DBLIB::date_string_make($sec,$min,$hour,$mday,$mon,$year);

  $order_id = ORDER::create( $c_id,
			     $grand_total,
			     $sub_total,
			     $tax_total,
			     $ship_total,
			     $gencomments,
			     $affiliate_id,
			     "null",
			     "no",
			     $order_log_date_str );

  # create shipment
  my $shipment_id = SHIPMENT::create(ship_type_str(), $ship_total, $ship_id);

  # create payment
  # XXX date fields need to go into the db unquoted
  my $payment_id = PAYMENT::create(payment_type_str(), $pf1, $pf2, $pf3, $pf4,
                                   'null', $grand_total);
  ORDER::payment_assoc($order_id, $payment_id);

  $ordernum = $order_id;
  $orderdate = "$hour:$min $mday/$mon/$year";

  # create orderlines

  for ($i = 0; $i < $#name + 1; $i++) {
    # get the options that the specified with each item, and insert it
    # along with the item.

    $tmptext = translate_widget_to_text($i);
    $tmptext = DBLIB::db_string_clean($tmptext);
    $foo = DBLIB::db_string_clean($name[$i]);

    $ol_id = ORDERLINE::create($order_id, ${$newname_ref}[$i], 
                               ${$newskus_ref}[$i], $price[$i], $qty[$i], 
                               $total[$i], $weight[$i], $volume[$i], 
                               $tmptext);

    # associate shipment
    ORDERLINE::shipment_assoc($ol_id, $shipment_id);

    # create discount--if any
    if (defined $discount[$i]) {
      if ($discount[$i] > 0) {
        my $discount_id = DISCOUNT::create("null", $discount[$i]);
        ORDERLINE::discount_assoc($ol_id, $discount_id);
      }
    }
  }

  # log to the cat_order_stat table

  $foo = 0;
  if ($ship_total != -1) { $foo = $ship_total; }

  ORDERSTATS::orderstats_add_entry($order_id,$affiliate_id,
				   $order_log_date_str, 
                                   total_quantity(),$grand_total,
                                   $foo,$tax_total);

  DBLIB::db_transact_end();

  # email receipt

  email_receipt($dirty_email);

  # fax receipt

  # autobiller

  return 1;

}

#
# -----------------------------------------------------------------------------
#
# Yowsa!  This sure doesn't look very pretty.  Hmmmm.
#

sub email_receipt {
  my($to_address) = shift;

  my($i, $from_address, $mail_text);
  my($subject, $mail_header1, $mail_header2, $mail_footer);
  my($my_ship_text);

  $from_address = KNAR::knar_entry_get('BASKET_EMAIL_FROM_ADDRESS');
  $subject      = KNAR::knar_entry_get('BASKET_EMAIL_SUBJECT');
  $mail_header1 = KNAR::knar_entry_get('BASKET_EMAIL_HEADER1');
  $mail_header2 = KNAR::knar_entry_get('BASKET_EMAIL_HEADER2');
  $mail_footer  = KNAR::knar_entry_get('BASKET_EMAIL_FOOTER');

  $sub_total_txt   = KNAR::knar_entry_get('BASKET_EMAIL_SUB_TOTAL_TEXT');
  $tax_total_txt   = KNAR::knar_entry_get('BASKET_EMAIL_TAX_TOTAL_TEXT');
  $ship_total_txt  = KNAR::knar_entry_get('BASKET_EMAIL_SHIP_TOTAL_TEXT');
  $grand_total_txt = KNAR::knar_entry_get('BASKET_EMAIL_GRAND_TOTAL_TEXT');

  $ordernum_txt    = KNAR::knar_entry_get('BASKET_EMAIL_ORDERNUM_TEXT');
  $orderdate_txt   = KNAR::knar_entry_get('BASKET_EMAIL_ORDERDATE_TEXT');

  $call_for_shipping_txt = KNAR::knar_entry_get('BASKET_EMAIL_CALLSHIP_TEXT');


  my($RECEIPT_ORDER_LINE) =
"@<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< @<<<<<<< @<<<<< @<<<<<<< @<<<<<<<
";

  my($RECEIPT_TOTAL_LINE) = 
"                              @>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> @<<<<<<<
";


  #
  # assemble the message text
  #

  $mail_text = $mail_header1;
  $mail_text .= "        ${ordernum_txt} $ordernum\n";
  $mail_text .= "        ${orderdate_txt} $orderdate\n\n";
  $mail_text .= $mail_header2;

  for ($i=0; $i<$#name+1; $i++) {
    $mail_text .= MILTON::swrite($RECEIPT_ORDER_LINE,
                                 $name[$i],sprintf("%.2f",$price[$i]),$qty[$i],
                                 sprintf("%.2f",$discount[$i]),
                                 sprintf("%.2f",$total[$i]));
  }

  $mail_text .= "\n";
  $mail_text .= MILTON::swrite($RECEIPT_TOTAL_LINE,$sub_total_txt,
                               sprintf("%.2f",$sub_total));
  $mail_text .= MILTON::swrite($RECEIPT_TOTAL_LINE,$tax_total_txt,
                               sprintf("%.2f",$tax_total));

  if ($ship_total == -1) {
    $my_ship_text = $call_for_shipping_txt;
  } else {
    $my_ship_text = sprintf("%.2f",$ship_total);
  }

  $mail_text .= MILTON::swrite($RECEIPT_TOTAL_LINE,
                               $ship_total_txt . " " . ship_type_str() . ":",
                               $my_ship_text);

  $mail_text .= MILTON::swrite($RECEIPT_TOTAL_LINE,$grand_total_txt,
                               sprintf("%.2f",$grand_total));

  $mail_text .= $mail_footer;

  # mail the thing!

  SENDMAIL::sendmail($to_address, $from_address, $subject, $mail_text);
}

#
# -----------------------------------------------------------------------------
#

sub clean_all_text {
  my($i, @newname, @newskus);

  if (!defined($gencomments) || $gencomments eq "") {
    $gencomments=" ";
  }
  $gencomments = DBLIB::db_string_clean($gencomments);

  $pf1 = DBLIB::db_string_clean($pf1);
  $pf2 = DBLIB::db_string_clean($pf2);
  $pf3 = DBLIB::db_string_clean($pf3);
  # $pf4 might have valid 's in it, if it contains
  # an Oracle TO_DATE statement.  Don't escape it.
  # That would be bad.  
  # XXX is pf4 a user-modifyable variable?  Possible security problem.

  ## untaint $pf4 
  SEC::untaint_ref(\$pf4);## if $DBLIB::taint_check;

  $s_fname     = DBLIB::db_string_clean($s_fname);
  $s_lname     = DBLIB::db_string_clean($s_lname);
  $s_address1  = DBLIB::db_string_clean($s_address1);
  $s_address2  = DBLIB::db_string_clean($s_address2);
  $s_city      = DBLIB::db_string_clean($s_city);
  $s_state     = DBLIB::db_string_clean($s_state);
  $s_zip       = DBLIB::db_string_clean($s_zip);
  $s_country   = DBLIB::db_string_clean($s_country);

  $b_fname     = DBLIB::db_string_clean($b_fname);
  $b_lname     = DBLIB::db_string_clean($b_lname);
  $b_address1  = DBLIB::db_string_clean($b_address1);
  $b_address2  = DBLIB::db_string_clean($b_address2);
  $b_city      = DBLIB::db_string_clean($b_city);
  $b_state     = DBLIB::db_string_clean($b_state);
  $b_zip       = DBLIB::db_string_clean($b_zip);
  $b_country   = DBLIB::db_string_clean($b_country);

  $b_company   = DBLIB::db_string_clean($b_company);
  $b_email     = DBLIB::db_string_clean($b_email);
  $b_homephone = DBLIB::db_string_clean($b_homephone);
  $b_workphone = DBLIB::db_string_clean($b_workphone);

  for ($i = 0; $i < $#name + 1; $i++) {
    $newname[$i] = DBLIB::db_string_clean($name[$i]);
    $newskus[$i] = DBLIB::db_string_clean($skus[$i]);
  }

  return(\@newname, \@newskus);
}

#
# -----------------------------------------------------------------------------
#

1;
