# CONTAINER.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

package CONTAINER;

use strict;

require DBLIB;
require ITEM;
require PAGE;
require GROUP;
require THING;

# 
# AVAILABLE FUNCTIONS AND VARIABLES DEFINED IN THIS PACKAGE:
#
#   &container_get_all_type
#   &container_children_cid
#   &container_children_type
#   &container_children_all
#   &container_children_all_cid
#   &container_get_mid_level_item
#   &container_get_id
#   &container_create
#   &container_find
#   &container_parent
#   &container_thing_info
#   &container_info
#   &container_thing_name
#   &container_swap_with_previous
#   &container_swap_with_next
#   &container_set_seq_no
#   &container_move_children
#   &container_move_some_children
#   &container_get_next_id
#   &container_delete
#   &container_delete_item
#   &container_delete_children
#   &container_delete_child_item
#   &container_do_delete
#   &container_get_heirarchy
#   &container_get_lineage
#   &container_depth_trav
#   &container_verify_id

# for convenience
use vars qw(@thing @thing_field);

@thing = @THING::thing;

$thing_field[0] = 'foo!';
$thing_field[$THING::item] = 'item_id';
$thing_field[$THING::page] = 'page_id';
$thing_field[$THING::group] = 'group_id';


## traversal vars

use vars qw($container_t_table $container_t_id_field $container_t_parent_field $container_t_visit $container_t_order $container_t_return_ref);


#
# =============================================================================
#
# CONTAINER API
#
# =============================================================================
#

sub container_get_all_type {
    my($type) = @_;

    my($rows) = DBLIB::db_fetchall_arrayref( "
      select container_id, p_container_id, seq_no, $thing_field[$type]
      from container
      where c_type = '$type'");

    return $rows;
}

#
# -----------------------------------------------------------------------------
# Returns a listref of container_ids to all the children of the specified
# container, sorted by seq_no
sub container_children_cid {
    my($p_container_id) = @_;

    my($container_ids) = DBLIB::db_fetchcol_arrayref("
        select container_id
        from container
        where p_container_id = ?
        order by seq_no", 0, $p_container_id, 1);

    return $container_ids;
}

# 
# -----------------------------------------------------------------------------
# Wrapper around container_children_all, takes a cid instead of kind and thing 
# id
sub container_children_all_cid {
    my($cid) = @_;

    my($kind, $thing_id) = container_thing_info($cid);
    return container_children_all($kind, $thing_id);
}

#
# -----------------------------------------------------------------------------
# Returns a listref of container_ids to all the children of the specified
# thing, wherever they are in the container system, sorted by seq_no
sub container_children_all {
    my($kind, $thing_id) = @_;

    my($container_ids) = DBLIB::db_fetchcol_arrayref("
        select children.container_id
        from container parent, container children
        where parent.c_type = '$kind'
        and parent.$thing_field[$kind] = $thing_id
        and children.p_container_id = parent.container_id
        order by children.seq_no");

    return $container_ids;
}


#
# -----------------------------------------------------------------------------
# Like list_children, but only returns children of the specified kind
sub container_children_type {
    my($kind, $p_container_id) = @_;

    my($thing_ids) = DBLIB::db_fetchcol_arrayref("
        select $thing_field[$kind]
        from container 
        where c_type = '$kind'
        and p_container_id = '$p_container_id'
        order by seq_no");

    return $thing_ids;
}

#
# -----------------------------------------------------------------------------
# Given the parent container id, kind and id, returns the container id(s)
# of matching child, if it exists.  Returns a false value if not.
sub container_get_id {
    my($pcid, $kind, $id) = @_;

    my($cids) = DBLIB::db_fetchcol_arrayref("
        select container_id
	from container
        where p_container_id=? 
        and c_type=?
        and $thing_field[$kind]=?", 0,
        $pcid, 1, $kind, 1, $id, 1);

    $cids = [] if(ref($cids) ne 'ARRAY');
	
    return wantarray ? @$cids : $cids->[0];
}

#
# -----------------------------------------------------------------------------
#
#   takes: an item id, a container id
# returns: the container id of the container connecting the two, or
#          0 if no middle container is found
# example: call with itemid=1000, ppid=2 (2 is the manufacturer
#          container); it would return which mfr contains item
#          1000.
#
#   notes: there has got to be a better way to do this.
#  -J There is, in fact, a better way if you don't mind using subqueries.  
#  See the __old version of the function for the original method without subqueries.
#
sub container_get_mid_level_item {
  my($itemid, $ppid) = @_;

  # The idea is this:
  #  [ container_id = $ppid ]
  #      ^
  #      |
  #  [ p_container_id, container_id = ??? ]
  #      ^
  #      |
  #  [ p_container_id, item_id = $itemid ]

  my($cid, $pcid) = DBLIB::db_fetchrow_array(
		"select container_id, p_container_id 
		 from container parent
		 where container_id in (
			select p_container_id 
			from container
			where 
			p_container_id = parent.container_id 
			and parent.p_container_id = $ppid 
			and item_id = $itemid)");
 
  return 0 unless($pcid == $ppid);
  return($cid);
}

#
# -----------------------------------------------------------------------------
# Passed the parent's cid, a kind and id, creates a new container.
# Returns the cid of the new container
sub container_create {
  my($parent,$kind,$link)=@_;
  my($iid,$pid,$gid);  # Item ID, Page ID, Group ID

  if ($kind==$THING::item) {
    $iid=$link; $pid=0; $gid=0;
  } elsif ($kind==$THING::page) {
    $iid=0; $pid=$link; $gid=0;
  } elsif ($kind==$THING::group) {
    $iid=0; $pid=0; $gid=$link;
  }

  # turn off AutoCommit
  DBLIB::db_transact_begin();

  # get the next sequence number
  # XXX I'm not convinced that this is the way to do it.  Is this safe???

  my($seq_no)=DBLIB::db_fetchrow_array(
	      "select max(seq_no)
               from container
               where p_container_id='$parent'");

  if (!defined($seq_no) || $seq_no eq "") {
    $seq_no=0;
  } else {
    $seq_no++;
  }

  # get the next container_id

  my($nextid)=&container_get_next_id();

  # and insert the container

  DBLIB::db_do("insert into container 
           (container_id,p_container_id,seq_no,c_type,
           item_id,page_id,group_id)
           values
           ('$nextid','$parent','$seq_no','$kind','$iid','$pid','$gid')");

  DBLIB::db_transact_end();

  return $nextid;
}

#
#------------------------------------------------------------------------------
# Given a thing type and thing id, return a list of the containers that
# represent that thing
sub container_find {
    my($type, $thing_id) = @_;

    my($rows) = DBLIB::db_fetchcol_arrayref("
      select container_id
      from container
      where c_type = '$type' 
      and $thing_field[$type] = '$thing_id'");

    return $rows;
}

#
#------------------------------------------------------------------------------
# Returns the parent container for the specified container
sub container_parent {
    my($c_id) = @_;

    my($parent) = DBLIB::db_fetchrow_array("
        select p_container_id
        from container
        where container_id = '$c_id'");

    return $parent;
}

      
#
# -----------------------------------------------------------------------------
# Returns the item, page or group id for the container
sub container_thing_info {
    my($cid) = @_;

    my($p_cid, $seq, $type, $thing_id) = container_info($cid);

    return ($type, $thing_id);
}

#
# -----------------------------------------------------------------------------
# returns parent container id, sequence #, container type, and thing id
sub container_info {
    my($cid) = @_;

    my($p_cid, $seq_no, $type, $iid, $pid, $gid) = DBLIB::db_fetchrow_array("
        select p_container_id, seq_no, 
               c_type, item_id, page_id, group_id
        from container 
        where container_id = ?", 0, $cid, 1);

    my($thing_id);
    if($type eq $THING::item) {
	$thing_id = $iid;
    } elsif($type eq $THING::page) {
	$thing_id = $pid;
    } elsif($type eq $THING::group) {
	$thing_id = $gid;
    } else {
        MILTON::fatal_error("reality check bounced in container_info($type)");
    }

    return ($p_cid, $seq_no, $type, $thing_id);
}

#
# -----------------------------------------------------------------------------
# Given a cid, returns that thing's name
sub container_thing_name {
    my($cid) = @_;

    my($p_cid, $seq_no, $type, $thing_id) = container_info($cid);

    if($type eq $THING::item) {
	return ITEM::item_get_label($thing_id);
    } elsif($type eq $THING::page) {
	return PAGE::page_get_label($thing_id);
    } elsif($type eq $THING::group) {
	return GROUP::group_get_label($thing_id);
    } else {
      MILTON::fatal_error("Reality check bounced in container_thing_name.");
    }
}

#
# neither of the following routines actually does any swapping.  It just
# resequences the thing.
#
sub container_swap_with_previous {
  my($cid)=@_;

  my($pcid, $seq_no) = DBLIB::db_fetchrow_array(
                "select p_container_id,seq_no 
                 from container
                 where container_id=$cid");

  # don't let them swap if they're the first item!

  if ($seq_no == 0) {
    return;
  }

  # update the previous item

  my($orig_seq_no)=$seq_no;
  $seq_no--;

  DBLIB::db_do("update container
             set seq_no = $orig_seq_no
             where seq_no = $seq_no
             and p_container_id=$pcid");

  # update the original item

  DBLIB::db_do("update container
             set seq_no = $seq_no
             where container_id=$cid");

}

#
# -----------------------------------------------------------------------------
#

sub container_swap_with_next {
  my($cid)=@_;

  # get the container's information

  my($pcid, $seq_no) = DBLIB::db_fetchrow_array(
		"select p_container_id,seq_no 
                 from container
                 where container_id=$cid");

  # now get the maximum sequence number

  my($maxseq) = DBLIB::db_fetchrow_array(
             "select max(seq_no)
             from container
             where p_container_id=$pcid");

  # don't let them swap if they're the last element!

  if ($seq_no == $maxseq) {
    return;
  }

  # update the previous item

  my($orig_seq_no)=$seq_no;
  $seq_no++;

  DBLIB::db_do("update container
             set seq_no = $orig_seq_no
             where seq_no = $seq_no
             and p_container_id=$pcid");

  # update the original item

  DBLIB::db_do("update container
             set seq_no = $seq_no
             where container_id=$cid");
}

#
# -----------------------------------------------------------------------------
#

sub container_set_seq_no {
  my($cid, $seqid) = @_;

  DBLIB::db_do("update container 
                set seq_no = '$seqid'
                where container_id = '$cid'");
}

#
# -----------------------------------------------------------------------------
#
# Takes all of the children of a given container and makes them children
# of a different container
#
#   takes: old container id, new container id
# returns: nothing
#
sub container_move_children {
  my($oldpcid,$newpcid)=@_;

  DBLIB::db_do("update container
               set p_container_id = '$newpcid'
               where p_container_id = '$oldpcid'");
}

#
# -----------------------------------------------------------------------------
#
# Takes all of the children of a given container and makes them children
# of a different container
#
#   takes: old container id, new container id
# returns: nothing
#
sub container_move_some_children {
  my($newpcid, @cids) = @_;

  DBLIB::db_do("update container
               set p_container_id = '$newpcid'
               where container_id in ('" . join("','",@cids) . "')");
}

#
# -----------------------------------------------------------------------------
#
sub container_get_next_id {
  my($nextid)=DBLIB::seq_next_get('container_sq');
  return($nextid);
}

#
# -----------------------------------------------------------------------------

###################################################################
# Tree-traversal related code -J
# 

# Work with trees made from tables.
# 9 July 199
# Jason E. Holt
# for Akopia, corp.

#
# -----------------------------------------------------------------------------
# Delete a container and resequence its siblings
sub container_delete {
    my($cid) = @_;

    MILTON::fatal_error("Attempt to delete invalid container $cid") unless $cid;

    DBLIB::db_transact_begin();
    my($p_id, $seq_no) = DBLIB::db_fetchrow_array(
        "select p_container_id, seq_no 
         from container
         where container_id = '$cid'");

    # delete cid and all children
    CONTAINER::container_do_delete($cid);

    DBLIB::db_do(
        "update container
         set seq_no = seq_no-1
         where p_container_id = '$p_id'
         and seq_no > '$seq_no'");    

    DBLIB::db_transact_end();
}

#
# -----------------------------------------------------------------------------
# Delete all containers matching the given item_id
sub container_delete_item {
    my($iid) = @_;

    my($cids) = DBLIB::db_fetchall_arrayref("
        select container_id 
        from container
        where item_id = '$iid'");

    my($cid);

    foreach $cid (@$cids) {
	CONTAINER::container_delete($cid->[0]);
    }
}

#
# -----------------------------------------------------------------------------
sub container_delete_children {
    my($cid) = @_;

    my($children) = DBLIB::db_fetchall_arrayref("
        select container_id 
        from container where
        p_container_id = '$cid'");

    my($child);
    for(@$children) {
	CONTAINER::container_do_delete($child->[0]);
    }
}

#
# -----------------------------------------------------------------------------
# Given parent id and item_id, delete the first corresponding item and 
# all its children
sub container_delete_child_item {
    my($pid, $iid) = @_;

    my($cid) = DBLIB::fetchrow_array("
        select container_id
        from container 
        where item_id = '$iid' and
              p_container_id = '$pid'");

    CONTAINER::container_delete($cid);
}

#
# -----------------------------------------------------------------------------
# Lower level delete, doesn't resequence
sub container_do_delete {
    my($cid) = @_;

    # Traverse in postorder, starting from (rather, ending with)
    # $cid, deleting everything.
    CONTAINER::container_depth_trav(
	  "container", "container_id", "p_container_id", $cid,
		    \&container_visit_delete, "POST");

}


#
# -----------------------------------------------------------------------------
#

# Pass a container id, get back a reference to a list of array references. 
# Each of those arrays have the level of the node, followed by all the columns
# of the array
sub container_get_heirarchy {
  my($cid) = shift;
  
  my($retval) =  CONTAINER::container_depth_trav(
    "container", "container_id", "p_container_id", $cid, 
    \&container_visit_store, "PRE");

  return $retval;
}

#
# -----------------------------------------------------------------------------
# Passed a container_id, returns a ref to a list of container_ids beginning with
# 0 (the root node) and listing the descendants until and including $cid.
sub container_get_lineage {
  my($cid)=@_;

  my(@lineage);
  while($cid != 0) {
      unshift @lineage, $cid;
      ($cid) = DBLIB::db_fetchrow_array("
        select p_container_id
        from container 
        where container_id = '$cid'");
  }
  
  unshift @lineage, 0;
  return \@lineage;
}

######################################################################
# The generic tree-traversal routines

#
# -----------------------------------------------------------------------------
# Traverse a tree-in-a-table depth-first pre- or post-order, beginning with $root
# and calling $visit_sub for each node.
#
# $table is the table to traverse, $id_field is the field
# name of the identifier for each node, $parent_field is the field name for the
# parent pointer field, and $order is "PRE" or "POST" for pre- and post-order
# traversal.
sub container_depth_trav {
    my($table, $id_field, $parent_field, $root, $visit, $order) = @_;

    # use vars qw(...) at the top of this module allows us to make these
    # vars available to the traverse and visit routines without having to 
    # pass them all over the place

    local($container_t_table, $container_t_id_field, 
	  $container_t_parent_field, $container_t_visit, $container_t_order) = 
	  ($table, $id_field, $parent_field, $visit, $order);

    # A variable for the visit routines to put things in
    local($container_t_return_ref);

    MILTON::fatal_error("Invalid traversal type $order") unless(
         $order eq "PRE" or $order eq "POST");
    
    container_do_depth_trav($root, 0);

    return $container_t_return_ref;
}


#
# -----------------------------------------------------------------------------
# NOT A PUBLIC METHOD
# This routine only to be called by depth_trav.  Assumes the presence of
# variables named $container_t_table, $container_t_id_field,
# $container_t_parent_field, $container_t_visit, and 
# $container_t_order (which is "PRE" or "POST" for preorder and postorder)
sub container_do_depth_trav {
    my($root, $level) = @_;

    # Visit myself if we're doing preorder traversal
    &$container_t_visit($root, $level) if($container_t_order eq "PRE");

    my($children) = DBLIB::db_fetchall_arrayref(
      "select $container_t_id_field
       from $container_t_table 
       where $container_t_parent_field='$root'");

    # Visit all of them
    my($child);
    for $child (@$children) {
	container_do_depth_trav(@{$child}[0], $level+1);
    }

    # Visit after the kids if we're doing postorder
    &$container_t_visit($root, $level) if($container_t_order eq "POST");
}


#
# -----------------------------------------------------------------------------
# NOT A PUBLIC METHOD
# To be called by _do_depth_trav.  
# We can do whatever we want with $container_t_return_ref
sub container_visit_delete {
    my($root, $level) = @_;
    
    DBLIB::db_do("delete from $container_t_table 
              where $container_t_id_field = '$root'");
}

#
# -----------------------------------------------------------------------------
# NOT A PUBLIC METHOD
# To be called by _do_depth_trav.  
# We can do whatever we want with $container_t_return_ref
#
# This sub makes a list of array references, one for each node in the tree.
# Each array contains the level of the node (0 for root, 1 for the first 
# generation, etc.) as the first element, followed by the columns of the row.  
# A ref to the list of arrays is kept in $return_ref
sub container_visit_store {
    my($root, $level) = @_;

    my($row_ref) = [$level, DBLIB::db_fetchrow_array("
        select * 
        from $container_t_table 
        where $container_t_id_field = '$root'")];

    push @$container_t_return_ref, $row_ref;
}

#
# -----------------------------------------------------------------------------
#

sub container_verify_id {
  return DBLIB::verify_id('container_id', 'container', @_);
}


# This is a hack to automagically add new things to the container tree
# -J

require THINGTYPE;
require KNAR;
require SEC;
require LAYIM;

sub container_auto_add {
    my($kind, $id, $thingtype) = @_;



    my($type_name) = THINGTYPE::thingtype_get_name($kind, $thingtype);
    my(@commands) = split(/\n/, KNAR::knar_entry_get("Layout: $type_name"));

    open(BLEH, ">> /tmp/bleh.txt") or die($!);
    print BLEH "Type name: $type_name. Commands: @commands\n";
    close BLEH;

    for(@commands) {
	my(@tokens) = split(/:/, $_);

	open(BLEH, ">> /tmp/bleh.txt") or die($!);
	print BLEH "Kind: $kind, id: $id, type: $thingtype, tokens: @tokens\n";
	close BLEH;


	# Check for a constraint
	if($tokens[0] =~ /^(.*)==\s*(.*)/) {
	    next unless(SNIPPET::snip_get_byname($kind, $id, $1) eq $2);
	    shift @tokens;
	}

	for(@tokens) {
	    tr/\r//d;

	    if(s/^(\@)//) {
		my($picture) = KNAR::knar_entry_get($_);

		my($name);
		if($kind eq $THING::item) {
		    $name = ITEM::item_get_label($id);
		} elsif($kind eq $THING::page) {
		    $name = PAGE::page_get_label($id);
		} elsif($kind eq $THING::group) {
		    $name = GROUP::group_get_label($id);
		} else {
  MILTON::fatal_error("Reality check bounced in container_auto_add()");
                }

		$picture =~ s/!NEW!/$name/g;

		# Ignore the return value
	        LAYIM::layout_import($picture);
	    } else {
		next unless( SEC::is_id($_) or $_ eq "0" );
		next unless container_verify_id(\$_);
	    
		container_create($_, $kind, $id);
	    }
	}
    }
}

############################################
1;


