# CUSTOMER.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

package CUSTOMER;

use strict;

require ADDRESS;
require MILTON;
require DBLIB;

use vars qw ( %listall 
	      %create 
	      %delete 
	      %get_info 
	      %set_attr 
	      %address_assoc
	      %address_deassoc
	      %address_listall );

## =============================================================================
##  This file is the public API for customer functionality.
##  It also--for now contains all of the implementation specific functionality.
##  The general layout is that, based on some parameter, a code reference
##  will be called by the general function.
## =============================================================================


## =============================================================================
##  CUSTOMER API
## =============================================================================
##  These are the avaliable functions
##
##  listall               returns arrayref of all customer id's
##  create                creates a new customer
##  delete                deletes a customer
##  get_info              returns all of the info for a specific customer
##  set_attr              allows setting of various customer attributes
##  verify_id             verifies legit customer id
##  address_assoc         associates a customer address
##  address_deassoc       deassociates a customer address
##  address_listall       lists all of the address id's for a customer
##

## =============================================================================
##  IMPLEMENTATIONS
## =============================================================================

## -----------------------------------------------------------------------------
##   func: listall
##   desc: lists all customer's
##  param: none
##    ret: arrayref to customer's
##  notes:
## -----------------------------------------------------------------------------

sub listall_local {
    
    my $sqlstr = "select id, name, company, email, home_phone, work_phone
                  from customer";

    my $ref = DBLIB::db_fetchall_arrayref($sqlstr);

    return $ref;
}

## -----------------------------------------------------------------------------
##   func: create
##   desc: create new customer
##  param: name, company, email, h_phone, w_phone
##    ret: id/ undef
##  notes:
## -----------------------------------------------------------------------------

sub create_local {
    my ($name, $company, $email, $h_phone, $w_phone) = @_;
    
    ## get next customer id
    my $id = DBLIB::seq_next_get("customer_sq");

    my $sqlstr = "insert into customer 
                  (id, name, company, email, home_phone, work_phone)
                  values
                  ($id, '$name', '$company', '$email', '$h_phone', '$w_phone')";

  DBLIB::db_do($sqlstr);

    return $id;
}

## -----------------------------------------------------------------------------
##   func: delete
##   desc: deletes a customer from the db
##  param: id
##    ret: 1/ undef
##  notes: don't do this
## -----------------------------------------------------------------------------

sub delete_local {
    my ($id) = @_;

  DBLIB::db_transact_begin();

    ## clear out customer_info
    my $sqlstr = "delete from customer_info where customer_id = $id";
    
  DBLIB::db_do($sqlstr);

    ## make sure fk is removed from order
    $sqlstr = "select id from cat_order where customer_id = $id";

    my $id_ref = DBLIB::db_fetchall_arrayref($sqlstr);
    my ($row, $value);
    
    foreach $row (@$id_ref) {
	($value) = @$row;
      OR DER::set_attr($value, ("customer_id" => 0));
    }

    ## clean out any customer addresses

    $sqlstr = "select address_id from customer_address where customer_id = $id";
    
    $id_ref = DBLIB::db_fetchall_arrayref($sqlstr);

    $sqlstr = "delete from customer_address where customer_id = $id";
  DBLIB::db_do($sqlstr);

    foreach $row (@$id_ref) {
	($value) = @$row;
      ADDRESS::delete($value);
    }


    ## finally remove the customer

    $sqlstr = "delete from customer where id = $id";

  DBLIB::db_do($sqlstr);

  DBLIB::db_transact_end();

    return 1;
}

## -----------------------------------------------------------------------------
##   func: get_info
##   desc: returns all of the info about a customer
##  param: id
##    ret: name, company, email, h_phone, w_phone
##  notes:
## -----------------------------------------------------------------------------

sub get_info_local {
    my ($id) = @_;
    
    my $sqlstr = "select name, company, email, home_phone, work_phone
                  from customer where id = $id";
    
    my @info = DBLIB::db_fetchrow_array($sqlstr);
    
    return @info;
}

## -----------------------------------------------------------------------------
##   func: set_attr
##   desc: change attributes of a specific customer
##  param: id, attrs
##    ret: 1/ undef
##  notes:
## -----------------------------------------------------------------------------

sub set_attr_local {
    my ($id, %attrs) = @_;
    
    ## create sqlstr
    my $sqlstr = "update customer set ";
    
    my ($key, $val);

    foreach $key (keys(%attrs)) {
	$sqlstr .= "$key = '$attrs{$key}', ";
    }

    chop $sqlstr;
    chop $sqlstr;
    
    $sqlstr .= " where id = $id";

  DBLIB::db_do($sqlstr);

    return 1;
}


## ----------------------------------------------------------------------------
##   func: verify_id
##   desc: verifies avalable customer id
##  param: customer_id
##    ret: 1/ 0 
##  notes: 
## ----------------------------------------------------------------------------

sub verify_id {
    return DBLIB::verify_id("id", "customer", @_);
}



## -----------------------------------------------------------------------------
##   func: address_assoc
##   desc: associates an address with a customer
##  param: customer id, address_id, address type (shipping, billing)
##    ret: 1/ undef
##  notes:
## -----------------------------------------------------------------------------

sub address_assoc_local {
    my ($customer_id, $address_id, $addr_type) = @_;

    ##  associate id's
    my $sqlstr = "insert into customer_address 
                  (customer_id, address_id, address_type)
                  values
                  ($customer_id, $address_id, '$addr_type')";

  DBLIB::db_do($sqlstr);

    return $1;
}

## -----------------------------------------------------------------------------
##   func: address_deassoc
##   desc: deassociates an address from a customer
##  param: customer_id, address_id
##    ret: 1/ undef
##  notes:
## -----------------------------------------------------------------------------

sub address_deassoc_local {
    my ($customer_id, $address_id) = @_;

    my $sqlstr = "delete from customer_address 
                  where customer_id = $customer_id
                  and address_id = $address_id";

  DBLIB::db_do($sqlstr);

    return 1;
}

## -----------------------------------------------------------------------------
##   func: address_listall
##   desc: lists all of the addresses associated with a customer
##         and the type of address that it is
##  param: $customer_id
##    ret: reference to the list
##  notes:
## -----------------------------------------------------------------------------

sub address_listall_local {
    my ($id) = @_;

    my $sqlstr = "select address.id, address_type, name, street1, street2, city, state, zip, country
                  from customer_address, address
                  where customer_address.customer_id = $id 
                  and customer_address.address_id = address.id";

    my $ref = DBLIB::db_fetchall_arrayref($sqlstr);
    
    return $ref;
}

## =============================================================================
##  IMPLEMENTATION HASHES
## =============================================================================

%listall             = ( "LOCAL" => \&listall_local, );
%create              = ( "LOCAL" => \&create_local, );
%delete              = ( "LOCAL" => \&delete_local, );
%get_info            = ( "LOCAL" => \&get_info_local, );
%set_attr            = ( "LOCAL" => \&set_attr_local, );
%address_assoc       = ( "LOCAL" => \&address_assoc_local, );
%address_deassoc     = ( "LOCAL" => \&address_deassoc_local, );
%address_listall     = ( "LOCAL" => \&address_listall_local, );

## =============================================================================
##  API CALLS
## =============================================================================

sub listall {
    return MILTON::generic_call(\%listall, @_);
}

sub create {
    return MILTON::generic_call(\%create, @_);
}

sub delete {
    return MILTON::generic_call(\%delete, @_);
}

sub get_info {
    return MILTON::generic_call(\%get_info, @_);
}

sub set_attr {
    return MILTON::generic_call(\%set_attr, @_);
}

sub address_assoc {
    return MILTON::generic_call(\%address_assoc, @_);
}

sub address_deassoc {
    return MILTON::generic_call(\%address_deassoc, @_);
}

sub address_listall {
    return MILTON::generic_call(\%address_listall, @_);
}

1;




