# DBDOWNLOAD.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

#
# This downloader generates a file suitable for use in the database uploader.
# It does not take into account everything that needs to be done; some objects
# in the system cannot be exported in that way, because the database uploader
# currently does not support them: tax, shipping, payment, and 
# affiliates all fall into this category.
#
# layout data is exported in a separate module (LAYEX.pm).
#

#
# 
# AVAILABLE FUNCTIONS AND VARIABLES DEFINED IN THIS PACKAGE:
#
# &db_download
#
#
# =============================================================================
#

package DBDOWNLOAD;

use strict;
require ITEM;
require PAGE;
require DBLIB;
require GROUP;
require STOCK;
require MATRIX;
require THINGTYPE;
require THING;
require SEC;

use vars qw(%stat_set);

#
# -----------------------------------------------------------------------------
#
# expects a reference to a hash containing which sections you want
# exported.  if the value is defined, the objects are exported.
# currently supported values are:
#
# 'do_m_d_r' 		= save matrices
# 'do_page_types' 	= save page types
# 'do_item_types' 	= save item types
# 'do_group_types' 	= save group types
# 'do_pages' 		= save pages
# 'do_items' 		= save items
# 'do_groups' 		= save groups
# 'do_stock' 		= save stock cells
# 'do_knar'		= save knar data
#
# returns a reference to a hash of statistics.
#

sub db_download {
  my($some_error);
  my($opt_ref) = shift;

  $some_error = 0;
  if (!defined($opt_ref)) {
    $opt_ref = {};
  }

  stat_set_initialize();

  if (!open_file()) {
    return \%stat_set;
  }

  if (!write_headers()) { $some_error = 1; }

  if ($$opt_ref{'do_m_d_r'}) {
    if (!save_m_d_r()) { $some_error = 1; }
  }

  if ($$opt_ref{'do_page_types'}) {
    if (!save_page_types()) { $some_error = 1; }
  }

  if ($$opt_ref{'do_item_types'}) {
    if (!save_item_types()) { $some_error = 1; }
  }

  if ($$opt_ref{'do_group_types'}) {
    if (!save_group_types()) { $some_error = 1; }
  }

  if ($$opt_ref{'do_pages'}) {
    if (!save_pages()) { $some_error = 1; }
  }

  if ($$opt_ref{'do_groups'}) {
    if (!save_groups()) { $some_error = 1; }
  }

  if ($$opt_ref{'do_items'}) {
    if (!save_items()) { $some_error = 1; }
  }

  if ($$opt_ref{'do_stock'}) {
    if (!save_stockdata()) { $some_error = 1; }
  }

  if ($$opt_ref{'do_knar'}) {
    if (!save_knar()) { $some_error = 1; }
  }

  if (!close_file()) { $some_error = 1; }

  return \%stat_set;
}

#
# -----------------------------------------------------------------------------
#

sub open_file {
  my($fname);

  $fname = "$ENV{'TALLYMAN_PATH'}/stores/$ENV{'TALLYMAN_SID'}/tmp/dbdownload.txt";
  SEC::untaint_ref(\$fname);

  if (!open(DOWNLOAD, ">$fname")) {
    stat_set_add_error("Could not open file $fname for writing: $!");
    return 0;
  }

  select(DOWNLOAD);
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub close_file {
  my($fname);

  $fname = "$ENV{'TALLYMAN_PATH'}/stores/$ENV{'TALLYMAN_SID'}/tmp/dbdownload.txt";
  SEC::untaint_ref(\$fname);

  if (!close(DOWNLOAD)) {
    stat_set_add_error("Could not close file $fname for writing: $!");
    return 0;
  }
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub write_headers {
  my($time);
  $time = localtime(time());

  print DOWNLOAD <<EOTEXT;
#
#  TALLYMAN DATABASE EXPORT FILE v1.0
#  Generated $time
#
#  In order to re-create your store, you will need to run this file through
#  the database uploader.
#
## show_debug_log;
#
# =============================================================================
#

EOTEXT
}

#
# -----------------------------------------------------------------------------
#

sub save_m_d_r {
  my($ref, $refs, $tmp);

  print DOWNLOAD <<EOTEXT;
#
#  ----------------------------------------------------------------------------
#  Export matrices / dimensions / ranges
#
EOTEXT

  # returns 0: id, 1: name
  $refs = MATRIX::matrix_list_all();

  if ($#{$refs} < 0) {
    print DOWNLOAD "#  (no matrices to export)\n#\n\n\n";
    stat_set_add_warning('No matrices to export!');
    return 1;
  }

  print DOWNLOAD "## 1_type=cmatrix; 1_clean=u\n";
  print DOWNLOAD "## 2_type=cdim; 2_clean=u\n";
  print DOWNLOAD "## 3_type=crange; 3_clean=u\n";

  foreach $ref (@$refs) {
    $stat_set{'matrices'}++; 
    save_dims($ref);
  }

  print DOWNLOAD "\n\n";

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub save_dims {
  my($matrix_ref) = shift;
  my($ref, $refs, $tmp);

  $refs = MATRIX::dim_list_matrix($matrix_ref->[0]);

  if ($#{$refs} < 0) {
    $tmp = clean_str($matrix_ref->[1]);
    print DOWNLOAD "$tmp\t\t\n";
    stat_set_add_warning("No dimensions to export for matrix $matrix_ref->[1]!");
    return 1;
  }

  foreach $ref (@$refs) {
    $stat_set{'dimensions'}++; 
    save_ranges($matrix_ref, $ref);
  }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub save_ranges {
  my($matrix_ref, $dim_ref) = @_;
  my($tmp1, $tmp2, $tmp3, $ref, $refs);

  $refs = MATRIX::range_list_dim($dim_ref->[0]);

  if ($#{$refs} < 0) {
    $tmp1 = clean_str($matrix_ref->[1]);
    $tmp2 = clean_str($dim_ref->[1]);
    print DOWNLOAD "$tmp1\t$tmp2\t\n";
    stat_set_add_warning("No ranges to export for dimension $dim_ref->[1]!");
    return 1;
  }

  foreach $ref (@$refs) {

    $stat_set{'ranges'}++;

    $tmp1 = clean_str($matrix_ref->[1]);
    $tmp2 = clean_str($dim_ref->[1]);
    $tmp3 = clean_str($ref->[1]);

    print DOWNLOAD "$tmp1\t$tmp2\t$tmp3\n";
  }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub save_page_types {
  return save_thingtype($THING::page, 'page');
}

sub save_item_types {
  return save_thingtype($THING::item, 'item');
}

sub save_group_types {
  return save_thingtype($THING::group, 'group');
}

#
# -----------------------------------------------------------------------------
#

sub save_thingtype {
  my($type, $typestr) = @_;
  my($ref, $refs, $tmp1, $tmp2, @nofield_list, @yesfield_list);

  print DOWNLOAD <<EOTEXT;
#
#  ----------------------------------------------------------------------------
#  Export ${typestr} types
#
EOTEXT

  # returns 0: id, 1: name, 2: label
  $refs = THINGTYPE::thingtype_listall($type);

  if ($#{$refs} < 0) {
    print DOWNLOAD "#  (no ${typestr} types to export)\n#\n\n\n";
    stat_set_add_warning("No ${typestr} types to export!");
    return 1;
  }

  foreach $ref (@$refs) {
    if (check_for_fields($type, $ref)) {
      push @yesfield_list, $ref;
    } else {
      push @nofield_list, $ref;
    }
  }

  #
  # save thingtypes with fields
  #

  if ($#yesfield_list > -1) {
    print DOWNLOAD <<EOTEXT;
#
#  -- Export ${typestr} types with fields. 
#
## 1_type=${typestr}_thingtype; 1_field=name; 1_clean=u
## 2_type=${typestr}_thingtype; 2_field=label; 2_clean=u
## 3_type=${typestr}_sniptype;  3_field=name; 3_clean=u
## 4_type=${typestr}_sniptype;  4_field=label; 4_clean=u
## 5_type=${typestr}_sniptype;  5_field=default; 5_clean=u
## 6_type=${typestr}_sniptype;  6_field=widgettype; 6_clean=u
## 7_type=${typestr}_sniptype;  7_field=widgetinfo; 7_clean=u
## 8_type=${typestr}_sniptype;  8_field=widgetinfo_split; 8_clean=u
EOTEXT

    foreach $ref (@yesfield_list) {
      save_sniptypes($type, $ref);
      $stat_set{"${typestr}types"}++;
    }
  }

  #
  # now save thingtypes with no fields
  #
  
  if ($#nofield_list > -1) {
    print DOWNLOAD <<EOTEXT;
#
#  -- Export ${typestr} types without fields. 
#
## 1_type=c${typestr}type_name; 1_clean=u
## 2_type=c${typestr}type_label; 2_clean=u
EOTEXT

    foreach $ref (@nofield_list) {
      $tmp1 = clean_str($ref->[1]);
      $tmp2 = clean_str($ref->[2]);
      print DOWNLOAD "$tmp1\t$tmp2\n";
      $stat_set{"${typestr}types"}++;
    }
  }

  print DOWNLOAD "\n\n";

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub check_for_fields {
  my($type, $thingtype_ref) = @_;
  my($sniptype_ids);

  $sniptype_ids = THINGTYPE::thingtype_get_sniptypes($type,
                                                     $thingtype_ref->[0]);

  if ($#{$sniptype_ids} > -1) {
    return 1;
  }

  return 0;
}

sub save_sniptypes {
  my($type, $thingtype_ref) = @_;
  my($sniptype_ids, $tmp1, $tmp2, $sniptype);
  my($index_name, $label, $default_val, $accessmethod, $wt, $widgetinfo);

  $sniptype_ids = THINGTYPE::thingtype_get_sniptypes($type,
                                                     $thingtype_ref->[0]);

  if ($#{$sniptype_ids} > -1) {
    $tmp1 = clean_str($thingtype_ref->[1]);
    $tmp2 = clean_str($thingtype_ref->[2]);

    foreach $sniptype (@$sniptype_ids) {
      ($index_name, $label, $default_val, $accessmethod, $wt, $widgetinfo) 
        = SNIPTYPE::sniptype_get_all($type, $sniptype); 

      $index_name = clean_str($index_name);
      $label = clean_str($label);
      $default_val = clean_str($default_val);
      $accessmethod = clean_str($accessmethod);
      $wt = clean_str($wt);

      if ($wt == $SNIPTYPE::selectionbox ||
          $wt == $SNIPTYPE::radio) {
        $widgetinfo = join(",", @$widgetinfo);
        $widgetinfo =~ s/\n//g;
      } else {
        $widgetinfo =~ s/,/ /g;
        $widgetinfo =~ s/\n/,/g;
      }
      $widgetinfo =~ s/\t//g;

      if ($wt == $SNIPTYPE::normal) {
        $wt = "normal";
      } elsif ($wt == $SNIPTYPE::checkbox) {
        $wt = "checkbox";
      } elsif ($wt == $SNIPTYPE::selectionbox) {
        $wt = "selectionbox";
      } elsif ($wt == $SNIPTYPE::radio) {
        $wt = "radio";
      } elsif ($wt == $SNIPTYPE::textarea) {
        $wt = "textarea";
      } elsif ($wt == $SNIPTYPE::password) {
        $wt = "password";
# XXX not yet implemented
#      } elsif ($wt == $SNIPTYPE::file_upload) {
#        $wt = "file_upload";
      } elsif ($wt == $SNIPTYPE::hidden) {
        $wt = "hidden";
      }

      print DOWNLOAD "$tmp1\t$tmp2\t$index_name\t$label\t$default_val\t";
      print DOWNLOAD "$wt\t$widgetinfo\t,\n";

    }
  }
}

#
# -----------------------------------------------------------------------------
#

sub save_pages {
  return save_things($THING::page, 'page', \&PAGE::page_get_label);
}

#
# -----------------------------------------------------------------------------
#

sub save_groups {
  return save_things($THING::group, 'group', \&GROUP::group_get_label);
}

#
# -----------------------------------------------------------------------------
#

sub save_items {
  return save_things($THING::item, 'item', \&ITEM::item_get_label);
}

#
# -----------------------------------------------------------------------------
#

sub save_things {
  my($type, $typestr, $labelfunc) = @_;
  my($thingtypes, $thingtype, $tt_name);
  my($sniptypes, $sniptype);
  my($thing_ids, $thing_id);
  my($tmp, $i);

  print DOWNLOAD <<EOTEXT;
#
#  ----------------------------------------------------------------------------
#  Export ${typestr}s
#
EOTEXT

  # foreach thing type, export all objects of that type

  $thingtypes = THINGTYPE::thingtype_listall($type);
  foreach $thingtype (@$thingtypes) {
    $sniptypes = THINGTYPE::thingtype_get_sniptypes($type, $thingtype->[0]);

    $thing_ids = THINGTYPE::thingtype_get_things($type, $thingtype->[0]);

    # no things of this type?  skip it.
    if ($#{$thing_ids} < 0) {
      next;
    }

    $tt_name = $thingtype->[1];
    $tt_name = clean_str($tt_name);

    print DOWNLOAD "#\n#  -- Export ${typestr}s of type \"$tt_name\"\n#\n";
    print DOWNLOAD "## 1_type=${typestr}; 1_field=Name; 1_attr=mcu; 1_clean=u\n";
    print DOWNLOAD "## 2_type=${typestr}_thingtype; 2_field=name; 2_clean=u\n";

    $i = 3;
    foreach $sniptype (@$sniptypes) {
      $tmp = SNIPTYPE::sniptype_get_name($type, $sniptype);
      $tmp = clean_str($tmp);
      print DOWNLOAD "## ${i}_type=${typestr}; ${i}_field=${tmp}; ${i}_attr=cu; ${i}_clean=u\n";
      $i++;
    }

    foreach $thing_id (@$thing_ids) {
      $tmp = &$labelfunc($thing_id);
      $tmp = clean_str($tmp);
      print DOWNLOAD "$tmp\t$tt_name";

      foreach $sniptype (@$sniptypes) {
        $tmp = SNIPTYPE::sniptype_get_name($type, $sniptype);
        if ($tmp eq "_widget_typenum" || $tmp eq "_widget_text") {
          next;
        } 
        $tmp = SNIPPET::snip_get($type, $thing_id, $sniptype); 
        $tmp = clean_str($tmp);
        print DOWNLOAD "\t$tmp";
      }

      $stat_set{$typestr.'s'}++;

      print DOWNLOAD "\n";

    }

    print DOWNLOAD "\n";
  }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub save_stockdata {
  my($item_id, $item_ids);
  my($label, $sku, $price, $weight, $volume, $atp, $rt, $drq, $stock_id,
     $mix_id, $cell_id);
  my($matrix_id, $dim, $dims, %sorted_items, $i, $j, $tmp, $tmp2);
  my($gridref, $height, $width);
  my(@nonmixed_items);

  print DOWNLOAD <<EOTEXT;
#
#  ----------------------------------------------------------------------------
#  Export stock data
#
EOTEXT

  $item_ids = ITEM::item_list_all();

  #
  # here, we sort all items by matrix.
  #

  foreach $item_id (@$item_ids) {
    ($label, $sku, $price, $weight, $volume, $atp, $rt, $drq, $stock_id,
     $mix_id, $cell_id) = ITEM::item_get_info($item_id->[0]);

    if ($mix_id != 0) {
      $matrix_id = MATRIX::matrix_get_id_from_mix_id($mix_id);
      if (defined($sorted_items{$matrix_id})) {
        ${$sorted_items{$matrix_id}}[$#{$sorted_items{$matrix_id}}+1] = $item_id->[0];
      } else {
        $sorted_items{$matrix_id} = [ $item_id->[0] ];
      }
    } else {
      push @nonmixed_items, $item_id->[0];
    }


  }

  foreach $matrix_id (sort(keys(%sorted_items))) {
    if ($matrix_id == 0) {
      next;
    }

    #
    # generate a header for each matrix (that has been used at least once)
    #

    $tmp = MATRIX::matrix_get_label($matrix_id);
    print DOWNLOAD "#\n#  -- Export stockdata for matrix \"$tmp\"\n#\n";

    $tmp2 = clean_str($tmp);

    $dims = MATRIX::dim_list_matrix($matrix_id);
    print DOWNLOAD "## 1_type=item;  1_attr=mc; 1_field=Name; 1_clean=u\n";

    $i = 2;
    foreach $dim (@$dims) {
      $tmp = clean_str($dim->[1]);
      print DOWNLOAD "## ${i}_type=range; ${i}_attr=c; ${i}_clean=u; ${i}_dim=${tmp}; ${i}_matrix=${tmp2}\n";
      $i++;
    }
    print_stock_header($i);

    #
    # now spit out each item that uses that matrix
    #

    $item_ids = $sorted_items{$matrix_id};

    foreach $item_id (@$item_ids) {
      ($label, $sku, $price, $weight, $volume, $atp, $rt, $drq, $stock_id,
       $mix_id, $cell_id) = ITEM::item_get_info($item_id);

      # spit out name

      $tmp2 = $label;
      $tmp2 = clean_str($tmp2);

      # spit out characteristics for this variant

      ($gridref, $height, $width) = ITEM::mix_get_grid($mix_id);

      for ($i=0; $i<$height; $i++) {
        print DOWNLOAD $tmp2 . "\t";
        for ($j=1; $j<$width; $j++) {
          $tmp = MATRIX::range_get_label($gridref->[$i][$j]) . "\t";
          $tmp = clean_str($tmp);
          print DOWNLOAD $tmp . "\t";
        }

        # now spit out the stock cell      
        ($sku, $price, $weight, $volume, $atp, $rt, $drq) 
          = STOCK::get_info($gridref->[$i][0]);

        $sku = clean_str($sku);
        print DOWNLOAD "$sku\t";

        $price = clean_str($price);
        print DOWNLOAD "$price\t";

        $weight = clean_str($weight);
        print DOWNLOAD "$weight\t";

        $volume = clean_str($volume);
        print DOWNLOAD "$volume\t";

        $atp = clean_str($atp);
        print DOWNLOAD "$atp\t";

        $rt = clean_str($rt);
        print DOWNLOAD "$rt\t";

        $drq = clean_str($drq);
        print DOWNLOAD "$drq";

        print DOWNLOAD "\n";

        $stat_set{'stocks'}++;
      }

    }

    print DOWNLOAD "\n";
  }

  #
  # now, spit out all non-mixed items
  #

  print DOWNLOAD "#\n#  -- Export stockdata for non-mixed items\n#\n";
  print DOWNLOAD "## 1_type=item; 1_attr=mc; 1_field=Name; 1_clean=u\n";
  print_stock_header(2);

  foreach $item_id (@nonmixed_items) {
    ($label, $sku, $price, $weight, $volume, $atp, $rt, $drq, $stock_id,
    $mix_id, $cell_id) = ITEM::item_get_info($item_id);

    $label = clean_str($label);
    print DOWNLOAD "$label\t";

    $sku = clean_str($sku);
    print DOWNLOAD "$sku\t";

    $price = clean_str($price);
    print DOWNLOAD "$price\t";

    $weight = clean_str($weight);
    print DOWNLOAD "$weight\t";

    $volume = clean_str($volume);
    print DOWNLOAD "$volume\t";

    $atp = clean_str($atp);
    print DOWNLOAD "$atp\t";

    $rt = clean_str($rt);
    print DOWNLOAD "$rt\t";

    $drq = clean_str($drq);
    print DOWNLOAD "$drq";

    print DOWNLOAD "\n";
    $stat_set{'stocks'}++;
  }

  print DOWNLOAD "\n";

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub print_stock_header {
  my($i) = shift;

  print DOWNLOAD "## ${i}_type=stock; ${i}_attr=u; ${i}_clean=u; ${i}_field=SKU\n";
  $i++;

  print DOWNLOAD "## ${i}_type=stock; ${i}_attr=u; ${i}_clean=u; ${i}_field=Price\n";
  $i++;

  print DOWNLOAD "## ${i}_type=stock; ${i}_attr=u; ${i}_clean=u; ${i}_field=Weight\n";
  $i++;

  print DOWNLOAD "## ${i}_type=stock; ${i}_attr=u; ${i}_clean=u; ${i}_field=Volume\n";
  $i++;

  print DOWNLOAD "## ${i}_type=stock; ${i}_attr=u; ${i}_clean=u; ${i}_field=atp_qty\n";
  $i++;

  print DOWNLOAD "## ${i}_type=stock; ${i}_attr=u; ${i}_clean=u; ${i}_field=reserve_threshold\n";
  $i++;

  print DOWNLOAD "## ${i}_type=stock; ${i}_attr=u; ${i}_clean=u; ${i}_field=def_reserve_qty\n";

}

#
# -----------------------------------------------------------------------------
#

sub save_knar {
  my($knar_id, $knar_ids);
  my($tmp1, $tmp2, $tmp3);

  print DOWNLOAD <<EOTEXT;
#
#  ----------------------------------------------------------------------------
#  Export knar data
#
EOTEXT

  $knar_ids = KNAR::knar_entry_list_all();

  if ($#{$knar_ids} < 0) {
    print DOWNLOAD "#  (no knar entries to export)\n#\n\n\n";
    stat_set_add_warning("No knar entries to export!");
    return 1;
  }

  print DOWNLOAD "## 1_type=cknarkey; 1_clean=u\n";
  print DOWNLOAD "## 2_type=cknarval; 2_clean=u\n";
  print DOWNLOAD "## 3_type=cknargroup; 3_clean=u\n";

  foreach $knar_id (@$knar_ids) {
    $tmp1 = clean_str($knar_id->[1]);
    $tmp2 = clean_str($knar_id->[2]);
    $tmp3 = clean_str($knar_id->[3]);

    print DOWNLOAD "$tmp1\t$tmp2\t$tmp3\n";
    $stat_set{'knar'}++;
  }

  print DOWNLOAD "\n\n";
  return 1;
}

#
# -----------------------------------------------------------------------------
#
#    UTILITY FUNCTIONS
# 
# -----------------------------------------------------------------------------
#

sub clean_str {
  my($tmp) = shift;
  clean_str_ref(\$tmp);
  return $tmp;
}

sub clean_str_ref {
  my($tmp) = shift;

  $$tmp = SEC::encode_uri($$tmp);

#  $$tmp =~ s/\t//g;
#  $$tmp =~ s/\n//g;
#  $$tmp =~ s/\r//g;
}

sub stat_set_initialize {
  %stat_set = undef;

  $stat_set{'warnings'} 	= 0;
  $stat_set{'errors'} 		= 0;

  $stat_set{'items'} 		= 0;
  $stat_set{'pages'} 		= 0;
  $stat_set{'groups'} 		= 0;
  $stat_set{'stocks'} 		= 0;

  $stat_set{'matrices'} 	= 0;
  $stat_set{'dimensions'} 	= 0;
  $stat_set{'ranges'} 		= 0;

  $stat_set{'pagetypes'} 	= 0;
  $stat_set{'itemtypes'} 	= 0;
  $stat_set{'grouptypes'} 	= 0;

  $stat_set{'knar'} 		= 0;
}

sub stat_set_add_warning {
  my($msg) = shift;
  my($wnum);

  # check to see if it's defined, so that warnings that are exactly the
  # same aren't repeated...

  if (!defined($stat_set{"warn_$msg"})) {
    $stat_set{'warnings'}++;
    $wnum = $stat_set{'warnings'};
    $stat_set{"warning_$wnum"} = $msg;
    $stat_set{"warn_$msg"} = 1;
  }
}

sub stat_set_add_error {
  my($msg) = shift;
  my($wnum);

  # check to see if it's defined, so that errors that are exactly the
  # same aren't repeated...

  if (!defined($stat_set{"error_$msg"})) {
    $stat_set{'errors'}++;
    $wnum = $stat_set{'errors'};
    $stat_set{"error_$wnum"} = $msg;
    $stat_set{"error_$msg"} = 1;
  }
}

#
# -----------------------------------------------------------------------------
#

1;
