# DBLIB.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

##-----------------------------------------------------------------------------
##  This library packages up DBI calls so that functionality can be handeled
##  elsewhere with one call.  It also handles error conditions.  It has 
##  initial, but largely unimplemented, and unused support for non-fatal
##  database errors.  Due to various factors, we are forgoing an object
##  oriented approach for the time being, in favor of a looser implementation.
##-----------------------------------------------------------------------------

package  DBLIB;

use strict;
use DBI;

require CONFIG;

# Used for taintedness checks
require SEC;

use vars qw($check_taint $log_db_access);
$check_taint = 1;
$log_db_access = 0;

require MILTON;

use vars qw(%db_types %db_pre_connect %dbh_count %dbh_ref %db_exec %db_str_clean %db_seq_next %db_date_str %trans_beg %trans_end %trans_rb %trans_count);

use vars qw($db_default $errlog $current_storeid %stores);

###############################################################################
##-----------------------------------------------------------------------------
##  FUNCTIONS AVAILABLE
##
##  db_connect
##  db_disconnect
##  db_connect_real
##  db_do
##  db_fetchrow_array
##  db_fetchrow_arrayref
##  db_fetchall_arrayref
##  db_fetchcol_arrayref
##  seq_next_get
##  db_string_clean
##  date_string_make
##  verify_id
##  db_cleanup
##  db_transact_begin
##  db_transact_end
##  db_transact_rollback
##  local_error
##  non_fatal_error
##
##-----------------------------------------------------------------------------
###############################################################################




###############################################################################
##-----------------------------------------------------------------------------
##  Here is where all of the database specific information will be 
##  concentrated.  Each DB will have a number of variables and code 
##  references associated with it.  These will then be collected in 
##  a hash with the DB string name as the key.  These are as follows:
##  
##  HASHES:
##  %db_types           all of the DB types avaliable
##  %dbh_count          refernce count of database handles given out
##  %dbh_ref            refernces to database handles
##  %db_connect         all of the @DB_connect arrays
##  %db_pre_connect     code references to what to do before connecting
##  %db_exec            code references to the &DB_exec subs
##  %db_str_clean       code references to the &DB_str_clean subs
##  %db_seq_next        code references to the &DB_seq_next subs
##  %db_date_str        code referneces to the &DB_date_str subs
##  %trans_beg          code for begining a transaction
##  %trans_end          code for ending a transaction
##  %trans_rb           code for cancelling a transaction
##  %trans_count        reference count for DB transactions
##
##  CODE REFERENCES:
##  &DB_exec            code to execute right after connecting with DBI
##  &DB_str_clean       code to clean up (escape) strings for DB access
##  &DB_seq_next        code to access the next value of a named sequence
##  &DB_date_str        code to return a DB acceptable string from a given
##                      date format
##  &DB_trans_beg       code to begin transaction
##  &DB_trans_end       code to end a transaction
##  &db_trans_rb        code to rollback a transaction
##
##  ARRAYS:
##  @DB_connect         information necessary for DBI to connect
##                      ($DB_data_source, $DB_username, $DB_password)
##  VARS:
##-----------------------------------------------------------------------------
###############################################################################

###############################################################################
##-----------------------------------------------------------------------------
##  DATABASES
##-----------------------------------------------------------------------------
###############################################################################


##-----------------------------------------------------------------------------
##  ORACLE
##  This is Oracle on Linux 8.0.5.0 or something like that
##-----------------------------------------------------------------------------

sub oracle_pre_connect {
  $ENV{'ORACLE_HOME'} = CONFIG::get_entry('oracle_homedir');
  $ENV{'ORACLE_SID'} = CONFIG::get_entry('oracle_sid');
}

sub oracle_exec { return 1; }

sub oracle_str_clean { 
    my ($str) = @_;
    $str =~ s/'/''/g;
                      # don't call more than once with the same string, eh?
    return $str;
}

sub oracle_seq_next {
  my ($seq_name) = shift;
  my ($sqlstr) = "select ${seq_name}.nextval from dual";

  my ($nextid) = db_fetchrow_array($sqlstr);
  return($nextid);
}

sub oracle_date_str { 
  my($sec,$min,$hour,$day,$mon,$year) = @_;

  return("TO_DATE('${sec}:${min}:${hour} ${day}/${mon}/${year}','SS:MI:HH24 DD/MM/YYYY')");
}

sub oracle_trans_beg { return undef; }
sub oracle_trans_end { return undef; }
sub oracle_trans_rb { return undef; }

##-----------------------------------------------------------------------------
##  PG
##  PostgreSQL v6.5.1
##-----------------------------------------------------------------------------

sub pg_pre_connect {
}

sub pg_exec { return 1; }

sub pg_str_clean {
    my ($str) = @_;

    $str =~ s/(['"\\\0])/\\$1/g; #"'

    return $str;
 }

sub pg_seq_next { 
    my ($seq_name) = @_;
    my $sqlstr = "select nextval('$seq_name')";
    my ($next_val) = db_fetchrow_array($sqlstr);

    return $next_val;
 }

sub pg_date_str { 
  my($sec,$min,$hour,$day,$mon,$year) = @_;

  my(@months) = qw(January February March April May June July August
		   September October November December);

  fatal_error("Month out of range: $mon") unless($mon > 0 and $mon < 13);

  return "'$months[$mon - 1] $day, $year $hour:$min:$sec'";
}


sub pg_trans_beg {
    my ($nff) = @_;
    
    ## we just need to turn the DBI autocommit off
    my $dbh = db_connect("PG", $nff);

    $dbh->{AutoCommit} = 0;

    return 1;
}

sub pg_trans_end {
    my ($nff) = @_;
    
    ## run a commit
    my $dbh = $dbh_ref{'PG'};
    $dbh->commit() || local_error("Cannot commit database.", $nff);
    
    ## turn DBI autocommit back on
    $dbh->{AutoCommit} = 1;

    return 1;
}

sub pg_trans_rb {
    my ($nff) = @_;

    ## run a rollback
    my $dbh = $dbh_ref{"PG"};
    $dbh->rollback() || local_error("Cannot complete database rollback.", $nff);

    ## turn DBI autocommit back on
    $dbh->{AutoCommit} = 1;

    return 1;
}


##-----------------------------------------------------------------------------
##  HASHES
##
##  If you change the defaults here, remember to change them in the
##  swap_in_new_store_settings function below
##
##-----------------------------------------------------------------------------

%db_types       = ( 'ORACLE' => "oracle", 
		    'PG'     => "postrgres", );

%db_pre_connect = ( 'ORACLE' => \&oracle_pre_connect,
		    'PG'     => \&pg_pre_connect, );


%dbh_count      = ( 'ORACLE' => 0,
		    'PG'     => 0, ); 

%dbh_ref        = ( 'ORACLE' => undef,
		    'PG'     => undef, ); 

$db_exec        {'ORACLE'} = \&oracle_exec;
$db_exec	{'PG'}     = \&pg_exec;

%db_str_clean   = ( 'ORACLE' => \&oracle_str_clean,
		    'PG'     => \&pg_str_clean, );

%db_seq_next    = ( 'ORACLE' => \&oracle_seq_next,
		    'PG'     => \&pg_seq_next, );

%db_date_str    = ( 'ORACLE' => \&oracle_date_str,
		    'PG'     => \&pg_date_str, );

%trans_beg      = ( 'ORACLE' => \&oracle_trans_beg,
		    'PG'     => \&pg_trans_beg, );

%trans_end      = ( 'ORACLE' => \&oracle_trans_end,
		    'PG'     => \&pg_trans_end, );

%trans_rb       = ( 'ORACLE' => \&oracle_trans_rb,
		    'PG'     => \&pg_trans_rb, );

%trans_count    = ( 'ORACLE' => 0,
		    'PG'     => 0, );


###############################################################################
##-----------------------------------------------------------------------------
##  FUNCTIONS
##-----------------------------------------------------------------------------
###############################################################################


##-----------------------------------------------------------------------------
##   func:  db_connect
##  param:  $database     : which database to use, Oracle, Postgres, etc
##          $nff          : non_fatal_flag (1/undef)
##    ret:  $dbh/undef 
##  notes:  for performance purposes, this will actually be a persistent
##          connection, although functions should connect and disconnect
##          as though this were actually occuring
##    def:  if $database, not passed, uses $db_default
##-----------------------------------------------------------------------------

sub db_connect {
    my ($database, $nff) = @_;

    check_for_store_swapping();

    ## make sure $database has been passed or not
    if (!defined ($database) || $database =~ /^\d$/) {
	$nff = $database;
	$database = $db_default;
    }

    ## there are two conditions for which we wish to open a database 
    ## handle for real:
    ##    if the reference count is at zero
    ##    if the dbh has timed out

    ## increment the reference count for this dbh
    $dbh_count{$database}++;

    if (($dbh_count{$database} == 1) || (!(($dbh_ref{$database})->ping))) {
	db_connect_real($database, $nff);
    }

    
    ## return the database handle
    return $dbh_ref{$database};
}


##-----------------------------------------------------------------------------
##   func:  db_disconnect
##  param:  $dbh          : the database handle to disconnect
##    ret:  1/undef       
##  notes:
##-----------------------------------------------------------------------------

sub db_disconnect {
    my ($dbh) = @_;
    
    ## this doesn't really do anything

    return 1;
}

##-----------------------------------------------------------------------------
##   func:  db_connect_real
##  param:  $database     : same as db_connect
##          $nff          : same as db_connect
##    ret:  1/undef    
##  notes:  this function will connect for real 
##          and sets the $dbh_ref{$database} handle
##    def:  this function expects to be called only internally--no default
##-----------------------------------------------------------------------------

sub db_connect_real {
    my ($database, $nff) = @_;

##  this gets called in db_connect--don't call connect_real from anywhere
##    check_for_store_swapping();

    &{$db_pre_connect{$database}}();

    my @connect;
    $connect[0] = CONFIG::get_entry("${database}_db_connect_str");
    $connect[1] = CONFIG::get_entry("${database}_db_username");
    $connect[2] = CONFIG::get_entry("${database}_db_password");

    $dbh_ref{$database} = DBI->connect(@connect);

    return local_error("Unable to connect to database: " . $DBI::errstr, $nff) if (!defined($dbh_ref{$database}));

    ## execute database specific code references
    return local_error("Cannot execute database specific code.", $nff) if (!defined(&{$db_exec{$database}}($database)));
    
    ## set up some handle defaults

    # Wow.  This is strange.  Apparently, the oracle DBD has a compiled in
    # max longreadlen of 2^^16-1.  Anything more than that, and you start to
    # get weird silent errors.  
    # Turns out it's a bug in DBD::Oracle 1.01 and earlier.  We'll
    # set it to 512k-1 for now; buggy DBD::Oracles will still see
    # 65535, fixed ones will see 512k-1.  If you need larger or smaller
    # / knar entries, etc., you can lower this value accordingly.

    $dbh_ref{$database}->{LongReadLen} = (512 * 1024) -1;
    $dbh_ref{$database}->{LongTruncOk} = 0;

    return 1;
}

##-----------------------------------------------------------------------------
## A hack to the db_fetch* and db_do routines allows passing named parameters.
## Pass the nonfatal flag, then pass pairs of parameters.  The first
## will be bound to the ? in the SQL statement, and the second is a flag
## indicating if the first should be converted to an INT before being
## passed to the database.
## Example:
##
## db_do("update foo set bar='toast is yummy' where id=42");
##    becomes
## db_do("update foo set bar=? where id=?", 0, "toast is yummy", 0, 42, 1);
##
##-----------------------------------------------------------------------------

##-----------------------------------------------------------------------------
##   func:  db_do
##  param:  $database, $sqlstr, $nff
##    ret:  1/undef
##  notes:
##-----------------------------------------------------------------------------
sub old_db_do {
    ## declare local variables
    my ($database, $sqlstr, $nff, $dbh, $sth);

    check_for_store_swapping();

    ## make sure $dbh is a database handle
    if (!$db_types{$_[0]}) {
	## assume that $database was not passed
	($sqlstr, $nff) = @_;
	$database = $db_default;
    } else {
	## assume it was
	($database, $sqlstr, $nff) = @_;
    }

    MILTON::fatal_error("Tainted: $sqlstr") if( $check_taint 
						and SEC::is_tainted($sqlstr));

    $dbh = db_connect($database, $nff);

    ## prepare and execute sqlstr
    $sth = $dbh->prepare($sqlstr) || return local_error($dbh->errstr, $nff);
    $sth->execute() || return local_error($dbh->errstr, $nff);
    $sth->finish();

    db_disconnect($dbh);

    if ($log_db_access) {
      open(HOHUM,">>/tmp/hohum");
      print HOHUM $sqlstr . "\n";
      close(HOHUM);
    }

    return 1;
}

sub db_do {

    ## declare local variables
    my ($database, $sqlstr, $nff, $dbh, $sth, @named_parms);

    check_for_store_swapping();

    if ($db_types{$_[0]}) {
	$database = shift @_;
    } else {
	$database = $db_default;
    }

    $sqlstr = shift @_;
    $nff = shift @_;
    @named_parms = @_;

    # There should be an even number of named parms.  
    MILTON::fatal_error("Incorrect number of parameters to db_do") unless
        ($#named_parms % 2);
	
    MILTON::fatal_error("Tainted: $sqlstr") if( $check_taint 
						and SEC::is_tainted($sqlstr));

    $dbh = db_connect($database, $nff);

    ## prepare and execute sqlstr, caching requests with named
    ## parameters
    if($#named_parms > -1) {
	$sth = $dbh->prepare_cached($sqlstr) || 
	    return local_error($dbh->errstr, $nff);

	# Bind parms, with any flagged parms as integers
	my($i, $parm_num);
	$parm_num = 1;
	for($i = 0; $i < $#named_parms; $i+=2) {
	    if($named_parms[$i+1]) {
		$sth->bind_param($parm_num, $named_parms[$i], 
			       DBI::SQL_INTEGER);
	    } else {
		$sth->bind_param($parm_num, $named_parms[$i]);
	    }

	    $parm_num++;
	}
    } else {
	$sth = $dbh->prepare($sqlstr) || 
	    return local_error($dbh->errstr, $nff);
    }

    $sth->execute() || return local_error($dbh->errstr, $nff);
    $sth->finish();

    db_disconnect($dbh);

    if ($log_db_access) {
      open(HOHUM,">>/tmp/hohum");
      print HOHUM $sqlstr . "\n";
      close(HOHUM);
    }

    return 1;
}

##-----------------------------------------------------------------------------
##   func:  db_fetchrow_array
##  param:  $database, $sqlstr, $nff
##    ret:  \@array              : a reference to the array returned
##  notes:
##-----------------------------------------------------------------------------
sub db_fetchrow_array {

    ## declare local variables
    my ($database, $sqlstr, $nff, $dbh, $sth, @selected, @named_parms);

    check_for_store_swapping();

    if ($db_types{$_[0]}) {
	$database = shift @_;
    } else {
	$database = $db_default;
    }

    $sqlstr = shift @_;
    $nff = shift @_;
    @named_parms = @_;

    # There should be an even number of named parms.  
    MILTON::fatal_error("Incorrect number of parameters to db_do") unless
        ($#named_parms % 2);
	
    MILTON::fatal_error("Tainted: $sqlstr") if( $check_taint 
						and SEC::is_tainted($sqlstr));

    $dbh = db_connect($database, $nff);

    ## prepare and execute sqlstr, caching requests with named
    ## parameters
    if($#named_parms > -1) {
	$sth = $dbh->prepare_cached($sqlstr) || 
	    return local_error($dbh->errstr, $nff);

	# Bind parms, with any flagged parms as integers
	my($i, $parm_num);
	$parm_num = 1;
	for($i = 0; $i < $#named_parms; $i+=2) {
	    if($named_parms[$i+1]) {
		$sth->bind_param($parm_num, $named_parms[$i], 
			       DBI::SQL_INTEGER);
	    } else {
		$sth->bind_param($parm_num, $named_parms[$i]);
	    }

	    $parm_num++;
	}
    } else {
	$sth = $dbh->prepare($sqlstr) || 
	    return local_error($dbh->errstr, $nff);
    }

    $sth->execute() || return local_error($dbh->errstr, $nff);
    @selected = $sth->fetchrow_array();
    $sth->finish();

    db_disconnect($dbh);

    if ($log_db_access) {
      open(HOHUM,">>/tmp/hohum");
      print HOHUM $sqlstr . "\n";
      close(HOHUM);
    }

    return @selected;
}

sub old_db_fetchrow_array {
    ## declare local variables
    my ($database, $sqlstr, $nff, $sth, @array, $dbh);

    check_for_store_swapping();

    ## make sure $dbh is a database handle
    if (!$db_types{$_[0]}) {
	## assume that $database was not passed
	($sqlstr, $nff) = @_;
	$database = $db_default;
    } else {
	## assume it was
	($database, $sqlstr, $nff) = @_;
    }

    MILTON::fatal_error("Tainted: $sqlstr") if( $check_taint 
						and SEC::is_tainted($sqlstr));

    $dbh = db_connect($database, $nff);

    $sth = $dbh->prepare($sqlstr) || return local_error($dbh->errstr, $nff);
    $sth->execute() || return local_error($dbh->errstr, $nff);
    @array = $sth->fetchrow_array();
    $sth->finish();

    db_disconnect($dbh);

    if ($log_db_access) {
      open(HOHUM,">>/tmp/hohum");
      print HOHUM $sqlstr . "\n";
      close(HOHUM);
    }

    return @array;
}

##-----------------------------------------------------------------------------
##   func:  db_fetchrow_arrayref
##  param:  $database, $sqlstr, $nff
##    ret:  $array/ undef                : reference to returned array
##  notes:
##-----------------------------------------------------------------------------
sub db_fetchrow_arrayref {

    ## declare local variables
    my ($database, $sqlstr, $nff, $dbh, $sth, $selected, @named_parms);

    check_for_store_swapping();

    if ($db_types{$_[0]}) {
	$database = shift @_;
    } else {
	$database = $db_default;
    }

    $sqlstr = shift @_;
    $nff = shift @_;
    @named_parms = @_;

    # There should be an even number of named parms.  
    MILTON::fatal_error("Incorrect number of parameters to db_do") unless
        ($#named_parms % 2);
	
    MILTON::fatal_error("Tainted: $sqlstr") if( $check_taint 
						and SEC::is_tainted($sqlstr));

    $dbh = db_connect($database, $nff);

    ## prepare and execute sqlstr, caching requests with named
    ## parameters
    if($#named_parms > -1) {
	$sth = $dbh->prepare_cached($sqlstr) || 
	    return local_error($dbh->errstr, $nff);

	# Bind parms, with any flagged parms as integers
	my($i, $parm_num);
	$parm_num = 1;
	for($i = 0; $i < $#named_parms; $i+=2) {
	    if($named_parms[$i+1]) {
		$sth->bind_param($parm_num, $named_parms[$i], 
			       DBI::SQL_INTEGER);
	    } else {
		$sth->bind_param($parm_num, $named_parms[$i]);
	    }

	    $parm_num++;
	}
    } else {
	$sth = $dbh->prepare($sqlstr) || 
	    return local_error($dbh->errstr, $nff);
    }

    $sth->execute() || return local_error($dbh->errstr, $nff);
    $selected = $sth->fetchrow_arrayref();
    $sth->finish();

    db_disconnect($dbh);

    if ($log_db_access) {
      open(HOHUM,">>/tmp/hohum");
      print HOHUM $sqlstr . "\n";
      close(HOHUM);
    }

    return $selected if (defined ($selected));
    return \();
}



sub old_db_fetchrow_arrayref {
    ##declare local vars
    my ($database, $sqlstr, $nff, $sth, $array, $dbh);

    check_for_store_swapping();

    ## make sure $dbh is a database handle
    if (!$db_types{$_[0]}) {
	## assume that $database was not passed
	($sqlstr, $nff) = @_;
	$database = $db_default;
    } else {
	## assume it was
	($database, $sqlstr, $nff) = @_;
    }

    MILTON::fatal_error("Tainted: $sqlstr") if( $check_taint 
						and SEC::is_tainted($sqlstr));

    $dbh = db_connect($database, $nff);

    $sth = $dbh->prepare($sqlstr) || return local_error($dbh->errstr, $nff);
    $sth->execute() || return local_error($dbh->errstr, $nff);
    $array = $sth->fetchrow_arrayref();
    $sth->finish();

    db_disconnect($dbh);

    if ($log_db_access) {
      open(HOHUM,">>/tmp/hohum");
      print HOHUM $sqlstr . "\n";
      close(HOHUM);
    }

    return $array if (defined ($array));
    return \();
}

##-----------------------------------------------------------------------------
##   func:  db_fetchall_arrayref
##  param:  $database, $sqlstr, $nff
##    ret:  $ref/ undef
##  notes:
##-----------------------------------------------------------------------------
sub db_fetchall_arrayref {

    ## declare local variables
    my ($database, $sqlstr, $nff, $dbh, $sth, $selected, @named_parms);

    check_for_store_swapping();

    if ($db_types{$_[0]}) {
	$database = shift @_;
    } else {
	$database = $db_default;
    }

    $sqlstr = shift @_;
    $nff = shift @_;
    @named_parms = @_;

    # There should be an even number of named parms.  
    MILTON::fatal_error("Incorrect number of parameters to db_do") unless
        ($#named_parms % 2);
	
    MILTON::fatal_error("Tainted: $sqlstr") if( $check_taint 
						and SEC::is_tainted($sqlstr));

    $dbh = db_connect($database, $nff);

    ## prepare and execute sqlstr, caching requests with named
    ## parameters
    if($#named_parms > -1) {
	$sth = $dbh->prepare_cached($sqlstr) || 
	    return local_error($dbh->errstr, $nff);

	# Bind parms, with any flagged parms as integers
	my($i, $parm_num);
	$parm_num = 1;
	for($i = 0; $i < $#named_parms; $i+=2) {
	    if($named_parms[$i+1]) {
		$sth->bind_param($parm_num, $named_parms[$i], 
			       DBI::SQL_INTEGER);
	    } else {
		$sth->bind_param($parm_num, $named_parms[$i]);
	    }

	    $parm_num++;
	}
    } else {
	$sth = $dbh->prepare($sqlstr) || 
	    return local_error($dbh->errstr, $nff);
    }

    $sth->execute() || return local_error($dbh->errstr, $nff);
    $selected = $sth->fetchall_arrayref();
    $sth->finish();

    db_disconnect($dbh);

    if ($log_db_access) {
      open(HOHUM,">>/tmp/hohum");
      print HOHUM $sqlstr . "\n";
      close(HOHUM);
    }

    return $selected if (defined ($selected));
    return \();
}


sub old_db_fetchall_arrayref {
    ##declare local vars
    my ($database, $sqlstr, $nff, $sth, $array, $dbh);

    check_for_store_swapping();

    ## make sure $dbh is a database handle
    if (!$db_types{$_[0]}) {
	## assume that $database was not passed
	($sqlstr, $nff) = @_;
	$database = $db_default;
    } else {
	## assume it was
	($database, $sqlstr, $nff) = @_;
    }

    MILTON::fatal_error("Tainted: $sqlstr") if( $check_taint 
						and SEC::is_tainted($sqlstr));

    $dbh = db_connect($database, $nff);

    $sth = $dbh->prepare($sqlstr) || return local_error($dbh->errstr, $nff);
    $sth->execute() || return local_error($dbh->errstr, $nff);
    $array = $sth->fetchall_arrayref();
    $sth->finish();

    db_disconnect($dbh);

    if ($log_db_access) {
      open(HOHUM,">>/tmp/hohum");
      print HOHUM $sqlstr . "\n";
      close(HOHUM);
    }

    return $array;
}

##-----------------------------------------------------------------------------
##   func:  db_fetchcol_arrayref
##  param:  $database, $sqlstr, $nff
##    ret:  $ref/ undef
##  notes:  wrapper around db_fetchcol_arrayref, returns a ref to an array of
##          scalars, just like DBI::selectcol_arrayref
##-----------------------------------------------------------------------------
sub db_fetchcol_arrayref {

    my($rows) = db_fetchall_arrayref(@_);

    return $rows unless(defined($rows));

    my(@col);
    my($row);
    for $row (@$rows) {
	push @col, $row->[0];
    }

    return \@col;
}

##-----------------------------------------------------------------------------
##   func:  seq_next_get
##  param:  $database, $seq_name, $nff
##    ret:  $next_num/ undef
##  notes:  this takes either both a $dbh and $database or neither and the
##          default is assumed
##-----------------------------------------------------------------------------

sub seq_next_get {
    my ($database, $seq_name, $nff, $next_num);

    check_for_store_swapping();

    ## make sure $dbh is a database handle
    if (!$db_types{$_[0]}) {
	## assume that $database was not passed
	($seq_name, $nff) = @_;
	$database = $db_default;
    } else {
	## assume it was
	($database, $seq_name, $nff) = @_;
    }

    ## call the DB specific code
    $next_num = &{$db_seq_next{$database}}($seq_name, $nff);

    if ($log_db_access) {
      open(HOHUM,">>/tmp/hohum");
      print HOHUM "requested sequence: $seq_name\n";
      close(HOHUM);
    }
    
    return $next_num;

}

##-----------------------------------------------------------------------------
##   func:  verify_id
##  param:  $obj_typeid, $obj_table, \$id, $fatal_error
##    ret:  1 / 0
##  notes:  Attempts to verify that a specified id is of valid construction,
##          and refers to an extant object in the database.
##          If $fatal_error is defined, the function returns a 1 or calls
##          MILTON::fatal_error.
##          If $fatal_error is not defined, either 1 or 0 is returned.
##-----------------------------------------------------------------------------

sub verify_id {
    my ($database, $obj_typeid, $obj_table, $id, $fatal_error, $nid, $sqlstr);

    check_for_store_swapping();

    if (!$db_types{$_[0]}) {
	## assume that $database was not passed
	($obj_typeid, $obj_table, $id, $fatal_error) = @_;
	$database = $db_default;
    } else {
	## assume it was
	($database, $obj_typeid, $obj_table, $id, $fatal_error) = @_;
    }


    if ( !SEC::is_id($$id) ) {
	if (defined($fatal_error)) {
	    MILTON::fatal_error("Missing / invalid id.  field:[$obj_typeid] table:[$obj_table] id:[$$id]");
	} else {
	    return 0;
	}
    }    

    SEC::untaint_ref($id) if $check_taint;

    $sqlstr = "select ${obj_typeid} from ${obj_table}
             where ${obj_typeid} = '$$id'";

    ($nid) = db_fetchrow_array($database, $sqlstr);

    if ( !SEC::is_id($nid) ) {
	if (defined($fatal_error)) {
	    MILTON::fatal_error("No such record. [$obj_table] [$obj_typeid] [$id]");
	} else {
	    return 0;
	}
    }    
    
    return 1;
}


##-----------------------------------------------------------------------------
##   func:  db_string_clean
##  param:  $database, $string
##    ret:  $string/ undef
##  notes:  this takes a string and returns one properly escaped for the 
##          called database
##-----------------------------------------------------------------------------

sub db_string_clean {
    my ($database, $str);

    check_for_store_swapping();

    ## make sure $dbh is a database handle
    if (!$db_types{$_[0]}) {
	## assume that $database was not passed
	($str) = @_;
	$database = $db_default;
    } else {
	## assume it was
	($database, $str) = @_;
    }

    $str = &{$db_str_clean{$database}}($str);

    SEC::untaint_ref(\$str) if $check_taint;

    return $str;
}

##-----------------------------------------------------------------------------
##   func:  date_string_make
##  param:  $database, $sec, $min, $hr, $day, $mon, $yr
##    ret:  $string/ undef
##  notes:  this takes a string and returns one properly escaped for the 
##          called database
##-----------------------------------------------------------------------------

sub date_string_make {
    my ($database, $str, $sec, $min, $hr, $day, $mon, $yr);

    check_for_store_swapping();

    ## make sure $dbh is a database handle
    if (!$db_types{$_[0]}) {
	## assume that $database was not passed
	($sec, $min, $hr, $day, $mon, $yr) = @_;
	$database = $db_default;
    } else {
	## assume it was
	($database, $sec, $min, $hr, $day, $mon, $yr) = @_;
    }

    $str = &{$db_date_str{$database}}($sec, $min, $hr, $day, $mon, $yr);
    return $str;
}

##-----------------------------------------------------------------------------
##   func:  db_cleanup
##  param:  none
##    ret:  none
##  notes:  this will close any open database connections and rollback any
##          open transactions--don't call this unless you are on your way
##          out--you have been warned.
##-----------------------------------------------------------------------------

sub db_cleanup {

    my ($db, $dbh);

    ## we will iterate though the db types
    foreach $db (keys %db_types) {
	
	if (defined ($dbh_ref{$db})) {
	    ## make sure any open transactions are rolled back
	    if ($trans_count{$db} > 0) {
		## let's try not to cause any infinite recuresion
		print STDERR "Closing transactions for $db.\n";
		db_transact_rb($db, 1);
	    }
	    
	    ## make sure that any open database handles are closed for real
	    
	    if (($dbh_count{$db} != 0) && ($dbh_ref{$db}->ping())) {
		print STDERR "Closing database handle for $db.\n";
		$dbh_ref{$db}->disconnect();
	    }
	}
    }
}
	


###############################################################################
##-----------------------------------------------------------------------------
##  TRANSACTION STUFF
##
##  This stuff is a little different, make sure you read the notes.  We have
##  two hashes.  One for begin and one for end.  These will contain the
##  sql strings to accomplish this.  A more sophisticated approach would
##  have actuall functions to call in each case.  Someday.
##-----------------------------------------------------------------------------
###############################################################################

##-----------------------------------------------------------------------------
##   func:  db_transact_begin
##  param:  $database, $nff
##    ret:  1/ undef
##  notes:  call this with either the $dbh and $database, or nothing
##    def:  this is ref counted so only the top level transaction
##          is actually activated
##-----------------------------------------------------------------------------

sub db_transact_begin {
    my ($database, $nff);
    
    check_for_store_swapping();

    ## make sure $dbh is a database handle
    if (!$db_types{$_[0]}) {
	## assume that $database was not passed
	($nff) = @_;
	$database = $db_default;
    } else {
	## assume it was
	($database, $nff) = @_;
    }
    
    $trans_count{$database} = 0 unless (defined ($trans_count{$database}));

    ## this is reference counted, so only the top level transaction is on
    if ($trans_count{$database} == 0) {

	## call the db specific code
	return undef if (!defined (&{$trans_beg{$database}}($nff)));
    }

    $trans_count{$database}++;

    return 1;
}    

##-----------------------------------------------------------------------------
##   func:  db_transact_end
##  param:  $database, $nff
##    ret:  1/ undef
##  notes:  it is up to you to make sure that this is called after 
##          transact_beg the right number of times.  I can't be responsible 
##          for your mistakes.
##-----------------------------------------------------------------------------

sub db_transact_end {
    my ($database, $nff);
    
    ## see if we have a $database passed
    if (!$db_types{$_[0]}) {
	## assume that $database was not passed
	($nff) = @_;
	$database = $db_default;
    } else {
	## assume it was
	($database, $nff) = @_;
    }


    ## only run the code if this is the outer transaction end

    if ($trans_count{$database} == 1) {
	## in this case, actually close the transaction

	return undef if (!defined(&{$trans_end{$database}}($nff)));
    }

    $trans_count{$database}--;

    return 1;
}

##-----------------------------------------------------------------------------
##   func:  db_transact_rb
##  param:  $database, $nff
##    ret:  1/ undef
##  notes:  this will clean up after a failed transaction
##          checking for errors and dealing with them is the Right Thing(tm)
##-----------------------------------------------------------------------------

sub db_transact_rb {
    my ($database, $nff);
    
    ## make sure we have an actual database type
    if (!$db_types{$_[0]}) {
	## assume that $database was not passed
	($nff) = @_;
	$database = $db_default;
    } else {
	## assume it was
	($database, $nff) = @_;
    }

    ## clean out the reference count
    $trans_count{$database} = 0;

    return undef if (!defined(&{$trans_rb{$database}}($nff)));

    return 1;
}



###############################################################################
##-----------------------------------------------------------------------------
##  ERROR SECTION
##
##  This is where the error function(s) hang out
##-----------------------------------------------------------------------------
###############################################################################


##-----------------------------------------------------------------------------
##   func:  local_error
##  param:  $errstr, $nff
##    ret:  1 or not at all
##  notes:  this calls fatal_error of non_fatal_error, depending of $nff
##-----------------------------------------------------------------------------


sub local_error {
    my($errstr, $nff) = @_;

    return &non_fatal_error($errstr) if ($nff);
    MILTON::fatal_error($errstr);
}

##-----------------------------------------------------------------------------
##   func:  non_fatal_error
##  param:  $errstr
##    ret:  undef--this is the return condition of failed functions
##  notes:  this will silently log the error message and continue on, make 
##          you have r/w permissions on the errlog you choose
##-----------------------------------------------------------------------------

sub non_fatal_error {
    my ($errstr) = @_;

    $errlog = "$ENV{'TALLYMAN_PATH'}/stores/$ENV{'TALLYMAN_SID'}/logs/error.log";

    if ($check_taint) {
      SEC::untaint_ref(\$errlog);
    }

    if (open(ERRORLOG,">>$errlog")) {
	my($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst)=localtime(time);
	print ERRORLOG "$mday/$mon/$year $hour:$min:$sec $errstr\n";
	print STDERR "$mday/$mon/$year $hour:$min:$sec $errstr\n";
	close(ERRORLOG);
    } else {
	print STDERR "\nCan't open error_log!";
    }
    
    return undef;
}

##-----------------------------------------------------------------------------
##   func:  swap_in_new_store_settings
##  param:  none
##    ret:  nothing
##-----------------------------------------------------------------------------

sub swap_in_new_store_settings {
  my($sid) = $current_storeid;

  # swap out the old store

  if ($sid ne '') {
    $stores{$sid . 'db_default'}  = $db_default;

    $stores{$sid . 'db_connect'}  = {};
    $stores{$sid . 'dbh_count'}   = {};
    $stores{$sid . 'dbh_ref'}     = {};
    $stores{$sid . 'trans_count'} = {};

    %{$stores{$sid . 'dbh_count'}}   = %dbh_count;
    %{$stores{$sid . 'dbh_ref'}}     = %dbh_ref;
    %{$stores{$sid . 'trans_count'}} = %trans_count;
  }

  $current_storeid = $ENV{'TALLYMAN_SID'};
  SEC::untaint_ref(\$current_storeid);
  $sid = $current_storeid;

  if (!defined($stores{$sid . 'db_default'})) {

    # we have to load a new store

    $db_default = CONFIG::get_entry('db_default');

    %dbh_count  = ( 'ORACLE' => 0, 
                    'PG'     => 0, 
                    'SB'     => 0, );

    %dbh_ref    = ( 'ORACLE' => undef, 
                    'PG'     => undef, 
                    'SB'     => undef, );

    %trans_count     = ( 'ORACLE' => 0,
                         'PG'     => 0,
                         'SB'     => 0, );

  } else {

    # swap in an old store

    $db_default  =   $stores{$sid . 'db_default'};
    %dbh_count   = %{$stores{$sid . 'dbh_count'}};
    %dbh_ref     = %{$stores{$sid . 'dbh_ref'}};
    %trans_count = %{$stores{$sid . 'trans_count'}};
  }
}

##-----------------------------------------------------------------------------
##   func:  check_for_store_swapping
##  param:  none
##    ret:  none
##-----------------------------------------------------------------------------

sub check_for_store_swapping {
  if (!$ENV{'TALLYMAN_SID'}) {
    MILTON::fatal_error("No TALLYMAN_SID environment variable set!");
  }

  if (!$ENV{'TALLYMAN_PATH'}) {
    MILTON::fatal_error("No TALLYMAN_PATH environment variable set!");
  }

  if ($current_storeid ne $ENV{'TALLYMAN_SID'}) {
    swap_in_new_store_settings();
  }
}

##-----------------------------------------------------------------------------
##   func:  fatal_error
##  param:  $errstr
##    ret:  not at all
##  notes:  this one just dies forever with a nice HTML error message
##          this goes back into milton--sorry for the mixup
##-----------------------------------------------------------------------------

## return a one to keep the other packages happy
1;
