# ITEM.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

package ITEM;

use strict;

require KNAR;
require DBLIB;
require MATRIX;
require SNIPPET;
require SNIPTYPE;
require CONTAINER;
require THINGTYPE;
require STOCK;

# 
# AVAILABLE FUNCTIONS AND VARIABLES DEFINED IN THIS PACKAGE:
#
#   &item_list_all
#   &item_create
#   &item_create_plain
#   &item_create_mix
#   &item_delete
#   &item_get_info
#   &item_get_label
#   &item_get_id
#   &item_set_label
#   &item_set_attr
#   &item_set_mixid_and_stockid
#   &item_find_by_name
#   &item_get_all_stock_ids
#   &item_verify_id
#   &item_widget_analysis_get
#   &item_widget_analysis_do
#
#   &stock_deallocate
#
#   &mix_create
#   &mix_delete
#   &mix_delete_existing_cells
#   &mix_get_grid
#   &mix_get_grid_with_dimid
#   &mix_lookup_cell_given_item
#   &mix_update_price
#   &mix_verify_id
#
#   &cell_create_mix
#   &cell_delete
#   &cell_find_given_ranges
#   &cell_get_range_labels
#   &cell_verify_id
#
#   &node_create_cell
#   &node_list_cell
#

#
# =============================================================================
#
# ITEM API
#
# =============================================================================
#

use vars qw($ITEM_WIDGET_THRESHOLD);

$ITEM_WIDGET_THRESHOLD = KNAR::knar_entry_get('ITEM_WIDGET_THRESHOLD');

if (!defined($ITEM_WIDGET_THRESHOLD)) {
    $ITEM_WIDGET_THRESHOLD = 10;
}
 
#
# -----------------------------------------------------------------------------
#
# get a list of all item, sorted by label
#   takes: nothing
# returns: arrayref of id, label arrayrefs
#

sub item_list_all {

  my($sqlstr) ="select item_id, label
                from item
                where item_id <> 0
                order by label";

  return DBLIB::db_fetchall_arrayref($sqlstr);
}

#
# -----------------------------------------------------------------------------
# create a new item, unattached to a stock item or a mix
#   takes: label
# returns: id of new item
#   notes: you will have to call item_set_stockid_and_mixid at a later
#          time to attach this item to something useful.
#

sub item_create {
  my($label, $type_id) = @_;

  my($itemid) = DBLIB::seq_next_get("item_sq");

  # create the item

  DBLIB::db_do("insert into item
                (item_id, label, stock_id, mix_id)
                values
                ($itemid, '$label', 0, 0)");

  # add to the connecting table
  THINGTYPE::thingtype_add_thing($THING::item, $type_id, $itemid);

  return $itemid;
}

#
# -----------------------------------------------------------------------------
# create a new item
#   takes: label,sku,price,weight,volume,
#          atp_qty,reserve_threshold,def_reserve_qty, type_id
# returns: id of new item
#

sub item_create_plain {
  my($label, $sku, $price, $weight, $volume, $atp, $rt, $drq, $type_id) = @_;

  my ($inv_id) = STOCK::inv_create_local($sku, $price, $weight, $volume, $atp,
                                         $rt, $drq);
  my ($stock_id) = STOCK::assoc($inv_id, "LOCAL");

  # get next item id

  my $itemid = DBLIB::seq_next_get("item_sq");

  # create the item
  DBLIB::db_do("insert into item
                (item_id, label, stock_id)
                values
                ($itemid, '$label', $stock_id)");

  # add to the connecting table
  THINGTYPE::thingtype_add_thing($THING::item, $type_id, $itemid);

  return $itemid;
}

#
# -----------------------------------------------------------------------------
#
# create a new item attached to a mix
#   takes: label, mix_id
# returns: id of new item
#

sub item_create_mix {
  my($label, $mix_id, $type_id) = @_;

  # get the next item id

  my $itemid = DBLIB::seq_next_get("item_sq");

  # insert the item

  DBLIB::db_do("insert into item
                (item_id, label, mix_id)
                values
                ($itemid, '$label', '$mix_id')");

  # add to the connecting table
  THINGTYPE::thingtype_add_thing($THING::item, $type_id, $itemid);

  return $itemid;
}

#
# -----------------------------------------------------------------------------
#
# deletes an item
#   takes: item_id
# returns: nothing
#

sub item_delete {
  my($id) = @_;

  DBLIB::db_transact_begin();

  # get the stock id & mix id

  my($stockid, $mixid) = DBLIB::db_fetchrow_array("select stock_id, mix_id 
                                                   from item 
                                                   where item_id = '$id'");

  if ($mixid !=0) {
    &item_delete_mix($id, $mixid);

  } elsif ($stockid != 0) {
    &item_delete_plain($id, $stockid);

  } else { # hmmm. both are 0?  I suppose it's possible.

    &item_delete_supporting_stuff($id);    

    DBLIB::db_do("delete from item where item_id = '$id'");
  }

  DBLIB::db_transact_end();
}

#
# -----------------------------------------------------------------------------
#
# deletes a plain item -- NOT A PUBLIC INTERFACE
#   takes: item_id, stock_id
# returns: nothing
#   notes: XXX deletes from the stock table as well.  That'll have to change.
#

sub item_delete_plain {
  my($itemid, $stockid)=@_;
   
  # delete all of the supporting stuff

  &item_delete_supporting_stuff($itemid);
   
  # delete from the item table

  DBLIB::db_do("delete from item where item_id = '$itemid'");

  # delete from the stock table

  STOCK::deassoc($stockid);
}

#
# -----------------------------------------------------------------------------
#
# deletes a mixed item -- NOT A PUBLIC INTERFACE
#   takes: item_id, mix_id
# returns: nothing
#

sub item_delete_mix {
  my($itemid, $mixid) = @_;

  # delete all of the supporting stuff.

  &item_delete_supporting_stuff($itemid);

  # delete all of the mix stuff
  ITEM::item_set_mixid_and_stockid($itemid,0,0);
  ITEM::mix_delete($mixid);

  # delete from the item table

  DBLIB::db_do("delete from item where item_id = '$itemid'");
}

#
# -----------------------------------------------------------------------------
#
# deletes everything that depends on an item key
#

sub item_delete_supporting_stuff {
  my($itemid) = @_;

  # delete from snip

  SNIPPET::snip_delete_all($THING::item, $itemid);

  # delete from the thing<->type table

  THINGTYPE::thingtype_delete_thing_to_type($THING::item, $itemid);

  # delete from container system

  CONTAINER::container_delete_item($itemid);

  # delete from item_image

  #ITEMIMAGE::itemimage_delete_all($itemid);
}

#
# -----------------------------------------------------------------------------
#
# returns the information of the specified item.
# if the item is a mix, it returns the information of the first available
# cell.
#
#   takes: item_id
# returns: label, sku, price, weight, volume, atp_qty, reserve_threshhold,
#          def_reserve_qty, stock_id, mix_id, cell_id
#

sub item_get_info {
  my($item_id) = @_;
  my($sku, $price, $weight, $volume, $atp, $rt, $drq, $invid);

  # get the label, mix_id and stock_id

  my($label, $stock_id, $mix_id) = DBLIB::db_fetchrow_array(
	      "select label, stock_id, mix_id 
               from item where item_id = '$item_id'");

  my($cell_id) = 0;

  # is it a mix?  Find the first cell and retrieve its data

  if($mix_id != 0) {
    ($stock_id, $cell_id) = DBLIB::db_fetchrow_array(
            "select stock_id, cell_id from cell
             where mix_id = '$mix_id'");    

    # XXX hmmm. theoretically, it doesn't violate any fk constraints if
    # a mix exists, but no cells are present.  What do we do?  Well, something
    # is probably wrong; let's delete the mix, create a new local stock
    # item, and set up the item to be a simple product.
    # The other option would be to delete the mix, and then return a mix
    # and stock id of 0.  That might be the better solution... Hmmmmmmm.

    if (!defined($stock_id)) {
      item_set_mixid_and_stockid($item_id, 0, 0);     
      mix_delete($mix_id);
      $mix_id = 0;
      $cell_id = 0;
 
      $invid = STOCK::inv_create_local('AUTOCREATED -- VACUOUS MIX', 0, 0, 0,
 				       0, 0, 0);
      $stock_id = STOCK::assoc($invid, 'LOCAL');

      item_set_mixid_and_stockid($item_id, 0, $stock_id);     
    }
    
  }

  # now that we have the stock_id, get the rest of the info

  ($sku, $price, $weight, $volume, $atp, $rt, $drq) 
     = STOCK::get_info($stock_id);

  return($label, $sku, $price, $weight, $volume, $atp, $rt, $drq, $stock_id,
         $mix_id, $cell_id);
}

#
# -----------------------------------------------------------------------------
#
# returns the label of the specified item
#   takes: item_id
# returns: label
#

sub item_get_label {
  my($item_id) = @_;

  my($label)= DBLIB::db_fetchrow_array(
                   "select label from item where item_id = ?",
				       0, $item_id, 1);

  return($label);
}

#
# -----------------------------------------------------------------------------
# Given an item name, returns its id.
sub item_get_id {
    my($name) = @_;

    my($id) =  DBLIB::db_fetchrow_array("
        select item_id from item 
        where label=?", 0, $name, 0);

    return $id;
}


#
# -----------------------------------------------------------------------------
#
# sets the label of the specified item
#   takes: item_id, new label
# returns: nothing
#

sub item_set_label {
  my($item_id, $new_label)=@_;

  DBLIB::db_do("update item
                set label = '$new_label'
                where item_id = '$item_id'");
}

#
# -----------------------------------------------------------------------------
#
# sets any column of an item to the specified value.
# 

sub item_set_attr {
  my($iid, %vals) = @_;
  my($str, $key, @vkeys);

  @vkeys = keys(%vals);
  if ($#vkeys < 0) {
    return;
  }

  $str = "update item set ";
  foreach $key (@vkeys) {
    $str .= "$key='$vals{'$key'}', ";
  }
  chop $str;
  chop $str;

  $str .= " where item_id = '$iid'";

  DBLIB::db_do($str);
}

#
# -----------------------------------------------------------------------------
# sets the mixid and stockid of the specified item
#   takes: itemid,mixid,stockid
# returns: nothing
#

sub item_set_mixid_and_stockid {
  my($itemid, $mixid, $stockid)=@_;

  DBLIB::db_do("update item
                set mix_id = '$mixid', stock_id='$stockid'
                where item_id = '$itemid'");
}

#
# -----------------------------------------------------------------------------
#
# $command specifies how to search: substring, exact, and/or case 
# insensitive ('s', 'e', and 'i', respectively).  Substring and exact
# are mutually exclusive; if both are specified, substring takes precedence.
#
# @ids is a set of item_ids that can used to restrict the search.
#

sub item_find_by_name {
  my($label, $command, @ids) = @_;
  my($column, $op, $sqlstr, $ids_str);

  $column = "label";
  if ($command =~ /i/) {
    $label =~ tr/A-Z/a-z/;
    $column = "lower(label)";
  }

  # XXX hmmm. does 'like' work for databases other than oracle???

  $op = " = '$label'";
  if ($command =~ /s/) {
    $op = " like '%$label%'";
  }

  $sqlstr = "select item_id, label 
             from item 
             where $column $op";

  if ($#ids > -1) {
    my $ids_str = join(",", @ids);
    $sqlstr .= " and item_id in ($ids_str)";
  }

  return(DBLIB::db_fetchall_arrayref($sqlstr));
}

#
# -----------------------------------------------------------------------------
#
# gets a list of all stock ids associated with an item (either simple or
# mixed).
#

sub item_get_all_stock_ids {
  my($item_id) = shift;
  my($sqlstr, @hohum);

  $sqlstr = "select stock_id, mix_id from item where item_id = '$item_id'";
  my($stock_id, $mix_id) = DBLIB::db_fetchrow_array($sqlstr);

  if ($mix_id != 0) {
    $sqlstr = "select stock_id from cell where mix_id = '$mix_id'";
    return(DBLIB::db_fetchall_arrayref($sqlstr));
  } else {
    $hohum[0] = [$stock_id];
    return (\@hohum);
  }
}

#
# -----------------------------------------------------------------------------
#

sub item_verify_id {
  return DBLIB::verify_id('item_id', 'item', @_);
}

#
# -----------------------------------------------------------------------------
#

sub item_widget_analysis_get {
  my($item_id) = shift;

  #
  # try really hard to make sure that the correct snippets exist
  #

  my($wt_type_id) = SNIPTYPE::sniptype_get_id($THING::item, '_widget_text');
  if (!defined($wt_type_id) || $wt_type_id == 0) {
    item_widget_analysis_do($item_id);
    my($wt_type_id) = SNIPTYPE::sniptype_get_id($THING::item, '_widget_text');
    if (!defined($wt_type_id) || $wt_type_id == 0) {
      MILTON::fatal_error("Could not perform widget analysis on $item_id!");
    }
  }

  my($wn_type_id) = SNIPTYPE::sniptype_get_id($THING::item, '_widget_typenum');
  if (!defined($wt_type_id) || $wt_type_id == 0) {
    item_widget_analysis_do($item_id);
    my($wn_type_id) = SNIPTYPE::sniptype_get_id($THING::item,
                                                '_widget_typenum');
    if (!defined($wn_type_id) || $wn_type_id == 0) {
      MILTON::fatal_error("Could not perform typenum analysis on $item_id!");
    }
  }

  my($typenum) = SNIPPET::snip_get($THING::item, $item_id, $wn_type_id);
  my($typetext) = SNIPPET::snip_get($THING::item, $item_id, $wt_type_id);

  return($typenum, $typetext);
}

#
# -----------------------------------------------------------------------------
#

sub item_widget_analysis_do {
  my($item_id) = shift;
  my($num, $text) = item_real_analyze_widgets($item_id);

#  $num = DBLIB::db_string_clean($num);
#  $text = DBLIB::db_string_clean($text);

  # make sure certain snippets exist

  my($numid) = item_widget_ensure_sniptype_exists($item_id, '_widget_typenum');
  my($textid) = item_widget_ensure_sniptype_exists($item_id, '_widget_text');

  # store the results of the analysis

  SNIPPET::snip_update($THING::item, $item_id, $numid, $num);  
  SNIPPET::snip_update($THING::item, $item_id, $textid, $text);  
}

sub item_widget_ensure_sniptype_exists {
  my($item_id, $sname) = @_;

  my($stid) = SNIPTYPE::sniptype_get_id($THING::item, $sname);
  if (!defined($stid) || $stid == 0) {

    my($types) = THINGTYPE::thingtype_get_types($THING::item, $item_id);

    $stid = SNIPTYPE::sniptype_create($THING::item, $types->[0], 
                                          $sname, 'system field',
                                          '', 'local', $SNIPTYPE::hidden, '');

#    THINGTYPE::thingtype_append($THING::item, $types->[0], $stid);
  }

  return($stid);
}

#
# -----------------------------------------------------------------------------
#
# THIS IS NOT A PUBLIC INTERFACE
#

sub item_real_analyze_widgets {
  my($item_id) = shift;
  my(@skip_col, $tmpstr, $tmpstr2, $result);
  my($i, $rj, $j, $base);

  #
  # get the mix_id
  #

  my($label, $sku, $price, $weight, $volume, $atp, $rt, $drq, $stock_id,
     $mix_id, $cell_id) = item_get_info($item_id);

  if ($mix_id == 0) {
    return(0, "NO WIDGETS NEEDED");
  }

  my($gridref, $height, $width) = mix_get_grid($mix_id);

  #
  # simple case: the total grid is small.  Just give 'em a single
  # selection box with all permutations.
  #

  if ($height < $ITEM_WIDGET_THRESHOLD) {
    return(1, item_analysis_simple_case($gridref, $height, $width));
  }

  #
  # reduction case: all of the nodes in a column of the grid are the same.
  # obviously, they don't need a choice, since there's only one option.
  #

  COLUMN: for ($j=1; $j<$width; $j++) {

    $base = $gridref->[0][$j];
    for ($i=0; $i<$height; $i++) {
      if ($gridref->[$i][$j] != $base) {
        next COLUMN;
      }
    }

    # if we get here, the whole column is the same.
    # add it to the skip list

    $skip_col[$j] = 1;
  }

  #
  # separation case
  # 

  # build an n-dimensional product cube.  If there are any holes, we'll have
  # to use the wizard.

  my(@dimcount,%phash);

  # dimcount is the number of unique elements in each dimension.
  # if we multiple them all together, that should be equal to the
  # height of our mix grid.  If they're not equal, it means that there
  # are holes in the grid; we therefore have to use a wizard.

  # first, determine uniqueness of items.  These actually turn out to 
  # be the labels of the possible positions of the cube. Kind of like
  # coordinates into the cube.  Think about it.
  #
  # (cube is actually a bad word, because it implies only three dimensions.  
  # think of it as an n-dimensional solid)
  #
  # here, we double-map the thing; that is, we map going from label to 
  # position, and from position to label.
  #
  # This was a lot of fun to code. 
  #

  for ($j=1; $j<$width; $j++) {
    $dimcount[$j] = 0;
    if ($skip_col[$j]) { next; }

    for ($i=0; $i<$height; $i++) {
      if (!defined($phash{"${j}_$gridref->[$i][$j]"})) {
        $phash{"${j}_$gridref->[$i][$j]"} = $dimcount[$j];
        $phash{"${j}_$dimcount[$j]"}=$gridref->[$i][$j];
        $dimcount[$j]++;
      }
    }
  }

  # check to see if it's fully populated

  my($total) = 1;
  for ($j=1; $j<$width; $j++) {
    if (!$skip_col[$j]) {
      $total *= $dimcount[$j];
    }
  }

  if ($total == $height) {

    # yes; just print out all unique possibilities

    $result = "";

    for ($j=1; $j<$width; $j++) {
      $rj = $j-1;

      if ($skip_col[$j]) {
        $result .= "<input type=hidden name=\"widget_%%_$rj\" value=\"$gridref->[0][$j]\">\n";

      } else {

        $tmpstr = "<select name=\"widget_%%_$rj\">\n";

        for ($i=0; $i<$dimcount[$j]; $i++) {
          $total = $phash{"${j}_$i"};
          $tmpstr .= "<option value=\"$total\">";
          $tmpstr2 = MATRIX::range_get_label($total);
          $tmpstr2 =~ s/%//g;
          $tmpstr .= $tmpstr2;
          $tmpstr .= "</option>\n";
        }

        $tmpstr .= "</select>\n";
        $result .= $tmpstr;
      }
    }

    return(2, $result);

  } else {

    # since the wizard isn't an option at this point, we need to give
    # them the option to dig themselves into a hole.  It's really too bad.
    # But we did the best we could.

    return(2, item_analysis_full_matrix_case($mix_id));

#    # um.  still no good?  well, it's too complicated, then, so we just
#    # return a wizard button.
#
#   return (3,"<input type=submit name=\"wiz%%\" value=\"Change options\">\n");
  }
}

#
# -----------------------------------------------------------------------------
#
# THIS IS NOT A PUBLIC INTERFACE
# total number of mix'ed variations is less than ITEM_WIDGET_THRESHOLD.
# print out a single selection box with all options.
#

sub item_analysis_simple_case {
  my($gridref, $height, $width) = @_;

  my($result) = '<select name="widget_simple_%%">';
  my($tmpstr);
  my($i,$j);

  for ($i=0; $i<$height; $i++) {
    $tmpstr="";
    for ($j=1; $j<$width; $j++) {
      $tmpstr .= MATRIX::range_get_label($gridref->[$i][$j]) . "--";
    }
    chop $tmpstr; chop $tmpstr;
    $tmpstr =~ s/%//g;
    $result .= "<option value=\"$gridref->[$i][0]\">$tmpstr</option>\n";
  }

  $result .= "</select>\n";

  return ($result);
}

#
# -----------------------------------------------------------------------------
#
# THIS IS NOT A PUBLIC INTERFACE
# total number of mix'ed variations is less than ITEM_WIDGET_THRESHOLD.
# print out a single selection box with all options.
#

sub item_analysis_full_matrix_case {
  my($mix_id) = shift;
  my($matrix_id) = MATRIX::matrix_get_id_from_mix_id($mix_id);
  my($result, $tmpstr, $dimref, $rangeref, $k, $j);

  $dimref = MATRIX::dim_list_matrix($matrix_id);
  $k=0;

  while ($k < $#{$dimref}+1) {
    $rangeref = MATRIX::range_list_dim($dimref->[$k][0]);
    $result .= "<select name=\"widget_%%_$k\">\n";

    $j=0;
    while ($j < $#{$rangeref}+1) {
      $tmpstr = $rangeref->[$j][0];
      $tmpstr =~ s/%//g;
      $result .= "<option value=\"$tmpstr\"> $rangeref->[$j][1]\n";
      $j++;
    }
    $result .= "</select>";
    $k++;
  }

  return ($result);
}


#
# =============================================================================
#
# MIX API
#
# =============================================================================
#

#
# -----------------------------------------------------------------------------
# create a mix
#   takes: nothing
# returns: mix_id
#

sub mix_create {

  my($nextid) = DBLIB::seq_next_get("mix_sq");

  DBLIB::db_do("insert into mix (mix_id, next_cell_seq_no)
                values
                ($nextid,'0')");

  return($nextid); 
}

#
# -----------------------------------------------------------------------------
# delete a mix
#   takes: mix_id
# returns: nothing
#

sub mix_delete {
  my($mixid) = @_;

  DBLIB::db_transact_begin();
  ITEM::mix_delete_existing_cells($mixid);

  DBLIB::db_do("delete from mix where mix_id = '$mixid'");
  DBLIB::db_transact_end();
}

#
# -----------------------------------------------------------------------------
# deletes all cells, nodes, and stock items that depend on a mix
#   takes: mix_id
# returns: nothing
#

sub mix_delete_existing_cells {
  my($mixid) = @_;

  my($rows) = DBLIB::db_fetchall_arrayref(
	 "select cell_id, stock_id from cell where cell.mix_id = '$mixid'");

  my($row, $cellid, $stockid);

  for $row (@$rows) {
    $cellid = $row->[0];
    $stockid = $row->[1];
      
    DBLIB::db_do("delete from node where node.cell_id = '$cellid'");

    DBLIB::db_do("delete from cell where cell.cell_id = '$cellid'");

    STOCK::deassoc($stockid);
  }
}
#
# -----------------------------------------------------------------------------
# returns the mix grid for the specified mix
#
#   takes: mix_id
# returns: a reference to a two dimensional array: 
#            stockid, rangeid, rangeid...
#            stockid, rangeid, rangeid...
#            ...
#          , the height of the grid and the the width of the grid.
#

sub mix_get_grid {
  return real_mix_get_grid(0, @_);
}

sub mix_get_grid_with_dimid {
  return real_mix_get_grid(1, @_);
}

sub real_mix_get_grid {
  my($op, $mix_id) = @_;
  my(@grid, $stock_id, $cell_id);
  my($gridtotal, $rangetotal, $sqlstr);
  my($rows, $row, $range_row, $range_ids);

  $sqlstr = "select cell_id, stock_id
             from cell
             where mix_id = '$mix_id'
             order by seq_no";

  $rows = DBLIB::db_fetchall_arrayref($sqlstr);

  $gridtotal=0;
  foreach $row (@$rows) {
    $cell_id = $row->[0];
    $stock_id = $row->[1];

    $grid[$gridtotal][0] = $stock_id;

    # note that it is _essential_ that we order by dimension_id here
    # that's because otherwise the order in which the ranges are returned
    # is not guaranteed to be consistent between cells.
    # for example, you might get the following:
    #   cell 0: men's, right-handed, light
    #   cell 1: men's, left-handed, stiff
    #   cell 2: men's, light, left-handed
    # if that happens, the widget_analyzer will come up with some weird
    # results.

    $sqlstr = "select node.range_id";

    if ($op == 1) {
      $sqlstr .= ", dimension.dimension_id";
    }

    $sqlstr .= " from node, range, dimension 
                where node.cell_id = '$cell_id'
                and node.range_id = range.range_id
                and range.dimension_id = dimension.dimension_id
                order by node.cell_id, dimension.dimension_id";

    $range_ids = DBLIB::db_fetchall_arrayref($sqlstr);

    $rangetotal=1;

    foreach $range_row (@$range_ids) {
      $grid[$gridtotal][$rangetotal] = $range_row->[0];
      $rangetotal++;

      if ($op == 1) {
        $grid[$gridtotal][$rangetotal] = $range_row->[1];
        $rangetotal++;
      }
    }

    $gridtotal++;
  }

  return(\@grid, $gridtotal, $rangetotal);
}

#
# -----------------------------------------------------------------------------
#
#   takes: stockid, mixid
# returns: the cell that points to the stock in the given mix
#

sub mix_lookup_cell_given_item {
  my($stockid, $mixid) = @_;

  my($cellid)= DBLIB::db_fetchrow_array(
              "select cell_id from cell 
               where stock_id = '$stockid'
               and mix_id = '$mixid'");

  return($cellid);  
}

#
# -----------------------------------------------------------------------------
# Updates the price of all of the stock items associated with a mix.
#   takes: mix_id, price
# returns: nothing
#

sub mix_update_price {
  my($mixid, $price_new) = @_;

  DBLIB::db_transact_begin();
  
  my $sqlstr = "select stock_id from cell where cell.mix_id = '$mixid'";
  my $id_refs = DBLIB::db_fetchall_arrayref($sqlstr);
  
  ## iterate through each stock and ATTEMPT to update the price

  my $stock_id;
  foreach $stock_id (@$id_refs) {
    STOCK::set_attr($stock_id, ('price' => $price_new));
  }

  DBLIB::db_transact_end();
}

#
# -----------------------------------------------------------------------------
#

sub mix_verify_id {
  return DBLIB::verify_id('mix_id', 'mix', @_);
}

#
# =============================================================================
#
# CELL API
#
# =============================================================================
#

#
# -----------------------------------------------------------------------------
# create a cell
#   takes: mix_id, stock_id
# returns: cell_id
#

sub cell_create_mix {
  my($mix_id, $stock_id) = @_;

  # XXX Should we be using a sequence here?

  # get the next mix seq no

  my($next_mix_seq) = DBLIB::db_fetchrow_array(
            "select next_cell_seq_no from mix where mix_id='$mix_id'");

  # increment it and put it back

  $next_mix_seq++;
  DBLIB::db_do("update mix 
           set next_cell_seq_no=$next_mix_seq 
           where mix_id='$mix_id'");

  $next_mix_seq--;

  # get the next cell id

  my($next_id) = DBLIB::seq_next_get("cell_sq");

  # insert the cell

  DBLIB::db_do("insert into cell
           (cell_id, mix_id, seq_no, stock_id) values
           ($next_id, $mix_id, $next_mix_seq, $stock_id)");

  return($next_id); 
}

#
# -----------------------------------------------------------------------------
# delete a cell
#   takes: cell_id
# returns: nothing
#   notes: XXX deletes associated stock as well.  That'll have to change.
#              (hmmm?  I think I was thinking of multi-lingual catalogs...
#              but I think I was wrong)
#

sub cell_delete {
  my($cell_id) = @_;

  DBLIB::db_transact_begin();

  # delete associated nodes

  DBLIB::db_do("delete from node where cell_id='$cell_id'");

  # get the stock id for the cell

  my($stock_id) = DBLIB::db_fetchrow_array(
                 "select stock_id from cell where cell_id='$cell_id'");

  # delete the cell itself

  DBLIB::db_do("delete from cell where cell_id='$cell_id'");

  # and delete from the stock table

  STOCK::deassoc($stock_id);

  DBLIB::db_transact_end();
}

#
# -----------------------------------------------------------------------------
#
#
# This function finds a cell that has all of nodes specified.  This is used
# as a reverse lookup function; given all of the options, which is the
# variation that matches?   ...pretty crazy, if you ask me.  And I wrote it.
#
# returns: node.cell_id, stock_access.stock_id
#
# XXX there is probably a better way to do this.  these nested selects are
#     rather inelegant.  But I couldn't think of any other way.
#

sub cell_find_given_ranges {
  my($mixid, @nodes) = @_;
  my($i);

  if ($#nodes < 0) { return (); }

  my($sqlstr)= "
                select node.cell_id, stock_access.stock_id
                from node, cell, mix, stock_access
                where node.cell_id = cell.cell_id
                and stock_access.stock_id = cell.stock_id
                and cell.mix_id = mix.mix_id
                and mix.mix_id = '$mixid'
                and node.range_id = '$nodes[0]' ";

  for ($i=1; $i < $#nodes+1; $i++) {
    $sqlstr.=  "
                and node.cell_id in
                (select node.cell_id
                from node
                where node.range_id = '$nodes[$i]' ";
  }

  for ($i=1; $i < $#nodes+1; $i++) {
    $sqlstr .= ")";
  }

  return(DBLIB::db_fetchrow_array($sqlstr));
}

#
# -----------------------------------------------------------------------------
#

sub cell_get_range_labels {
  my($cell_id) = shift;
  my($ridref,$label,$i);

  $ridref = node_list_cell($cell_id);

  for ($i=0; $i<$#{$ridref}+1; $i++) {
    $label .= MATRIX::range_get_label($ridref->[$i]) . " -- ";
  }
  chop $label;
  chop $label;
  chop $label;
  chop $label;

  return($label);
}

#
# -----------------------------------------------------------------------------
#

sub cell_verify_id {
  return DBLIB::verify_id('cell_id', 'cell', @_);
}

#
# =============================================================================
#
# NODE API
#
# =============================================================================
#

#
# -----------------------------------------------------------------------------
# create a node
#   takes: cell_id,range_id
# returns: nothing
#

sub node_create_cell {
  my($cell_id, $range_id) = @_;

  DBLIB::db_do("insert into node
               (cell_id, range_id) values
               ($cell_id,$range_id)");
}

#
# -----------------------------------------------------------------------------
# returns a list of ranges that correspond to a given cell.
#   takes: cell_id
# returns: \@range_ids
#

sub node_list_cell {
  my($cell_id) = @_;
  my(@rids, $rid);

  my($rows) = DBLIB::db_fetchall_arrayref("select range_id from node
               where cell_id='$cell_id'");

  for $rid (@$rows) {
    push @rids,$rid->[0];
  }

  return(\@rids);
}

#
# -----------------------------------------------------------------------------
#

1;
