# KNAR.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

# 'Knar is Not A Registry.'
#
# As is typical in English with a word that begins in two consonants, the
# first being 'k', the 'k' is silent (compare with 'knife' and 'knight,' for 
# example).  While pronouncing 'knar,' the 'a' sound must be short (similar
# to the word 'are'); the final 'r' sound can be drawn out ('knarrrrrr.').
# Suffixing the pronounciation with two more 'r' sounds ('Ohhhhhh.  Knarrrrrr.
# Arrrr!  Arrrr!') is highly regarded.
# 
#

package KNAR;

use strict;
use DBLIB;

#
# AVAILABLE FUNCTIONS AND VARIABLES DEFINED IN THIS PACKAGE:
#
#  &knar_entry_list_all
#  &knar_entry_create
#  &knar_entry_delete
#  &knar_entry_get
#  &knar_get_by_id
#  &knar_get_by_name
#  &knar_entry_set
#  &knar_entry_set_by_name
#
#  &knar_group_list_all
#  &knar_group_get_all
#  &knar_varify_id
#
# =============================================================================
#

my(%cache_name_val, %cache_id_name, %cache_name_grp, %cache_name_id);
my($flush, $USE_CACHE);

$USE_CACHE = 0;

#
# -----------------------------------------------------------------------------
#

sub knar_entry_list_all {
  my($sqlstr) = 'select knar_id, key, value, group_str
                 from knar
                 where knar_id <> 0
                 order by key, group_str';

  return(DBLIB::db_fetchall_arrayref($sqlstr));
}

#
# -----------------------------------------------------------------------------
#

sub knar_entry_create {
  my($key, $value, $group) = @_;

  my($nextid) = DBLIB::seq_next_get('knar_sq');

  my($sqlstr) = "insert into knar (knar_id, key, value, group_str) 
                 values
                 (?,?,?,?)";

  DBLIB::db_do($sqlstr,0, $nextid,1, $key,0, $value,0, $group,0);


  if ($USE_CACHE) {
    if ($group =~ 'cachable') {
      $cache_name_val{$key} = $value;
      $cache_name_grp{$key} = $group;
      $cache_name_id{$key} = $nextid;
      $cache_id_name{$nextid} = $key;
    }
  }

  return($nextid);  
}

#
# -----------------------------------------------------------------------------
#

sub knar_entry_delete {
  my ($id) = shift;

  my($key, $val, $grp) = knar_get_by_id($id);

  my($sqlstr) = "delete from knar where knar_id = ?";

  DBLIB::db_do($sqlstr,0, $id,1);

  if ($USE_CACHE) {
    delete $cache_name_val{$key};
    delete $cache_name_grp{$key};
    delete $cache_name_id{$key};
    delete $cache_id_name{$id};
  }
}

#
# -----------------------------------------------------------------------------
# 

sub knar_get_by_id {
  my($id) = shift;
  my($key, $value, $group);

  if ($USE_CACHE) {
    $key = $cache_id_name{$id};
    if (defined($key)) {
      $value = $cache_name_val{$key};
      $group = $cache_name_grp{$key};
      return ($key, $value, $group);
    }
  }

  ($key, $value, $group) = DBLIB::db_fetchrow_array(
      "select key, value, group_str from knar where knar_id = ?", 0
						    , $id,1);

  if ($USE_CACHE) {
    if ($group =~ 'cachable') {
      $cache_name_val{$key} = $value;
      $cache_name_grp{$key} = $group;
      $cache_name_id{$key} = $id;
      $cache_id_name{$id} = $key;
    }
  }

  return ($key, $value, $group);
}

#
# -----------------------------------------------------------------------------
# 
sub old_knar_get_by_name {
  my($key) = shift;
  my($id, $value, $group);

  if ($USE_CACHE) {
    $id = $cache_name_id{$key};
    if (defined($id)) {
      $value = $cache_name_val{$key};
      $group = $cache_name_grp{$key};
#      open(HOHUM,">>/tmp/hohum");
#      print HOHUM "using cached values for '$key'\n";
#      close(HOHUM);
      return ($id, $value, $group);
    }
  }
    
  ($id, $value, $group) = DBLIB::db_fetchrow_array(
      "select knar_id, value, group_str from knar where key = '$key'");

  if ($USE_CACHE) {
    if ($group =~ 'cachable') {
      $cache_name_val{$key} = $value;
      $cache_name_grp{$key} = $group;
      $cache_name_id{$key} = $id;
      $cache_id_name{$id} = $key;
#      open(HOHUM,">>/tmp/hohum");
#      print HOHUM "caching values for '$key'\n";
#      close(HOHUM);
    }
  }

  return ($id, $value, $group);
}

sub knar_get_by_name {
  my($key) = shift;
  my($id, $value, $group);

  if ($USE_CACHE) {
    $id = $cache_name_id{$key};
    if (defined($id)) {
      $value = $cache_name_val{$key};
      $group = $cache_name_grp{$key};
#      open(HOHUM,">>/tmp/hohum");
#      print HOHUM "using cached values for '$key'\n";
#      close(HOHUM);
      return ($id, $value, $group);
    }
  }
    
  ($id, $value, $group) = DBLIB::db_fetchrow_array(
      "select knar_id, value, group_str from knar where key = ?",
						   0, $key,0);

  if ($USE_CACHE) {
    if ($group =~ 'cachable') {
      $cache_name_val{$key} = $value;
      $cache_name_grp{$key} = $group;
      $cache_name_id{$key} = $id;
      $cache_id_name{$id} = $key;
#      open(HOHUM,">>/tmp/hohum");
#      print HOHUM "caching values for '$key'\n";
#      close(HOHUM);
    }
  }

  return ($id, $value, $group);
}


#
# -----------------------------------------------------------------------------
#

sub knar_entry_get {
  my ($key) = shift;

  my($id, $value, $group) = knar_get_by_name($key);

  return($value);
}

#
# -----------------------------------------------------------------------------
#

sub knar_entry_set {
  my ($id, $key, $value, $group) = @_;

  if ($USE_CACHE) {
    my($tmpid, $tmp1, $tmp2) = knar_get_by_name($key);
    if (defined($tmpid) && ($group =~ 'cachable')) {
      $cache_name_val{$key} = $value;
      $cache_name_grp{$key} = $group;
      $cache_name_id{$key} = $id;
      $cache_id_name{$id} = $key;
    }
  }

  my($sqlstr) = "update knar set key = ?, value = ?,
                 group_str = ? where knar_id = ?";

  DBLIB::db_do($sqlstr,0, $key,0, $value,0, $group,0, $id,1);
}

#
# -----------------------------------------------------------------------------
#

sub knar_entry_set_by_name {
  my ($key, $value, $group) = @_;
  my ($id, $tmp1, $tmp2);

  ($id, $tmp1, $tmp2) = knar_get_by_name($key);

  if (!defined($id)) {
    $id = knar_entry_create(@_);  

  } else {
    my($sqlstr) = "update knar set value = ?,
                   group_str = ? where key = ?";

    DBLIB::db_do($sqlstr,0, $value,0, $group,0, $key,0);
  }

  if ($USE_CACHE) {
    if ($group =~ 'cachable') {
      $cache_name_val{$key} = $value;
      $cache_name_grp{$key} = $group;
      $cache_name_id{$key} = $id;
      $cache_id_name{$id} = $key;
    }
  }
}

#
# -----------------------------------------------------------------------------
#

sub knar_group_list_all {
  my($sqlstr) = 'select distinct(group_str) from knar order by group_str';

  return(DBLIB::db_fetchall_arrayref($sqlstr));
}

#
# -----------------------------------------------------------------------------
#

sub knar_group_get_all {
  my($group) = shift;

  my($sqlstr) = "select knar_id, key, value
                 from knar
                 where group_str = ?
                 order by key";

  return(DBLIB::db_fetchall_arrayref($sqlstr,0, $group,0));
}

#
# -----------------------------------------------------------------------------
#

## ----------------------------------------------------------------------------
##   func: knar_verify_id
##   desc: verifies avalable knar id
##  param: knar_id
##    ret: 1/ 0 
##  notes: 
## ----------------------------------------------------------------------------

sub knar_verify_id {
    return DBLIB::verify_id("knar_id", "knar", @_);
}

1;
