# LAYIM.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

# The layout importer
package LAYIM;

use strict;
use DBLIB;
use THING;
use ITEM;
use PAGE;
use GROUP;
use CONTAINER;

# =============================================================================
#
# LAYIM API
#
# =============================================================================
#

#
# -----------------------------------------------------------------------------
# Pass a string like this:
# !ROOT!
#   Quicklinks
#     Categories Quicklinks
#           Home Page
#     Navigation Quicklinks
#           Site Map
#   Pages
#      Products
#
# Or like this:
# !ROOT!
#   !G!Quicklinks
# ...
#   Pages
#      !P!Products
# ...
#
# Creates containers accordingly. 
# Returns ($status, $reason), where $status is true for success.
#
# Multiple identical children will be created if specified, but
# if a child specified already existed in the container system,
# another will *not* be created (even if specified more than once in
# the picture).
#
# Multiple !ROOT!s can also be specified.  Grokking the usefulness
# of this is left as an exercise to the reader.
#
# Bugs: If you get your picture from Netscape, Netscape will tend to
# munge your text.  So any things with multiple or trailing whitespaces in
# their names won't be matched properly.
#
sub layout_import {
    my($picture) = @_;

    my(@lines) = split(/\n/, $picture);

    my(%created); # Containers we created during this call

    my(@column); # Which parents are at which indentation levels

    DBLIB::db_transact_begin();

    my $line_no = 0;
    for(@lines) {
	$line_no++;

	my($name, $kind, $id, $level);

	($level, $name) = layout_parse($_);

	splice(@column, $level, 1 + $#column - $level);

#	print("Working on $name at level $level.\n");

	if($name eq '!ROOT!') {
	    $column[$level]{name} = "Root Node";
	    $column[$level]{cid} = 0;
	} else {

	    ($kind, $id) = layout_id($name);

	    if($kind == -1) {
	        DBLIB::db_transact_rb();
		return(0, "Error: $id for '$name' at line $line_no.\n");
	    }

	    my $cid;
	    my $i;
	    for($i = $level-1; $i >=0; $i--) {
		next unless exists $column[$i]{cid};

		if(exists($column[$i]{child_level}) and
			  $column[$i]{child_level} != $level) {

		    DBLIB::db_transact_rb();
		    return(0,"Inconsistent indentation of child '$name'.\n");
		}

		$column[$i]{child_level} = $level;
		
#		print "'$name' is a child of $column[$i]{name}.\n";

		my($pcid) = $column[$i]{cid};

		# Create if it exists because we created it ourselves,
		# or if it doesn't exist
		if(exists($created{$pcid, $kind, $id}) or
		   !($cid=CONTAINER::container_get_id($pcid, $kind, $id))) {

#		    print "Creating for $name.\n";

		    $created{$pcid, $kind, $id} = 1;
		    $cid = CONTAINER::container_create($pcid, $kind, $id);
		} else {
		    # Ignore
#		    print "Ignoring for $name.\n";
		}

		last;
	    }

	    if($i == -1) {
	        DBLIB::db_transact_rb();
		return(0, "Indentation error at line $line_no.\n");
	    }

	    $column[$level]{name} = $name;
	    $column[$level]{cid} = $cid;
	}
    }

    DBLIB::db_transact_end();
    return( 1, "Success." );
}

#
# -----------------------------------------------------------------------------
#
sub layout_id {
    my($name) = @_;

    my($kind);
    ($kind, $name) = ($name =~ /^(![PIG]!)?(.*)/);

    $kind = '';
    $kind = $THING::item if( $kind eq '!I!' );
    $kind = $THING::page if( $kind eq '!P!' );
    $kind = $THING::group if( $kind eq '!G!' );

    if($kind) {
	my($id);

	if($kind == $THING::item) {
	    $id = ITEM::item_get_id($name);
	} elsif($kind == $THING::page) {
	    $id = PAGE::page_get_id($name);
	} elsif($kind == $THING::group) {
	    $id = GROUP::group_get_id($name);
	}

	return($kind, $id) if $id;
	return(-1, "Not found");

    } else {
	my($item_id, $page_id, $group_id);

	$item_id = ITEM::item_get_id($name);
	$page_id = PAGE::page_get_id($name);
	$group_id = GROUP::group_get_id($name);
	
#    print("iid: $item_id\tpid: $page_id\tgid: $group_id\t$name\n");
	return(-1, "Ambiguous") if($item_id && $page_id or
				   $item_id && $group_id or
				   $page_id && $group_id); # Ambiguous

	return($THING::item, $item_id) if $item_id;
	return($THING::page, $page_id) if $page_id;
	return($THING::group, $group_id) if $group_id;
	return(-1, "Not found");
    }
}

#
# -----------------------------------------------------------------------------
#
sub layout_parse {
    my($line) = @_;

    chomp $line;
    $line =~ s/\r//g;

    my($level, $name) = ($line =~ /^(\s*)(.+)/);

    return(length($level), $name);
}

1;
