# MATRIX.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

package MATRIX;

use strict;

require DBLIB;


#
# 
# AVAILABLE FUNCTIONS AND VARIABLES DEFINED IN THIS PACKAGE:
#
#  &matrix_list_all
#  &matrix_create
#  &matrix_delete
#  &matrix_get_label
#  &matrix_set_label
#  &matrix_find
#  &matrix_get_id_from_mix_id
#  &matrix_verify_id
#
#  &dim_list_all
#  &dim_list_matrix
#  &dim_create_matrix
#  &dim_create
#  &dim_add_matrix
#  &dim_del
#  &dim_del_matrix
#  &dim_get_label
#  &dim_set_label
#  &dim_find_matrix
#  &dim_verify_id
#
#  &range_list_all
#  &range_list_dim
#  &range_create_dim
#  &range_del
#  &range_get_label
#  &range_set_label
#  &range_find
#  &range_find_dim
#  &range_find_matrix
#  &range_verify_id
#
# =============================================================================
#

#
# -----------------------------------------------------------------------------
# get a list of all matrices, sorted by label
#   takes: nothing
# returns: $array_ref (ids, labels, next_dim_seq_nos)
#
sub matrix_list_all {

  my($sqlstr) = 'select matrix_id, label, next_dim_seq_no 
                 from matrix
                 where matrix_id <> 0
                 order by label';

  my $array_ref = DBLIB::db_fetchall_arrayref($sqlstr);

  return $array_ref;
}

#
# -----------------------------------------------------------------------------
# create a new matrix
#   takes: label
# returns: id of new matrix
#
sub matrix_create {
  my ($label) = shift;

  my ($nextid) = DBLIB::seq_next_get('matrix_sq');

  my $sqlstr = "insert into matrix 
                (matrix_id, label, next_dim_seq_no) values
                ($nextid,'$label','0')";

  DBLIB::db_do($sqlstr);

  return ($nextid);
}

#
# -----------------------------------------------------------------------------
# deletes a matrix
#   takes: matrix_id
# returns: nothing
#
sub matrix_delete {
  my ($id) = shift;

  my($sqlstr) = "delete from matrix where matrix_id = '$id'";

  DBLIB::db_do($sqlstr);
}

#
# -----------------------------------------------------------------------------
# returns the label of the specified matrix
#   takes: matrix_id
# returns: label
#
sub matrix_get_label {
  my ($matrix_id) = shift;

  my($sqlstr) = "select label from matrix where matrix_id = '$matrix_id'";

  my ($label) = DBLIB::db_fetchrow_array($sqlstr);

  return ($label);
}

#
# -----------------------------------------------------------------------------
# sets the label of the specified matrix
#   takes: matrix_id, new label
# returns: nothing
#
sub matrix_set_label {
  my ($matrix_id) = shift;
  my ($new_label) = shift;

  my($sqlstr) = "update matrix 
                 set label = '$new_label'
                 where matrix_id = '$matrix_id'";

  DBLIB::db_do($sqlstr);
}

#
# -----------------------------------------------------------------------------
# finds a matrix by name
#   takes: matrix_id, new label
# returns: nothing
#
sub matrix_find {
  my ($matrix_label) = shift;

  my($sqlstr) = "select matrix_id from matrix where label = '$matrix_label'";

  my($mid) = DBLIB::db_fetchrow_array($sqlstr);

  return($mid);
}

#
# -----------------------------------------------------------------------------
# finds a matrix id given a mix id
#   takes: mix_id
# returns: matrix_id
#
sub matrix_get_id_from_mix_id {
  my ($mix_id) = shift;

  my($sqlstr) = "select matrix_dimension.matrix_id 
                 from mix,cell,node,range,dimension,matrix_dimension
                 where mix.mix_id = '$mix_id'
                 and cell.mix_id = mix.mix_id
                 and node.cell_id = cell.cell_id
                 and node.range_id = range.range_id
                 and range.dimension_id = dimension.dimension_id
                 and matrix_dimension.dimension_id = dimension.dimension_id";

  my($mid) = DBLIB::db_fetchrow_array($sqlstr);

  return($mid);
}

#
# -----------------------------------------------------------------------------
#

sub matrix_verify_id {
  return DBLIB::verify_id('matrix_id', 'matrix', @_);
}


#
# =============================================================================
#
# DIMENSION API
#
# =============================================================================
#

#
# -----------------------------------------------------------------------------
# get a list of all dimensions, sorted by label
#   takes: nothing
# returns: $array_ref (dim_ids, dim_labels, dim_next_range_seq_no)
#
sub dim_list_all {
  my($sqlstr) = 'select dimension_id, label, next_range_seq_no
                 from dimension
                 where dimension_id <> 0
                 order by label';

  my $array_ref = DBLIB::db_fetchall_arrayref($sqlstr);

  return $array_ref;
}

#
# -----------------------------------------------------------------------------
# get a list of all dimensions within a matrix, ordered by sequence number
#   takes: matrix_id
# returns: $array_ref (dim_ids, dim_labels, dim_next_range_seq_no)
#
sub dim_list_matrix {
  my ($matrix_id) = shift;

  my($sqlstr) = "select matrix_dimension.dimension_id, dimension.label, 
                        dimension.next_range_seq_no, matrix_dimension.seq_no
                   from matrix_dimension, dimension
                  where matrix_dimension.matrix_id='$matrix_id'
                    and matrix_dimension.dimension_id = dimension.dimension_id
               order by matrix_dimension.seq_no";

  my $array_ref = DBLIB::db_fetchall_arrayref($sqlstr);

  return $array_ref;
}

#
# -----------------------------------------------------------------------------
# create a new dimension in a specific matrix
#   takes: label, matrix_id
# returns: id of new dimension
#
sub dim_create_matrix {
  my ($label) = shift;
  my ($matrix_id) = shift;

  my ($dim_id) = dim_create($label);
  dim_add_matrix($dim_id, $matrix_id);

  return($dim_id);
}

#
# -----------------------------------------------------------------------------
# create a new dimension
#   takes: dim_label
# returns: id of new dimension
#
sub dim_create {
  my ($label) = shift;

  my $nextid = DBLIB::seq_next_get('dimension_sq');

  # insert into the database

  my $sqlstr = "insert into dimension
                (dimension_id, label, next_range_seq_no) values
                ($nextid,'$label','0')";

  DBLIB::db_do($sqlstr); 

  return($nextid);
}

#
# -----------------------------------------------------------------------------
# add a dimension to a specific matrix
#   takes: dim_id, matrix_id
# returns: nothing
#
sub dim_add_matrix {
  my ($dim_id) = shift;
  my ($matrix_id) = shift;

  DBLIB::db_transact_begin();

  # get the current sequence number

  my($sqlstr) = "select next_dim_seq_no 
                 from matrix 
                 where matrix_id = '$matrix_id'";

  my ($seqno) = DBLIB::db_fetchrow_array($sqlstr);

  # increment the sequence number and store it

  $seqno++;

  $sqlstr = "update matrix 
             set next_dim_seq_no='$seqno'
             where matrix_id='$matrix_id'";

  DBLIB::db_do($sqlstr);

  $seqno--;

  # insert the matrix->dimension link

  $sqlstr = "insert into matrix_dimension 
             (matrix_id, dimension_id, seq_no) 
             values ($matrix_id, $dim_id, $seqno)";

  DBLIB::db_do($sqlstr);

  DBLIB::db_transact_end();
}

#
# -----------------------------------------------------------------------------
# delete a dimension completely
#   takes: dim_id
# returns: nothing
#   notes: XXX the error handling here isn't pretty.
#          this function will fail spectacularly if any child records still
#          depend on the dimension deleted
#          XXX we need to reorder the matrix->range sequences!!!
#
sub dim_del {
  my ($dim_id) = shift;


  DBLIB::db_transact_begin();

  # delete associated ranges
  my ($sqlstr) = "delete from range where dimension_id = '$dim_id'";
  DBLIB::db_do($sqlstr);

  # delete from matrix_dimension table
  $sqlstr = "delete from matrix_dimension where dimension_id = '$dim_id'";
  DBLIB::db_do($sqlstr);

  # delete from dimension table
  $sqlstr = "delete from dimension where dimension_id = '$dim_id'";
  DBLIB::db_do($sqlstr);
 
  DBLIB::db_transact_end();
}

#
# -----------------------------------------------------------------------------
# delete a dimension only from specified matrix
#   takes: dim_id, matrix_id
# returns: nothing
#   notes: XXX we need to reorder the matrix->range sequences!!!
#
sub dim_del_matrix {
  my ($dim_id) = shift;
  my ($matrix_id) = shift;

  # delete from matrix_dimension table

  my($sqlstr) = "delete from matrix_dimension where dimension_id = '$dim_id'";

  DBLIB::db_do($sqlstr);
}

#
# -----------------------------------------------------------------------------
# returns the label of the specified dimension
#   takes: dim_id
# returns: label
#
sub dim_get_label {
  my ($dim_id) = shift;

  my($sqlstr) = "select label from dimension where dimension_id = '$dim_id'";

  my ($label) = DBLIB::db_fetchrow_array($sqlstr);

  return($label);
}

#
# -----------------------------------------------------------------------------
# sets the label of the specified dimension
#   takes: dim_id, new label
# returns: nothing
#
sub dim_set_label {
  my ($dim_id) = shift;
  my ($new_label) = shift;

  my($sqlstr)="update dimension
               set label = '$new_label'
               where dimension_id = '$dim_id'";

  DBLIB::db_do($sqlstr);
}

#
# -----------------------------------------------------------------------------
# finds a dimension within a matrix, by name
#   takes: dim_id, new label
# returns: nothing
#
sub dim_find_matrix {
  my ($dim_str, $matrix_id) = @_;

  my($sqlstr) = "select dimension.dimension_id
                 from dimension, matrix_dimension
                 where dimension.dimension_id = matrix_dimension.dimension_id
                 and dimension.label = '$dim_str'
                 and matrix_dimension.matrix_id = '$matrix_id'";

  my ($did) = DBLIB::db_fetchrow_array($sqlstr);

  return $did;
}

#
# -----------------------------------------------------------------------------
#

sub dim_verify_id {
  return DBLIB::verify_id('dimension_id', 'dimension', @_);
}


#
# =============================================================================
#
# RANGE API
#
# =============================================================================
#

#
# -----------------------------------------------------------------------------
# returns a list of all ranges, sorted by label
#   takes: nothing
# returns: $array_ref (range_ids, dim_ids, seq_no, labels)
#
sub range_list_all {
  my($sqlstr) = 'select range_id,dimension_id, label, seq_no
                 from range
                 where range_id <> 0
                 order by label';

  my $array_ref = DBLIB::db_fetchall_arrayref($sqlstr);

  return $array_ref;
}

#
# -----------------------------------------------------------------------------
# returns a list of all ranges within a specific dimension
#   takes: dim_id
# returns: \@range_ids, \@labels, \@seq_nos
#
sub range_list_dim {
  my ($dim_id) = shift;

  my($sqlstr) = "select range_id, label, seq_no
                 from range
                 where dimension_id = '$dim_id'
                 order by seq_no";

  my $array_ref = DBLIB::db_fetchall_arrayref($sqlstr);

  return $array_ref;
}


#
# -----------------------------------------------------------------------------
# creates a range within a specified dimension
#   takes: label, dim_id
# returns: id of new range
#
sub range_create_dim {
  my ($label) = shift;
  my ($dim_id) = shift;

  DBLIB::db_transact_begin();

  # get the next available range_id number
  my $nextid = DBLIB::seq_next_get('range_sq');

  # get the next available sequence number
  my $sqlstr = "select next_range_seq_no 
                from dimension 
                where dimension_id = '$dim_id'";

  my ($nextseq) = DBLIB::db_fetchrow_array($sqlstr);

  # increment the sequence number
  $nextseq++;
  $sqlstr = "update dimension 
             set next_range_seq_no = '$nextseq'
             where dimension_id = '$dim_id'";
  DBLIB::db_do($sqlstr);

  $nextseq--;

  # insert into the database

  $sqlstr = "insert into range
             (range_id, dimension_id, label, seq_no) values
             ($nextid, $dim_id, '$label', $nextseq)";

  DBLIB::db_do($sqlstr);

  DBLIB::db_transact_end();

  return ($nextid);
}

#
# -----------------------------------------------------------------------------
# deletes a range
#   takes: range_id
# returns: nothing
#   notes: XXX we need to reorder the dimension->range sequences!!!
#
sub range_del {
  my ($range_id) = shift;

  my ($sqlstr) = "delete from range where range_id = '$range_id'";
  DBLIB::db_do($sqlstr);
}

#
# -----------------------------------------------------------------------------
# returns the label of the specified range
#   takes: range_id
# returns: label
#
sub range_get_label {
  my ($range_id) = shift;

  my($sqlstr) = "select label from range where range_id = '$range_id'";

  my ($label) = DBLIB::db_fetchrow_array($sqlstr);

  return($label);
}

#
# -----------------------------------------------------------------------------
# sets the label of the specified dimension
#   takes: dim_id, new label
# returns: nothing
#
sub range_set_label {
  my ($range_id) = shift;
  my ($new_label) = shift;

  my($sqlstr)="update range
               set label = '$new_label'
               where range_id = '$range_id'";

  DBLIB::db_do($sqlstr);
}

#
# -----------------------------------------------------------------------------
# returns the range id of the first range with the corresponding label
#   takes: label
# returns: range_id
#
sub range_find {
  my ($label) = shift;

  my($sqlstr) = "select range_id from range where label = '$label'";

  my ($range_id) = DBLIB::db_fetchrow_array($sqlstr);

  return($range_id);
}

#
# -----------------------------------------------------------------------------
# returns the range id of the first range with the corresponding label
#   takes: label, dimid
# returns: range_id
#

sub range_find_dim {
  my ($label, $dimid) = @_;

  my($sqlstr) = "select range_id 
                 from range 
                 where label = '$label'
                 and dimension_id = '$dimid'";

  my ($range_id) = DBLIB::db_fetchrow_array($sqlstr);

  return($range_id);
}

#
# -----------------------------------------------------------------------------
#
# finds the rangeid of a range given it's label and the matrix it's in.
#
#   takes: label, matrixid
# returns: rangeid
#

sub range_find_matrix {
  my($label) = shift;
  my($matrixid) = shift;

  my($sqlstr) = "select range_id 
                 from range,dimension,matrix_dimension
                 where range.label = '$label'
                 and range.dimension_id = dimension.dimension_id
                 and matrix_dimension.dimension_id = dimension.dimension_id
                 and matrix_dimension.matrix_id = '$matrixid'";

  my ($range_id) = DBLIB::db_fetchrow_array($sqlstr);

  return($range_id);
}

#
# -----------------------------------------------------------------------------
#

sub range_verify_id {
  return DBLIB::verify_id('range_id', 'range', @_);
}

#
# -----------------------------------------------------------------------------
#


1;
