# ORDER.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

package ORDER;

use strict;

require MILTON;
require DBLIB;
require ORDERLINE;
require ORDERSTATS;
require AFFILIATES;
require PAYMENT;

use vars qw ( %listall 
	      %create 
	      %delete 
	      %get_info 
	      %set_attr 
	      %cc_info_clear
	      %payment_assoc
	      %payment_deassoc
	      %payment_listall 
	      %discount_assoc
	      %discount_deassoc
	      %discount_listall );

## ============================================================================
##  This file is the public API for order functionality.
##  It also--for now contains all of the implementation specific functionality.
##  The general layout is that, based on some parameter, a code reference
##  will be called by the general function.
## ============================================================================


## ============================================================================
##  ORDER API
## ============================================================================
##  These are the avaliable functions
##
##  listall            returns arrayref of all orders
##  create             creates a new order
##  delete             deletes a order
##  get_info           returns all of the info for a specific order
##  set_attr           allows setting of various order attributes
##  verify_id          verifies correct order_id
##  cc_info_clear      clears cc info from the order
##  payment_assoc      associates an order payment
##  payment_deassoc    deassociates an order from an payment
##  payment_listall    lists all of the payment id's for a customer
##  discount_assoc     associate an order discount
##  discount_deassoc   deassociates an order from an discount
##  discount_listall   list all of the discounts for an order

## ============================================================================
##  IMPLEMENTATIONS
## ============================================================================

## ----------------------------------------------------------------------------
##   func: listall
##   desc: lists all of the order id's
##  param: $arch_flag
##    ret: a reference to an array with the id's
##  notes: if the arch_flag is set then archived orders are returned
##         otherwise unarchived ones are returned
## ----------------------------------------------------------------------------

sub listall_local {
    my ($arch_flag) = @_;
    
    ## set archived flag correctly

    if (defined ($arch_flag)) {
	$arch_flag = "yes";
    } else {
	$arch_flag = "no";
    }

    my $sqlstr = "select id, customer_id, total, subtotal, tax, shipping,
                  general_ins, affiliate, status, archived, date_rec
                  from cat_order where archived = '$arch_flag'
                  order by id";
    my $ref = DBLIB::db_fetchall_arrayref($sqlstr);

    return $ref;
}
    
## ----------------------------------------------------------------------------
##   func: create
##   desc: create a new order
##  param: customer_id, total, subtotal, tax, shipping, gen_ins, 
##         affiliate, status, archive, date_rec
##    ret: order_id
##  notes:
## ----------------------------------------------------------------------------

sub create_local {
    my ($customer_id, $total, $subtotal, $tax, $shipping, $gen_ins, 
	$aff, $status, $arch, $date_rec) = @_;

    ## get a new order id
    my $id = DBLIB::seq_next_get("cat_order_sq");

    my $sqlstr = "insert into cat_order 
                  (id, customer_id, total, subtotal, tax, shipping, general_ins,
                   affiliate, status, archived, date_rec)
                  values
                  ('$id', '$customer_id', '$total', '$subtotal', '$tax', '$shipping', '$gen_ins',
                   '$aff', '$status', '$arch', $date_rec)";

  DBLIB::db_do($sqlstr);

    return $id;
}

## ----------------------------------------------------------------------------
##   func: delete
##   desc: delete an order
##  param: order id, bogus_flag
##    ret: 1 /undef
##  notes: 
## ----------------------------------------------------------------------------

sub delete_local {
    my ($id, $bogus_flag) = @_;
    
    DBLIB::db_transact_begin();

    ## take out any orderlines associated with the order
    my $sqlstr = "select id from orderline where cat_order_id = $id";
    my $id_ref = DBLIB::db_fetchall_arrayref($sqlstr);

    my ($row, $val);

    foreach $row (@$id_ref) {
	($val) = @$row;
      ORDERLINE::delete($val);
    }

    ## take out any payments associated with the order
    $sqlstr = "select payment_id from cat_order_payment where cat_order_id = $id";
    $id_ref = DBLIB::db_fetchall_arrayref($sqlstr);
    
    foreach $row (@$id_ref) {
	($val) = @$row;
      PAYMENT::delete($val);
    }

    ## take out any discounts associated with the order
    $sqlstr = "select discount_id from cat_order_discount where cat_order_id = $id";
    $id_ref = DBLIB::db_fetchall_arrayref($sqlstr);
    
    foreach $row (@$id_ref) {
	($val) = @$row;
      DISCOUNT::delete($val);
    }

    ## check for bogus orders
    if (defined ($bogus_flag)) {
	## clear out orderstats for bogus order
      ORDERSTATS::orderstats_delete_entry($id);

	## fix affiliates
	$sqlstr = "select affiliate, total from cat_order where id = $id";
	my ($affiliate_id, $total) = DBLIB::db_fetchrow_array($sqlstr);
      AFFILIATES::affiliates_subtract_bogus_order($affiliate_id, $total);
    }

    ## now take out the order
    $sqlstr = "delete from cat_order where id = $id";
  DBLIB::db_do($sqlstr);

  DBLIB::db_transact_end();

    return 1;
}
 
## ----------------------------------------------------------------------------
##   func: get_info
##   desc: return information about an order
##  param: order id
##    ret: customer_id, total, subtotal, tax, shipping, general_ins, 
##         affiliate, status, archived, date_rec
##  notes:
## ----------------------------------------------------------------------------

sub get_info_local {
    my ($id) = @_;

    my $sqlstr = "select customer_id, total, subtotal, tax, shipping,
                  general_ins, affiliate, status, archived, date_rec
                  from cat_order where id = $id";

    my @info = DBLIB::db_fetchrow_array($sqlstr);

    return @info;
}

## ----------------------------------------------------------------------------
##   func: set_attr 
##   desc: allows to change attrs of an order
##  param: order_id, attrs hash
##    ret: 1/ undef
##  notes:
## ----------------------------------------------------------------------------

sub set_attr_local {
    my ($id, %attrs) = @_;

    ## create sqlstr
    my $sqlstr = "update cat_order set ";
    
    my ($key, $val);

    foreach $key (keys(%attrs)) {
	$sqlstr .= "$key = '$attrs{$key}', ";
    }

    chop $sqlstr;
    chop $sqlstr;
    
    $sqlstr .= " where id = $id";

  DBLIB::db_do($sqlstr);

    return 1;
}

## ----------------------------------------------------------------------------
##   func: verify_id
##   desc: verifies avalable order id
##  param: order_id
##    ret: 1/ 0 
##  notes: 
## ----------------------------------------------------------------------------

sub verify_id {
    return DBLIB::verify_id("id", "cat_order", @_);
}

## ----------------------------------------------------------------------------
##   func: cc_info_clear 
##   desc: clears out all credit card info from the order
##  param: order_id
##    ret: 1/ undef
##  notes: I'll write this later
## ----------------------------------------------------------------------------

## ----------------------------------------------------------------------------
##   func: payment_assoc
##   desc: associates an oreder payment
##  param: order_id, payment_id
##    ret: 1 /undef
##  notes:
## ----------------------------------------------------------------------------

sub payment_assoc_local {
    my ($order_id, $payment_id) = @_;

    ## make entry in cat_order_payment
    my $sqlstr = "insert into cat_order_payment
                  (cat_order_id, payment_id)
                  values
                  ($order_id, $payment_id)";

  DBLIB::db_do($sqlstr);

    return 1;
}

## ----------------------------------------------------------------------------
##   func: payment_deassoc
##   desc: associates an oreder payment
##  param: payment_id
##    ret: 1 /undef
##  notes:
## ----------------------------------------------------------------------------

sub payment_deassoc_local {
    my ($id) = @_;

    my $sqlstr = "delete from cat_order_payment where payment_id = $id";

  DBLIB::db_do($sqlstr);

    return 1;
}

## ----------------------------------------------------------------------------
##   func: payment_listall
##   desc: lists all payments associated with an order
##  param: order id
##    ret: reference to payment's
##  notes:
## ----------------------------------------------------------------------------

sub payment_listall_local {
    my ($id) = @_;
    
    my $sqlstr = "select payment.id, type_label, field1, field2, field3, field4, field5, amnt
                  from payment, cat_order_payment 
                  where cat_order_payment.cat_order_id = $id
                  and cat_order_payment.payment_id = payment.id";

    my $id_ref = DBLIB::db_fetchall_arrayref($sqlstr);

    return $id_ref;
}

## ----------------------------------------------------------------------------
##   func: discount_assoc
##   desc: associtates a discount with an order
##  param: order id, discount_id
##    ret: 1 /undef
##  notes:
## ----------------------------------------------------------------------------

sub discount_assoc_local {
    my ($order_id, $discount_id) = @_;

    my $sqlstr = "insert into payment_discount 
                  (cat_order_id, discount_id)
                  values
                  ($order_id, $discount_id)";

  DBLIB::db_do($sqlstr);

    return 1;
}

## ----------------------------------------------------------------------------
##   func: discount_deassoc
##   desc: deassocitates a discount from an order
##  param: discount_id
##    ret: 1 /undef
##  notes:
## ----------------------------------------------------------------------------

sub discount_deassoc_local {
    my ($id) = @_;

    my $sqlstr = "delete from cat_order_discount where discount_id = $id";
  DBLIB::db_do($sqlstr);

    return 1;
}

## ----------------------------------------------------------------------------
##   func: discount_listall
##   desc: lists all discounts associated with an order
##  param: order id
##    ret: reference to payment id's
##  notes:
## ----------------------------------------------------------------------------

sub discount_listall_local {
    my ($id) = @_;
    
    my $sqlstr = "select discount.id, label, amnt
                  from cat_order_discount, discount
                  where cat_order_discount.cat_order_id = $id
                  and cat_order_discount.discount_id = discount.id";

##    my $sqlstr = "select discount_id from cat_order_discount where cat_order_id = $id";
    my $ref = DBLIB::db_fetchall_arrayref($sqlstr);

    return $ref;
}

## ============================================================================
##  IMPLEMENTATION HASHES
## ============================================================================

%listall          = ( "LOCAL" => \&listall_local, );
%create           = ( "LOCAL" => \&create_local, );
%delete           = ( "LOCAL" => \&delete_local, );
%get_info         = ( "LOCAL" => \&get_info_local, );
%set_attr         = ( "LOCAL" => \&set_attr_local, );
%payment_assoc    = ( "LOCAL" => \&payment_assoc_local, );
%payment_deassoc  = ( "LOCAL" => \&payment_deassoc_local, );
%payment_listall  = ( "LOCAL" => \&payment_listall_local, );
%discount_assoc   = ( "LOCAL" => \&discount_assoc_local, );
%discount_deassoc = ( "LOCAL" => \&discount_deassoc_local, );
%discount_listall = ( "LOCAL" => \&discount_listall_local, );


## ============================================================================
##  API CALLS
## ============================================================================


sub listall {
    return MILTON::generic_call(\%listall, @_);
}

sub create {
    return MILTON::generic_call(\%create, @_);
}

sub delete {
    return MILTON::generic_call(\%delete, @_);
}

sub get_info {
    return MILTON::generic_call(\%get_info, @_);
}

sub set_attr {
    return MILTON::generic_call(\%set_attr, @_);
}

sub payment_assoc {
    return MILTON::generic_call(\%payment_assoc, @_);
}

sub payment_deassoc {
    return MILTON::generic_call(\%payment_deassoc, @_);
}

sub payment_listall {
    return MILTON::generic_call(\%payment_listall, @_);
}

sub discount_assoc {
    return MILTON::generic_call(\%discount_deassoc, @_);
}

sub discount_deassoc {
    return MILTON::generic_call(\%discount_deassoc, @_);
}

sub discount_listall {
    return MILTON::generic_call(\%discount_listall, @_);
}

##############################################################################
##  func:  print_order_ref
## param:  order id, string ref
##   ret:  none
## notes: 
##############################################################################

sub print_order_ref {
    my ($ref, $str) = @_;

    ## get order info
    my ( $id, $customer_id, $total, $subtotal, $tax_total, $shipping_total, 
	 $general_ins, $affiliate, $status, $archived, $date_rec);  

    my $size = $#{$ref}  + 1;
    my $count = 0;

    ## print info
    while ($count < $size) {
	($id, $customer_id, $total, $subtotal, $tax_total, $shipping_total, 
	  $general_ins, $affiliate, $status, $archived, $date_rec)
	    = @{$ref->[$count]};

	
	$$str .= "ORDER\t$id\n";
	$$str .= "DATE\t$date_rec\n";
	$$str .= "TOTAL\t$total\n";
	$$str .= "TAX_TOTAL\t$tax_total\n";
	$$str .= "SHIPPING_TOTAL\t$shipping_total\n";
	$$str .= "AFFILIATE\t$affiliate\n";
	$$str .= "ARCHIVED\t$archived\n";
	$$str .= "STATUS\t$status\n";
	$$str .= "GENERAL_INS\t$general_ins\n";

	print_discount_list("order", $id, $str);
	print_payment_list("order", $id, $str);
	print_customer($customer_id, $str);
	print_orderline_list($id, $str);

	$count++;
    }
}

##############################################################################
##  func:  print_customer
## param:  customer id
##   ret:  none
## notes: 
##############################################################################

sub print_customer {
    my ($id, $str) = @_;

    ## get customer info
    my ( $name, $company, $email, $h_phone, $w_phone ) = CUSTOMER::get_info($id);

    ## print info
    $$str .= "CUSTOMER\t$id\n";
    $$str .= "NAME\t$name\n";
    $$str .= "COMPANY\t$company\n";
    $$str .= "EMAIL\t$email\n";
    $$str .= "HOME_PHONE\t$h_phone\n";
    $$str .= "WORK_PHONE\t$w_phone\n";

    print_address_list($id, $str);
}
    
##############################################################################
##  func:  print_orderline_list
## param:  order id
##   ret:  none
## notes: 
##############################################################################

sub print_orderline_list {
    my ($id, $str) = @_;

    my $ref = ORDERLINE::listall($id);

    my $size = $#{$ref} + 1;
    my $count = 0;

    my ($ol_id, $co_id, $label, $sku, $price, $qty, $total, $weight, $volume, $info);

    while ($count < $size) {
	($ol_id, $co_id, $label, $sku, $price, $qty, $total, $weight, $volume, $info)
	    = @{$ref->[$count]};
	
	## $$str .= info
	$$str .= "ORDERLINE\t$id\n";
	$$str .= "LABEL\t$label\n";
	$$str .= "SKU\t$sku\n";
	$$str .= "PRICE\t$price\n";
	$$str .= "QTY\t$qty\n";
	$$str .= "TOTAL\t$total\n";
	$$str .= "WEIGHT\t$weight\n";
	$$str .= "VOLUME\t$volume\n";
	$$str .= "INFO\t$info\n";

	print_discount_list("orderline", $ol_id, $str);
	print_tax_list($ol_id, $str);
	print_payment_list("orderline", $ol_id, $str);
	print_shipment_list($ol_id, $str);
	
	$count++;
    }
}

##############################################################################
##  func:  print_tax_list
## param:  orderline id
##   ret:  none
## notes: 
##############################################################################

sub print_tax_list {
    my ($id, $str) = @_;

    my $ref = ORDERLINE::tax_listall($id);
    
    my $count = 0;
    my $size = $#{$ref} + 1;

    my ($tax_id, $label, $amnt);

    while ($count < $size) {
	($tax_id, $label, $amnt) = @{$ref->[$count]};

	$$str .= "TAX\t$label\n";
	$$str .= "AMOUNT\t$amnt\n";

	$count++;
    }
}
    
##############################################################################
##  func:  print_payment_list
## param:  (order | orderline) id
##   ret:  none
## notes: 
##############################################################################
    
sub print_payment_list {
    my ($type, $id, $str) = @_;

    my $ref;

    $ref = ORDER::payment_listall($id) if ($type eq "order");
    $ref = ORDERLINE::payment_listall($id) if ($type eq "orderline");

    my $count = 0;
    my $size = $#{$ref} + 1;

    my ($payment_id, $label, $f1, $f2, $f3, $f4, $f5, $amnt);

    while ($count < $size) {
	($payment_id, $label, $f1, $f2, $f3, $f4, $f5, $amnt)
	    = @{$ref->[$count]};

	$$str .= "PAYMENT\t$label\n";
	$$str .= "FIELD1\t$f1\n";
	$$str .= "FIELD2\t$f2\n";
	$$str .= "FIELD3\t$f3\n";
	$$str .= "FIELD4\t$f4\n";
	$$str .= "FIELD5\t$f5\n";
	$$str .= "AMOUNT\t$amnt\n";
	
	$count++;
    }
}

##############################################################################
##  func:  print_discount_list
## param:  (order | orderline) id
##   ret:  none
## notes: 
##############################################################################

sub print_discount_list {
    my ($type, $id, $str) = @_;

    my $ref;

    $ref = ORDER::discount_listall($id) if ($type eq "order");
    $ref = ORDERLINE::discount_listall($id) if ($type eq "orderline");

    my $count = 0;
    my $size = $#{$ref} + 1;

    my ($d_id, $label, $amnt);

    while ($count < $size) {
	($d_id, $label, $amnt) = @{$ref->[$count]};

	$$str .= "DISCOUNT\t$label\n";
	$$str .= "AMOUNT\t$amnt";

	$count++;
    }
}

##############################################################################
##  func:  print_shipment_list
## param:  orderline id
##   ret:  none
## notes: 
##############################################################################

sub print_shipment_list {
    my ($id, $str) = @_;

    my $ref = ORDERLINE::shipment_listall($id);

    my $count = 0;
    my $size = $#{$ref} + 1;

    my ($ship_id, $label, $cost, $address_id);

    while ($count < $size) {
	($ship_id, $label, $cost, $address_id) = @{$ref->[$count]};

	my ($name, $s1, $s2, $city, $state, $zip, $country) = ADDRESS::get_info($address_id);

	$$str .= "SHIPMENT\t$label\n";
	$$str .= "COST\t$cost\n";
	$$str .= "ADDRESS\t$address_id\n";
	$$str .= "NAME\t$name\n";
	$$str .= "STREET1\t$s1\n";
	$$str .= "STREET2\t$s2\n";
	$$str .= "CITY\t$city\n";
	$$str .= "STATE\t$state\n";
	$$str .= "POSTAL_CODE\t$zip\n";
	$$str .= "COUNTRY\t$country\n";

	$count++;
    }
}

##############################################################################
##  func:  print_address_list
## param:  customer id
##   ret:  none
## notes: 
##############################################################################

sub print_address_list {
    my ($id, $str) = @_;

    my $ref = CUSTOMER::address_listall($id);

    my ($addr_id, $label, $name, $s1, $s2, $city, $state, $zip, $country);

    my $count = 0;
    my $size = $#{$ref} + 1;

    while ($count < $size) {
	($addr_id, $label, $name, $s1, $s2, $city, $state, $zip, $country) = @{$ref->[$count]};

	$$str .= "ADDRESS\t$label\n";
	$$str .= "NAME\t$name\n";
	$$str .= "STREET1\t$s1\n";
	$$str .= "STREET2\t$s2\n";
	$$str .= "CITY\t$city\n";
	$$str .= "STATE\t$state\n";
	$$str .= "POSTAL_CODE\t$zip\n";
	$$str .= "COUNTRY\t$country\n";

	$count++;
    }
}

1;









