# ORDERLINE.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

package ORDERLINE;

use strict;

require MILTON;
require DBLIB;
require SHIPMENT;
require TAX;
require ORDER;
require DISCOUNT;
require PAYMENT;



use vars qw ( %listall 
	      %create 
	      %delete 
	      %get_info 
	      %set_attr 
	      %payment_assoc
	      %payment_deassoc
	      %payment_listall
	      %shipment_assoc
	      %shipment_deassoc
	      %shipment_listall
	      %tax_assoc
	      %tax_deassoc
	      %tax_listall
	      %discount_assoc
	      %discount_deassoc
	      %discount_listall );

## ============================================================================
##  This file is the public API for order functionality.
##  It also--for now contains all of the implementation specific functionality.
##  The general layout is that, based on some parameter, a code reference
##  will be called by the general function.
## ============================================================================


## ============================================================================
##  ORDERLINE API
## ============================================================================
##  These are the avaliable functions
##
##  create           create a new orderline
##  delete           delete an orderline
##  listall          lists all orderlines
##  get_info         get info about an orderline
##  set_attr         set attrs of an orderline
##  verify_id        verify an orderline id
##  payment_assoc    associates an orderline with a payment
##  payment_deassoc  deassocitates a payment from an orderline
##  payment_listall  lists all the payments for an orderline
##  shipment_assoc   associates a shipment with an orderline
##  shipment_deassoc deassocitates a shipment from an orderline
##  shipment_listall list the shipments for an orderline
##  tax_assoc        associates a tax with an orderline
##  tax_deassoc      deassociates a tax from an orderline
##  tax_listall      lists all taxes associated with an ordeline
##  discount_assoc   associates a discount with an orderline
##  discount_deassoc deassociates a discount from an ordeline
##  discount_listall lists all discounts associated with an orderline



## ============================================================================
##  IMPLEMENTATIONS
## ============================================================================

## ----------------------------------------------------------------------------
##   func: create
##   desc: create a new orderline
##  param: order_id, sku, price, qty, weight, volume, total, info
##    ret: orderline_id 
##  notes:
## ----------------------------------------------------------------------------

sub create_local {
    my ($order_id, $label, $sku, $price, $qty, $total, $weight, $vol, $info) = @_;

    my ($id) = DBLIB::seq_next_get("orderline_sq");

    my $sqlstr = "insert into orderline
                  (id, cat_order_id, label, sku, price, qty, total, weight, volume, info)
                  values
                  ($id, $order_id, '$label', '$sku', $price, $qty, $total, $weight, $vol, '$info')";

  DBLIB::db_do($sqlstr);
    
    return $id;
}

## ----------------------------------------------------------------------------
##   func: delete
##   desc: delete an existing orderline
##  param: orderline_id
##    ret: 1/ undef
##  notes:
## ----------------------------------------------------------------------------

sub delete_local {
    my ($id) = @_;

    DBLIB::db_transact_begin();

    ## take out any payments associtated with the orderline
    my $sqlstr = "select payment_id from orderline_payment where orderline_id = $id";
    my $id_ref = DBLIB::db_fetchall_arrayref($sqlstr);

    my ($row, $val);
    
    foreach $row (@$id_ref) {
	($val) = @$row;
        PAYMENT::delete($val);
    }

    ## take out any shipments associated with the orderline
    $sqlstr = "select shipment_id from orderline_shipment where orderline_id = $id";
    $id_ref = DBLIB::db_fetchall_arrayref($sqlstr);

    foreach $row (@$id_ref) {
	($val) = @$row;
        SHIPMENT::delete($val);
    }
    

    ## take out any taxes associated with the orderline
    $sqlstr = "select tax_id from orderline_tax where orderline_id = $id";
    $id_ref = DBLIB::db_fetchall_arrayref($sqlstr);

    foreach $row (@$id_ref) {
	($val) = @$row;
        TAX::delete($val);
    }

    ## take out any discounts associated with the orderline
    $sqlstr = "select discount_id from orderline_discount where orderline_id = $id";
    $id_ref = DBLIB::db_fetchall_arrayref($sqlstr);

    foreach $row (@$id_ref) {
	($val) = @$row;
        DISCOUNT::delete($val);
    }

    $sqlstr = "delete from orderline where id = $id";
    DBLIB::db_do($sqlstr);

    DBLIB::db_transact_end();

    return 1;
}

## ----------------------------------------------------------------------------
##   func: listall
##   desc: lists all orderlines associated with an order
##  param: order_id
##    ret: reference to array of orderlines
##  notes:
## ----------------------------------------------------------------------------

sub listall_local {
    my ($id) = @_;
    
    my $sqlstr = "select id, cat_order_id, label, sku, price, qty, total, weight, volume, info
                  from orderline where cat_order_id = $id";

    my $ref = DBLIB::db_fetchall_arrayref($sqlstr);

    return $ref;
}

## ----------------------------------------------------------------------------
##   func: get_info
##   desc: returns the info for an orderline
##  param: orderline_id
##    ret: id, order_id, sku, price, qty, total, weight, vol, info
##  notes:
## ----------------------------------------------------------------------------

sub get_info_local {
    my ($id) = @_;
    
    my $sqlstr = "select id, cat_order_id, label, sku, price, qty, total, weight, volume, info
                  from orderline where id = $id";

    my @info = DBLIB::db_fetchrow_array($sqlstr);

    return @info;
}


## ----------------------------------------------------------------------------
##   func: set_attr 
##   desc: set attributes of an orderline
##  param: id, attrs
##    ret: 1/ undef
##  notes:
## ----------------------------------------------------------------------------

sub set_attr_local {
    my ($id, %attrs) = @_;
    
    ## create sqlstr
    my $sqlstr = "update orderline set ";
    
    my ($key, $val);

    foreach $key (keys(%attrs)) {
	$sqlstr .= "$key = '$attrs{$key}', ";
    }

    chop $sqlstr;
    chop $sqlstr;
    
    $sqlstr .= " where id = $id";

  DBLIB::db_do($sqlstr);

    return 1;
}


## ----------------------------------------------------------------------------
##   func: verify_id
##   desc: verifies avalable orderline id
##  param: order_id
##    ret: 1/ 0 
##  notes: 
## ----------------------------------------------------------------------------

sub verify_id {
    return DBLIB::verify_id("id", "orderline", @_);
}


## ----------------------------------------------------------------------------
##   func: payment_assoc
##   desc: associates a payment with an orderline
##  param: orderline_id, payment_id
##    ret: 1/ undef
##  notes:
## ----------------------------------------------------------------------------

sub payment_assoc_local {
    my ($orderline_id, $payment_id) = @_;

    my $sqlstr = "insert into orderline_payment
                  (orderline_id, payment_id)
                  values
                  ($orderline_id, $payment_id)";

  DBLIB::db_do($sqlstr);

    return 1;
}

## ----------------------------------------------------------------------------
##   func: payment_deassoc
##   desc: deassociates a payment from an orderline
##  param: orderline_id, payment_id
##    ret: 1/ undef
##  notes:
## ----------------------------------------------------------------------------

sub payment_deassoc_local {
    my ($orderline_id, $payment_id) = @_;

    my $sqlstr = "delete from orderline_payment
                  where orderline_id = $orderline_id
                  and payment_id = $payment_id";

  DBLIB::db_do($sqlstr);

    return 1;
}
## ----------------------------------------------------------------------------
##   func: payment_listall
##   desc: list all payments associtated with an orderline
##  param: orderline_id
##    ret: payment_id, type_label, f1, f2, f3, f4, f5, amnt
##  notes:
## ----------------------------------------------------------------------------

sub payment_listall_local {
    my ($id) = @_;

    my $sqlstr = "select payment.id, type_label, field1, field2, field3, field4,
                  field5, amnt
                  from orderline_payment, payment
                  where orderline_payment.orderline_id = $id
                  and orderline_payment.payment_id = payment.id";

    my $ref = DBLIB::db_fetchall_arrayref($sqlstr);

    return $ref;
}

## ----------------------------------------------------------------------------
##   func: shipment_assoc
##   desc: associates a shipment with an orderline
##  param: orderline_id, shipment_id
##    ret: 1/ undef
##  notes:
## ----------------------------------------------------------------------------

sub shipment_assoc_local {
    my ($orderline_id, $shipment_id) = @_;

    my $sqlstr = "insert into orderline_shipment
                  (orderline_id, shipment_id)
                  values
                  ($orderline_id, $shipment_id)";

  DBLIB::db_do($sqlstr);

    return 1;
}

## ----------------------------------------------------------------------------
##   func: shipment_deassoc
##   desc: deassociates a shipment from an orderline
##  param: orderline_id, shipment_id
##    ret: 1/ undef
##  notes:
## ----------------------------------------------------------------------------

sub shipment_deassoc_local {
    my ($orderline_id, $shipment_id) = @_;

    my $sqlstr = "delete from orderline_shipment
                  where orderline_id = $orderline_id
                  and shipment_id = $shipment_id";

  DBLIB::db_do($sqlstr);

    return 1;
}
## ----------------------------------------------------------------------------
##   func: shipment_listall
##   desc: list all shipments associtated with an orderline
##  param: orderline_id
##    ret: shipment_id, type_label, cost, address_id
##  notes:
## ----------------------------------------------------------------------------

sub shipment_listall_local {
    my ($id) = @_;

    my $sqlstr = "select shipment.id, type_label, cost, address_id
                  from orderline_shipment, shipment
                  where orderline_shipment.orderline_id = $id
                  and orderline_shipment.shipment_id = shipment.id";

    my $ref = DBLIB::db_fetchall_arrayref($sqlstr);

    return $ref;
}

## ----------------------------------------------------------------------------
##   func: tax_assoc
##   desc: associates a tax with an orderline
##  param: orderline_id, tax_id
##    ret: 1/ undef
##  notes:
## ----------------------------------------------------------------------------

sub tax_assoc_local {
    my ($orderline_id, $tax_id) = @_;

    my $sqlstr = "insert into orderline_tax
                  (orderline_id, tax_id)
                  values
                  ($orderline_id, $tax_id)";

  DBLIB::db_do($sqlstr);

    return 1;
}

## ----------------------------------------------------------------------------
##   func: tax_deassoc
##   desc: deassociates a tax from an orderline
##  param: orderline_id, tax_id
##    ret: 1/ undef
##  notes:
## ----------------------------------------------------------------------------

sub tax_deassoc_local {
    my ($orderline_id, $tax_id) = @_;

    my $sqlstr = "delete from orderline_tax
                  where orderline_id = $orderline_id
                  and tax_id = $tax_id";

  DBLIB::db_do($sqlstr);

    return 1;
}

## ----------------------------------------------------------------------------
##   func: tax_listall
##   desc: list all taxes associtated with an orderline
##  param: orderline_id
##    ret: tax_id, type_label, amnt
##  notes:
## ----------------------------------------------------------------------------

sub tax_listall_local {
    my ($id) = @_;

    my $sqlstr = "select tax.id, type_label, amnt
                  from orderline_tax, tax
                  where orderline_tax.orderline_id = $id
                  and orderline_tax.tax_id = tax.id";

    my $ref = DBLIB::db_fetchall_arrayref($sqlstr);

    return $ref;
}

## ----------------------------------------------------------------------------
##   func: discount_assoc
##   desc: associates a discount with an orderline
##  param: orderline_id, discount_id
##    ret: 1/ undef
##  notes:
## ----------------------------------------------------------------------------

sub discount_assoc_local {
    my ($orderline_id, $discount_id) = @_;

    my $sqlstr = "insert into orderline_discount
                  (orderline_id, discount_id)
                  values
                  ($orderline_id, $discount_id)";

  DBLIB::db_do($sqlstr);

    return 1;
}

## ----------------------------------------------------------------------------
##   func: discount_deassoc
##   desc: deassociates a discount from an orderline
##  param: orderline_id, discount_id
##    ret: 1/ undef
##  notes:
## ----------------------------------------------------------------------------

sub discount_deassoc_local {
    my ($orderline_id, $discount_id) = @_;

    my $sqlstr = "delete from orderline_discount
                  where orderline_id = $orderline_id
                  and discount_id = $discount_id";

  DBLIB::db_do($sqlstr);

    return 1;
}
## ----------------------------------------------------------------------------
##   func: discount_listall
##   desc: list all discounts associtated with an orderline
##  param: orderline_id
##    ret: discount_id, label,amnt
##  notes:
## ----------------------------------------------------------------------------

sub discount_listall_local {
    my ($id) = @_;

    my $sqlstr = "select discount.id, label, amnt
                  from orderline_discount, discount
                  where orderline_discount.orderline_id = $id
                  and orderline_discount.discount_id = discount.id";

    my $ref = DBLIB::db_fetchall_arrayref($sqlstr);

    return $ref;
}


## ============================================================================
##  IMPLEMENTATION HASHES
## ============================================================================

%listall              = ( "LOCAL" => \&listall_local, );
%create               = ( "LOCAL" => \&create_local, );
%delete               = ( "LOCAL" => \&delete_local, );
%get_info             = ( "LOCAL" => \&get_info_local, );
%set_attr             = ( "LOCAL" => \&set_attr_local, );
%payment_assoc        = ( "LOCAL" => \&payment_assoc_local, );
%payment_deassoc      = ( "LOCAL" => \&payment_deassoc_local, );
%payment_listall      = ( "LOCAL" => \&payment_listall_local, );
%shipment_assoc       = ( "LOCAL" => \&shipment_assoc_local, );
%shipment_deassoc     = ( "LOCAL" => \&shipment_deassoc_local, );
%shipment_listall     = ( "LOCAL" => \&shipment_listall_local, );
%tax_assoc            = ( "LOCAL" => \&tax_assoc_local, );
%tax_deassoc          = ( "LOCAL" => \&tax_deassoc_local, );
%tax_listall          = ( "LOCAL" => \&tax_listall_local, );
%discount_assoc       = ( "LOCAL" => \&discount_assoc_local, );
%discount_deassoc     = ( "LOCAL" => \&discount_deassoc_local, );
%discount_listall     = ( "LOCAL" => \&discount_listall_local, );

## ============================================================================
##  API CALLS
## ============================================================================

sub listall {
    return MILTON::generic_call(\%listall, @_);
}

sub create {
    return MILTON::generic_call(\%create, @_);
}

sub delete {
    return MILTON::generic_call(\%delete, @_);
}

sub get_info {
    return MILTON::generic_call(\%get_info, @_);
}

sub set_attr {
    return MILTON::generic_call(\%set_attr, @_);
}

sub shipment_assoc {
    return MILTON::generic_call(\%shipment_assoc, @_);
}

sub shipment_deassoc {
    return MILTON::generic_call(\%shipment_deassoc, @_);
}

sub shipment_listall {
    return MILTON::generic_call(\%shipment_listall, @_);
}

sub payment_assoc {
    return MILTON::generic_call(\%payment_assoc, @_);
}

sub payment_deassoc {
    return MILTON::generic_call(\%payment_deassoc, @_);
}

sub payment_listall {
    return MILTON::generic_call(\%payment_listall, @_);
}

sub tax_assoc {
    return MILTON::generic_call(\%tax_assoc, @_);
}

sub tax_deassoc {
    return MILTON::generic_call(\%tax_deassoc, @_);
}

sub tax_listall {
    return MILTON::generic_call(\%tax_listall, @_);
}

sub discount_assoc {
    return MILTON::generic_call(\%discount_assoc, @_);
}

sub discount_deassoc {
    return MILTON::generic_call(\%discount_deassoc, @_);
}

sub discount_listall {
    return MILTON::generic_call(\%discount_listall, @_);
}

1;
