# REGEN.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

package REGEN;

use strict;
require THING;
require SNIPPET;
require ITEM;
require PAGE;
require GROUP;
require CONTAINER;
require KNAR;
require SEC;

# @regen_stack is used to store information on what we're currently rendering,
# what methods attrib() should try, and where the output of this template
# should go.  Since all the recursion happens through do_page, we could
# also have kept it in a my() variable there.
use vars qw(%filehash $print_sub $regen_level @regen_stack);
use vars qw($cur_cid $cur_thing $cur_kind 
            $cur_out_location $cur_out_type);


# 
# ----------------------------------------------------------------
# Takes list refs, each consisting of a scalar ref, attribute
# name, and optional default value.  For each attribute name, 
# checks for a snippet with that name, failing that a KNAR
# entry with that name, and failing that uses the supplied default.
# Result is stored in the scalar referenced.  Example:
#
# attrib(
#        [\$num_wheels, 'Number of Wheels', '3'     ],
#        [\$color,      'Color',            ''      ],
#        [\$weasel,     'Weasel type',      'ferret'] );
#
# Attributes like "Name" which are internal to the things and
# not stored as snippets are treated like snippets anyway so
# that you don't have to worry about which are which.
sub attrib {
    return attrib_cid($cur_cid, @_);
}

# 
# ----------------------------------------------------------------
# Same as attrib(), but takes a container id as its first arg.
sub attrib_cid {
    my($cid, @rows) = @_;

    return undef unless SEC::is_id($cid);

    for (@rows) {
	my($result_ref, $name, $default) = @$_;

	$$result_ref = 
	    snippet_cid($cid, $name) || knar_get($name) || $default;
    }
}


# ----------------------------------------------------------------
# Same as attrib(), but only considers snippets and supplied
# defaults, not the KNAR table. 
#
sub snippets {
    return snippets_cid($cur_cid, @_);
}


# 
# ----------------------------------------------------------------
# Same as snippets(), but takes a container id as its first arg.
sub snippets_cid {
    my($cid, @rows) = @_;

    return undef unless SEC::is_id($cid);

    for (@rows) {
	my($result_ref, $name, $default) = @$_;

	$$result_ref = 
	    snippet_cid($cid, $name) || $default;
    }
}

# 
# ----------------------------------------------------------------
# Takes the name of a snippet for the current thing, and returns
# its value.  
# NOTE: This is *different* from the SNIPPET::snip_get_byname
#       routine - snip_get_byname doesn't automatically take care of internal
#       attributes like "Price" and "Name", which aren't actually snippets.
sub snippet {
    return snippet_cid($cur_cid, @_);
}

# 
# ----------------------------------------------------------------
# Same as snippet(), but takes a container id as its first arg.
{
# This is just in the interest of speed; this way, it's only done once.

my(%item_property);
%item_property = (
  Name                       => 1,
  SKU                        => 2,
  Price                      => 3,
  Weight                     => 4,
  Volume                     => 5,
  ATP                        => 6,
  "Reserve Threshold"        => 7,
  "Default Reserve Quantity" => 8,
		  );

sub snippet_cid {
    my($cid, $name) = @_;

    return undef unless SEC::is_id($cid);
    return undef unless(length($name) > 0);

    my($kind, $id) = CONTAINER::container_thing_info($cid);

    if($kind eq $THING::item) {
	if($item_property{$name}) {
	    my(@item_info) = ITEM::item_get_info($id);
	    return $item_info[$item_property{$name}-1];
	} else {
	    return SNIPPET::snip_get_byname($kind, $id, $name);
	}

    } elsif($kind eq $THING::page) {
	if($name eq "Name") {
	    return PAGE::page_get_label($id);
	} else {
	    return SNIPPET::snip_get_byname($kind, $id, $name);
	}

    } elsif($kind eq $THING::group) {
	if($name eq "Name") {
	    return GROUP::group_get_label($id);
	} else {
	    return SNIPPET::snip_get_byname($kind, $id, $name);
	}
    }

    MILTON::fatal_error("Reality check bounced in REGEN.pm");
} # end snippet_cid()
} # end block for my() var.


# 
# ----------------------------------------------------------------
# Return the cid, kind (which will be $THING::item, $THING::page, or
# $THING::group), and thing id for the thing currently being 
# regenerated.
sub my_cid { return $cur_cid; }
sub my_kind { return $cur_kind; }
sub my_id { return $cur_thing; }

# 
# ----------------------------------------------------------------
# Returns ($kind, $id) for the cid provided
sub thing_info_cid {
    my($cid) = @_;

    return undef unless SEC::is_id($cid);
    return CONTAINER::container_thing_info($cid);
}

# 
# ----------------------------------------------------------------
# Returns just the kind for the cid provided
sub thing_kind_cid {
    my($cid) = @_;

    return undef unless SEC::is_id($cid);
    my($kind, $id) = CONTAINER::container_thing_info($cid);
    return $kind;
}

# 
# ----------------------------------------------------------------
# Returns the KNAR entry of the specified name
sub knar_get {
    my($name) = @_;

    return undef unless(length($name) > 0);

    return KNAR::knar_entry_get($name);
}

# 
# ----------------------------------------------------------------
# Returns the cid of the child of the current container which has
# the name specified.  Not a deep search.
sub find_child {
    my($name) = @_;

    return find_child_cid($cur_cid, $name);
}

# 
# ----------------------------------------------------------------
# Like find_child(), but searches children of the container specified.
sub find_child_cid {
    my($cid, $name) = @_;

    return undef unless SEC::is_id($cid);
    my $all_children = CONTAINER::container_children_cid($cid);

    for (@$all_children) {
	my($label) = CONTAINER::container_thing_name($_);
	return $_ if $label eq $name;
    }
}

# 
# ----------------------------------------------------------------
# Returns an arrayref of cids of all children containers of the
# current container.
sub children {
    return children_cid($cur_cid);
}

# 
# ----------------------------------------------------------------
# Like children(), but lists children of the container specified.
sub children_cid {
    my($cid) = @_;
    return undef unless($cid eq "0" or SEC::is_id($cid));
    return CONTAINER::container_children_cid($cid);
}

# 
# ----------------------------------------------------------------
# Returns the children of the container specified which are groups
sub group_children {
    my($cid) = @_;
    return undef unless($cid eq "0" or SEC::is_id($cid));
    return CONTAINER::container_children_type($THING::group, $cur_cid);
}

# 
# ----------------------------------------------------------------
# Returns the children of the container specified which are pages
sub page_children {
    my($cid) = @_;
    return undef unless($cid eq "0" or SEC::is_id($cid));
    return CONTAINER::container_children_type($THING::page, $cur_cid);
}

# 
# ----------------------------------------------------------------
# Returns the children of the container specified which are items
sub item_children {
    my($cid) = @_;
    return undef unless($cid eq "0" or SEC::is_id($cid));
    return CONTAINER::container_children_type($THING::item, $cur_cid);
}

# 
# ----------------------------------------------------------------
# Like image_fn_cid, but for the current thing.
sub image_fn {
    return image_fn_cid($cur_cid, @_);
}

# 
# ----------------------------------------------------------------
# Returns a URL to an image of the specified container suitable
# for placing in an <IMG> tag.
# If no other parameters are passed, checks only for a "Link Image"
# snippet.  Otherwise, checks for a snippet of the thing for each
# extra parameter passed.
# Examples:
# image_fn_cid($item_cid);  # only checks for "Link Image"
# image_fn_cid($item_cid, "Link Image", "Main Image", "Image");
#
sub image_fn_cid {
    my($cid, @snip_names) = @_;

    push(@snip_names, "Link Image") unless($#snip_names >= 0);

    return undef unless SEC::is_id($cid);
    my($kind, $id) = CONTAINER::container_thing_info($cid);
    MILTON::fatal_error("Reality check bounced!") unless SEC::is_id($id);

    my($fn);
    for(@snip_names) {
	next unless(length($_) > 0);

	$fn = SNIPPET::snip_get_byname($kind, $id, $_) || 
	    knar_get($_);
	last if $fn;
    }

    my($dir, $img_dir);
    $dir = SNIPPET::snip_get_byname($kind, $id, 'Directory') ||
	knar_get("Directory");

    $img_dir = SNIPPET::snip_get_byname($kind, $id, 'Image Directory') ||
	knar_get("Image Directory");

    return "" unless $fn;

    return knar_get("REGEN_OUTPUT_URL") . 
	($dir ? "/$dir" : '') .
	($img_dir ? "/$img_dir":'') . "/$fn";
}

# 
# ----------------------------------------------------------------
# Returns a link for the cid specified, suitable for an <A HREF=...>
# tag.
sub link_fn_cid {
    my($cid) = @_;

    return undef unless SEC::is_id($cid);
    my($kind, $id) = CONTAINER::container_thing_info($cid);
    MILTON::fatal_error("Reality check bounced!") unless SEC::is_id($id);

    my($fn) = SNIPPET::snip_get_byname($kind, $id, "Filename");

    my($dir);
    $dir = SNIPPET::snip_get_byname($kind, $id, 'Directory') ||
	knar_get("Directory");

    return "" unless $fn;

    return knar_get("REGEN_OUTPUT_URL") . 
	($dir ? "/$dir" : '') . "/$fn";
}

#
# ----------------------------------------------------------------
# Returns a URL for an <A HREF=...> tag instructing the shopping
# basket to add the current item to the shopping cart.
sub order_link {
    return order_link_cid($cur_cid);
}

# 
# ----------------------------------------------------------------
# Like order_link(), but returns a link for the item whose container
# id is specified
sub order_link_cid {
    my($cid) = @_;

    return undef unless SEC::is_id($cid);

    my($kind, $id) = CONTAINER::container_thing_info($cid);

    if($kind ne $THING::item) {

	my $name = snippet_cid($cid, "Name");
	if($kind eq $THING::page) {
	    warning("order_link_cid: Can't make an order link for page $name");
	} elsif($kind eq $THING::group) {
           warning("order_link_cid: Can't make an order link for group $name");
	} else {
	    warning("order_link_cid: Invalid container $cid");
	}

	return "error";
    }

    return KNAR::knar_entry_get('TALLYMAN_SECURE_URL') . 
      "/cart.epl?add=ok&pnum=$id";
}

# 
# ----------------------------------------------------------------
# Returns a URL to the redirection script.  Used for the "Go!" button 
# in dropdown quicklink boxes.  Normally they use Javascript; the
# "Go!" button allows folks without Javascript to get there via
# the redirector.
sub redir_link {
    my $url = KNAR::knar_entry_get('REDIRECTOR_URL');
    warning("REDIRECTOR_URL knar entry not set.") unless $url;
    return $url;
}






##########################################################################
##########################################################################
# Under-the-hood stuff.

sub status {
    my $text = join("", @_);
    &{$print_sub}( "  " x $regen_level, "$text\n" );
}

sub warning {
    my $text = join("", @_); 
    &{$print_sub}( "  " x $regen_level, "<font color=\"#FF0000\">Warning: $text</font>\n");
}


sub regen_init {
    my($code_ref) = @_;

    $regen_level = 0;
    $print_sub = $code_ref;
}

sub regen_all {
    status("Generating site...");

    do_page(MILTON::template_dir() . "/generic.epl", "return", "", 0,
	    $THING::group, 0, []);

    status("Regen finished.");
}

    
sub render_cid {
    my($cid, $args, $other_template) = @_;

    unless(SEC::is_id($cid)) {
	warning("Can't render: Invalid cid $cid.");
	return undef;
    }

    my($kind, $thing_id) = CONTAINER::container_thing_info($cid);

    return render_thing($kind, $thing_id, $cid, $args, $other_template);
}

sub render_thing {
    my($kind, $thing_id, $cid, $args, $other_template) = @_;

    unless(SEC::is_id($thing_id)) {
	warning("Can't render: Invalid thing_id $thing_id.");
	return undef;
    }

    unless(SEC::is_id($cid)) {
	warning("Can't render: Invalid cid $cid.");
	return undef;
    }

    my($thing_name) = snippet_cid($cid, "Name");

    my($template_fn) = $other_template ||
      SNIPPET::snip_get_byname($kind, $thing_id, "Template");

    unless($template_fn) {
        warning("Couldn't regen $thing_name (cid: $cid).  No template.");
	return undef;
    }

    my($template_path) = MILTON::template_dir() . "/$template_fn";
    SEC::untaint_ref(\$template_path);

    my($location);

    # If there's a "Filename" snippet, use that for output
    my($filename) = SNIPPET::snip_get_byname($kind, $thing_id, "Filename");
    my($output_type) = "file";

    my($output_dir);
    if($filename and SEC::is_filename($filename)) {

    #    if($fn ne "" and $fn =~ m|^[^/]|) {
	my $output_path = KNAR::knar_entry_get('REGEN_OUTPUT_DIR');
	attrib_cid($cid, [\$output_dir, 'Directory', '']);

	if($output_path) {
	    if($output_dir) {
		$location = "$output_path/$output_dir/$filename";
	    } else {
		$location = "$output_path/$filename";
	    }

            SEC::untaint_ref(\$location);
	} else {
	    warning("KNAR entry REGEN_OUTPUT_DIR not found.");
	    $location = "";
	    $output_type = "return";
	}
    #    }

    } elsif($filename and !SEC::is_filename($filename)){
	$location = "";
	$output_type = "return";
	warning("Filename field invalid: '$filename'");


    # If not, use the HTML snippet if it exists for this thing
    } elsif(grep {SNIPTYPE::sniptype_get_name($kind, $_) eq "HTML"} 
	     @{THINGTYPE::thingtype_get_thing_sniptypes($kind, $thing_id)}) {

	$location = "HTML";
	$output_type = "snippet";

    # Or just return the text to the caller
    } else {
	    
	$location = "";
	$output_type = "return";
    }


    if($output_type eq "snippet") {
	status("Generating $location for \"$thing_name\" using $template_fn");

    } elsif($output_type eq "return") {
        status("Returning output of $template_fn for \"$thing_name\"");

    } elsif($output_type eq "file") {
	
	status("Generating \"<a href=\"". 
	       KNAR::knar_entry_get("REGEN_OUTPUT_URL").
	       ($output_dir ? "/$output_dir" : '') .
	       "/$filename\">$thing_name</a>\" with $template_fn");
    }

    do_page($template_path, $output_type, $location, $cid, $kind, $thing_id,
	    [$kind, $thing_id, $cid, $location, $args]);
}


#
# -----------------------------------------------------------------------------
# This where much of the magic happens.  We load the template of the thing
# to render, set up the global variables with info on the current thing,
# call Execute(), put the results where they need to be, 
# then restore everything to the way it was.
sub do_page {
  my($infn,$out_type,$out_location,$cid,$kind, $thing_id,$param)=@_;
  my($hohum,$filecontents);

  unless(SEC::is_id($thing_id) or $thing_id eq 0) {
      warning("Can't render: Invalid thing_id $thing_id.");
      return undef;
  }
  
  unless(SEC::is_id($cid) or $cid eq "0") {
      warning("Can't render: Invalid cid $cid.");
      return undef;
  }
  
  # Cache the templates(!).  This is really redundant, since Embperl
  # will cache them itself if you don't pass the "input" parm.
  if (!defined($filehash{$infn})) {
      $filecontents=REGEN::emit_file($infn);
      if (!defined($filecontents)) { return 0; }

      $filehash{$infn} = $filecontents;
  } else {
      $filecontents = $filehash{$infn};
  }

  $regen_level++;

#status("Pushing: $cur_out_location, $cur_out_type, $cur_kind, $cur_thing");

  push @regen_stack, 
      [$cur_out_type, 
       $cur_out_location, 
       $cur_cid,
       $cur_kind, 
       $cur_thing ];

  ($cur_out_type,$cur_out_location,$cur_cid,$cur_kind,$cur_thing) = 
  ($out_type,    $out_location,    $cid,    $kind,    $thing_id);

  HTML::Embperl::Execute ({ inputfile  => "$infn",
                            options    => 256 | 512 | 16384,
                            param      => $param,
			    input      => \$filecontents,
                            mtime      => 42,
                            output     => \$hohum});

  if($hohum =~ m|<TITLE>Embperl Error|) {
      $cur_out_type = "return";
      $cur_out_location = "";

      warning("Embperl reported the following errors with this page:");
      my @lines = split(/\n/, $hohum);
      for (@lines) {
	  next unless /ERR: /;
	  warning($_);
      }
  }

  my $retval = 1;

  if($cur_out_location) {
#      if(SEC::is_tainted($cur_out_location)) {
#	  SEC::untaint_ref(\$cur_out_location);
#	    warning("$cur_out_location tainted.");
#	}

      if($cur_out_type eq "file") {
	  if (!open(TMP,">$cur_out_location")) {
	      warning("Cannot open file $cur_out_location: $!"); 
	      $retval = 0;
	  }
	  print TMP $hohum;
	  close(TMP);
      } elsif($cur_out_type eq "snippet") {
	  my($html_id) = SNIPTYPE::sniptype_get_id($cur_kind, "HTML");
	  if(!SEC::is_id($html_id)) {
	      warning("Can't find HTML snippet.");
	      $retval = 0;
	  } else {
#	      $hohum = DBLIB::db_string_clean($hohum);
	      SNIPPET::snip_update($cur_kind, $cur_thing, $html_id, $hohum);
	  }
      } elsif($cur_out_type eq "return") {
	  $retval = $hohum;
      } else {
	  warning("Reality check bounced in do_page($cur_out_type).");
      }
  }

  my($popped_vars) = pop @regen_stack;
  ($cur_out_type,$cur_out_location, $cur_cid, $cur_kind, $cur_thing) 
      = @$popped_vars;

#  status("Popped: $cur_out_location, $cur_kind, $cur_thing");
  $regen_level--;

  return($retval);
}

sub set_output_location {
    my($type, $loc) = @_;

    if($type ne "file" and $type ne "snippet" and $type ne "return") {
   warning("Invalid set_output_location: '$type'.  Output location unchanged");
    }

    $cur_out_type = $type;
    $cur_out_location = $loc;
}


#
# -----------------------------------------------------------------------------
#
# returns the contents of a file
#   takes: filename
# returns: the contents of the file
#
sub emit_file {
  my($fn)=@_;
  my($result,$length);

  if (!open(TMP,"$fn")) { 
    warning("Cannot open file $fn: $!"); 
    return(undef);
  }

  seek(TMP,0,2);
  $length=tell(TMP);
  seek(TMP,0,0);
  read TMP,$result,$length;
  close(TMP);
  return($result);
}


1;

