# SEC.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

# Security checks for untrusted data
# (also the encode_uri() sub)

require DBLIB;
require MILTON;
package SEC;
use strict;

#
# -----------------------------------------------------------------------------
# -----------------------------------------------------------------------------
# First arg to these require_* functions must be an array ref.  The corresponding
# entry of it will be listed as the reason for a fatal_error if one of the
# args doesn't meet the requirement.  Any args passed as scalar references will
# be untainted if untainting is turned on.

#
# -----------------------------------------------------------------------------
# a handy function to check the type of a number of variables, and clean
# the strings.
#
# example call:
#
# SEC::require_and_clean('SKU',    'STR',   \$fdat{'sku'},
#                        'Price',  'FLOAT', \$fdat{'price'},
#                        'Weight', 'FLOAT', \$fdat{'weight'},
#                        'Volume', 'FLOAT', \$fdat{'volume'},
#                        'ATP',    'INT',   \$fdat{'atp'},
#                        'RES',    'INT',   \$fdat{'res'},
#                        'DRQ',    'INT',   \$fdat{'drq'});
#

sub require_and_clean {
  my($i, $name, $type, $val);
  my(@params) = @_;

  for ($i=0; $i<($#params+1)/3; $i++) {
    $name = $params[$i*3 + 0];
    $type = $params[$i*3 + 1];
    $val = $params[$i*3 + 2];

    if ($type eq 'INT') {
      require_int(["$name must be composed of integer characters."], $val);

    } elsif ($type eq 'FLOAT') {
      require_float(["$name must be composed of float characters."], $val);

    } elsif ($type eq 'ID') {
      require_id(["$name must be composed of ID characters."], $val);

    } elsif ($type eq 'WORD') {
      require_word(["$name must be composed of word characters."], $val);

    } elsif ($type eq 'WORDS') {
      require_words(["$name must be composed of word characters."], $val);

    } elsif ($type eq 'FILENAME') {
      require_filename(["$name must be composed of filename characters."], 
                       $val);

    } elsif ($type eq 'STR') {
      if ($$val eq "") {
#        MILTON::fatal_error("$name must not be empty.");
      } else {
        ${$params[$i*3 +2]} = DBLIB::db_string_clean(${$params[$i*3 + 2]});
      }

    } else {
      MILTON::fatal_error('Unrecognized cleaning type!');
    }

  }

}

#
# -----------------------------------------------------------------------------
#
sub require_int {
    return do_require(\&is_int, @_);
}

#
# -----------------------------------------------------------------------------
#
sub require_float {
    return do_require(\&is_float, @_);
}

#
# -----------------------------------------------------------------------------
#
sub require_id {
    return do_require(\&is_id, @_);
}

#
# -----------------------------------------------------------------------------
# Here a "word" is something like a C identifier; single word with no 
# whitespace
sub require_word {
    return do_require(\&is_word, @_);
}

#
# -----------------------------------------------------------------------------
# Here the definition of a word is a little more broad, but something has to
# be there, and only a few punctuation characters are allowed.
sub require_words {
    return do_require(\&is_words, @_);
}

#
# -----------------------------------------------------------------------------
#
sub require_filename {
    return do_require(\&is_filename, @_);
}

#
# -----------------------------------------------------------------------------
# 
sub do_require {
    my($check_sub, $err_msg_ref, @vals) = @_;

    MILTON::fatal_error("SEC::require_* called without err strings") 
	unless(ref($err_msg_ref) eq "ARRAY");

    my(@err_msg) = @$err_msg_ref;

    my($val);
    foreach $val (@vals) {
	my($msg) = shift @err_msg;

	if(ref($val) eq "SCALAR" or
	   ref($val) eq "LVALUE") {
	    MILTON::fatal_error($msg) unless(&{$check_sub}($$val));
	    untaint_ref($val) if $DBLIB::check_taint;
	} elsif(!ref($val)) {
    	    MILTON::fatal_error($msg) unless(&{$check_sub}($val));
	} else {
	    MILTON::fatal_error("Invalid arg $val passed to require_*");
	}
    }

    return 1;
}


#
# -----------------------------------------------------------------------------
#
sub is_int {
    my($val) = @_;

    return 1 if($val =~ /^[0-9]+$/ and substr($val, -1, 1) ne "\n");
    return 0;
}

#
# -----------------------------------------------------------------------------
#
sub is_float {
    my($val) = @_;

    # pretty weird match, eh?  It's to handle four possible float formats:
    # xxxx xxxx.yyyy xxxx. .yyyy

    return 1 if($val =~ /^[-+]?([0-9]+|[0-9]+\.[0-9]*|[0-9]*\.[0-9]+)$/ and substr($val, -1, 1) ne "\n");
    return 0;
}

#
# -----------------------------------------------------------------------------
#
sub is_id {
    my($val) = @_;

    return 1 if($val =~ /^[0-9]+$/ 
		and substr($val, -1, 1) ne "\n"
		and $val >= 1000);
    return 0;
}

#
# -----------------------------------------------------------------------------
#
sub is_word {
    my($val) = @_;

    return 1 if($val =~ /^[a-zA-Z_][a-zA-Z0-9_]*$/ 
		and substr($val, -1, 1) ne "\n");
    return 0;
}

#
# -----------------------------------------------------------------------------
#
sub is_words {
    my($val) = @_;

    return 1 if($val =~ m|^[\w\s\-/;:,.<>?=+_)(*&\%\$\#\@\!~]+$| 
		and substr($val, -1, 1) ne "\n");
    return 0;
}

#
# -----------------------------------------------------------------------------
#
sub is_filename {
    my($val) = @_;

    return 1 if($val =~ m|^(/?[a-zA-Z_0-9]+[a-zA-Z0-9.\-_]*)+$| 
		and substr($val, -1, 1) ne "\n");
    return 0;
}


#
# -----------------------------------------------------------------------------
#
# Returns untainted copies of the values in the array for which a ref
# was passed.  Does *NOT* untaint the values of the array itself.
sub untaint {
    my(@args) = @_;

    foreach (@args) {
	/(.*)/s;
	$_ = $1;
    }

    return @args if wantarray;
    return $args[0];
}

#
# -----------------------------------------------------------------------------
# Passed some variable references, untaints the variables referred to.
# The \() notation described in the perlref manpage might be very helpful
# when calling this function.
sub untaint_ref {
    foreach (@_) {
        MILTON::fatal_error("untaint_ref arg not a scalar ref(".ref($_).")") 
	    unless(ref($_) eq "SCALAR" or
		   ref($_) eq "LVALUE");

	$$_ =~ /(.*)/s;
	$$_ = $1;
    }

    return undef;
}

#
# -----------------------------------------------------------------------------
# From the perlsec manpage
sub is_tainted {
    no strict;
    return ! eval {
	$foo = join('',@_), kill 0;
	1;
    };
}

#
# -----------------------------------------------------------------------------
# Returns a conservative URI-encoding of the string passed
#
# -----------------------------------------------------------------------------
#
sub encode_uri {
    my($text) = @_;

    $text =~ s/([^a-zA-Z0-9_])/'%' . uc(unpack('H2', $1))/eg;
    return $text;
}
