# SHIPPING.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

package SHIPPING;

use strict;

require DBLIB;
require SEC;

#
# AVAILABLE FUNCTIONS AND VARIABLES DEFINED IN THIS PACKAGE:
#
#   &shippingtype_get_label
#   &shippingtype_list_all
#   &shippingtype_list_all_available
#   &shippingtype_get_info
#   &shippingtype_set_info
#   &shippingtype_delete
#   &shippingtype_create
#   &shippingtype_verify_id
#
#   &shippingalg_get_label
#   &shippingalg_get_info
#   &shippingalg_list_all
#   &shippingalg_needs_table
#   &shippingalg_verify_id
#
#   &shipbytotal_list_all
#   &shipbytotal_create
#   &shipbytotal_delete
#   &shipbytotal_set_info
#   &shipbytotal_find_applicable_range
#   &shipbytotal_verify_id
#
#   &zb_ups_calc_charge_ground
#   &zb_ups_calc_charge_next_day_air
#   &zb_ups_calc_charge_next_day_air_saver
#   &zb_ups_calc_charge_second_day_air
#   &zb_ups_calc_charge_second_day_air_am
#   &zb_ups_calc_charge_three_day_select
#   &zb_ups_calc_charge
#   &zb_ups_calc_zone
#   &zb_ups_calc_rate
#   &zb_ups_needs_surcharge
#
# =============================================================================
#

use vars qw(%shipfunc);

%shipfunc = ('shipping_calculate_charge_by_price' =>
              \&shipping_calculate_charge_by_price,
  
             'shipping_calculate_charge_by_weight' =>
              \&shipping_calculate_charge_by_weight,

             'shipping_calculate_charge_by_volume' =>
              \&shipping_calculate_charge_by_volume,

             'shipping_calculate_charge_by_quantity' =>
              \&shipping_calculate_charge_by_quantity,

             'zb_ups_calc_charge_ground' =>
              \&zb_ups_calc_charge_ground,

             'zb_ups_calc_charge_next_day_air' =>
              \&zb_ups_calc_charge_next_day_air,

             'zb_ups_calc_charge_next_day_air_saver' =>
              \&zb_ups_calc_charge_next_day_air_saver,

             'zb_ups_calc_charge_second_day_air' =>
              \&zb_ups_calc_charge_second_day_air,

             'zb_ups_calc_charge_second_day_air_am' =>
              \&zb_ups_calc_charge_second_day_air_am,

             'zb_ups_calc_charge_three_day_select' =>
              \&zb_ups_calc_charge_three_day_select );

#
# -----------------------------------------------------------------------------
#
# This function is passed a specific shipping type.  It then looks up
# the algorithm in the database, grabs the function call, and calls it.
# The called function frobs whatever data it needs to, and returns an
# appropriate charge.
#
# XXX these functions really shouldn't frob BASKET's data directly, but
#     they do, and I couldn't think of a better way to do it.
#

sub shipping_calculate_charge {
  my($id) = shift;
  my($str,$charge);

  $charge = 0;

  my($label, $available, $calculated, $algorithm_id, $min, $extra) 
     = shippingtype_get_info($id);

  if ($available eq "no" || $calculated eq "no") {
    return -1;
  }

  my($alglabel,$algfunc) = shippingalg_get_info($algorithm_id);

  # this will call the shipping algorithm's function, which will calculate
  # a base charge.  It might return -1.
  # it is assumed that the code is correct; the function that enters it
  # should ensure that!

  $charge = &{$shipfunc{$algfunc}}($id);

  # we then apply any user-defined shipping logic.
  # if the above eval failed for some reason (we got a -1), we don't
  # apply anything; we just allow the error to fall through.

  if ($charge != -1) {
    eval($extra);
  } else {
    return -1;
  }

  # if the calculation of a shipping charge was successful, we test it
  # against the minimum allowable value specified by the user

  if ($charge < $min) {
    return ($min);
  } else {
    return ($charge);
  }
}

#
# -----------------------------------------------------------------------------
#

sub shipping_calculate_charge_by_price {
  return(shipping_calculate_charge_with_range(@_,$BASKET::sub_total));
}

sub shipping_calculate_charge_by_weight {
  return(shipping_calculate_charge_with_range(@_,BASKET::total_weight()));
}

sub shipping_calculate_charge_by_volume {
  return(shipping_calculate_charge_with_range(@_,BASKET::total_volume()));
}

sub shipping_calculate_charge_by_quantity {
  return(shipping_calculate_charge_with_range(@_,BASKET::total_quantity()));
}

sub shipping_calculate_charge_with_range {
  my($id, $range) = @_;
  my($flat, $percent, $ship_total);

  # find_applicable_range returns -1 if no applicable range was found

  ($flat,$percent) = shipbytotal_find_applicable_range($id,$range);

  if ($flat==-1 && $percent==-1) { return (-1); }

  if ($flat==0) { 
    $ship_total=$BASKET::sub_total * $percent; 
  } else { 
    $ship_total=$flat;
  }

  return ($ship_total);
}

#
# =============================================================================
#
# SHIPPINGTYPE API
#
# =============================================================================
#
# gets a list of AVAILABLE shipping types
#   (see shippingtype_list_all to get a full list)
#   takes: nothing
# returns: \@ids, \@labels, \@available, \@calculated, \@algorithm,
#          \@minimum, \@extra_alg
#   notes: available and calculated are either 'yes' or 'no'
#

sub shippingtype_list_all_available {
  my($sqlstr) = "select shippingtype_id,label,available,calculated,
                        shippingalg_id,minimum,extra_alg
                 from shippingtype 
                 where shippingtype_id <> 0
                 and available = 'yes'";

  return(DBLIB::db_fetchall_arrayref($sqlstr));
}

#
# -----------------------------------------------------------------------------
# gets a list of ALL shipping types
#   (see shippingtype_list_all_available to get a just available items)
#   takes: nothing
# returns: \@ids, \@labels, \@available, \@calculated, \@algorithm
#   notes: available and calculated are either 'yes' or 'no'
#

sub shippingtype_list_all {
  my($sqlstr) = "select shippingtype_id,label,available,calculated,
                        shippingalg_id,minimum,extra_alg
                 from shippingtype 
                 where shippingtype_id <> 0";

  return(DBLIB::db_fetchall_arrayref($sqlstr));
}

#
# -----------------------------------------------------------------------------
# gets the info of a specific shipping type
#   takes: shippingtype_id
# returns: $label, $available, $calculated,$algorithm,$minimum,$extra_alg
#   notes: available and calculated are either 'yes' or 'no'
#

sub shippingtype_get_info {
  my($id) = shift;

  my($sqlstr) = "select label,available,calculated,shippingalg_id,
                        minimum,extra_alg
                 from shippingtype 
                 where shippingtype_id = '$id'";

  return(DBLIB::db_fetchrow_array($sqlstr));
}

#
# -----------------------------------------------------------------------------
# sets the info of a specific shipping type
#   takes: shippingtype_id, label, available, calculated, algorithm,
#          minimum, extra_alg
# returns: nothing
#   notes: available and calculated are forced to be either 'yes' or 'no'
#

sub shippingtype_set_info {
  my($id,$label,$available,$calculated,$algorithm,$min,$extra) = @_;

  if (!($available eq 'yes' || $available eq 'no')) {
    $available='no';  # off by default
  }

  if (!($calculated eq 'yes' || $calculated eq 'no')) {
    $calculated='no'; # off by default
  }

  my($sqlstr) = "update shippingtype 
                 set label='$label',
                 available='$available',
                 calculated='$calculated',
                 shippingalg_id='$algorithm',
                 minimum='$min',
                 extra_alg='$extra'
                 where shippingtype_id = '$id'";

  DBLIB::db_do($sqlstr);
}

#
# -----------------------------------------------------------------------------
# returns the label of a shipping type
#   takes: $shippingtype_id
# returns: $label
#
sub shippingtype_get_label {
  my($id) = shift;
  my($sqlstr) = "select label from shippingtype 
                 where shippingtype_id = '$id'";
  my($label) = DBLIB::db_fetchrow_array($sqlstr);
  return $label;
}

#
# -----------------------------------------------------------------------------
#
# creates a shipping type
#   takes: label, available, calculated, shipping algorithm id,
#          minimum, extra_alg
# returns: id of new shipping type
#   notes: available and calculated are forced to be either 'yes' or 'no'
#

sub shippingtype_create {
  my($label, $available, $calculated, $algid, $min, $extra) = @_;
  my($sqlstr);

  DBLIB::db_transact_begin();

  # get the next available id

  my($nextid) = DBLIB::seq_next_get('shippingtype_sq');

  if (!($available eq 'yes' || $available eq 'no')) {
    $available='no';  # off by default
  }

  if (!($calculated eq 'yes' || $calculated eq 'no')) {
    $calculated='no'; # off by default
  }

  $sqlstr = "insert into shippingtype (shippingtype_id, label, available,
             calculated, shippingalg_id, minimum, extra_alg)
             values
             ('$nextid', '$label', '$available', '$calculated',
             '$algid', '$min', '$extra')";

  DBLIB::db_do($sqlstr);

  DBLIB::db_transact_end();

  return($nextid);
}


#
# -----------------------------------------------------------------------------
#

sub shippingtype_delete {
  my($id) = shift;

  # delete all dependant entries in shipbytotal

  my($sqlstr) = "delete from shipbytotal where shippingtype_id = '$id'";
  DBLIB::db_do($sqlstr);

  # delete the actual method

  $sqlstr = "delete from shippingtype where shippingtype_id = '$id'";
  DBLIB::db_do($sqlstr);
}

#
# -----------------------------------------------------------------------------
#

sub shippingtype_verify_id {
  return DBLIB::verify_id('shippingtype_id', 'shippingtype', @_);
}

#
# =============================================================================
#
# SHIPPINGALG API
#
# =============================================================================
#
# gets a list of all shipping algorithms
#   takes: nothing
# returns: \@id, \@label
#

sub shippingalg_list_all {
  my($sqlstr) = "select shippingalg_id, label
                 from shippingalg
                 where shippingalg_id <> 0
                 order by shippingalg_id";

  return(DBLIB::db_fetchall_arrayref($sqlstr));
}

#
# -----------------------------------------------------------------------------
# returns the label of a shipping algorithm
#   takes: $shippingalg_id
# returns: $label
#

sub shippingalg_get_label {
  my($id) = shift;

  my($sqlstr) = "select label from shippingalg
                 where shippingalg_id = '$id'";

  my($str) = DBLIB::db_fetchrow_array($sqlstr);
  return($str);
}

#
# -----------------------------------------------------------------------------
# returns the information of an algorithm
#   takes: $shippingalg_id
# returns: $label, $func
#

sub shippingalg_get_info {
  my($id) = shift;

  my($sqlstr) = "select label,function_name
                 from shippingalg
                 where shippingalg_id = '$id'";

  return(DBLIB::db_fetchrow_array($sqlstr));
}

#
# -----------------------------------------------------------------------------
# tells you whether or not this shipping algorithm needs table entries
#   takes: $shippingalg_id
# returns: 1 or 0
#

sub shippingalg_needs_table {
  my($id) = shift;

  my($sqlstr) = "select needs_table
                 from shippingalg
                 where shippingalg_id = '$id'";

  my($token) = DBLIB::db_fetchrow_array($sqlstr);

  if ($token eq 'Y') {
    return 1;
  }

  return 0;
}

#
# -----------------------------------------------------------------------------
#

sub shippingalg_verify_id {
  return DBLIB::verify_id('shippingalg_id', 'shippingalg', @_);
}

#
# =============================================================================
#
# SHIP BY TOTAL API
#
# =============================================================================
#
#
# gets a list of all shipbytotal ranges for a specified shipping type,
# ordered by start_range
#
#   takes: shippingtype_id
# returns: \@ids, \@start_range, \@end_range, \@flat, \@percent
#

sub shipbytotal_list_all {
  my($shippingtype_id) = shift;

  my($sqlstr)="select shipbytotal_id, start_range, end_range, flat, percent
               from shipbytotal
               where shippingtype_id = '$shippingtype_id'
               order by start_range, end_range";

  return(DBLIB::db_fetchall_arrayref($sqlstr));
}

#
# -----------------------------------------------------------------------------
#

sub shipbytotal_delete {
  my($id) = shift;
  my($sqlstr) = "delete from shipbytotal
                 where shipbytotal_id = '$id'";
  DBLIB::db_do($sqlstr);
}

#
# -----------------------------------------------------------------------------
#

sub shipbytotal_create {
  my($id, $start, $end, $flat, $percent) = @_;

  # get the next available id
  my($nextid) = DBLIB::seq_next_get('shipbytotal_sq');

  # and insert
  my($sqlstr) = "insert into shipbytotal (shipbytotal_id, shippingtype_id,
                 start_range,end_range,flat,percent)
                 values
                 ('$nextid', '$id', '$start', '$end', '$flat', '$percent')";

  DBLIB::db_do($sqlstr);

  return($nextid);
}

#
# -----------------------------------------------------------------------------
#

sub shipbytotal_set_info {
  my($shipbytotal_id, $shippingtype_id, $start, $end, $flat, $percent) = @_;

  my($sqlstr) = "update shipbytotal set
                 shippingtype_id = '$shippingtype_id',
                 start_range = '$start',
                 end_range = '$end',
                 flat = '$flat',
                 percent = '$percent'
                 where shipbytotal_id = '$shipbytotal_id'";

  DBLIB::db_do($sqlstr);
}

#
# -----------------------------------------------------------------------------
#
# returns the flat and percent values corresponding to a specified value.
#   takes: shippingtype_id, dollar
# returns: $flat, $percent;  returns -1 if no corresponding range was found.
#

sub shipbytotal_find_applicable_range {
  my($id, $val) = @_;

  my($sqlstr) = "select flat,percent from shipbytotal
                 where end_range > '$val' and start_range <= '$val' 
                 and shippingtype_id='$id'";

  my($flat,$percent) = DBLIB::db_fetchrow_array($sqlstr);

  if (!defined($flat)) {
    $flat = -1;
    $percent = -1;
  }

  return($flat,$percent);
}

#
# -----------------------------------------------------------------------------
#

sub shipbytotal_verify_id {
  return DBLIB::verify_id('shipbytotal_id', 'shipbytotal', @_);
}

#
# =============================================================================
#
# UPS ALGORITHMS
#
# =============================================================================
#
# algorithms provided by UPS:
#   Ground
#   Next Day Air
#   Next Day Air Saver
#   2nd Day Air
#   2nd Day Air A.M.
#   3 Day Select
#
# -----------------------------------------------------------------------------
#
# For each of the following algorithms:
#   takes: a zipcode and a weight
# returns: -1 if the service is not available, otherwise the cost of the
#          shipment
#

sub zb_ups_calc_charge_ground {
  return(zb_ups_calc_charge($BASKET::s_zip,BASKET::total_weight(),'ground'));
}

sub zb_ups_calc_charge_next_day_air {
  return(zb_ups_calc_charge($BASKET::s_zip,BASKET::total_weight(),'nextday'));
}

sub zb_ups_calc_charge_next_day_air_saver {
  return(zb_ups_calc_charge($BASKET::s_zip,BASKET::total_weight(),'nextdaysaver'));
}

sub zb_ups_calc_charge_second_day_air {
  return(zb_ups_calc_charge($BASKET::s_zip,BASKET::total_weight(),'secondday'));
}

sub zb_ups_calc_charge_second_day_air_am {
  return(zb_ups_calc_charge($BASKET::s_zip,BASKET::total_weight(),'seconddayam'));
}

sub zb_ups_calc_charge_three_day_select {
  return(zb_ups_calc_charge($BASKET::s_zip,BASKET::total_weight(),'threedayselect'));
}

sub zb_ups_calc_charge {
  my($zipcode, $weight, $type) = @_;
  my($dest_zone, $sub_total);

  # does the zipcode contain any non-digit characters?
  #could be a canadian zip code, or somesuch.  Still valid, but we
  # can't process it.
  if ($zipcode =~ /\D/) {
    return (-1);
  }
  SEC::untaint_ref(\$zipcode);

  $dest_zone = zb_ups_calc_zone($zipcode,$type);
  if ($dest_zone == 0) {
    return (-1);
  }

  $sub_total = zb_ups_calc_rate($weight,$dest_zone);
  if (zb_ups_needs_surcharge($zipcode)) {
    $sub_total += 1.00;
  }

  return($sub_total);
}

#
# -----------------------------------------------------------------------------
#
# figure out what zone a specified zip code lies in
#

sub zb_ups_calc_zone {
  my($zipcode, $type) = @_;

  # calculate 3 digit version of zipcode

  $zipcode =~ s/^(\d{3})(.*)$/$1/;

  # try normal zone

  my($sqlstr) = "select $type 
                 from zb_ups_zone_chart
                 where start_range <= '$zipcode' and end_range >= '$zipcode'";

  my($zone) = DBLIB::db_fetchrow_array($sqlstr);  

  # no luck?  Well, if it's 'nextday' or 'secondday' we can try the weird zones

  if ((!defined($zone) || $zone==0) && 
      ($type eq 'nextday' || $type eq 'secondday')) {

    $sqlstr = "select $type
               from zb_ups_weird_zone
               where zipcode = '$zone'";

    $zone = DBLIB::db_fetchrow_array($sqlstr);  
  }

  if (!defined($zone)) { $zone = 0; }

  return($zone);
}

#
# -----------------------------------------------------------------------------
#
# given the weight of the shipment and the zone, calculate the charge
#

sub zb_ups_calc_rate {
  my($weight, $dest_zone) = @_;

  my($sqlstr) = "select r$dest_zone
                 from zb_ups_shipping_rates
                 where weight = '$weight'";

  my($charge) = DBLIB::db_fetchrow_array($sqlstr);

  # didn't find it in the table?  Hmmmm.  It might be too heavy, then.
  # Or something.

  if (!defined($charge)) { $charge = -1; }

  return($charge);
}

#
# -----------------------------------------------------------------------------
#
# if the zipcode exists in this table, a surcharge is needed
#

sub zb_ups_needs_surcharge {
  my($zipcode) = shift;

  my($sqlstr) = "select zipcode
                 from zb_ups_shipping_surcharge
                 where zipcode = '$zipcode'";

  my($hohum) = DBLIB::db_fetchrow_array($sqlstr);

  if (defined($hohum)) {
    return(1);
  } else {
    return(0);
  }
}

#
# -----------------------------------------------------------------------------
#

1;
