# SNIPPET.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

package SNIPPET;

use strict;

require DBLIB;
require THING;
require THINGTYPE;
require SNIPTYPE;

#  Methods available in this package:

#   &snip_create 
#   &snip_do_create 
#   &snip_delete 
#   &snip_do_delete 
#   &snip_delete_all 
#   &snip_delete_all_type 
#   &snip_get 
#   &snip_get_byname
#   &snip_do_get 
#   &snip_update 
#   &snip_do_update 

# Some variables for convenience...
my @snip_sq = @THING::snip_sq;
my @snip_table = @THING::snip_table;

#
# =============================================================================
#
# SNIPPET API
#
# =============================================================================
#


# -----------------------------------------------------------------------------

# Call the method for the snippet type
sub snip_create {
    my($thing_type, $thing_id, $type_id, $text) = @_;

    my($method) = SNIPTYPE::sniptype_get_method($thing_type, $type_id);

    MILTON::fatal_error("Not a code ref") unless(ref($method) eq "CODE");

    my($status, $msg) = 
	&{$method}("c", $thing_type, $thing_id, 
				     $type_id, $text);

    return 1 if $status == $USER_MILTON::ok;

    MILTON::fatal_error($msg);
}

#
# -----------------------------------------------------------------------------
# creates a new snippet
#   takes: thing type, thing id, snippet type id, snippet text
# returns: true
#
sub snip_do_create {
  my($thing_type, $thing_id, $sniptypeid, $text) = @_;

  # get the next snippet id

  my($nextsnipid)= DBLIB::seq_next_get("$snip_sq[$thing_type]");

  # insert the snippet

  DBLIB::db_do("insert into $snip_table[$thing_type] 
                (thing_id,snip_id,sniptype_id,text)
                values
                (?, ?, ?, ?)", 0, 
	       $thing_id, 1, $nextsnipid, 1, $sniptypeid, 1, $text, 0);

  return(1);
}


# -----------------------------------------------------------------------------
# Call the delete method for the snippet type to delete the snippet associated
# with the supplied thing and sniptype
sub snip_delete {
    my($thing_type, $thing_id, $type_id) = @_;

    my($method) = SNIPTYPE::sniptype_get_method($thing_type, $type_id);

    MILTON::fatal_error("Not a code ref") unless(ref($method) eq "CODE");

    my($status, $msg) = 
	&{$method}("d", $thing_type, $thing_id, $type_id);
		   
    return if $status == $USER_MILTON::ok;
    MILTON::fatal_error($msg);
}

#
# -----------------------------------------------------------------------------
# deletes a local snippet from the snippet table
#   takes: thing type, thing id, type
# returns: nothing
#

sub snip_do_delete {
  my($thing_type, $thing_id, $sniptypeid)=@_;

  # Delete the snippet

  DBLIB::db_do("delete from $snip_table[$thing_type] 
      where thing_id = '$thing_id' and
      sniptype_id = '$sniptypeid'");


  return;
}

#
# -----------------------------------------------------------------------------
# deletes all snippets associated with a thing
#   takes: thing type, thing id
# returns: nothing
#

sub snip_delete_all {
  my($thing_type, $thing_id)=@_;

  my($types) = THINGTYPE::thingtype_get_thing_sniptypes($thing_type,
							       $thing_id);
  my($sniptype_id);
  foreach $sniptype_id (@$types) {
      snip_delete($thing_type, $thing_id, $sniptype_id);
  }
}


#
# -----------------------------------------------------------------------------
# deletes all snippets associated with a sniptype
#   takes: thing type, snippet type
# returns: nothing
#

sub snip_delete_all_type {

  my($thing_type, $sniptype_id)=@_;

  my($things) = THINGTYPE::thingtype_get_sniptype_things($thing_type,
							 $sniptype_id);

  my($thing);
  foreach $thing (@$things) {
      snip_delete($thing_type, $thing, $sniptype_id);
  }

  return;
}

# 
# -----------------------------------------------------------------------------
# Wrapper around snip_get, allows snippets to be gotten by sniptype name
sub snip_get_byname {
    my($kind, $id, $snipname) = @_;

    my($sniptype_id) = SNIPTYPE::sniptype_get_id($kind, $snipname);

    return undef unless($sniptype_id >= 1000);

    return snip_get($kind, $id, $sniptype_id);
}

# 
# -----------------------------------------------------------------------------
# gets a snippet, given a thingtype, thing id and type id
#

sub snip_get {
    my($thing_type, $thing_id, $sniptype_id) = @_;

    my($method) = SNIPTYPE::sniptype_get_method($thing_type, $sniptype_id);

    MILTON::fatal_error("Not a code ref") unless(ref($method) eq "CODE");

    my($status, $msg) = 

	&{$method}("r", $thing_type, $thing_id, 
				     $sniptype_id);

    return($msg) if $status == $USER_MILTON::ok;
    MILTON::fatal_error($msg);
}

# (called from USER_MILTON for local snippets)
# -----------------------------------------------------------------------------
# gets a snippet from the local table
#   takes: thing type, thing id, snippet type id
#
sub old_snip_do_get {
  my($thing_type, $thing_id, $sniptype_id)=@_;

  my($text) = DBLIB::db_fetchrow_array("
      select text
      from $snip_table[$thing_type] 
      where sniptype_id = '$sniptype_id'
      and thing_id = '$thing_id'");

  return $text;
}

sub snip_do_get {
  my($thing_type, $thing_id, $sniptype_id)=@_;

  my($text) = DBLIB::db_fetchrow_array("
      select text
      from $snip_table[$thing_type] 
      where sniptype_id = ?
      and thing_id = ?", 0, $sniptype_id, 1, $thing_id, 1);

  return $text;
}


# 
# -----------------------------------------------------------------------------
# updates a snippet, given a thing id, sniptype id, and the text of the snippet
#
sub snip_update {
    my($thing_type, $thing_id, $type_id, $text) = @_;

    my($method) = SNIPTYPE::sniptype_get_method($thing_type, $type_id);

    MILTON::fatal_error("Not a code ref") unless(ref($method) eq "CODE");

    my($status, $msg) = 
	&{$method}("u", $thing_type, $thing_id, 
				     $type_id, $text);

    return if $status == $USER_MILTON::ok;
    MILTON::fatal_error($msg);
}


#
# -----------------------------------------------------------------------------
# updates a snippet record (called from USER_MILTON for local snippets)
#   takes: thing type, thing id, type name, data
# returns: nothing
#   notes: automatically creates the snippet if none exists
#
sub old_snip_do_update {
  my($thing_type, $thing_id, $sniptype_id, $data)=@_;

  my($snip_id) = DBLIB::db_fetchrow_array("select snip_id
      from $snip_table[$thing_type]
      where sniptype_id = '$sniptype_id'
      and thing_id = '$thing_id'");

  if(!$snip_id) {
      SNIPPET::snip_create($thing_type, $thing_id, $sniptype_id, $data);
  } else {
      DBLIB::db_do("update $snip_table[$thing_type] 
                    set text = '$data'
                    where snip_id = '$snip_id'");
  }

  return;
}

sub snip_do_update {
  my($thing_type, $thing_id, $sniptype_id, $data)=@_;

  my($snip_id) = DBLIB::db_fetchrow_array("select snip_id
      from $snip_table[$thing_type]
      where sniptype_id = '$sniptype_id'
      and thing_id = '$thing_id'");

  if(!$snip_id) {
      SNIPPET::snip_create($thing_type, $thing_id, $sniptype_id, $data);
  } else {
      DBLIB::db_do("update $snip_table[$thing_type] 
                    set text = ?
                    where snip_id = ?", 0, $data, 0, $snip_id, 1);
  }

  return;
}



# -----------------------------------------------------------------------------

1;







