# SNIPTYPE.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

# XXX add code to check whether a snippet exists for a thing

package SNIPTYPE;

use strict;
use DBLIB;
require SNIPPET;
require THING;
require THINGTYPE;
require USER_MILTON;

#
# =============================================================================
#
# INTERFACE STUFF
#
# =============================================================================
#
#  Methods available in this package:
#
#    &sniptype_list_all 
#    &sniptype_create 
#    &sniptype_delete 
#    &sniptype_update 
#    &sniptype_get 
#    &sniptype_get_default 
#    &sniptype_get_method 
#    &sniptype_get_method_description 
#    &sniptype_get_widget 
#    &sniptype_get_all 
#    &sniptype_get_id 
#    &sniptype_get_label 
#    &sniptype_get_name 
#    &sniptype_verify_id
#
#  Use these to specify widget types:
#   $normal
#   $checkbox
#   $selectionbox
#   $radio
#   $textarea
#   $password
#   $file_upload <- not yet implemented
#   $hidden

use vars      qw( $normal $checkbox $selectionbox $radio $textarea 
		  $password $hidden );

use vars      qw( %valid_widget );

# Widget type numbers 
$normal = 0;
$checkbox = 1;
$selectionbox = 2;
$radio = 3;
$textarea = 4;
$password = 5;
# $file_upload = 6;
$hidden = 7;

# Make it easy to see if a widget type is supported
%valid_widget = (
		 $normal => 1,
		 $checkbox => 1,
		 $selectionbox => 1,
		 $radio => 1,
		 $textarea => 1,
		 $password => 1,
#		 $file_upload => 1,
                 $hidden => 1,
		 );

# For convenience...
my @sniptype = @THING::sniptype;
my @sniptype_sq  = @THING::sniptype_sq;
my @type_to_snip = @THING::type_to_snip;

# This avoids a warning on startup:
{ my($foo) = $#THING::sniptype + $#THING::sniptype_sq + $#THING::type_to_snip }

#
# =============================================================================
#
# SNIPTYPE API
#
# =============================================================================
#

#
# -----------------------------------------------------------------------------
# returns a listref to sniptype_id, name, label listrefs for all sniptypes
# for the supplied thing type
#
sub sniptype_list_all {
    my($thing_type) = @_;

    my($rows) = DBLIB::db_fetchall_arrayref("
        select sniptype_id, index_name, label
        from $sniptype[$thing_type]
        where sniptype_id <> 0");

    return $rows;
}


#
# -----------------------------------------------------------------------------
# returns the new snippet type's id
#
sub sniptype_create {
    my($thing_type, $type_id,
       $name, $label, 
       $defaultval, 
       $accessmethod,
       $widgettype, $widgetinfo) = @_;

    my($sniptype_id) = DBLIB::seq_next_get($sniptype_sq[$thing_type]);

    MILTON::fatal_error("Invalid widget type $widgettype.") unless
	$valid_widget{$widgettype};

    DBLIB::db_do("
        insert into $sniptype[$thing_type]
                   (sniptype_id, index_name, label, default_val, 
                    accessmethod, widgettype, widgetinfo)
        values     ($sniptype_id, '$name', '$label', '$defaultval',
                    '$accessmethod', $widgettype, '$widgetinfo')");

    THINGTYPE::thingtype_append($thing_type, $type_id, $sniptype_id);

    return $sniptype_id;
}


#
# -----------------------------------------------------------------------------
#
#
sub sniptype_delete {
    my($thing_type, $sniptype_id  ) = @_;

    # Delete all snippets of the type
    SNIPPET::snip_delete_all_type($thing_type, $sniptype_id);

    # Delete all entries in the table connecting types to sniptypes
    THINGTYPE::thingtype_delete_type_to_sniptype($thing_type, $sniptype_id);

    # Delete the snippet type
    DBLIB::db_do("
        delete from $sniptype[$thing_type]
        where sniptype_id = $sniptype_id");
}


#
# -----------------------------------------------------------------------------
# Updates the snippet type having sniptype_id
#
sub sniptype_update {
    my($thing_type, 
       $sniptype_id, $name, $label,
       $defaultval, $accessmethod, $widgettype, $widgetinfo) = @_;

  DBLIB::db_do("
        update $sniptype[$thing_type]
        set index_name = '$name'
        where sniptype_id = $sniptype_id");

  DBLIB::db_do("
        update $sniptype[$thing_type]
        set label = '$label'
        where sniptype_id = $sniptype_id");

  DBLIB::db_do("
        update $sniptype[$thing_type]
        set default_val = '$defaultval'
        where sniptype_id = $sniptype_id");

  DBLIB::db_do("
        update $sniptype[$thing_type]
        set accessmethod = '$accessmethod'
        where sniptype_id = $sniptype_id");

  DBLIB::db_do("
        update $sniptype[$thing_type]
        set widgettype = $widgettype
        where sniptype_id = $sniptype_id");

  DBLIB::db_do("
        update $sniptype[$thing_type]
        set widgetinfo = '$widgetinfo'
        where sniptype_id = $sniptype_id");

}


#
# -----------------------------------------------------------------------------
# Returns the name and label for the sniptype with sniptype_id
#
sub sniptype_get {
    my($thing_type, $sniptype_id) = @_;

    my($name, $label) = DBLIB::db_fetchrow_array("
        select index_name, label
        from $sniptype[$thing_type]
        where sniptype_id = $sniptype_id");

    return ($name, $label);
}

#
# -----------------------------------------------------------------------------
# Returns the default value for the sniptype with sniptype_id
#
sub sniptype_get_default {
    my($thing_type, $sniptype_id) = @_;

    my($default) = DBLIB::db_fetchrow_array("
        select default_val
        from $sniptype[$thing_type]
        where sniptype_id = $sniptype_id");

    return ($default);
}

#
# -----------------------------------------------------------------------------
# Returns the access method for the sniptype with sniptype_id
#
sub sniptype_get_method {
    my($thing_type, $sniptype_id) = @_;

    my($accessmethod) = DBLIB::db_fetchrow_array("
        select accessmethod
        from $sniptype[$thing_type]
        where sniptype_id = $sniptype_id");

    MILTON::fatal_error("Invalid user-defined method $accessmethod.") unless
        (ref($USER_MILTON::snippet_method{$accessmethod}) eq "CODE");
	
    return ($USER_MILTON::snippet_method{$accessmethod});
}

#
# -----------------------------------------------------------------------------
# Returns the access method's description for the sniptype with sniptype_id
#
sub sniptype_get_method_description {
    my($thing_type, $sniptype_id) = @_;

    my($methodtag) = DBLIB::db_fetchrow_array("
        select accessmethod
        from $sniptype[$thing_type]
        where sniptype_id = $sniptype_id");

    MILTON::fatal_error("Invalid user-defined method $methodtag.") unless
	($USER_MILTON::snippet_desc{$methodtag});

    return $USER_MILTON::snippet_desc{$methodtag};
}

#
# -----------------------------------------------------------------------------
# Returns the widget info for the sniptype with sniptype_id
#
sub sniptype_get_widget {
    my($thing_type, $sniptype_id) = @_;

    my($wtype, $winfo) = DBLIB::db_fetchrow_array("
        select widgettype, widgetinfo
        from $sniptype[$thing_type]
        where sniptype_id = $sniptype_id");

    return ($wtype, parse_winfo($wtype, $winfo));
}

sub parse_winfo {
    my($type, $info) = @_;

    # Assume (for now) that the info is extra text ready to be inserted into
    # the appropriate <INPUT ...> html tag
    if($type eq $normal or
       $type eq $checkbox or
       $type eq $textarea or
#       $type eq $file_upload or
       $type eq $password or
       $type eq $hidden) {

	return $info;

    # Otherwise, that it's a list of valid options
    } elsif($type eq $radio or
	    $type eq $selectionbox) {

	my(@options) = split(/\r?\n/, $info);
	return \@options;
    } else {
        MILTON::fatal_error("Unknown widget type $type");
    }
}

#
# -----------------------------------------------------------------------------
# Returns name, label, defaultval, accessmethod, widgettype, widgetinfo
# for the sniptype with sniptype_id
#
sub sniptype_get_all {
    my($thing_type, $sniptype_id) = @_;

    my(@info) = DBLIB::db_fetchrow_array("
        select index_name, label, default_val, accessmethod, widgettype,
               widgetinfo
        from $sniptype[$thing_type]
        where sniptype_id = $sniptype_id
        and sniptype_id <> 0");

    return undef unless $#info > 0;
    $info[5] = parse_winfo($info[4], $info[5]);
    return (@info);
}


#
# -----------------------------------------------------------------------------
# Given a sniptype name, returns its id, or undef if no such field exists.
#
sub old_sniptype_get_id {
    my($thing_type, $name) = @_;

    my($sniptype_id) = DBLIB::db_fetchrow_array("
        select sniptype_id 
        from $sniptype[$thing_type]
        where index_name = '$name'");

    if (!defined($sniptype_id) || $sniptype_id==0) { 
	return undef;
    }

    return $sniptype_id;
}

sub sniptype_get_id {
    my($thing_type, $name) = @_;

    my($sniptype_id) = DBLIB::db_fetchrow_array("
        select sniptype_id 
        from $sniptype[$thing_type]
        where index_name = ?", 0, $name, 0);

    if (!defined($sniptype_id) || $sniptype_id==0) { 
	return undef;
    }

    return $sniptype_id;
}



#
# -----------------------------------------------------------------------------
#
#
sub sniptype_get_label {
    my($thing_type, $sniptype_id) = @_;

    my($name, $label) = sniptype_get($thing_type, $sniptype_id);
    return $label;
}


#
# -----------------------------------------------------------------------------
#
#
sub sniptype_get_name {
    my($thing_type, $sniptype_id) = @_;

    my($name, $label) = sniptype_get($thing_type, $sniptype_id);
    return $name;
}

#
# -----------------------------------------------------------------------------
#

sub sniptype_verify_id {
  my($thing_type) = shift;

  return DBLIB::verify_id('sniptype_id', $sniptype[$thing_type], @_);
}

#
# -----------------------------------------------------------------------------
#

1;
