# STATE.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

#  Keep track of shopping basket state
#  -Jason E. Holt 16Jul1999

# BUGS
# I actually quite like this module; it's fairly terse and elegant, IMHO.
# The biggest cruft is using unpack() to freeze the data; makes it very
# hard for humans to read when browsing the database.
# 
# Last minute change:
# state_expire() is currently called in state_touch().  This is very 
# inefficient, and should be moved elsewhere (perhaps a cron job) for
# sites with any serious amount of shopping basket traffic.

package STATE;

use strict;
use FileHandle;
require MILTON;
require DBLIB;

#
# =============================================================================
#
# INTERFACE STUFF
#
# =============================================================================
#

# 
# AVAILABLE FUNCTIONS AND VARIABLES DEFINED IN THIS PACKAGE:
#
#


# file-private lexicals go here
use vars qw($random_dev $random_bytes $cookie_env $freeze_delim $expire);

$random_dev = '/dev/urandom';
$random_bytes = 10;
$cookie_env = 'HTTP_COOKIE';
$freeze_delim = ",";
$expire = 1 * 24 * 60 * 60; # Expire untouched state entries after 1 day


#
# =============================================================================
#
# STATE API
#
# =============================================================================
#

#
# -----------------------------------------------------------------------------
#

# Returns a truly random string, in the format ccc.._nnn....
# where nnn... is the hex representation of $random_bytes worth of random
# data, and ccc... is the decimal representation of its 16-bit checksum
sub state_gen_id {

    my($id);
    my($random_data);

    MILTON::fatal_error("$random_dev not available") unless(-r $random_dev);
    my($fh) = new FileHandle;
    $fh->open("< $random_dev") or 
      MILTON::fatal_error("$random_dev not available: $!");

    # Generate ids until a unique one is generated.
    # With $random_bytes==10, there are 2^80 possible
    # ids, so this virtually never loop more than once.
    # If you don't mind risking a duplicate id, taking out
    # the DBLIB call might significantly speed this function up.
    my($existing_id);
    do {

        MILTON::fatal_error("Error reading from $random_dev") unless
	  (sysread($fh, $random_data, $random_bytes) == $random_bytes);

	  ## random_data needs to be untainted at this point
	SEC::untaint_ref(\$random_data);

	$id = unpack('%16C*', $random_data) . '_' . unpack("H*", $random_data);

	$existing_id = DBLIB::db_fetchrow_array("
            select id from basket_state
            where id=?", 0, $id,0);

    } while($existing_id);

	
    $fh->close;

    return $id;

}


#
# -----------------------------------------------------------------------------
#
sub state_create_session {

    my($id) = state_gen_id();

    my($now) = time;
    
    DBLIB::db_do("
        insert into basket_state(id,accessed,data) 
        values(?,?,'hohumdiddlything')", 0, 
		 $id,0, $now,0);

    return $id;
}

#
# -----------------------------------------------------------------------------
#
sub state_touch {
    my($id) = @_;

    MILTON::fatal_error("state_touch called with invalid id") unless
      state_valid_id($id);

    my($now) = time;

    DBLIB::db_do("
        update basket_state
        set accessed = ?
        where id = ?", 0, $now,0, $id,0);

    state_expire();
}

#
# -----------------------------------------------------------------------------
# Returns the value of the cookie named, iff that value is a valid id.
sub state_get_state_cookie {
    my($cookie_name) = @_;

    my($cookie) = $ENV{$cookie_env};
    return( undef ) unless $cookie;

    $cookie =~ /\b$cookie_name\=(\S+)/;
    my($val) = $1;
    $val =~ s/;//g;

    return undef unless state_valid_id($val);

    return $val;
}

#
# -----------------------------------------------------------------------------
# Returns the value of the cookie named.
sub state_get_cookie {
    my($cookie_name) = @_;

    my($cookie) = $ENV{$cookie_env};
    return( undef ) unless $cookie;

    $cookie =~ /\b$cookie_name\=(\S+)/;
    my($val) = $1;
    $val =~ s/;//g;

    return $val;
}

#
# -----------------------------------------------------------------------------
#
# Passed an id in the form returned by state_gen_id, returns true iff
# it's in the proper form and the checksum is correct.
sub state_valid_id {
    my($id) = @_;

    my($sum, $val) = ($id =~ /^([0-9]+)_([0-9a-fA-F]{2,})$/);

    (length($sum) && length($val)) or return undef;

    return undef unless($sum == unpack('%16C*', pack('H*', $val)));

    return 1;
}

#
# -----------------------------------------------------------------------------
#
# Passed an id in the form returned by state_gen_id, returns true iff
# it's in the database
sub state_id_exists {
    my($id) = @_;

    return 0 unless state_valid_id($id);

    my($existing_id);
    $existing_id = DBLIB::db_fetchrow_array("
            select id from basket_state
            where id=?", 0, $id,0);

    return( $existing_id ? 1 : 0 );
}

    
#
# -----------------------------------------------------------------------------
# Delete state entries which haven't been accessed recently enough
sub state_expire {

    my($too_old) = time - $expire;

#    DBLIB::db_do("
#        delete from basket_state
#        where accessed < ?", 0, $too_old,0);

    DBLIB::db_do("
        delete from basket_state
        where accessed < '$too_old'");
}


#
# -----------------------------------------------------------------------------
# Return a reference to the hash of state for the supplied id
sub state_get {
    my($id) = @_;

    MILTON::fatal_error("state_get passed invalid id: $id") unless
      state_valid_id($id);

    my($frozen_hash) = DBLIB::db_fetchrow_array("
        select data
        from basket_state
        where id=?", 0, $id,0) or return undef;

    state_touch($id);

    my($data_ref) = state_unfreeze($frozen_hash);
    return $data_ref;
}
 
#
# -----------------------------------------------------------------------------
# Store a hash of data for $id 
sub state_put {
    my($id, $hashref) = @_;

    MILTON::fatal_error("state_put passed invalid id") unless
      state_valid_id($id);
    
    my($frozen_hash) = state_freeze($hashref);

    DBLIB::db_do("
        update basket_state
        set data = ?
        where id = ?", 0, $frozen_hash,0, $id,0);

    state_touch($id);

}

#
# -----------------------------------------------------------------------------
#
sub state_unfreeze {
    my($frozen) = @_;

    my(%unfrozen) = split($freeze_delim, $frozen);

    for(keys %unfrozen) {
	$unfrozen{$_} = pack("H*", $unfrozen{$_});
    }

    return \%unfrozen;
}


#
# -----------------------------------------------------------------------------
#
sub state_freeze {
    my($hashref) = @_;

    my(%freeze) = %$hashref;

    for(keys %freeze) {
	$freeze{$_} = unpack('H*', $freeze{$_});
    }

    return join($freeze_delim, %freeze);
}


1;






