#
# THINGTYPE.pm
# copyright (c) 1999 akopia, inc.
#
########################################################################
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public
#    License as published by the Free Software Foundation.
#    
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    General Public License for more details.
########################################################################

# XXX thingtype_create has some code to verify the existance of things
# to be pointed to by a connecting table.  Do we want to do this manually
# in all appropriate functions, or let the foriegn key constraints take
# care of it?

# XXX get_thing_sniptypes doesn't preserve snip_order

# XXX default/acessmethod/widget stuff are all per-sniptype; when multiple 
# XXX thingtypes share a sniptype, they also share this info.  Might be better
# XXX to let that work per-instance, so that each thingtype could have a different
# XXX default value, for instance.  OTOH, that would make matters confusing when,
# XXX say, a thing had 2 thingtypes which both shared a sniptype.  What default
# XXX to use?

package THINGTYPE;

use strict;

require DBLIB;
require MILTON;
require THING;
require SNIPTYPE;

#
# =============================================================================
#
# INTERFACE STUFF
#
# =============================================================================
#
#  Methods available in this package:
#   &thingtype_listall 
#   &thingtype_create 
#   &thingtype_delete_completely   
#   &thingtype_delete_thing_to_type 
#   &thingtype_delete_type_to_sniptype 
#   &thingtype_append 
#   &thingtype_subtract 
#   &thingtype_snip_reorder 
#   &thingtype_subtract_prep 
#   &thingtype_get_id 
#   &thingtype_get_name
#   &thingtype_set_name
#   &thingtype_get_label 
#   &thingtype_set_label 
#   &thingtype_get_sniptypes 
#   &thingtype_get_thing_sniptypes 
#   &thingtype_get_sniptype_things 
#   &thingtype_get_things 
#   &thingtype_get_types 

# Some variables for convenience...
my @type_table = @THING::type_table;
my @type_sq = @THING::type_sq;
my @type_to_snip = @THING::type_to_snip;
my @thing_to_type = @THING::thing_to_type;

#
# =============================================================================
#
# THINGTYPE API
#
# =============================================================================
#

#
# -----------------------------------------------------------------------------
# returns a listref for [type id, name, label] listrefs for the given thing 
# thing type
sub thingtype_listall {
    my($thing_type) = @_;
    
    my($types) = DBLIB::db_fetchall_arrayref("
        select * from $type_table[$thing_type]
        where type_id <> 0
        ");

    return $types;
}

#
# -----------------------------------------------------------------------------
# creates a new type
#   takes: thing type, type name, type description, optional arrayref
#          of snippet type ids to associate with this type.  snippet order
#          numbers will be assigned in the order they show up in the list.
# returns: type_id of the new type
#
sub thingtype_create {
  my($thing_type, $name, $label, $sniptyperef) = @_;

  my($type_id)= DBLIB::seq_next_get($type_sq[$thing_type]);

  # insert the type
  DBLIB::db_do("insert into $type_table[$thing_type] (type_id,name,label)
           values($type_id, '$name', '$label')");

  if(ref($sniptyperef) eq "ARRAY") {

      # Make a hash so we can verify the existance of all the
      # sniptypes we try to insert.
      my($all_sniptypes) = SNIPTYPE::sniptype_list_all($thing_type);
      my(%snip_exists) = map { $_->[0] => 1 } @$all_sniptypes;

      my($sniptype_id);
      my($order_num) = 1;
      foreach $sniptype_id (@$sniptyperef) {

     	  MILTON::fatal_error("Unknown sniptype $sniptype_id.") unless
	      $snip_exists{$sniptype_id};

	  DBLIB::db_do("insert into $type_to_snip[$thing_type] 
                                (type_id, sniptype_id, snip_order)
                    values($type_id, $sniptype_id, $order_num)");
	  $order_num++;
      }
  }

  return($type_id);
}


#
# -----------------------------------------------------------------------------
# deletes an object type, all snippet types associated only with that object
# type, all snippets associated with those snippet types, and all things
# associated only with that object type.  
#
#   takes: thing type, type id
# returns: nothing
#
sub thingtype_delete_completely {  
  my($thing_type, $type_id)=@_;

  # Delete all the things which have only our type.
  # As a side effect, all entries in the thing<->type table
  # will be removed, as well as all the snippets for each thing.
  my($things) = DBLIB::db_fetchcol_arrayref("
      select thing_id
      from $thing_to_type[$thing_type]
      where type_id = $type_id");

  my($thing);
  foreach $thing (@$things) {
      
      # skip if it has another type as well
      my($types) = DBLIB::db_fetchcol_arrayref("
          select type_id
          from $thing_to_type[$thing_type]
          where thing_id = $thing");
      next if($#{$types} != 0);  # Next if there was other than 1 type listed

      if($thing_type == $THING::item) {
	ITEM::item_delete($thing);
      } elsif($thing_type == $THING::page) {
	PAGE::page_delete($thing);
      } elsif($thing_type == $THING::group) {
	GROUP::group_delete($thing);
      } else {

	MILTON::fatal_error("Unknown thing type $thing_type.");

      }
  }
  
  # Delete all the sniptypes which are only associated with our type
  # As a side effect, all entries in the type<->sniptype table
  # will be removed as well.
  
  my($sniptypes) = DBLIB::db_fetchcol_arrayref("
                 select sniptype_id
                 from $type_to_snip[$thing_type]
                 where type_id = $type_id");

  my($sniptype_id);
  foreach $sniptype_id (@$sniptypes) {

      # skip if it has another type as well
      my($types) = DBLIB::db_fetchcol_arrayref("
                 select sniptype_id
                 from $type_to_snip[$thing_type]
                 where sniptype_id = $sniptype_id");
      next if($#{$types} != 0);  # Next if there was other than 1 type listed

      SNIPTYPE::sniptype_delete($thing_type, $sniptype_id);

  }
  
  # Delete the actual type
  DBLIB::db_do("
      delete from $type_table[$thing_type]
      where type_id = $type_id");
}

#
# -----------------------------------------------------------------------------
# Given a thing type and id, delete all entries in the table connecting
# things and their types (presumably in preparation for deleting that
# thing).
sub thingtype_delete_thing_to_type {
    my($thing_type, $thing_id) = @_;

    # Now delete from the thing<->type table
    DBLIB::db_do("
      delete from $thing_to_type[$thing_type]
      where thing_id = $thing_id");
}

#
# -----------------------------------------------------------------------------
# Given thing type and sniptype id, delete all entries in table connecting
# sniptypes and associated thing types (presumably in preparation for 
# deleting that snippet type.)
sub thingtype_delete_type_to_sniptype {
    my($thing_type, $sniptype_id) = @_;


    # Reorder the snip_order fields for each entry
    my($entries) = DBLIB::db_fetchall_arrayref("
      select type_id, snip_order 
      from $type_to_snip[$thing_type] 
      where sniptype_id = $sniptype_id");

    my($entry);
    foreach $entry (@$entries) {
      DBLIB::db_do("
          update $type_to_snip[$thing_type] 
          set snip_order = snip_order-1
          where type_id = $entry->[0]
          and snip_order > $entry->[1]");
    }


    # delete all entries in the type<->snippet table
    DBLIB::db_do("
      delete from $type_to_snip[$thing_type]
      where sniptype_id = $sniptype_id");

}

# 
# -----------------------------------------------------------------------------
# Adds a thing to the thing->type connecting table.
sub thingtype_add_thing {
    my($thing_type, $type_id, $thing_id) = @_;

    DBLIB::db_do("
        insert into $thing_to_type[$thing_type](thing_id, type_id)
        values($thing_id, $type_id)");
}

#
# -----------------------------------------------------------------------------
# adds a snippet type to a thing type (eg, add "ShaftType" snippet type to the
# "Golf Club" item type) 
# takes: thing type, type name, snippet type name.  
#        order number will be the current max + 1
# returns: order number for this entry
sub thingtype_append {
    my($thing_type, $type_id, $sniptype_id) = @_;

    DBLIB::db_transact_begin();

    my($order) = DBLIB::db_fetchrow_array("
        select max(snip_order) 
        from $type_to_snip[$thing_type]
        where type_id = $type_id");

#    $order || ($order = 1);
    if (!defined($order) || ($order==0)) {
      $order = 1;
    } else {
      $order++;
    }

    DBLIB::db_do("
        insert into $type_to_snip[$thing_type](type_id, sniptype_id, snip_order)
        values('$type_id', '$sniptype_id', '$order')");

    DBLIB::db_transact_end();
    
    return $order;
}


#
# -----------------------------------------------------------------------------
# Remove a snippet type from a thing type.  If the thingtype is the only type
# using the sniptype, it is deleted as normal via the SNIPTYPE library.
# Otherwise, remove any snippets which exist only in that thing type.  
# IE, some things may have more than one
# thingtype, and more than one may share a certain sniptype.  Any snippets
# thusly shared between more than one type are not be deleted, since the
# other thingtype's things still needs them. 
sub thingtype_subtract {
    my($thing_type, $type_id, $sniptype) = @_;

    # get the sniptype order number so we can resequence later
    my($snip_order) = DBLIB::db_fetchrow_array("
        select snip_order
        from $type_to_snip[$thing_type] 
        where type_id = $type_id
        and sniptype_id = $sniptype");

    # if any other thingtypes are using the sniptype,
    # don't delete it.
    my($rows) = DBLIB::db_fetchcol_arrayref("
        select type_id
        from $type_to_snip[$thing_type]
        where sniptype_id = $sniptype");

    if($#{$rows} > 0) { # More than one thingtype uses the sniptype.

	my($things) = thingtype_subtract_prep($thing_type, $type_id, 
					      $sniptype);

	# delete all appropriate snippets first
	my($thing);
	foreach $thing (@$things) {
	  SNIPPET::snip_delete($thing_type, $thing, $sniptype);
	}
    
	# delete the sniptype association with the thingtype
        DBLIB::db_do("
          delete from $type_to_snip[$thing_type]
          where type_id = $type_id
          and sniptype_id = $sniptype");

	# resequence the remaining sniptypes
        DBLIB::db_do("
          update $type_to_snip[$thing_type] 
          set snip_order = snip_order-1
          where type_id = $type_id
          and snip_order > $snip_order");

    } elsif($#{$rows} == 0) { # Just one thingtype uses us.  Delete normally.
	# All snippets of the type and the entry in the type<->sniptype
	# table will be removed (and the type<->sniptype table resequenced)
        SNIPTYPE::sniptype_delete($thing_type, $sniptype);
    } else {
        MILTON::fatal_error("Reality check bounced in thingtype_subtract.");
    }

}

#
# -----------------------------------------------------------------------------
# Returns a listref of things for which snippets of the given sniptype would
# be deleted if thingtype_subtract were called.
sub thingtype_subtract_prep {
    my($thing_type, $type_id, $sniptype) = @_;

    # find all other thingtypes which use the sniptype
    my($thingtypes) = DBLIB::db_fetchcol_arrayref("
        select type_id
        from $type_to_snip[$thing_type]
        where sniptype_id = $sniptype
        and type_id <> $type_id");

    my(%intersecting) = map { $_ => 1 } @$thingtypes;

    # find all things which have the thingtype
    my($things) = thingtype_get_things($thing_type, $type_id);
 
    # delete snippets for any things which don't also have another 
    # type which uses the sniptype
    my($thing);
    my(@things_to_delete);
    foreach $thing (@$things) {

	my($types) = thingtype_get_types($thing_type, $thing);

	my($delete) = 1;

	my($type);
	foreach $type (@$types) {
	    if($intersecting{$type}) {
		$delete = 0;
		last;
	    }
	}

	push(@things_to_delete, $thing) if $delete;
    }

    return \@things_to_delete;
}

#
# -----------------------------------------------------------------------------
# Reorder the sniptypes for a type in the order supplied. 
# All the sniptypes for the type must be supplied, or we flail.
sub thingtype_snip_reorder {
    my($thing_type, $type_id, $sniptypes) = @_;

    my($rows) = DBLIB::db_fetchcol_arrayref("
        select type_id 
        from $type_to_snip[$thing_type]
        where type_id = $type_id");

    MILTON::fatal_error("Wrong number of args.") unless($#{$rows} == $#{$sniptypes});

    my($sniptype);
    my($i) = 1;
    foreach $sniptype (@$sniptypes) {
        DBLIB::db_do("
            update $type_to_snip[$thing_type]
            set snip_order = $i
            where type_id = $type_id
            and sniptype_id = $sniptype");

        $i++;
    }
}



#
# -----------------------------------------------------------------------------
# Given a thing type and a type name, returns the type id
sub thingtype_get_id {
    my($thing_type, $type) = @_;

    my($type_id) = DBLIB::db_fetchrow_array("
        select type_id
        from $type_table[$thing_type]
        where name = '$type'");

    return $type_id;
}

#
# -----------------------------------------------------------------------------
# Given a thing type and type id, returns a ref to an array of snippet type
# ids which are associated with that type, sorted by snip_order.
sub thingtype_get_sniptypes {
    my($thing_type, $type_id) = @_;

    my($sniptypes) = DBLIB::db_fetchcol_arrayref("
        select sniptype_id
        from $type_to_snip[$thing_type]
        where type_id = '$type_id'
        order by snip_order");

    return $sniptypes;
}

#
# -----------------------------------------------------------------------------
# Given a thing type and thing id, returns a listref of unique snippet
# types which that thing has
sub thingtype_get_thing_sniptypes {
    my($thing_type, $thing_id) = @_;

    my($types) = thingtype_get_types($thing_type, $thing_id);

    return([]) unless($#$types >=0);

    # Just return the sniptypes if the thing only has zero or one thingtype
    if($#$types == 0) {
	my($sniptypes) = thingtype_get_sniptypes($thing_type, $types->[0]);
	return $sniptypes;
    }

    # If the thing has multiple thingtypes (!), union them all together and return
    my($type, %unique_sniptypes);
    foreach $type (@$types) {
	my($sniptypes) = thingtype_get_sniptypes($thing_type, $type);

	my($sniptype);
	foreach $sniptype (@$sniptypes) {
	    $unique_sniptypes{$sniptype} = 1;
	}
    }

    # XXX if a thing has multiple types, the sniptypes will *not* be sorted
    my(@type_keys) = keys(%unique_sniptypes);

    return \@type_keys;
}

#
# -----------------------------------------------------------------------------
# Given a thing type and sniptype id, returns a list of things which should have
# snippets of that type
sub thingtype_get_sniptype_things {
    my($thing_type, $sniptype_id) = @_;

    # All types which use the sniptype
    my($types) = DBLIB::db_fetchcol_arrayref("
        select type_id
        from $type_to_snip[$thing_type]
        where sniptype_id = $sniptype_id");

    my(%thing_hash);
    my($type);
    foreach $type (@$types) {

	# All things which have each type
	my($things) = DBLIB::db_fetchcol_arrayref("
            select thing_id 
            from $thing_to_type[$thing_type]
            where type_id = '$type'");

	my($thing);
	foreach $thing (@$things) {
	    $thing_hash{$thing}++;
	}
    }

    my(@thinglist) = keys %thing_hash;
    return \@thinglist;
}

#
# -----------------------------------------------------------------------------
# Given a thing type and a type id, returns the label for that type
sub thingtype_get_label {
    my($thing_type, $type_id) = @_;

    my($label) =  DBLIB::db_fetchrow_array("
        select label from $type_table[$thing_type]
        where type_id = '$type_id'");

    return $label;
}

#
# -----------------------------------------------------------------------------
# 
sub thingtype_set_label {
  my($thing_type, $type_id, $label) = @_;

  DBLIB::db_do("update $type_table[$thing_type] set label='$label'
                where type_id = '$type_id'");
}

#
# -----------------------------------------------------------------------------
# Given a thing type and a type id, returns the name for that type
sub thingtype_get_name {
    my($thing_type, $type_id) = @_;

    my($name) =  DBLIB::db_fetchrow_array("
        select name from $type_table[$thing_type]
        where type_id = '$type_id'");

    return $name;
}

#
# -----------------------------------------------------------------------------
# 
sub thingtype_set_name {
  my($thing_type, $type_id, $name) = @_;

  DBLIB::db_do("update $type_table[$thing_type] set name='$name'
                where type_id = '$type_id'");
}

#
# -----------------------------------------------------------------------------
# Given a thing type and a type id, returns a ref to a list of things
# which have that type
sub thingtype_get_things {
    my($thing_type, $type_id) = @_;

    my($things) = DBLIB::db_fetchcol_arrayref("
        select thing_id 
        from $thing_to_type[$thing_type]
        where type_id = $type_id");

    return $things;
}

#
# -----------------------------------------------------------------------------
# Given a thing type and a thing id, returns a listref of type ids 
# which that thing has.
sub thingtype_get_types {
    my($thing_type, $thing_id) = @_;

    my($types) = DBLIB::db_fetchcol_arrayref("
        select type_id 
        from $thing_to_type[$thing_type]
        where thing_id = $thing_id");

    return $types;
}

#
# -----------------------------------------------------------------------------
#

sub thingtype_verify_id {
  my($thing_type) = shift;

  return DBLIB::verify_id('type_id', $type_table[$thing_type], @_);
}

#
# -----------------------------------------------------------------------------
#

1;

