# inst_db.pm
#   copyright (c) 1999 akopia, inc.
#
######################################################################
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of version 2 of the GNU General Public
#   License as published by the Free Software Foundation.
#    
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#   General Public License for more details.
######################################################################

package INST_DB;

#
# DATABASE-DEPENDENT FUNCTIONS
#
#  oracle_create_tables
#  oracle_create_user
#  oracle_drop_user
#
#  postgres_create_tables
#  postgres_create_user
#  postgres_drop_user
#

use vars qw(@sql_file_list);

@sql_file_list = (
  "make_stock.sql",
  "make_item_matrix.sql",
  "make_item_mix.sql",

  "make_item.sql",
  "make_page.sql",
  "make_group.sql",

  "make_snippets.sql",
  "make_formatting.sql",

  "make_zb_shipping.sql",

  "make_customer.sql",
  "make_shipping.sql",
  "make_payment.sql",
  "make_tax.sql",
  "make_affiliate.sql",
  "make_order_stat.sql",

  "make_order.sql",
  "make_knar.sql",
  "make_access.sql",

  "make_basket_state.sql",
  "make_test_data.sql"
);

#
# -----------------------------------------------------------------------------
#

sub load_file {
  my($fn) = shift;
  my($result, $length);

  if (!open(TMP,"$fn")) { 
    print STDOUT "      Error loading file $fn: $!\n";
    return(undef);
  }

  seek(TMP,0,2);
  $length=tell(TMP);
  seek(TMP,0,0);
  read TMP,$result,$length;
  close(TMP);
  return($result);  
}

#
# -----------------------------------------------------------------------------
#

sub execute_sql_data {
  my($dbh, $data) = @_;
  my(@commands, $command);

  # redirect stdout and stderr.  postgres DBD prints weird things sometimes.

  open(SAVEOUT, ">&STDOUT");
  open(SAVEERR, ">&STDERR");

  open(STDOUT, ">/dev/null") || print "Can't redirect stdout to /dev/null!";
  open(STDERR, ">/dev/null") || print "Can't redirect stderr to /dev/null!";

  # strip out comments
  $data =~ s/--.*$//mg;

  # strip off leading and trailing whitespace
  $data =~ s/^\s+//mg;
  $data =~ s/\s+$//mg;

  @commands = split(/;/, $data);

  foreach $command (@commands) {
#    print STDOUT "        - Executing [$command]\n";

    if ($command =~ /commit/i) {
      $dbh->commit;
    } else {
      $sth = $dbh->prepare($command) || db_error_2($dbh->errstr);
      $sth->execute() || db_error_2($dbh->errstr);
      $sth->finish();
    }
  }

  $dbh->commit;

  # de-redirect stdout and stderr

  close(STDOUT);
  close(STDERR);

  open(STDOUT, ">&SAVEOUT");
  open(STDERR, ">&SAVEERR");

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub db_error {
  my($msg) = shift;
  print STDOUT "  Error in database access: $msg\n";
  return 0;
}

sub db_error_2 {
  my($msg) = shift;
  print SAVEOUT "  Error in database access: $msg\n";
  return 0;
}

#
# =============================================================================
#
#   ORACLE
#
# =============================================================================
# 

sub oracle_create_tables {
  my($gconf, $store) = @_;
  my($sqlstr, $dbh, $fn, $username, $password, $cstr);

  main::do_log("    - Creating oracle tables: ");

  $ENV{'ORACLE_SID'} = $$store{'oracle_sid'};
  $ENV{'ORACLE_HOME'} = $$store{'oracle_home'};

  $username = $$store{'db_username'};
  $password = $$store{'db_password'};
  $cstr = $$store{'dbi_connect_str'};

  $dbh = DBI->connect($cstr, $username, $password) 
    || return db_error($DBI::errstr);

  foreach $fn (@sql_file_list) {
    main::do_log('.');
    if (!oracle_process_file($gconf, $store, $fn, $dbh)) {
      main::do_log("\n");
      return 0;
    }
  }

  $dbh->disconnect;

  main::do_log("\n");
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub oracle_process_file {
  my($gconf, $store, $fn, $dbh) = @_;
  my($data, @commands, $command, $sth, $fn2);

  $dbh->{AutoCommit} = 0;

  $fn2 = $$gconf{'basedir'} . '/make/oracle/' . $fn;

#  print STDOUT "      - Processing $fn2\n";

  $data = load_file($fn2);

  if (!defined($data)) {
    print STDOUT "      - Error: file $fn2 contained no statements!\n";
    return 0;
  }

  return execute_sql_data($dbh, $data);
}

#
# -----------------------------------------------------------------------------
#

sub oracle_create_user {
  my($gconf, $store) = @_;
  my($sqlstr, $dbh, $sth, $username, $password, $cstr);

  main::do_log("    - Creating oracle user...\n");

  $ENV{'ORACLE_SID'} = $$store{'oracle_sid'};
  $ENV{'ORACLE_HOME'} = $$store{'oracle_home'};

  $username = $$store{'oracle_dba_username'};
  $password = $$store{'oracle_dba_password'};
  $cstr = $$store{'dbi_connect_str'};

  $dbh = DBI->connect($cstr, $username, $password) 
    || return db_error($DBI::errstr);

  $sqlstr = "
    CREATE USER $$store{'db_username'} IDENTIFIED BY $$store{'db_password'}
    DEFAULT TABLESPACE $$store{'oracle_default_tablespace'}
    TEMPORARY TABLESPACE $$store{'oracle_temp_tablespace'}";
  $sth = $dbh->prepare($sqlstr) || return db_error($dbh->errstr);
  $sth->execute() || return db_error($dbh->errstr);
  $sth->finish();

  $sqlstr = "GRANT CONNECT TO $$store{'db_username'}";
  $sth = $dbh->prepare($sqlstr) || return db_error($dbh->errstr);
  $sth->execute() || return db_error($dbh->errstr);
  $sth->finish();

  $sqlstr = "GRANT RESOURCE TO $$store{'db_username'}";
  $sth = $dbh->prepare($sqlstr) || return db_error($dbh->errstr);
  $sth->execute() || return db_error($dbh->errstr);
  $sth->finish();

  $dbh->disconnect;

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub oracle_drop_user {
  my($gconf, $store) = @_;
  my($sqlstr, $dbh, $sth, $username, $password, $cstr);

  main::do_log("    - Dropping oracle user...\n");

  $ENV{'ORACLE_SID'} = $$store{'oracle_sid'};
  $ENV{'ORACLE_HOME'} = $$store{'oracle_home'};

  $username = $$store{'oracle_dba_username'};
  $password = $$store{'oracle_dba_password'};
  $cstr = $$store{'dbi_connect_str'};

  $dbh = DBI->connect($cstr, $username, $password) 
    || return db_error($DBI::errstr);

  $sqlstr = "DROP USER $$store{'db_username'} CASCADE";

  $sth = $dbh->prepare($sqlstr); # || return db_error($dbh->errstr);
  $sth->execute(); # || return db_error($dbh->errstr);
  $sth->finish();

  $dbh->disconnect;

  return 1;
}

#
# =============================================================================
#
#   POSTGRESQL
#
# =============================================================================
#

sub postgres_create_tables {
  my($gconf, $store) = @_;
  my($sqlstr, $dbh, $fn, $username, $password, $cstr);

  main::do_log("    - Creating postgres tables: ");

  $username = $$store{'db_username'};
  $password = $$store{'db_password'};
  $cstr = $$store{'dbi_connect_str'};

  $dbh = DBI->connect($cstr, $username, $password) 
    || return db_error($DBI::errstr);

  foreach $fn (@sql_file_list) {
    main::do_log('.');
    if (!postgres_process_file($gconf, $store, $fn, $dbh)) {
      main::do_log("\n");
      return 0;
    }
  }

  $dbh->disconnect;

  main::do_log("\n");
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub postgres_process_file {
  my($gconf, $store, $fn, $dbh) = @_;
  my($data, @commands, $command, $sth, $fn2);

  $dbh->{AutoCommit} = 0;

  $fn2 = $$gconf{'basedir'} . '/make/postgresql/' . $fn;

#  print STDOUT "      - Processing $fn2\n";

  $data = load_file($fn2);

  if (!defined($data)) {
    print STDOUT "      - Error: file $fn2 contained no statements!\n";
    return 0;
  }

  return execute_sql_data($dbh, $data);
}

#
# -----------------------------------------------------------------------------
#

sub postgres_create_user {
  my($gconf, $store) = @_;
  my($sqlstr, $dbh, $sth, $username, $password, $cstr);

  main::do_log("    - Creating postgres user...\n");

  $username = $$store{'postgres_dba_username'};
  $password = $$store{'postgres_dba_password'};
  $cstr = $$store{'dbi_connect_str'};

  $dbh = DBI->connect($cstr, $username, $password) 
    || return db_error($DBI::errstr);

  $sqlstr = "CREATE USER $$store{'db_username'} WITH PASSWORD 
             $$store{'db_password'}";

  $sth = $dbh->prepare($sqlstr) || return db_error($dbh->errstr);
  $sth->execute() || return db_error($dbh->errstr);
  $sth->finish();

  $sqlstr = "CREATE DATABASE $$store{'db_username'}";

  $sth = $dbh->prepare($sqlstr) || return db_error($dbh->errstr);
  $sth->execute() || return db_error($dbh->errstr);
  $sth->finish();

  $dbh->disconnect;

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub postgres_drop_user {
  my($gconf, $store) = @_;
  my($sqlstr, $dbh, $sth, $username, $password, $cstr);

  main::do_log("    - Dropping postgres user...\n");

  $username = $$store{'postgres_dba_username'};
  $password = $$store{'postgres_dba_password'};
  $cstr = $$store{'dbi_connect_str'};

  $dbh = DBI->connect($cstr, $username, $password) 
    || return db_error($DBI::errstr);

  $sqlstr = "DROP USER $$store{'db_username'}";

  $sth = $dbh->prepare($sqlstr); # || return db_error($dbh->errstr);
  $sth->execute(); # || return db_error($dbh->errstr);
  $sth->finish();

  $sqlstr = "DROP DATABASE $$store{'db_username'}";

  $sth = $dbh->prepare($sqlstr); # || return db_error($dbh->errstr);
  $sth->execute(); # || return db_error($dbh->errstr);
  $sth->finish();

  $dbh->disconnect;

  return 1;
}
 
#
# -----------------------------------------------------------------------------
#

1;
