#!/usr/bin/perl

# install.pl
#   copyright (c) 1999 akopia, inc.
#
######################################################################
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of version 2 of the GNU General Public
#   License as published by the Free Software Foundation.
#    
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#   General Public License for more details.
######################################################################

#
# INSTALL OUTLINE
#
# Config software (how are you running the software? offline | mod_perl)
#
# Create a new store
#     System prompts for parameters
#         database
#         url
#         dir
#         store
#     Store is created
#         create directories
#             logs
#             templates
#             tmp
#         copy data
#             templates
#         create per-store config file
#         set up db
#             create database user (if we have a privileged account)
#             connect as user
#             run table make scripts
#         config embedded variables
#             epl
#             CONFIG.pm
#         change permissions
#             tmp directory
#             template directory
#             logs directory
#             html output directory
#         extra stuff
#             run the zb_ups_shipping data importer
#     ** show the user what he still has to do (apache config, permissions)    
#
# Edit an existing store
#     System prompts for parameters
#     Values are updated
#
# Delete an existing store
#     connect as user
#     delete tables
#     delete user
#     delete directories
#   

#
# -----------------------------------------------------------------------------
#

use strict;

eval {
  require "helptext.pm";
};
if ($@) {
  print "Can't find 'helptext.pm' -- please execute the installer from\n";
  print "the TallyMan tools directory.\n\n";
  exit 0;
}

require "inst_db.pm";   # database-dependant functions

package main;

my($VERBOSE);
my($choice, $state, @stores, %global_conf, %global_default, @existing_stores,
   $install_logfn, $install_logfn2, $tmp_logfn);

$VERBOSE = 0;
@existing_stores = ();



%global_default = %{load_hash('install.storedat')};

%global_conf = %{load_hash('install.globaldat')};

check_command_line_args(@ARGV);

check_for_required_packages();

print_welcome_screen();

find_existing_stores();



while ($choice ne 'q') {

  print_main_menu();

  $choice = get_choice();

  if ($choice eq 's') {
    global_config();

  } elsif ($choice eq 'c') {
    create_store();

  } elsif ($choice eq 'e') {
    edit_store();

  } elsif ($choice eq 'd') {
    delete_store();

  } elsif ($choice eq 'help') {
    print "\n\n  $H::welcome_screen\n\n\n";
    <STDIN>;

  }

}

exit 0;

#
# -----------------------------------------------------------------------------
#

sub global_config {
  my($tmp, $mp, $ol, $done, $str);
  my($tval, @tvals, $i, $base);

  semi_clear_screen();

  print STDOUT " >>  Main menu  >>  Configure software\n\n";

  $done = 0;

  # try to guess at the base directory based on the invocation directory

  if (!defined($global_conf{'basedir'}) || ($global_conf{'basedir'} eq "")) {

    if ($ARGV[0] !~ /^\//) {
      $base = `pwd`;
      chomp $base;
      $base .= $ARGV[0];
    } else {
      $base = $ARGV[0];
    }

    $base =~ s/\/$//;
    $base =~ s/\/tools$//;

    @tvals = split(/\//, $base);

    $i = 0;
    foreach $tval (@tvals) {
      if ($tval eq ".") {
        splice @tvals, $i, 1;
        $i--;
      } elsif ($tval eq "..") {
        splice @tvals, $i-1, 1;
        $i--;
      }
      $i++;
    }

    $global_conf{'basedir'} = join("/", @tvals);

  }

  while (!$done) {

    get_simple_string(\%global_conf, 'basedir',
                      'Enter the base directory of the software',
                      'Invalid directory.', \$H::global_basedir);

    if (($global_conf{'milton_lib_check'} eq 'notok') ||
        ($global_conf{'milton_lib_check'} ne $global_conf{'basedir'})) {

      if (do_milton_lib_check()) {
        $done = 1;
      }

    } else {
      $done = 1;
    }

  }

  $done = 0;
  while (!$done) {
    $tmp = $global_conf{'mode'};
    if (!defined($tmp) || $tmp eq "") {
      $tmp = 'MODPERL';
    }

    if ($tmp eq 'MODPERL') {
      $mp = '*'; $ol = ' ';
    } elsif ($tmp eq 'OFFLINE') {
      $mp = ' '; $ol = '*';
    }

    print STDOUT "\n\n  In what mode will you be running the software?

  [ 1${mp}]  --  mod_perl / apache

  [ 2${ol}]  --  offline

";
    print STDOUT "Please enter your selection or 'help' for more info: ";
    $str = get_choice();

    if ($str eq '1') {
      $global_conf{'mode'} = 'MODPERL';
      $done = 1;
    } elsif ($str eq '2') {
      $global_conf{'mode'} = 'OFFLINE';
      $done = 1;
    } elsif ($str eq '') {
      $global_conf{'mode'} = $tmp;
      $done = 1;
    } elsif ($str eq 'help') {
      print "\n\n  $H::global_mode\n\n\n";
    } else {
      print "\n\n  Invalid selection.\n\n";
    }
  }

  $global_conf{'config_done'} = 'ok';

  if ($global_conf{'find_existing_stores_done'} ne 'ok') {
    find_existing_stores();
  }

  save_hash(\%global_conf, 'install.globaldat');
}

#
# -----------------------------------------------------------------------------
#

sub create_store {
  my($choice);
  my($mystore) = {};

  if (!($global_conf{'config_done'} eq 'ok')) {
    print "\n\n  You must configure the software before you can create any stores.\n\n";
    <STDIN>;
    return;
  }

  %$mystore = %global_default;

  while ($choice ne 'q') {

    print_create_menu($mystore);

    $choice = get_choice();

    if ($choice eq 'd') {
      config_database($mystore);

    } elsif ($choice eq 'r') {
      config_dir($mystore);

    } elsif ($choice eq 'u') {
      config_url($mystore);

    } elsif ($choice eq 'm') {
      config_misc($mystore);

    } elsif ($choice eq 'i') {
      do_install($mystore);
      <STDIN>;

    } elsif ($choice eq 'f') {
      set_global_default($mystore);

    } elsif ($choice eq 'help') {
      print "\n\n  $H::create_screen\n\n\n";
      <STDIN>;

    }

  }
}

#
# -----------------------------------------------------------------------------
#

sub edit_store {
}

#
# -----------------------------------------------------------------------------
#

sub delete_store {
}

#
# =============================================================================
#
# CREATE STORE SUBROUTINES
#
# =============================================================================
#

sub config_database {
  my($store) = shift;
  my($done) = 0;
  my($str, $tmp, $o, $p);

  semi_clear_screen();
  print STDOUT "  >>  Main menu  >>  Create a new store  >>  Configure database\n\n";

  #
  # get database type
  #

  while (!$done) {
    $tmp = $$store{'db_type'};
    if (!defined($tmp) || $tmp eq "") {
      $tmp = 'ORACLE';
    }

    if ($tmp eq 'ORACLE') {
      $o = '*'; $p = ' ';
    } elsif ($tmp eq 'PG') {
      $o = ' '; $p = '*';
    }

    print STDOUT "  What type of database would you like to use?

  [ 1${o}]  --  Oracle

  [ 2${p}]  --  PostgreSQL

";
    print STDOUT "Please enter your selection or 'help' for more info: ";
    $str = get_choice();

    if ($str eq '1') {
      $$store{'db_type'} = 'ORACLE';
      $done = 1;
    } elsif ($str eq '2') {
      $$store{'db_type'} = 'PG';
      $done = 1;
    } elsif ($str eq '') {
      $$store{'db_type'} = $tmp;
      $done = 1;
    } elsif ($str eq 'help') {
      print "\n\n  $H::db_type\n\n\n";
    } else {
      print "\n\n  Invalid selection.\n\n";
    }
  }

  #
  # get username / password
  #

  get_simple_string($store, 'db_username', 'Enter the database username',
                    'Invalid username.', \$H::db_username);

  get_simple_string($store, 'db_password', 'Enter the database password',
                    'Invalid password.', \$H::db_password);

  #
  # get extra config tokens
  #

  if ($$store{'db_type'} eq 'ORACLE') {
    oracle_get_config_tokens($store);

  } elsif ($$store{'db_type'} eq 'PG') {
    postgres_get_config_tokens($store);
  }
  
  $$store{'db_done'} = 'ok';
}

#
# -----------------------------------------------------------------------------
#

sub postgres_get_config_tokens {
  my($store) = shift;

  if (!defined($$store{'dbi_connect_str'}) ||
      $$store{'dbi_connect_str'} eq "") {
    $$store{'dbi_connect_str'} = 'dbi:Pg:';
  }

  get_simple_string($store, 'dbi_connect_str', 'Enter DBI connect string',
                    'Invalid string.', \$H::dbi_connect_str);

  if (!defined($$store{'postgres_try_to_create_user'}) ||
      $$store{'postgres_try_to_create_user'} eq "") {
    $$store{'postgres_try_to_create_user'} = 'yes';
  }

  get_simple_string($store, 'postgres_try_to_create_user', 
                    'Should I try to create the user?',
                    'Please answer yes or no.',
                    \$H::postgres_try_to_create_user,
                    \&yes_no_checker);

  if ($$store{'postgres_try_to_create_user'} =~ /^yes$/i) {

    if (!defined($$store{'postgres_try_to_drop_user'}) ||
        $$store{'postgres_try_to_drop_user'} eq "") {
      $$store{'postgres_try_to_drop_user'} = 'no';
    }

    get_simple_string($store, 'postgres_try_to_drop_user', 
                      'Should I try to drop the user?',
                      'Please answer yes or no.',
                      \$H::postgres_try_to_drop_user,
                      \&yes_no_checker);

    get_simple_string($store, 'postgres_dba_username', 
                      'Postgres privileged user account',
                      'Please enter something', \$H::postgres_dba_username);

    get_simple_string($store, 'postgres_dba_password', 
                      'Postgres privileged user password',
                      'Please enter something', \$H::postgres_dba_password);

  }

  if (!defined($$store{'postgres_try_to_create_tables'}) ||
      $$store{'postgres_try_to_create_tables'} eq "") {
    $$store{'postgres_try_to_create_tables'} = 'yes';
  }

  get_simple_string($store, 'postgres_try_to_create_tables', 
                    'Should I try to create the tables?',
                    'Please answer yes or no.',
                    \$H::postgres_try_to_create_tables,
                    \&yes_no_checker);
}

#
# -----------------------------------------------------------------------------
#

sub oracle_get_config_tokens {
  my($store) = shift;

    if (!defined($$store{'dbi_connect_str'}) ||
        $$store{'dbi_connect_str'} eq "") {
      $$store{'dbi_connect_str'} = 'dbi:Oracle:';
    }

    get_simple_string($store, 'dbi_connect_str', 'Enter DBI connect string',
                      'Invalid string.', \$H::dbi_connect_str);

    if (!defined($$store{'oracle_home'}) ||
        $$store{'oracle_home'} eq "") {
      $$store{'oracle_home'} = $ENV{'ORACLE_HOME'};
    }

    if (!defined($$store{'oracle_sid'}) ||
        $$store{'oracle_sid'} eq "") {
      $$store{'oracle_sid'} = $ENV{'ORACLE_SID'};
    }

    if (!defined($$store{'oracle_try_to_create_user'}) ||
        $$store{'oracle_try_to_create_user'} eq "") {
      $$store{'oracle_try_to_create_user'} = 'yes';
    }

    get_simple_string($store, 'oracle_home', 'Enter the Oracle home directory',
                      'Invalid directory.', \$H::oracle_homedir);

    get_simple_string($store, 'oracle_sid', 'Enter the Oracle session id',
                      'Invalid session id.', \$H::oracle_sid);

    get_simple_string($store, 'oracle_try_to_create_user', 
                      'Should I try to create the user?',
                      'Please answer yes or no.',
                      \$H::oracle_try_to_create_user,
                      \&yes_no_checker);

    if ($$store{'oracle_try_to_create_user'} =~ /^yes$/i) {

      if (!defined($$store{'oracle_try_to_drop_user'}) ||
          $$store{'oracle_try_to_drop_user'} eq "") {
        $$store{'oracle_try_to_drop_user'} = 'no';
      }

      get_simple_string($store, 'oracle_try_to_drop_user', 
                        'Should I try to drop the user?',
                        'Please answer yes or no.',
                        \$H::oracle_try_to_drop_user,
                        \&yes_no_checker);

      get_simple_string($store, 'oracle_dba_username', 
                        'Oracle privileged user account',
                        'Please enter something', \$H::oracle_dba_username);

      get_simple_string($store, 'oracle_dba_password', 
                        'Oracle privileged user password',
                        'Please enter something', \$H::oracle_dba_password);

      if (!defined($$store{'oracle_default_tablespace'}) ||
          $$store{'oracle_default_tablespace'} eq "") {
        $$store{'oracle_default_tablespace'} = 'users';
      }

      if (!defined($$store{'oracle_temp_tablespace'}) ||
          $$store{'oracle_temp_tablespace'} eq "") {
        $$store{'oracle_temp_tablespace'} = 'temp';
      }

      get_simple_string($store, 'oracle_default_tablespace', 
                        'Oracle default tablespace',
                        'Please enter something',
                        \$H::oracle_default_tablespace);

      get_simple_string($store, 'oracle_temp_tablespace', 
                        'Oracle temporary tablespace',
                        'Please enter something', 
                        \$H::oracle_temp_tablespace);

    }

    if (!defined($$store{'oracle_try_to_create_tables'}) ||
        $$store{'oracle_try_to_create_tables'} eq "") {
      $$store{'oracle_try_to_create_tables'} = 'yes';
    }

    get_simple_string($store, 'oracle_try_to_create_tables', 
                      'Should I try to create the tables?',
                      'Please answer yes or no.',
                      \$H::oracle_try_to_create_tables,
                      \&yes_no_checker);

}

#
# -----------------------------------------------------------------------------
#

sub config_dir {
  my($store) = shift;

  semi_clear_screen();
  print STDOUT "  >>  Main menu  >>  Create a new store  >>  Configure directories\n\n";

  get_simple_string($store, 'regen_output_directory', 
                    'Enter the path where created HTML files should be placed',
                    'Invalid path.', \$H::regen_output_dir, \&path_checker);

  get_simple_string($store, 'regen_output_url', 
                    'Enter the url of the directory you just specified',
                    'Invalid url.', \$H::regen_output_url, \&url_checker);

  $$store{'dir_done'} = 'ok';
}

#
# -----------------------------------------------------------------------------
#

sub config_url {
  my($store) = shift;
  my($tmp);

  semi_clear_screen();
  print STDOUT "  >>  Main menu  >>  Create a new store  >>  Configure URLs\n\n";

  get_simple_string($store, 'admin_url', 
                    'Enter the url of the admin directory',
                    'Invalid url.', \$H::admin_url, \&url_checker);

  if ($global_conf{'mode'} eq "OFFLINE") {
    unless($$store{'admin_url'} =~ m{/epl$}) {
      $$store{'admin_url'} .= "/epl";
      print "\n\n (Since this is a CGI-mode install, appended /epl)\n\n";
    }
  }

  if (!defined($$store{'admin_secure_url'}) ||
      $$store{'admin_secure_url'} eq "") {
    $tmp = $$store{'admin_url'};
    $tmp =~ s/http:/https:/i;
    $$store{'admin_secure_url'} = $tmp;
  }

  get_simple_string($store, 'admin_secure_url', 
                    'Enter the secure url of the admin directory',
                    'Invalid url.', \$H::admin_secure_url, \&url_checker);

  if ($global_conf{'mode'} eq "OFFLINE") {
    unless($$store{'admin_secure_url'} =~ m{/epl$}) {
      $$store{'admin_secure_url'} .= "/epl";
      print "\n\n (Since this is a CGI-mode install, appended /epl)\n\n";
    }
  }

  if (!defined($$store{'tallyman_url'}) ||
      $$store{'tallyman_url'} eq "") {
    $tmp = $$store{'admin_url'};
    $tmp =~ s/admin/tallyman/i;
    $$store{'tallyman_url'} = $tmp;
  }

  get_simple_string($store, 'tallyman_url', 
                    'Enter the url of the tallyman directory',
                    'Invalid url.', \$H::tallyman_url, \&url_checker);

  if ($global_conf{'mode'} eq "OFFLINE") {
    unless($$store{'tallyman_url'} =~ m{/epl$}) {
      $$store{'tallyman_url'} .= "/epl";
      print "\n\n (Since this is a CGI-mode install, appended /epl)\n\n";
    }
  }

  if (!defined($$store{'tallyman_secure_url'}) ||
      $$store{'tallyman_secure_url'} eq "") {
    $tmp = $$store{'tallyman_url'};
    $tmp =~ s/http:/https:/i;
    $$store{'tallyman_secure_url'} = $tmp;
  }

  get_simple_string($store, 'tallyman_secure_url', 
                    'Enter the secure url of the tallyman directory',
                    'Invalid url.', \$H::tallyman_secure_url, \&url_checker);

  if ($global_conf{'mode'} eq "OFFLINE") {
    unless($$store{'tallyman_secure_url'} =~ m{/epl$}) {
      $$store{'tallyman_secure_url'} .= "/epl";
      print "\n\n (Since this is a CGI-mode install, appended /epl)\n\n";
    }
  }

  $$store{'url_done'} = 'ok';
}

#
# -----------------------------------------------------------------------------
#

sub config_misc {
  my($store) = shift;

  semi_clear_screen();
  print STDOUT "  >>  Main menu  >>  Create a new store  >>  Configure miscellaneous\n\n";

  get_simple_string($store, 'store_name', 
                    'Enter your store name',
                    'Store name must not be blank.', \$H::store_name);

  get_simple_string($store, 'store_id', 
                    'Enter a store id',
                    'Store id must not be blank.', \$H::store_id,
                    \&super_clean_checker);

  if (!defined($$store{'do_create_sample_store'}) ||
      $$store{'do_create_sample_store'} eq "") {
    $$store{'do_create_sample_store'} = 'yes';
  }

  if (!defined($$store{'do_zb_ups_import'}) ||
      $$store{'do_zb_ups_import'} eq "") {
    $$store{'do_zb_ups_import'} = 'yes';
  }

  get_simple_string($store, 'do_create_sample_store', 
                    'Do you want to create sample store data?',
                    'Please answer yes or no.', \$H::store_sample_data,
                    \&yes_no_checker);

  get_simple_string($store, 'do_zb_ups_import', 
                    'Do you want to import UPS zone-based shipping data?',
                    'Please answer yes or no.', \$H::zb_ups_importer,
                    \&yes_no_checker);

  get_simple_string($store, 'store_origin_zipcode', 
                    'Enter the origin zipcode of your store',
                    'Invalid zipcode.', \$H::store_origin_zipcode);

  get_simple_string($store, 'store_admin_email', 
                    'Enter the administrator\'s email address',
                    'Invalid email address', \$H::store_admin_email,
                    \&email_checker);

  get_simple_string($store, 'sendmail_bounceback', 
                    'Enter the bounceback email address',
                    'Invalid email address.', \$H::sendmail_bounceback,
                    \&email_checker);

  get_simple_string($store, 'basket_from_address', 
                    'Enter the shopping basket \'from\' email address',
                    'Invalid email address.', \$H::basket_from_address,
                    \&email_checker);

#  if (!defined($$store{'sendmail_path'}) || $$store{'sendmail_path'} eq "") {
#    $$store{'sendmail_path'} = '/usr/sbin/sendmail';
#  }
  
#  get_simple_string($store, 'sendmail_path', 
#                    'Enter the path to sendmail',
#                    'Invalid path.', \$H::sendmail_path, \&file_checker);

  $$store{'misc_done'} = 'ok';
}

#
# -----------------------------------------------------------------------------
#

sub do_log {
  print STDOUT @_;
  print STORELOG @_;
}

sub do_system {
  my($cmd) = shift;

  open (TMP, ">>$install_logfn2");
  print TMP "--------------------------------------------------------------\n";
  print TMP "Results of [$cmd]\n";
  print TMP "--------------------------------------------------------------\n";
  close (TMP);

  system($cmd . " >> $install_logfn2");
}

sub do_append_tmpfile {
  my($cmd) = shift;

  open (TMP, ">>$install_logfn2");
  print TMP "--------------------------------------------------------------\n";
  print TMP "Results of [$cmd]\n";
  print TMP "--------------------------------------------------------------\n";
  close (TMP);

  system("cat $tmp_logfn >> $install_logfn2");
}

#
# -----------------------------------------------------------------------------
#
 
sub do_install {
  my($store) = shift;
  my($timestr);

  $install_logfn = $global_conf{'basedir'} . "/tools/install.log";
  $install_logfn2 = $global_conf{'basedir'} . "/tools/install2.log";
  $tmp_logfn = $global_conf{'basedir'} . "/tools/install.tmp.log";

  if (!open (STORELOG, "> $install_logfn")) {
    print "\n\n  Error opening log file $install_logfn: $!\n\n";
    return 0;
  }

  select(STDOUT);
  $| = 1;

  select STORELOG;
  $| = 1;
  select STDOUT;

  $timestr = localtime(time);

  do_log("\n");
  do_log("# ----------------------------------------------------------------");
  do_log("\n\n");
  do_log("  - Creating store '$$store{'store_id'}' at $timestr...\n");
  do_log("  - A log of the process will be kept in\n");
  do_log("      [$install_logfn].\n");
  do_log("  - A log of all externally executed commands will be kept in\n");
  do_log("      [$install_logfn2].\n\n");

  do_install_real($store);

  close (STORELOG);
}

#
# -----------------------------------------------------------------------------
#

sub do_install_real {
  my($store) = shift;

  do_log("  - Checking configuration...\n");
  if (!check_to_make_sure_its_configured($store)) {
    return;
  }

  do_log("  - Creating directories...\n");
  if (!($$store{'di_create_dirs_done'} eq 'ok')) {
    if (!create_dirs($store)) { return 0; }
  }

  do_log("  - Copying data...\n");
  if (!($$store{'di_copy_data_done'} eq 'ok')) {
    if (!copy_data($store)) { return 0; }
  }

  do_log("  - Creating config file...\n");
  if (!($$store{'di_create_config_done'} eq 'ok')) {
    if (!create_config_file($store)) { return 0; }
  }

  do_log("  - Setting up database...\n");
  if (!($$store{'di_setup_db_done'} eq 'ok')) {
    if (!setup_db($store)) { return 0; }
  }

  do_log("  - Creating database objects...\n");
  if (!($$store{'di_create_db_objs_done'} eq 'ok')) {
    if (!create_db_objs($store)) { return 0; }
  }

  do_log("  - Configuring embedded variables...\n");
  if (!($$store{'di_config_embedded_vars_done'} eq 'ok')) {
    if (!config_embedded_vars($store)) { return 0; }
  }

  do_log("  - Changing permissions...\n");
  if (!($$store{'di_change_permissions_done'} eq 'ok')) {
    if (!change_permissions($store)) { return 0; }
  }

  do_log("  - Putting on the finishing touches...\n");
  if (!($$store{'di_extra_stuff_done'} eq 'ok')) {
    if (!extra_stuff($store)) { return 0; }
  }

  print STORELOG "\n\nSuccess!\n";
  print STDOUT <<EOTEXT;

  - Done!

EOTEXT

  semi_clear_screen();

  if ($global_conf{'mode'} eq "OFFLINE") {
    print STDOUT <<EOTEXT;
Congratulations!  TallyMan is now installed on your system.

You still need to make sure that your webserver is configured correctly.
Since you have decided to run in offline (or CGI) mode, you need to make
sure that CGI scripts can be executed from the following directories
(this can be done in Apache with the 'ScriptAlias' directive; consult
your documentation if you are running another webserver):

      $global_conf{'basedir'}/admin
      $global_conf{'basedir'}/tallyman
EOTEXT

  } elsif ($global_conf{'mode'} eq "MODPERL") {
    print STDOUT <<EOTEXT;
Congratulations!  TallyMan is now installed on your system.

You still need to make sure that your webserver is configured correctly.
Since you have decided to run in mod_perl / Apache mode, you need to 
add several directives to your httpd.conf.  Please see the file 
doc/apache_include.conf for an example of how to configure Apache.  Also,
you'll need to check out the doc/startup.pl script -- it _must_ be included
in Apache, because TallyMan needs to know where it can find its libraries.
The "use lib" line at the top needs to be edited to read:

  use lib "$global_conf{'basedir'}/lib";

Then make sure that the startup.pl script is actually executed by adding
a line like

  PerlScript startup.pl

to your httpd.conf.

EOTEXT

  }

  print STDOUT "Press [enter] to continue...";
  <STDIN>;

  semi_clear_screen();

  print STDOUT <<EOTEXT;

In order to access TallyMan, make sure your web server is running
and point your browser at:

      $$store{'admin_secure_url'}/index.epl

The first thing you'll want to do is go to the 'Configuration' section and 
check all of the system settings.  Pay special attention to your tax, 
shipping and payment options -- we haven't set up any, so you'll need to 
enter some to be accurate.  Note that you'll need to select 'Regenerate'
before the .html files making up your store will be updated.

Also, you need to make sure that all of the permissions are set correctly.
These directories need to be writable by the webserver user:

      $global_conf{'basedir'}/stores/$$store{'store_id'}/tmp
      $$store{'regen_output_directory'}

EOTEXT

  print STDOUT "Press [enter] to continue...";
  <STDIN>;
  semi_clear_screen();



  print STDOUT <<EOTEXT;

                 ** IMPORTANT NOTE ABOUT SECURITY **

Because the TallyMan scripts need to access the database account, the
database password that you have entered has been saved in your store's
configuration file (it may also have been saved to the default store
configuration file):

      $global_conf{'basedir'}/stores/$$store{'store_id'}/configuration
      $global_conf{'basedir'}/tools/install.storedat
      $global_conf{'basedir'}/tools/install.globaldat

YOU NEED TO MAKE SURE THAT THESE FILES ARE SECURE!

Remember, your customer's payment information will stored in this database
account, so everyone that knows the database username and password can
easily download all payment information.  It is essential that you allow
only trusted users to read these files.

On most systems, it should be sufficient to change the permissions to 
400 ('chmod 400 configuration') and make the webserver user the owner
of the files ('chown www:www configuration', for example).

EOTEXT

  print STDOUT "Press [enter] to continue...";
  <STDIN>;
  semi_clear_screen();




  print STDOUT <<EOTEXT;

               ** ANOTHER IMPORTANT NOTE ABOUT SECURITY **

Remember, TallyMan uses a web-based interface.  On most systems, the
HTTP server is a publicly accessible service.  Anyone can look at the
documents your server offers unless specific measures to the contrary
are taken.

It is essential that you somehow protect the directory
      $global_conf{'basedir'}/admin 
so that some sort of authentication is necessary, and only trusted
users are allowed to access the contents (remember, that's where the
order manager is kept; anyone that has access to the order manager can
view your customer's payment information).  This is typically done
with some sort of access control file (on apache servers, it is
usually called .htaccess) TallyMan ships with an example .htaccess
file in the admin/ directory.  It references a password file called
/usr/local/tallyman/secret/tallyman_passwd, which should contain the
usernames and encrypted passwords of the users you wish to allow into
the admin/ directory (you can create this file using Apaches htpasswd
utility.) Consult your server documentation for more information on
access control.

EOTEXT

  print STDOUT "Press [enter] to continue...";
  <STDIN>;
  semi_clear_screen();




  if ($$store{'do_create_sample_store'} eq "yes") {
    print STDOUT <<EOTEXT;

We've installed some sample store data for you to play with.  It's pretty
much an exact copy of the demo store running on http://www.tallyman.com:4242
You don't have to use any of it, though -- feel free to modify or delete
anything you see.  

EOTEXT
  }

  print STDOUT <<EOTEXT;

Have fun, and may you be blessed with more sales than you can handle!

Press the [return] key to continue...

EOTEXT


  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub set_global_default {
  my($mystore) = shift;
  my($choice);

  while ($choice ne 'q') {

    print_global_default_menu();

    $choice = get_choice();

    if ($choice eq 't') {
      %global_default = %$mystore;
      print STDOUT "  Done.  Press [enter] to continue...";
      <STDIN>;
      return;

    } elsif ($choice eq 'a') {
      %global_default = %$mystore;
      save_hash($mystore, 'install.storedat');
      print STDOUT "  Done.  Press [enter] to continue...";
      <STDIN>;
      return;

    } elsif ($choice eq 'help') {
      print "\n\n  $H::global_default\n\n\n";
      <STDIN>;
    }

  }

}

#
# -----------------------------------------------------------------------------
#

sub check_to_make_sure_its_configured {
  my($store) = shift;

  # check db

  if ($$store{'db_done'} ne 'ok') {
    print STDOUT "\n\n  You must configure your database before installing.\n\n";
    <STDIN>;
    return 0;

  } elsif ($$store{'dir_done'} ne 'ok') {
    print STDOUT "\n\n  You must configure your directories before installing.\n\n";
    <STDIN>;
    return 0;

  } elsif ($$store{'url_done'} ne 'ok') {
    print STDOUT "\n\n  You must configure your URLs before installing.\n\n";
    <STDIN>;
    return 0;

  } elsif ($$store{'misc_done'} ne 'ok') {
    print STDOUT "\n\n  You must configure miscellaneous options before installing.\n\n";
    <STDIN>;
    return 0;
  }

  if (!chdir($global_conf{'basedir'})) {
    print STDOUT "\n\n  Unable to chdir to $global_conf{'basedir'}: $!\n\n";
    <STDIN>;
    return 0;
  }

  return 1;
}

#
# =============================================================================
#
#   ACTUAL INSTALL ROUTINES
#
# =============================================================================
#

sub create_dirs {
  my($store) = shift;
  my($base, $dirname, $dir);

  $base = $global_conf{'basedir'} . '/stores/' . $$store{'store_id'};
  if (!create_dir($base)) {
    return 0;
  }

  foreach $dir ('logs','templates','tmp','templates/a','templates/b', 'templates/c') {
    $dirname = $base . '/' . $dir;
    if (!create_dir($dirname)) {
      return 0;
    }
  }

  if (!create_dir($$store{'regen_output_directory'})) {
    return 0;
  }

  if (!create_dir("$$store{'regen_output_directory'}/images")) {
    return 0;
  }

  if (!create_dir("$$store{'regen_output_directory'}/standard_images")) {
    return 0;
  }

  $$store{'di_create_dirs_done'} = 'ok';
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub create_dir {
  my($dirname) = shift;

  print STORELOG "    - Creating directory [$dirname]... ";

  if (opendir HOHUM, $dirname) {
    do_log("Directory [$dirname] already exists!\n");
    closedir HOHUM;
  } else {
    if (!mkdir($dirname, 0777)) {
      do_log("Error creating [$dirname]: $! \n");
      return 0;
    }
  }

  print STORELOG "Success.\n";
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub copy_data {
  my($store) = shift;
  my($srcbase, $destbase);

  # template data

  $srcbase = $global_conf{'basedir'} . '/stores/default';
  $destbase  = $global_conf{'basedir'} . '/stores/' . $$store{'store_id'};

  copy_all_files($srcbase . '/templates', $destbase . '/templates');
  copy_all_files($srcbase . '/templates/a', $destbase . '/templates/a');
  copy_all_files($srcbase . '/templates/b', $destbase . '/templates/b');
  copy_all_files($srcbase . '/templates/c', $destbase . '/templates/c');

  $$store{'di_copy_data_done'} = 'ok';
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub copy_all_files {
  my($src, $dest) = @_;
  my($fn);

  for (<$src/*>) {
    $fn = $_;
    if (opendir HOHUM, $fn) {
      closedir HOHUM;
    } else {
      print STORELOG "    - Copying file [$fn] to [$dest]...\n";
      do_system("cp $fn $dest");
    }
  }
}

#
# -----------------------------------------------------------------------------
#

sub create_config_file {
  my($store) = shift;
  my($fn);

  $fn = $global_conf{'basedir'} . '/stores/' . $$store{'store_id'} . 
        '/configuration';

  if (!open(CONFIG, ">$fn")) {
    do_log("Error creating [$fn]: $!\n");
    return 0;
  }

  print CONFIG 'db_default: ' . $$store{'db_type'} . "\n";
  print CONFIG $$store{'db_type'} . '_db_username: ' . 
               $$store{'db_username'} . "\n";
  print CONFIG $$store{'db_type'} . '_db_password: ' . 
               $$store{'db_password'} . "\n";
  print CONFIG $$store{'db_type'} . '_db_connect_str: ' . 
               $$store{'dbi_connect_str'} . "\n";

  if ($$store{'db_type'} eq 'ORACLE') {
    print CONFIG 'oracle_homedir: ' . $$store{'oracle_home'} . "\n";
    print CONFIG 'oracle_sid: ' . $$store{'oracle_sid'} . "\n";
  }

  close CONFIG;

  $$store{'di_create_config_file_done'} = 'ok';
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub setup_db {
  my($store) = shift;
  my($str, $dbtype, $rval);

  # generate the real scripts

  $dbtype = $$store{'db_type'};
  $dbtype =~ tr/A-Z/a-z/;

  if ($dbtype eq 'pg') {
    $dbtype = 'postgresql';
  }

  $str = $global_conf{'basedir'} . '/make/make_sql.pl ' . $dbtype;

  if (!chdir("./make")) {
    do_log("\n\n  Cannot chdir to ./make: $!\n\n");
    return 0;
  }

  do_log("    - Creating database table creation script files...\n");

  $rval = do_system($str);

  if ($rval == 1) {
    do_log("Error executing script creator: [$str] returned $rval\n");
    return 0;
  }

  #
  # create database user (if we have a privileged account)
  #

  if ($$store{'db_type'} eq 'ORACLE') {

    if (!chdir("./oracle")) {
      do_log("\n\n  Cannot chdir to ./oracle: $!\n\n");
      return 0;
    }

    if ($$store{'oracle_try_to_create_user'} eq 'yes') {

      if ($$store{'oracle_try_to_drop_user'} eq 'yes') {
        if (!INST_DB::oracle_drop_user(\%global_conf, $store)) {
          return 0;
        }
      }

      if (!INST_DB::oracle_create_user(\%global_conf, $store)) {
        return 0;
      }

    }

    if (!chdir("..")) {
      do_log("\n\n  Cannot chdir to ..: $!\n\n");
      return 0;
    }

  } elsif ($$store{'db_type'} eq 'PG') {

    if (!chdir("./postgresql")) {
      do_log("\n\n  Cannot chdir to ./postgresql: $!\n\n");
      return 0;
    }

    if ($$store{'postgres_try_to_create_user'} eq 'yes') {

      if ($$store{'postgres_try_to_drop_user'} eq 'yes') {
        if (!INST_DB::postgres_drop_user(\%global_conf, $store)) {
          return 0;
        }
      }

      if (!INST_DB::postgres_create_user(\%global_conf, $store)) {
        return 0;
      }

    }

    if (!chdir("..")) {
      do_log("\n\n  Cannot chdir to ..: $!\n\n");
      return 0;
    }

  }

  #
  # run table make scripts
  #

  if ($$store{'db_type'} eq 'ORACLE') {
    if ($$store{'oracle_try_to_create_tables'}) {

      if (!INST_DB::oracle_create_tables(\%global_conf, $store)) {
        return 0;
      }

    }
  } elsif ($$store{'db_type'} eq 'PG') {
    if ($$store{'postgres_try_to_create_tables'}) {

      if (!INST_DB::postgres_create_tables(\%global_conf, $store)) {
        return 0;
      }

    }
  }

  if (!chdir("..")) {
    do_log("\n\n  Cannot chdir to ..: $!\n\n");
    return 0;
  }

  $$store{'di_setup_db_done'} = 'ok';
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub create_db_objs {
  my($store) = shift;
  my($base, $mlib);

  $base = $global_conf{'basedir'};

  $ENV{'TALLYMAN_SID'} = $$store{'store_id'};
  $ENV{'TALLYMAN_PATH'} = $base;

  $mlib = $base . '/lib';
  push @INC, $mlib;

  if (!milton_lib_check('DBLIB')) { return 0; }
  if (!milton_lib_check('KNAR')) { return 0; }

  #
  # create shopping basket emailer entries
  #

  do_knar('BASKET_EMAIL_FROM_ADDRESS', $$store{'basket_from_address'});
  do_knar('BASKET_EMAIL_SUBJECT', 'Your receipt');

  do_knar('BASKET_EMAIL_HEADER1',
'
This is your receipt:

');

  do_knar('BASKET_EMAIL_HEADER2',
'Name                                         Price    Qty.   Discount Subtotal
------------------------------------------------------------------------------
');

  do_knar('BASKET_EMAIL_FOOTER',
"
------------------------------------------------------------------------------

You may mail any questions or comments about your order to 
        $$store{'basket_from_address'}

Thank you for shopping with $$store{'store_name'}!

");

  do_knar('BASKET_EMAIL_SUB_TOTAL_TEXT', 'Sub-total:');
  do_knar('BASKET_EMAIL_TAX_TOTAL_TEXT', 'Tax:');
  do_knar('BASKET_EMAIL_SHIP_TOTAL_TEXT', 'Shipping:');
  do_knar('BASKET_EMAIL_GRAND_TOTAL_TEXT','Grand Total:');
  do_knar('BASKET_EMAIL_ORDERNUM_TEXT', 'Order number:');
  do_knar('BASKET_EMAIL_ORDERDATE_TEXT', 'Order date:');
  do_knar('BASKET_EMAIL_CALLSHIP_TEXT', 'Call');

  #
  # create regeneration system entries
  #

  do_knar('REGEN_OUTPUT_DIR', $$store{'regen_output_directory'});
  do_knar('REGEN_OUTPUT_URL', $$store{'regen_output_url'});

  do_knar('ADMIN_DIR', $base . '/admin');
  do_knar('ADMIN_URL', $$store{'admin_url'});
  do_knar('ADMIN_SECURE_URL', $$store{'admin_secure_url'});

  do_knar('TALLYMAN_DIR', $base . '/tallyman');
  do_knar('TALLYMAN_URL', $$store{'tallyman_url'});
  do_knar('TALLYMAN_SECURE_URL', $$store{'tallyman_secure_url'});

  #
  # create misc. entries
  #

#  do_knar('SENDMAIL_PATH', $$store{'sendmail_path'});
  do_knar('SENDMAIL_BOUNCEBACK', $$store{'sendmail_bounceback'});

  #
  # create user interface entries
  #

  do_knar('MILTON_SCHEMENAME', 'Brown, Desolate, and Lifeless', 'cachable');
  do_knar('MILTON_COLOR1', '#ffffcc', 'cachable');
  do_knar('MILTON_COLOR2', '#cccc99', 'cachable');
  do_knar('MILTON_COLOR3', '#666633', 'cachable');
  do_knar('MILTON_COLOR4', '#336699', 'cachable');
  do_knar('MILTON_TITLEBARTEXTCOLOR', '#cccccc', 'cachable');

  do_knar('MILTON_LINKCOLOR','#000000', 'cachable');
  do_knar('MILTON_ALINKCOLOR', '#000000', 'cachable');
  do_knar('MILTON_VLINKCOLOR', '#000000', 'cachable');

  do_knar('MILTON_TEXTCOLOR', '#333333', 'cachable');
  do_knar('MILTON_BGCOLOR', '#ffffcc', 'cachable');
  do_knar('MILTON_IMAGE', 'images/tan_logo.gif', 'cachable');
  do_knar('MILTON_ERRORCOLOR', '#ffeeee', 'cachable');

  do_knar('BASKET_SHOW_DEBUG_FOOTER', 'n', 'cachable');
  do_knar('ITEM_WIDGET_THRESHOLD', '10', 'cachable');

  do_knar('REDIRECTOR_URL', "$$store{'tallyman_url'}/redirect.epl");

  do_knar('Template Image URL', "$$store{'regen_output_url'}/standard_images/");

  pop @INC;

  $$store{'di_create_db_objs_done'} = 'ok';
  return 1;
}

sub do_knar {
  my($key, $val, $grp) = @_;
  $val = DBLIB::db_string_clean($val);
  KNAR::knar_entry_set_by_name($key, $val, $grp);
}

#
# -----------------------------------------------------------------------------
#

sub config_embedded_vars {
  my($store) = shift;
  my(%evar, $base);

  #
  # epl.pristine -> epl
  #

  $evar{'MILTON_BASE_DIR'} = $global_conf{'basedir'};
  $evar{'MILTON_LIB_DIR'} = $global_conf{'basedir'} . '/lib';
  $evar{'MILTON_THIS_DIR'} = $global_conf{'basedir'} . '/admin';
  $evar{'MILTON_SID'} = $$store{'store_id'};

  $base = $global_conf{'basedir'};

  rewrite_embedded_vars($base . '/admin/epl.pristine', $base . '/admin/epl',
                        %evar);

  $evar{'MILTON_THIS_DIR'} = $global_conf{'basedir'} . '/tallyman';

  rewrite_embedded_vars($base . '/tallyman/epl.pristine',
                        $base . '/tallyman/epl', %evar);

  $$store{'di_config_embedded_vars_done'} = 'ok';
  return 1;
}

sub rewrite_embedded_vars {
  my($src, $dest, %evars) = @_;

  if (!open (SRCFILE, "<$src")) {
    do_log("  Error: could not open file $src: $!\n");
    return 0;
  }

  if (!open (DESTFILE, ">$dest")) {
    do_log("  Error: could not open file $src: $!\n");
    close(SRCFILE);
    return 0;
  }

  while (<SRCFILE>){
    s/\@\@\@_(\w+)_\@\@\@/exists($evars{$1}) ? $evars{$1} : $1/eg;
    print DESTFILE $_;
  }

  close(SRCFILE);
  close(DESTFILE);
  
}

#
# -----------------------------------------------------------------------------
#

sub change_permissions {
  my($store) = shift;

  # tmp directory
  # template directory
  # logs directory
  # html output directory

  $$store{'di_change_permissions_done'} = 'ok';
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub extra_stuff {
  my($store) = shift;
  my($path, $src, $dest);

  if ($$store{'do_zb_ups_import'} eq "yes") { 

    do_log("    - Executing the UPS data importer (this may take a while)...\n");

    # run the zb_ups_shipping data importer
    # usage: zb_ups_import <milton_path> <storeid> <data directory>
    #        <all | rates | surcharge | zone> [origin zipcode]

    $path = $global_conf{'basedir'} . "/tools/zb_ups_import.pl "
            . $global_conf{'basedir'} . " " . $$store{'store_id'} . " "
            . $global_conf{'basedir'} . "/data/zb_ups_shipping/us all "
            . $$store{'store_origin_zipcode'};

  #  do_log("$path");
    do_system($path);

  }

  # copy template images

  do_log("    - Copying template images...\n");

  $src = $global_conf{'basedir'} . "/data/sample_store/standard_images";
  $dest = $$store{'regen_output_directory'} . "/standard_images";
  copy_all_files($src, $dest);  

  #
  # install sample store data
  # 

  if ($$store{'do_create_sample_store'} eq "yes") {

    do_log("    - Installing sample store data...\n");

    # copy demo images

    $src = $global_conf{'basedir'} . "/data/sample_store/images";
    $dest = $$store{'regen_output_directory'} . "/images";
    copy_all_files($src, $dest);  

    #
    # import database objects
    #

    $path = $global_conf{'basedir'};

    $ENV{'TALLYMAN_SID'} = $$store{'store_id'};
    $ENV{'TALLYMAN_PATH'} = $path;

    $path = $path . '/lib';
    push @INC, $path;

    if (!milton_lib_check('DBLIB')) { return 0; }
    if (!milton_lib_check('DBUPLOAD')) { return 0; }
    if (!milton_lib_check('LAYIM')) { return 0; }

    # run db import
 
    $src = $global_conf{'basedir'} . "/data/sample_store/dbdownload.txt";
    $dest = $global_conf{'basedir'} . "/stores/" . $$store{'store_id'} .
            "/tmp/dbupload.txt";
    do_system("cp $src $dest");

    my($i_set, $stat_set);
    $i_set = {};  # just an empty instruction set.
    $stat_set = DBUPLOAD::dbupload_rip_it("\t", $i_set);  

    # run layout import

    my($data, $tmp, $errmsg);
    $data = load_file($global_conf{'basedir'} . "/data/sample_store/layoutdownload.txt");
    ($tmp, $errmsg) = LAYIM::layout_import($data);

    if ($tmp != 1) {
      do_log("      - Error installing layout data: $errmsg\n");
    }
  }

  $$store{'di_extra_stuff_done'} = 'ok';
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub print_welcome_screen {
  print STDOUT <<EOT;

TallyMan Installer Release 1.9 - Developer

(c) Copyright 1999, Akopia, Inc.  All Rights Reserved.
    See the LICENSE file for licensing details.


Welcome to the TallyMan installer!

This program is designed to simplify the configuration of your TallyMan
ecommerce system.  You can also use it later to change the technical
settings of your system, or create multiple stores using a single
copy of the software.

If you have not already done so, you need to have a web-server and an
SQL compliant database installed.  Make sure both are working properly
before proceeding with the install.

Any time you are prompted for something, you may respond with 'help,'
and the system will tell you more about what it wants.


Press the [return] key to continue...

EOT
  <STDIN>;
}

sub print_main_menu {
  my($config);

  $config = $global_conf{'config_done'} eq 'ok'?'*':' ';

  semi_clear_screen();
  print STDOUT <<EOT;
  >>  Main menu


  [ s${config}]  --  Configure software
EOT

  if ($config eq "*") {
    print STDOUT <<EOT;

  [ c ]  --  Create a new store
EOT
  }
  
#  if ($#existing_stores > -1) {
#    print STDOUT <<EOT;
#  [ e ]  --  Edit an existing store
#
#  [ d ]  --  Delete an existing store
#
#EOT
#  }

  print STDOUT <<EOT;


  [ q ]  --  Quit


EOT
  print STDOUT "Please enter your selection or 'help' for more info: ";
}

sub print_create_menu {
  my($db, $dir, $url, $misc);
  my($store) = shift;

  $db   = $$store{'db_done'} eq 'ok'?'*':' ';
  $dir  = $$store{'dir_done'} eq 'ok'?'*':' ';
  $url  = $$store{'url_done'} eq 'ok'?'*':' ';
  $misc = $$store{'misc_done'} eq 'ok'?'*':' ';

  semi_clear_screen();

  print STDOUT <<EOT;
  >>  Main menu  >>  Create a new store



  [ d${db}]  --  Configure database

  [ r${dir}]  --  Configure store directories

  [ u${url}]  --  Configure store URLs

  [ m${misc}]  --  Configure miscellaneous

  [ f ]  --  Use this store configuration as a global default


  [ i ]  --  Install store



  [ q ]  --  Back to main menu


EOT
  print STDOUT "Please enter your selection or 'help' for more info: ";
}

sub print_global_default_menu {
  semi_clear_screen();

  print STDOUT <<EOT;
  >>  Main menu  >>  Create a new store  >>  Set global default options



  [ t ]  --  Just for this session

  [ a ]  --  For this session and future sessions


  [ q ]  --  Cancel


EOT
  print STDOUT "Please enter your selection or 'help' for more info: ";
}

#
# -----------------------------------------------------------------------------
#

#
# per database-type operations we want to be able to perform:
#   create a user (optional)
#   delete a user (optional)
#   connect as a user
#   create tables
#   drop tables
#


sub save_hash {
  my($hashref, $fn) = @_;
  my($key);

  open(HOHUM,">./$fn");
  foreach $key (keys(%$hashref)) {
    if ($key !~ /^di_/) {
      print HOHUM "$key: $$hashref{$key}\n";
    }
  }
  close(HOHUM);
}

sub load_hash {
  my($fn) = shift;
  my($key, $val);
  my($hashref) = {};

  open(HOHUM,"<./$fn");
  while (<HOHUM>) {
    chomp;
    ($key, $val) = split(/: /, $_, 2);
    $$hashref{$key} = $val;
  }
  close(HOHUM);

  return $hashref;
}
   
#
# =============================================================================
#
# UTILITY FUNCTIONS
#
# =============================================================================
#

sub yes_no_checker {
  my($str_ref, $err) = @_;

  $$str_ref =~ tr/A-Z/a-z/;

  if ($$str_ref eq 'y') {
    $$str_ref = 'yes';
    return 1;
  }

  if ($$str_ref eq 'n') {
    $$str_ref = 'no';
    return 1;
  }

  if ($$str_ref eq 'yes' || $$str_ref eq 'no') {
    return 1;
  }

  print "\n\n  $err\n\n";
  return 0;
}

#
# -----------------------------------------------------------------------------
#

sub url_checker {
  my($str_ref, $err) = @_;

  if ($$str_ref =~ /\/$/) {
    print "\n\n  (trimmed trailing slash.)\n\n";
    $$str_ref =~ s/\/$//;
  }

  if ($$str_ref =~ /^(http:\/\/|https:\/\/)(.*)$/i) {
    return 1;
  } else {
    print "\n\n  $err\n\n";
    return 0;
  }
}

#
# -----------------------------------------------------------------------------
#

sub path_checker {
  my($str_ref, $err) = @_;

  if ($$str_ref =~ /\/$/) {
    print "\n\n  (trimmed trailing slash.)\n\n";
    $$str_ref =~ s/\/$//;
  }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub file_checker {
  my($str_ref, $err) = @_;

  if ($$str_ref =~ /\/$/) {
    print "\n\n  (trimmed trailing slash.)\n\n";
    $$str_ref =~ s/\/$//;
  }

  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub email_checker {
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub super_clean_checker {
  my($str_ref, $err) = @_;
  $$str_ref =~ s/[^a-zA-Z0-9]//g;
  return 1;
}

#
# -----------------------------------------------------------------------------
#

sub get_choice {
  my($str);
  $str = <STDIN>;
  strip_it(\$str);
  return $str;
}

#
# -----------------------------------------------------------------------------
#

sub strip_it {
  my($str_ref) = shift;
  chomp $$str_ref;
  $$str_ref =~ tr/A-Z/a-z/;
}

#
# -----------------------------------------------------------------------------
#

sub get_simple_string {
  my($store, $key, $prompt, $err, $helpref, $checker) = @_;
  my($done, $str, $default);

  $default = $$store{$key};

  $done = 0;
  while (!$done) {
    print STDOUT "\n\n$prompt [$default]: ";
    $str = <STDIN>;
    chomp $str;
    $done = 1;
    if ($str eq "") {
      if (!defined($default) || ($default eq "")) {
        print STDOUT "\n\n  $err\n\n";
        $done = 0;
      } else {
        $str = $default;
      }
    } elsif ($str =~ /^help$/) {
      $done = 0;
      print STDOUT "\n\n  $$helpref\n\n";
    }

    if (($done) && (ref($checker) eq 'CODE')) {
      $done = &$checker(\$str, $err);
    }
  }

  $$store{$key} = $str;
}

#
# -----------------------------------------------------------------------------
#

sub semi_clear_screen {
  print STDOUT <<EOT;



















EOT
}

#
# -----------------------------------------------------------------------------
#

sub find_existing_stores {
  my($dir) = $global_conf{'basedir'} . "/stores";
  my(@dirs, $tmp, $possible_error);

  $global_conf{'find_existing_stores_done'} = 'notok';

  $possible_error = 0;

  if ($VERBOSE) {
    print "Detecting existing stores...\n";
  }

  if (defined($global_conf{'basedir'})) {
    if (opendir (BASEDIR, $dir)) {
      @dirs = readdir(BASEDIR);
      closedir (BASEDIR);

      foreach $tmp (@dirs) {
        if ($tmp eq '.' || $tmp eq '..') { next; }

        if (opendir (TMPDIR, $dir . '/' . $tmp)) {
          if (open (HOHUM, $dir . '/' . $tmp . '/configuration')) {
            push @existing_stores, $tmp;
            if ($VERBOSE) { print "  Found storeid [$tmp]\n"; }
            close HOHUM;

          } else {
            if ($VERBOSE) {
              print "  [$tmp] has no 'configuration' file, skipping...\n";
              $possible_error = 1;
            }
          }
          closedir (TMPDIR);
        } else {
          if ($VERBOSE) {
            print "  Possible error: could not open directory $dir: $!\n";
            $possible_error = 1;
          }
        }
      }

      $global_conf{'find_existing_stores_done'} = 'ok';

    } else {
      print "  Could not open directory $dir: $!\n";
      print "  Could not detect existing stores.\n";
      print "  Is your TallyMan base directory set correctly?\n";

      # force them to re-configure the software
      $global_conf{'config_done'} = 'notok';
      $possible_error = 1;

      <STDIN>;
    }
  }

  if ( ($#existing_stores == -1) && $VERBOSE) {
    print "No stores detected.\n";
  }

  if ($VERBOSE) {
    print STDOUT "Done.  Press the [return] key to continue...\n";
    <STDIN>;
  }
}

#
# -----------------------------------------------------------------------------
#

sub check_command_line_args {
  my($arg);

  while (defined ($arg = shift)) {
    if ($arg eq '-h' || $arg eq '-?' || $arg eq '--help' ||
        $arg eq '-H' || $arg eq '-?' || $arg eq '?') {
      print STDOUT $H::usage_screen;
      exit(0);
    }

    if ($arg eq '-v' || $arg eq '-V' || $arg eq '--verbose') {
      $VERBOSE = 1;
    }  
  }

}

#
# -----------------------------------------------------------------------------
#

sub check_for_required_packages {
  my($missing_a_module) = 0;

  print "  Checking for DBI......";
  eval {
    require DBI;
    #Net::FTP->require_version('2.00');
  };

  if ($@) {
    print " failed.\n";
    $missing_a_module = 1;
    print <<EOT;
    $@
    TallyMan requires that DBI (along with a suitable DBD for your database)
    be installed on your system _before_ installation can proceed.  Please 
    consult the install documentation before continuing.

EOT
  } else {
    print " ok.\n";
  }


  print "  Checking for HTML::Embperl......";
  eval {
    require HTML::Embperl;
    #Net::FTP->require_version('2.00');
  };

  if ($@) {
    print " failed.\n";
    $missing_a_module = 1;
    print <<EOT;
    $@
    TallyMan relies on HTML::Embperl for all of its form interfaces.  It is
    required to exist _before_ installation can proceed.  Please consult the
    install documentation before continuing.

EOT
  } else {
    print " ok.\n";
  }

  if ($missing_a_module) {
    exit(1);
  }

}

#
# -----------------------------------------------------------------------------
#

sub do_milton_lib_check {
  my($mlib, $missing_a_module);

  $missing_a_module = 0;

  $mlib = $global_conf{'basedir'} . '/lib';

  push @INC, $mlib;

  #
  # I was going to try and iterate through all of the MILTON libraries,
  # but it turns out that some of them want to connect immediately to the
  # database.  Er. Um. That's bad, since we haven't necessarily configured
  # a store with information on how to connect to a database.  The point
  # here is just to make sure that we can access the libraries, so we 
  # one that doesn't require anything else, and is very simple.
  #

  if (!milton_lib_check('THING')) {
    $missing_a_module = 1;
  }

  pop @INC;

  if ($missing_a_module) {
    $global_conf{'milton_lib_check'} = 'notok';
  } else {
    $global_conf{'milton_lib_check'} = $global_conf{'basedir'} . '/lib';
  }

  return !$missing_a_module;
}

#
# -----------------------------------------------------------------------------
#

sub milton_lib_check {
  my($libname) = shift;

#  print "  Checking for $libname......";
  eval {
    require $libname . '.pm';
  };

  if ($@) {
    print "  Check for $libname failed.\n";
    print <<EOT;
    $@
    $libname.pm is a part of the TallyMan library.  It must be accessible.
    It is typically located in your $global_conf{'basedir'}/lib
    directory.  We can't seem to load it, which means that we won't be able to
    load any of the other libraries at run-time, either.

EOT
    return 0;
  } else {
#    print " ok.\n";
    return 1;
  }
}

#
# -----------------------------------------------------------------------------
#

sub load_file {
  my($fn) = shift;
  my($result, $length);

  if (!open(TMP,"$fn")) { 
    print STDOUT "      Error loading file $fn: $!\n";
    return(undef);
  }

  seek(TMP,0,2);
  $length=tell(TMP);
  seek(TMP,0,0);
  read TMP,$result,$length;
  close(TMP);
  return($result);  
}
