#!/usr/bin/perl

# zb_ups_import.pl
#   copyright (c) 1999 akopia, inc.
#
######################################################################
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of version 2 of the GNU General Public
#   License as published by the Free Software Foundation.
#    
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#   General Public License for more details.
######################################################################

# Zone-based shipping import script for UPS.

#
# -----------------------------------------------------------------------------
#
# make sure that MILTON can find us.
#

BEGIN {
  if (($#ARGV != 3) && ($#ARGV !=4)) {
    die "usage: zb_ups_import <milton_path> <storeid> <data directory>
       <all | rates | surcharge | zone>
       [origin zipcode]

       (origin zipcode is only needed for zone loading)\n";
  }

  if (!defined($ENV{'TALLYMAN_PATH'})) {
    $ENV{'TALLYMAN_PATH'} = $ARGV[0];
  }

  if (!defined($ENV{'TALLYMAN_SID'})) {
    $ENV{'TALLYMAN_SID'} = $ARGV[1];
  }
}

use strict;
use lib "$ENV{'TALLYMAN_PATH'}/lib";
use MILTON;
use DBLIB;

#
# -----------------------------------------------------------------------------
#

#
# set up our error handler
#

MILTON::set_error_handler(\&zb_fatal_error);

#
# parse command line options
#

use vars qw($all $rates $surcharge $zone $hohum);

$all=0;
$rates=0;
$surcharge=0;
$zone=0;

$hohum = $ARGV[3];
$hohum =~ tr/A-Z/a-z/;

if ($hohum eq "all") { $all = 1; }
elsif ($hohum eq "rates") { $rates = 1; }
elsif ($hohum eq "surcharge") { $surcharge = 1; }
elsif ($hohum eq "zone") { $zone = 1; }

if ($all==0 && $rates==0 && $surcharge==0 && $zone==0) {
  die "Invalid command line.\n";
}

#
# firstly, clean out the appropriate databases of any old files
#

&clean_out_database($all, $rates, $surcharge, $zone);

#
# ... and load the data
#

if ($rates || $all) {
  &load_rate_tables();
}

if ($surcharge || $all) {
  &load_surcharge_table();
}

if ($zone || $all) {
  &load_zone_table($ARGV[4]);
}

exit 0;

#
# -----------------------------------------------------------------------------
#
# expects filename xarea.csv
#

sub load_surcharge_table {
  my($sqlstr, @values, $val, );

  print "Loading surcharge areas (this may take a while)...\n";

  open(IN,"$ARGV[2]/xarea.csv")
    || &zb_fatal_error("$ARGV[2]/xarea.csv doesn't exist: $!");

  # skip the header

  while (<IN>) {
    $_ =~ s/\n//g;
    $_ =~ s/\r//g;
    if ($_ eq "") { last; }
  }
  <IN>;

  # process the data

  while (<IN>) {
    $_ =~ s/\n//g;
    $_ =~ s/\r//g;
    (@values) = split(/,/, $_);

    foreach $val (@values) {
      $sqlstr = "insert into zb_ups_shipping_surcharge values ('$val')";
      DBLIB::db_do($sqlstr);
    }
  }

  close(IN);
}


#
# -----------------------------------------------------------------------------
#
# Loads a set of rate tables.
#
# File names expected:
#   gndcomm.csv
#   1da.csv
#   1dasaver.csv
#   2da.csv
#   2dam.csv
#   3da.csv
#

sub load_rate_tables {
  my($val);
  my(@filelist) = ('gndcomm.csv',
                   '1da.csv',
                   '1dasaver.csv',
                   '2da.csv',
                   '2dam.csv',
                   '3ds.csv'); 

  print "Loading rates tables...\n";

  &prime_rates_database();

  foreach $val (@filelist) {
    &do_file("$ARGV[2]/$val");
  }
}

#
# -----------------------------------------------------------------------------
#
# primes the rates database by inserting 150 blank shipping entries.
#

sub prime_rates_database {
  my($sqlstr, $i);

  for ($i=1; $i<=150; $i++) {
    $sqlstr = "insert into zb_ups_shipping_rates values (
               '$i',
               '0','0','0','0','0','0','0',
               '0','0','0','0','0','0','0','0','0','0',
               '0','0','0','0','0','0','0',
               '0','0','0','0','0','0','0','0','0','0',
               '0','0','0','0','0','0','0',
               '0','0','0','0','0','0','0'
               )";

    DBLIB::db_do($sqlstr);
  }
}

#
# -----------------------------------------------------------------------------
#

sub do_file {
  my($fname) = shift;
  my($sqlstr, $i, $jed, @cols, $weight, @values);

  open(IN, $fname) || &zb_fatal_error("$fname doesn't exist: $!");

  # skip the header

  while (<IN>) {
    if ($_ =~ /Exceed/) {
      $_ =~ s/\n//g;
      $_ =~ s/\r//g;
      ($jed, @cols) = split(/,/, $_);
      last;
    }
  }

  # process the data

  DATALOOP: while (<IN>) {
    $_ =~ s/\n//g;
    $_ =~ s/\r//g;
    $_ =~ s/\$//g;

    ($weight, @values) = split(/,/, $_);
    if ($weight eq "Letter") { next DATALOOP; }

    $sqlstr = 'update zb_ups_shipping_rates set ';

    for ($i=0; $i<$#cols+1; $i++) {
      $sqlstr .= 'r' . $cols[$i] . '=' . $values[$i] . ', ';
    }
    chop $sqlstr;
    chop $sqlstr;

    $sqlstr .= ' where weight=' . $weight;

    DBLIB::db_do($sqlstr);
  }

  close(IN);
}

#
# -----------------------------------------------------------------------------
#
# Cleans out any old information in the database
# 

sub clean_out_database {
  my($do_all, $do_rates, $do_surcharge, $do_zone) = @_;
  my($sqlstr);

  print "Cleaning out database...\n";

  # clean out any old entries in the zb_ups_shipping_rates table

  if ($do_rates || $do_all) {
    $sqlstr = "delete from zb_ups_shipping_rates";
    DBLIB::db_do($sqlstr);
  }

  # clean out any old entries in the zb_ups_shipping_surcharge table

  if ($do_surcharge || $do_all) {
    $sqlstr = "delete from zb_ups_shipping_surcharge";
    DBLIB::db_do($sqlstr);
  }

  # clean out any old entries in the zb_ups_zone_chart and
  # zb_ups_weird_zone tables

  if ($do_zone || $do_all) {
    $sqlstr = "delete from zb_ups_zone_chart";
    DBLIB::db_do($sqlstr);

    $sqlstr = "delete from zb_ups_weird_zone";
    DBLIB::db_do($sqlstr);
  }
}

#
# -----------------------------------------------------------------------------
#

sub determine_zone_table {
  my($orig_zipcode) = shift;
  my($three_zc) = $orig_zipcode;
  my($two_zc) = $orig_zipcode;
  my($i);

  # calculate a three-digit zip code

  $three_zc =~ s/^(\d{3})(.*)$/$1/;

  # first, try xxx.csv

  if (open(IN,"$ARGV[2]/${three_zc}.csv")) { 
    close(IN);
    return("${three_zc}.csv");
  }

  # no luck?  try others.
  # iterate over files to determine which one applies

  $two_zc =~ s/^(\d{2})(.*)$/$1/;

  FILELOOP: for ($i=0; $i<10; $i++) {
    if (!open(IN,"$ARGV[2]/${two_zc}${i}.csv")) { next FILELOOP; }

    # what range of zipcodes does this file serve?

    while (<IN>) {
      if ($_ =~ /ZIP Codes (\d{3})-(\d{2}) to (\d{3})-(\d{2})/) {
        if ( ($orig_zipcode >= ("$1$2")) && ($orig_zipcode <= ("$3$4")) ) {
          close(IN);
          return("${two_zc}${i}.csv");
        }
      }
    }
    close(IN);
  }

  return("");
}

#
# -----------------------------------------------------------------------------
#

sub load_zone_table {
  my($origin_zone) = shift;
  my($sqlstr, $start, $end, $range, @values, $done, $fname, $i);
  my($nday, $sday, $val);

  print "Loading zone table...\n";

  $fname = &determine_zone_table($origin_zone);
  if ($fname eq "") {
    print "Could not determine which zone table applies to $origin_zone!\n";
  }

  $done = 0;
  while (!$done) {
    if ($fname eq "") {
      print "Please enter filename, or 'quit' to skip: ";
      $fname = <STDIN>;
      chomp $fname;
      if ($fname eq 'quit') { return; }
    }
    print "  filename is $fname\n";
 
    if (!open(IN,"$ARGV[2]/$fname")) {
      print "Could not open $ARGV[2]/$fname : $!";
      $fname = "";
    } else {
      last;
    }
  }

  # skip the header

  while (<IN>) {
    if ($_ =~ /ZONES/) {
      <IN>; <IN>;
      last;
    }
  }

  # process the data

  while (<IN>) {
    $_ =~ s/\n//g;
    $_ =~ s/\r//g;
    if ($_ eq "") { last; }

    ($range,@values)=split(/,/,$_);
    if ($range eq "") { last; }

    if ($range =~ /-/) {
      ($start,$end)=split(/-/,$range);
    } else {
      $start=$range;
      $end=$range;
    }

    # strip out things like '-', '[1]','[2]', and '[3]'

    for ($i=0; $i<$#values+1; $i++) {
      if ($values[$i] =~ /\D/) {
        $values[$i] = 0;
      }
    }

    $sqlstr = 'insert into zb_ups_zone_chart values (';
    $sqlstr .= join(',', ($start, $end, @values));
    $sqlstr .= ')';

    DBLIB::db_do($sqlstr);
  }

  #
  # now, process the footer.
  #

  while (<IN>) {
    if (/Zone (\d*).*Zone (\d*)/) {
      $nday = $1;
      $sday = $2;

      $_ = <IN>;
      $_ =~ s/\n//g;
      $_ =~ s/\r//g;
      (@values) = split(/,/,$_);
      foreach $val (@values) {
        $sqlstr = 'insert into zb_ups_weird_zone values (';
        $sqlstr .= join(',', $val, $nday, $sday);
        $sqlstr .= ')';
        DBLIB::db_do($sqlstr);
      }
    }
  }

  close(IN);
}

#
# -----------------------------------------------------------------------------
#

sub zb_fatal_error {
  my($msg) = shift;
  print STDERR "Fatal error: $msg\n";
  exit 1;
}
