﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/firehose/Firehose_EXPORTS.h>
#include <aws/firehose/model/CloudWatchLoggingOptions.h>
#include <aws/firehose/model/CopyCommand.h>
#include <aws/firehose/model/ProcessingConfiguration.h>
#include <aws/firehose/model/RedshiftRetryOptions.h>
#include <aws/firehose/model/RedshiftS3BackupMode.h>
#include <aws/firehose/model/S3DestinationDescription.h>
#include <aws/firehose/model/SecretsManagerConfiguration.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Firehose {
namespace Model {

/**
 * <p>Describes a destination in Amazon Redshift.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/RedshiftDestinationDescription">AWS
 * API Reference</a></p>
 */
class RedshiftDestinationDescription {
 public:
  AWS_FIREHOSE_API RedshiftDestinationDescription() = default;
  AWS_FIREHOSE_API RedshiftDestinationDescription(Aws::Utils::Json::JsonView jsonValue);
  AWS_FIREHOSE_API RedshiftDestinationDescription& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_FIREHOSE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Amazon Web Services credentials. For
   * more information, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Names (ARNs) and Amazon Web Services Service Namespaces</a>.</p>
   */
  inline const Aws::String& GetRoleARN() const { return m_roleARN; }
  inline bool RoleARNHasBeenSet() const { return m_roleARNHasBeenSet; }
  template <typename RoleARNT = Aws::String>
  void SetRoleARN(RoleARNT&& value) {
    m_roleARNHasBeenSet = true;
    m_roleARN = std::forward<RoleARNT>(value);
  }
  template <typename RoleARNT = Aws::String>
  RedshiftDestinationDescription& WithRoleARN(RoleARNT&& value) {
    SetRoleARN(std::forward<RoleARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The database connection string.</p>
   */
  inline const Aws::String& GetClusterJDBCURL() const { return m_clusterJDBCURL; }
  inline bool ClusterJDBCURLHasBeenSet() const { return m_clusterJDBCURLHasBeenSet; }
  template <typename ClusterJDBCURLT = Aws::String>
  void SetClusterJDBCURL(ClusterJDBCURLT&& value) {
    m_clusterJDBCURLHasBeenSet = true;
    m_clusterJDBCURL = std::forward<ClusterJDBCURLT>(value);
  }
  template <typename ClusterJDBCURLT = Aws::String>
  RedshiftDestinationDescription& WithClusterJDBCURL(ClusterJDBCURLT&& value) {
    SetClusterJDBCURL(std::forward<ClusterJDBCURLT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>COPY</code> command.</p>
   */
  inline const CopyCommand& GetCopyCommand() const { return m_copyCommand; }
  inline bool CopyCommandHasBeenSet() const { return m_copyCommandHasBeenSet; }
  template <typename CopyCommandT = CopyCommand>
  void SetCopyCommand(CopyCommandT&& value) {
    m_copyCommandHasBeenSet = true;
    m_copyCommand = std::forward<CopyCommandT>(value);
  }
  template <typename CopyCommandT = CopyCommand>
  RedshiftDestinationDescription& WithCopyCommand(CopyCommandT&& value) {
    SetCopyCommand(std::forward<CopyCommandT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the user.</p>
   */
  inline const Aws::String& GetUsername() const { return m_username; }
  inline bool UsernameHasBeenSet() const { return m_usernameHasBeenSet; }
  template <typename UsernameT = Aws::String>
  void SetUsername(UsernameT&& value) {
    m_usernameHasBeenSet = true;
    m_username = std::forward<UsernameT>(value);
  }
  template <typename UsernameT = Aws::String>
  RedshiftDestinationDescription& WithUsername(UsernameT&& value) {
    SetUsername(std::forward<UsernameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The retry behavior in case Firehose is unable to deliver documents to Amazon
   * Redshift. Default value is 3600 (60 minutes).</p>
   */
  inline const RedshiftRetryOptions& GetRetryOptions() const { return m_retryOptions; }
  inline bool RetryOptionsHasBeenSet() const { return m_retryOptionsHasBeenSet; }
  template <typename RetryOptionsT = RedshiftRetryOptions>
  void SetRetryOptions(RetryOptionsT&& value) {
    m_retryOptionsHasBeenSet = true;
    m_retryOptions = std::forward<RetryOptionsT>(value);
  }
  template <typename RetryOptionsT = RedshiftRetryOptions>
  RedshiftDestinationDescription& WithRetryOptions(RetryOptionsT&& value) {
    SetRetryOptions(std::forward<RetryOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon S3 destination.</p>
   */
  inline const S3DestinationDescription& GetS3DestinationDescription() const { return m_s3DestinationDescription; }
  inline bool S3DestinationDescriptionHasBeenSet() const { return m_s3DestinationDescriptionHasBeenSet; }
  template <typename S3DestinationDescriptionT = S3DestinationDescription>
  void SetS3DestinationDescription(S3DestinationDescriptionT&& value) {
    m_s3DestinationDescriptionHasBeenSet = true;
    m_s3DestinationDescription = std::forward<S3DestinationDescriptionT>(value);
  }
  template <typename S3DestinationDescriptionT = S3DestinationDescription>
  RedshiftDestinationDescription& WithS3DestinationDescription(S3DestinationDescriptionT&& value) {
    SetS3DestinationDescription(std::forward<S3DestinationDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data processing configuration.</p>
   */
  inline const ProcessingConfiguration& GetProcessingConfiguration() const { return m_processingConfiguration; }
  inline bool ProcessingConfigurationHasBeenSet() const { return m_processingConfigurationHasBeenSet; }
  template <typename ProcessingConfigurationT = ProcessingConfiguration>
  void SetProcessingConfiguration(ProcessingConfigurationT&& value) {
    m_processingConfigurationHasBeenSet = true;
    m_processingConfiguration = std::forward<ProcessingConfigurationT>(value);
  }
  template <typename ProcessingConfigurationT = ProcessingConfiguration>
  RedshiftDestinationDescription& WithProcessingConfiguration(ProcessingConfigurationT&& value) {
    SetProcessingConfiguration(std::forward<ProcessingConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon S3 backup mode.</p>
   */
  inline RedshiftS3BackupMode GetS3BackupMode() const { return m_s3BackupMode; }
  inline bool S3BackupModeHasBeenSet() const { return m_s3BackupModeHasBeenSet; }
  inline void SetS3BackupMode(RedshiftS3BackupMode value) {
    m_s3BackupModeHasBeenSet = true;
    m_s3BackupMode = value;
  }
  inline RedshiftDestinationDescription& WithS3BackupMode(RedshiftS3BackupMode value) {
    SetS3BackupMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration for backup in Amazon S3.</p>
   */
  inline const S3DestinationDescription& GetS3BackupDescription() const { return m_s3BackupDescription; }
  inline bool S3BackupDescriptionHasBeenSet() const { return m_s3BackupDescriptionHasBeenSet; }
  template <typename S3BackupDescriptionT = S3DestinationDescription>
  void SetS3BackupDescription(S3BackupDescriptionT&& value) {
    m_s3BackupDescriptionHasBeenSet = true;
    m_s3BackupDescription = std::forward<S3BackupDescriptionT>(value);
  }
  template <typename S3BackupDescriptionT = S3DestinationDescription>
  RedshiftDestinationDescription& WithS3BackupDescription(S3BackupDescriptionT&& value) {
    SetS3BackupDescription(std::forward<S3BackupDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon CloudWatch logging options for your Firehose stream.</p>
   */
  inline const CloudWatchLoggingOptions& GetCloudWatchLoggingOptions() const { return m_cloudWatchLoggingOptions; }
  inline bool CloudWatchLoggingOptionsHasBeenSet() const { return m_cloudWatchLoggingOptionsHasBeenSet; }
  template <typename CloudWatchLoggingOptionsT = CloudWatchLoggingOptions>
  void SetCloudWatchLoggingOptions(CloudWatchLoggingOptionsT&& value) {
    m_cloudWatchLoggingOptionsHasBeenSet = true;
    m_cloudWatchLoggingOptions = std::forward<CloudWatchLoggingOptionsT>(value);
  }
  template <typename CloudWatchLoggingOptionsT = CloudWatchLoggingOptions>
  RedshiftDestinationDescription& WithCloudWatchLoggingOptions(CloudWatchLoggingOptionsT&& value) {
    SetCloudWatchLoggingOptions(std::forward<CloudWatchLoggingOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The configuration that defines how you access secrets for Amazon Redshift.
   * </p>
   */
  inline const SecretsManagerConfiguration& GetSecretsManagerConfiguration() const { return m_secretsManagerConfiguration; }
  inline bool SecretsManagerConfigurationHasBeenSet() const { return m_secretsManagerConfigurationHasBeenSet; }
  template <typename SecretsManagerConfigurationT = SecretsManagerConfiguration>
  void SetSecretsManagerConfiguration(SecretsManagerConfigurationT&& value) {
    m_secretsManagerConfigurationHasBeenSet = true;
    m_secretsManagerConfiguration = std::forward<SecretsManagerConfigurationT>(value);
  }
  template <typename SecretsManagerConfigurationT = SecretsManagerConfiguration>
  RedshiftDestinationDescription& WithSecretsManagerConfiguration(SecretsManagerConfigurationT&& value) {
    SetSecretsManagerConfiguration(std::forward<SecretsManagerConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_roleARN;
  bool m_roleARNHasBeenSet = false;

  Aws::String m_clusterJDBCURL;
  bool m_clusterJDBCURLHasBeenSet = false;

  CopyCommand m_copyCommand;
  bool m_copyCommandHasBeenSet = false;

  Aws::String m_username;
  bool m_usernameHasBeenSet = false;

  RedshiftRetryOptions m_retryOptions;
  bool m_retryOptionsHasBeenSet = false;

  S3DestinationDescription m_s3DestinationDescription;
  bool m_s3DestinationDescriptionHasBeenSet = false;

  ProcessingConfiguration m_processingConfiguration;
  bool m_processingConfigurationHasBeenSet = false;

  RedshiftS3BackupMode m_s3BackupMode{RedshiftS3BackupMode::NOT_SET};
  bool m_s3BackupModeHasBeenSet = false;

  S3DestinationDescription m_s3BackupDescription;
  bool m_s3BackupDescriptionHasBeenSet = false;

  CloudWatchLoggingOptions m_cloudWatchLoggingOptions;
  bool m_cloudWatchLoggingOptionsHasBeenSet = false;

  SecretsManagerConfiguration m_secretsManagerConfiguration;
  bool m_secretsManagerConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace Firehose
}  // namespace Aws
