/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.january.dataset;

import java.io.Serializable;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import org.apache.commons.math3.util.MathArrays;
import org.eclipse.january.DatasetException;
import org.eclipse.january.dataset.AbstractCompoundDataset;
import org.eclipse.january.dataset.AbstractDataset;
import org.eclipse.january.dataset.BooleanDataset;
import org.eclipse.january.dataset.BroadcastPairIterator;
import org.eclipse.january.dataset.BroadcastUtils;
import org.eclipse.january.dataset.ByteDataset;
import org.eclipse.january.dataset.ComplexDoubleDataset;
import org.eclipse.january.dataset.ComplexFloatDataset;
import org.eclipse.january.dataset.CompoundByteDataset;
import org.eclipse.january.dataset.CompoundDataset;
import org.eclipse.january.dataset.CompoundDoubleDataset;
import org.eclipse.january.dataset.CompoundFloatDataset;
import org.eclipse.january.dataset.CompoundIntegerDataset;
import org.eclipse.january.dataset.CompoundLongDataset;
import org.eclipse.january.dataset.CompoundShortDataset;
import org.eclipse.january.dataset.DTypeUtils;
import org.eclipse.january.dataset.Dataset;
import org.eclipse.january.dataset.DatasetFactory;
import org.eclipse.january.dataset.DoubleDataset;
import org.eclipse.january.dataset.FloatDataset;
import org.eclipse.january.dataset.IDataset;
import org.eclipse.january.dataset.ILazyDataset;
import org.eclipse.january.dataset.IndexIterator;
import org.eclipse.january.dataset.IntegerDataset;
import org.eclipse.january.dataset.LongDataset;
import org.eclipse.january.dataset.Maths;
import org.eclipse.january.dataset.PositionIterator;
import org.eclipse.january.dataset.RGBDataset;
import org.eclipse.january.dataset.ShapeUtils;
import org.eclipse.january.dataset.ShortDataset;
import org.eclipse.january.dataset.Slice;
import org.eclipse.january.dataset.SliceIterator;
import org.eclipse.january.dataset.SliceND;
import org.eclipse.january.dataset.SliceNDIterator;
import org.eclipse.january.dataset.StringDataset;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatasetUtils {
    protected static final transient Logger utilsLogger = LoggerFactory.getLogger(DatasetUtils.class);

    public static Dataset append(IDataset a, IDataset b, int axis) {
        int[] bshape;
        int[] ashape = a.getShape();
        int rank = ashape.length;
        if (rank != (bshape = b.getShape()).length) {
            throw new IllegalArgumentException("Incompatible number of dimensions");
        }
        axis = ShapeUtils.checkAxis(rank, axis);
        int i = 0;
        while (i < rank) {
            if (i != axis && ashape[i] != bshape[i]) {
                throw new IllegalArgumentException("Incompatible dimensions");
            }
            ++i;
        }
        int[] nshape = new int[rank];
        int i2 = 0;
        while (i2 < rank) {
            nshape[i2] = ashape[i2];
            ++i2;
        }
        int n = axis;
        nshape[n] = nshape[n] + bshape[axis];
        int ot = DTypeUtils.getDType(b);
        int dt = DTypeUtils.getDType(a);
        Dataset ds = DatasetFactory.zeros(a.getElementsPerItem(), nshape, dt > ot ? dt : ot);
        IndexIterator iter = ds.getIterator(true);
        int[] pos = iter.getPos();
        while (iter.hasNext()) {
            int d = ashape[axis];
            if (pos[axis] < d) {
                ds.setObjectAbs(iter.index, a.getObject(pos));
                continue;
            }
            int n2 = axis;
            pos[n2] = pos[n2] - d;
            ds.setObjectAbs(iter.index, b.getObject(pos));
            int n3 = axis;
            pos[n3] = pos[n3] + d;
        }
        return ds;
    }

    public static <T extends Dataset> T put(T a, Dataset indices, Object values) {
        IndexIterator it = indices.getIterator();
        Dataset vd = DatasetFactory.createFromObject(values).flatten();
        int vlen = vd.getSize();
        int v = 0;
        while (it.hasNext()) {
            if (v >= vlen) {
                v -= vlen;
            }
            a.setObjectAbs((int)indices.getElementLongAbs(it.index), vd.getObjectAbs(v++));
        }
        return a;
    }

    public static <T extends Dataset> T put(T a, int[] indices, Object values) {
        int ilen = indices.length;
        Dataset vd = DatasetFactory.createFromObject(values).flatten();
        int vlen = vd.getSize();
        int i = 0;
        int v = 0;
        while (i < ilen) {
            if (v >= vlen) {
                v -= vlen;
            }
            a.setObjectAbs(indices[i], vd.getObjectAbs(v++));
            ++i;
        }
        return a;
    }

    public static <T extends Dataset> T take(T a, Dataset indices, Integer axis) {
        IntegerDataset indexes = (IntegerDataset)indices.flatten().cast(3);
        return DatasetUtils.take(a, indexes.getData(), axis);
    }

    public static <T extends Dataset> T take(T a, int[] indices, Integer axis) {
        Dataset result;
        if (indices == null || indices.length == 0) {
            utilsLogger.error("No indices given");
            throw new IllegalArgumentException("No indices given");
        }
        int[] ashape = a.getShape();
        int rank = ashape.length;
        int at = a.getDType();
        int ilen = indices.length;
        int is = a.getElementsPerItem();
        if (axis == null) {
            ashape = new int[]{ilen};
            result = DatasetFactory.zeros(is, ashape, at);
            Serializable src = a.getBuffer();
            int i = 0;
            while (i < ilen) {
                ((AbstractDataset)result).setItemDirect(i, indices[i], src);
                ++i;
            }
        } else {
            axis = a.checkAxis(axis);
            ashape[axis.intValue()] = ilen;
            result = DatasetFactory.zeros(is, ashape, at);
            int[] dpos = new int[rank];
            int[] spos = new int[rank];
            boolean[] axes = new boolean[rank];
            Arrays.fill(axes, true);
            axes[axis.intValue()] = false;
            Serializable src = a.getBuffer();
            int i = 0;
            while (i < ilen) {
                spos[axis.intValue()] = indices[i];
                dpos[axis.intValue()] = i;
                SliceIterator siter = a.getSliceIteratorFromAxes(spos, axes);
                SliceIterator diter = result.getSliceIteratorFromAxes(dpos, axes);
                while (siter.hasNext() && diter.hasNext()) {
                    ((AbstractDataset)result).setItemDirect(diter.index, siter.index, src);
                }
                ++i;
            }
        }
        result.setDirty();
        return (T)result;
    }

    public static Dataset tile(IDataset a, int ... reps) {
        boolean manyColumns;
        int i;
        int extraRank;
        int[] newShape;
        int rlen;
        int[] shape = a.getShape();
        int rank = shape.length;
        if (rank < (rlen = reps.length)) {
            newShape = new int[rlen];
            extraRank = rlen - rank;
            i = 0;
            while (i < extraRank) {
                newShape[i] = 1;
                ++i;
            }
            i = 0;
            while (i < rank) {
                newShape[i + extraRank] = shape[i];
                ++i;
            }
            shape = newShape;
            rank = rlen;
        } else if (rank > rlen) {
            int[] newReps = new int[rank];
            extraRank = rank - rlen;
            i = 0;
            while (i < extraRank) {
                newReps[i] = 1;
                ++i;
            }
            i = 0;
            while (i < rlen) {
                newReps[i + extraRank] = reps[i];
                ++i;
            }
            reps = newReps;
        }
        newShape = new int[rank];
        int i2 = 0;
        while (i2 < rank) {
            newShape[i2] = shape[i2] * reps[i2];
            ++i2;
        }
        Dataset tdata = DatasetFactory.zeros(a.getElementsPerItem(), newShape, DTypeUtils.getDType(a));
        if (rank == 1) {
            manyColumns = true;
        } else {
            boolean bl = manyColumns = shape[rank - 1] > 64;
        }
        if (manyColumns) {
            IndexIterator iter = tdata.getSliceIterator(null, null, shape);
            SliceIterator siter = (SliceIterator)tdata.getSliceIterator(null, shape, null);
            int[] pos = iter.getPos();
            while (iter.hasNext()) {
                siter.setStart(pos);
                tdata.setSlice((Object)a, siter);
            }
        } else {
            int[] skip = new int[rank];
            int i3 = 0;
            while (i3 < rank) {
                skip[i3] = reps[i3] == 1 ? newShape[i3] : shape[i3];
                ++i3;
            }
            Dataset aa = DatasetUtils.convertToDataset(a);
            IndexIterator ita = aa.getIterator(true);
            int[] pos = ita.getPos();
            int[] sstart = new int[rank];
            int extra = rank - pos.length;
            int i4 = 0;
            while (i4 < extra) {
                sstart[i4] = 0;
                ++i4;
            }
            SliceIterator siter = (SliceIterator)tdata.getSliceIterator(sstart, null, skip);
            while (ita.hasNext()) {
                int i5 = 0;
                while (i5 < pos.length) {
                    sstart[i5 + extra] = pos[i5];
                    ++i5;
                }
                siter.setStart(sstart);
                tdata.setSlice(aa.getObjectAbs(ita.index), siter);
            }
        }
        return tdata;
    }

    public static Dataset transpose(IDataset a, int ... axes) {
        return DatasetUtils.convertToDataset(a).transpose(axes);
    }

    public static Dataset swapAxes(IDataset a, int axis1, int axis2) {
        return DatasetUtils.convertToDataset(a).swapAxes(axis1, axis2);
    }

    public static <T extends Dataset> T sort(T a) {
        return DatasetUtils.sort(a, null);
    }

    public static <T extends Dataset> T sort(T a, Integer axis) {
        Dataset s = a.clone();
        return (T)s.sort(axis);
    }

    public static void sort(Dataset a, Dataset ... b) {
        if (!DTypeUtils.isDTypeNumerical(a.getDType())) {
            throw new UnsupportedOperationException("Sorting non-numerical datasets not supported yet");
        }
        DoubleDataset s = DatasetUtils.copy(DoubleDataset.class, a);
        int l = b == null ? 0 : b.length;
        DoubleDataset[] t = new DoubleDataset[l];
        int n = 0;
        int i = 0;
        while (i < l) {
            if (b[i] != null) {
                if (!DTypeUtils.isDTypeNumerical(b[i].getDType())) {
                    throw new UnsupportedOperationException("Sorting non-numerical datasets not supported yet");
                }
                t[i] = DatasetUtils.copy(DoubleDataset.class, b[i]);
                ++n;
            }
            ++i;
        }
        double[][] y = new double[n][];
        int i2 = 0;
        int j = 0;
        while (i2 < l) {
            if (t[i2] != null) {
                y[j++] = t[i2].getData();
            }
            ++i2;
        }
        MathArrays.sortInPlace((double[])s.getData(), (double[][])y);
        a.setSlice((Object)s, new Slice[0]);
        i2 = 0;
        while (i2 < l) {
            if (b[i2] != null) {
                b[i2].setSlice((Object)t[i2], new Slice[0]);
            }
            ++i2;
        }
    }

    public static IntegerDataset indexSort(Dataset a, Integer axis) {
        if (axis == null) {
            int size = a.getSize();
            Integer[] index = new Integer[size];
            int i = 0;
            while (i < size) {
                index[i] = i;
                ++i;
            }
            final Dataset f = a.flatten();
            Comparator<Integer> cmp = new Comparator<Integer>(){

                @Override
                public int compare(Integer o1, Integer o2) {
                    return Double.compare(f.getElementDoubleAbs(o1), f.getElementDoubleAbs(o2));
                }
            };
            Arrays.sort(index, cmp);
            return DatasetFactory.createFromObject(IntegerDataset.class, (Object)index);
        }
        axis = a.checkAxis(axis);
        int[] shape = a.getShapeRef();
        IntegerDataset id = DatasetFactory.zeros(IntegerDataset.class, shape);
        int size = shape[axis];
        Integer[] index = new Integer[size];
        int[] dShape = new int[shape.length];
        Arrays.fill(dShape, 1);
        dShape[axis.intValue()] = size;
        final DoubleDataset dd = DatasetFactory.zeros(DoubleDataset.class, dShape);
        Comparator<Integer> cmp = new Comparator<Integer>(){

            @Override
            public int compare(Integer o1, Integer o2) {
                return Double.compare(dd.getElementDoubleAbs(o1), dd.getElementDoubleAbs(o2));
            }
        };
        SliceND ds = new SliceND(dShape);
        SliceNDIterator it = new SliceNDIterator(new SliceND(shape), axis);
        int[] pos = it.getPos();
        int[] ipos = (int[])pos.clone();
        while (it.hasNext()) {
            dd.setSlice((Object)a.getSliceView(it.getCurrentSlice()), ds);
            int i = 0;
            while (i < size) {
                index[i] = i;
                ++i;
            }
            Arrays.sort(index, cmp);
            System.arraycopy(pos, 0, ipos, 0, pos.length);
            i = 0;
            while (i < size) {
                ipos[axis.intValue()] = i;
                id.set((Object)index[i], ipos);
                ++i;
            }
        }
        return id;
    }

    public static Dataset concatenate(IDataset[] as, int axis) {
        if (as == null || as.length == 0) {
            utilsLogger.error("No datasets given");
            throw new IllegalArgumentException("No datasets given");
        }
        IDataset a = as[0];
        if (as.length == 1) {
            return DatasetUtils.convertToDataset(a.clone());
        }
        int[] ashape = a.getShape();
        axis = ShapeUtils.checkAxis(ashape.length, axis);
        int at = DTypeUtils.getDType(a);
        int anum = as.length;
        int isize = a.getElementsPerItem();
        int i = 1;
        while (i < anum) {
            if (at != DTypeUtils.getDType(as[i])) {
                utilsLogger.error("Datasets are not of same type");
                break;
            }
            if (!ShapeUtils.areShapesCompatible(ashape, as[i].getShape(), axis)) {
                utilsLogger.error("Datasets' shapes are not equal");
                break;
            }
            int is = as[i].getElementsPerItem();
            if (isize < is) {
                isize = is;
            }
            ++i;
        }
        if (i < anum) {
            utilsLogger.error("Dataset are not compatible");
            throw new IllegalArgumentException("Datasets are not compatible");
        }
        i = 1;
        while (i < anum) {
            int n = axis;
            ashape[n] = ashape[n] + as[i].getShape()[axis];
            ++i;
        }
        Dataset result = DatasetFactory.zeros(isize, ashape, at);
        int[] start = new int[ashape.length];
        int[] stop = ashape;
        stop[axis] = 0;
        i = 0;
        while (i < anum) {
            IDataset b = as[i];
            int[] bshape = b.getShape();
            int n = axis;
            stop[n] = stop[n] + bshape[axis];
            result.setSlice((Object)b, start, stop, null);
            int n2 = axis;
            start[n2] = start[n2] + bshape[axis];
            ++i;
        }
        return result;
    }

    public static List<Dataset> split(Dataset a, int sections, int axis, boolean checkEqual) {
        int[] ashape = a.getShapeRef();
        axis = a.checkAxis(axis);
        int imax = ashape[axis];
        if (checkEqual && imax % sections != 0) {
            utilsLogger.error("Number of sections does not divide axis into equal parts");
            throw new IllegalArgumentException("Number of sections does not divide axis into equal parts");
        }
        int n = (imax + sections - 1) / sections;
        int[] indices = new int[sections - 1];
        int i = 1;
        while (i < sections) {
            indices[i - 1] = n * i;
            ++i;
        }
        return DatasetUtils.split(a, indices, axis);
    }

    public static List<Dataset> split(Dataset a, int[] indices, int axis) {
        int[] ashape = a.getShapeRef();
        axis = a.checkAxis(axis);
        int rank = ashape.length;
        int imax = ashape[axis];
        ArrayList<Dataset> result = new ArrayList<Dataset>();
        int[] nshape = (int[])ashape.clone();
        int is = a.getElementsPerItem();
        int oind = 0;
        int[] start = new int[rank];
        int[] stop = new int[rank];
        int[] step = new int[rank];
        int i = 0;
        while (i < rank) {
            start[i] = 0;
            stop[i] = ashape[i];
            step[i] = 1;
            ++i;
        }
        int[] nArray = indices;
        int n = indices.length;
        int n2 = 0;
        while (n2 < n) {
            int ind = nArray[n2];
            if (ind > imax) {
                result.add((Dataset)DatasetFactory.zeros(is, a.getClass(), 0));
            } else {
                nshape[axis] = ind - oind;
                start[axis] = oind;
                stop[axis] = ind;
                Object n3 = DatasetFactory.zeros(is, a.getClass(), nshape);
                IndexIterator iter = a.getSliceIterator(start, stop, step);
                a.fillDataset((Dataset)n3, iter);
                result.add((Dataset)n3);
                oind = ind;
            }
            ++n2;
        }
        if (imax > oind) {
            nshape[axis] = imax - oind;
            start[axis] = oind;
            stop[axis] = imax;
            Object n4 = DatasetFactory.zeros(is, a.getClass(), nshape);
            IndexIterator iter = a.getSliceIterator(start, stop, step);
            a.fillDataset((Dataset)n4, iter);
            result.add((Dataset)n4);
        }
        return result;
    }

    public static <T extends Dataset> T repeat(T a, int[] repeats, int axis) {
        int alen;
        Serializable buf = a.getBuffer();
        int[] shape = a.getShape();
        int rank = shape.length;
        int is = a.getElementsPerItem();
        if (axis >= rank) {
            utilsLogger.warn("Axis value is out of bounds");
            throw new IllegalArgumentException("Axis value is out of bounds");
        }
        if (axis < 0) {
            alen = a.getSize();
            axis = 0;
            rank = 1;
            shape[0] = alen;
        } else {
            alen = shape[axis];
        }
        int rlen = repeats.length;
        if (rlen != 1 && rlen != alen) {
            utilsLogger.warn("Repeats array should have length of 1 or match chosen axis");
            throw new IllegalArgumentException("Repeats array should have length of 1 or match chosen axis");
        }
        int i = 0;
        while (i < rlen) {
            if (repeats[i] < 0) {
                utilsLogger.warn("Negative repeat value is not allowed");
                throw new IllegalArgumentException("Negative repeat value is not allowed");
            }
            ++i;
        }
        int[] newShape = new int[rank];
        int i2 = 0;
        while (i2 < rank) {
            newShape[i2] = shape[i2];
            ++i2;
        }
        if (repeats.length == 1) {
            int n = axis;
            newShape[n] = newShape[n] * repeats[0];
        } else {
            int nlen = 0;
            int i3 = 0;
            while (i3 < alen) {
                nlen += repeats[i3];
                ++i3;
            }
            newShape[axis] = nlen;
        }
        Dataset rdata = DatasetFactory.zeros(is, newShape, a.getDType());
        Serializable nbuf = rdata.getBuffer();
        int csize = is;
        int i4 = axis + 1;
        while (i4 < rank) {
            csize *= newShape[i4];
            ++i4;
        }
        int nout = 1;
        int i5 = 0;
        while (i5 < axis) {
            nout *= newShape[i5];
            ++i5;
        }
        int oi = 0;
        int ni = 0;
        if (rlen == 1) {
            int i6 = 0;
            while (i6 < nout) {
                int j = 0;
                while (j < shape[axis]) {
                    int k = 0;
                    while (k < repeats[0]) {
                        System.arraycopy(buf, oi, nbuf, ni, csize);
                        ni += csize;
                        ++k;
                    }
                    oi += csize;
                    ++j;
                }
                ++i6;
            }
        } else {
            int i7 = 0;
            while (i7 < nout) {
                int j = 0;
                while (j < shape[axis]) {
                    int k = 0;
                    while (k < repeats[j]) {
                        System.arraycopy(buf, oi, nbuf, ni, csize);
                        ni += csize;
                        ++k;
                    }
                    oi += csize;
                    ++j;
                }
                ++i7;
            }
        }
        return (T)rdata;
    }

    public static <T extends Dataset> T resize(T a, int ... shape) {
        int size = a.getSize();
        Dataset rdata = DatasetFactory.zeros(a.getElementsPerItem(), shape, a.getDType());
        IndexIterator it = rdata.getIterator();
        while (it.hasNext()) {
            rdata.setObjectAbs(it.index, a.getObjectAbs(it.index % size));
        }
        return (T)rdata;
    }

    public static Dataset copy(IDataset d, int dtype) {
        Dataset a = DatasetUtils.convertToDataset(d);
        AbstractDataset c = null;
        try {
            switch (dtype) {
                case 9: {
                    c = new StringDataset(a);
                    break;
                }
                case 0: {
                    c = new BooleanDataset(a);
                    break;
                }
                case 1: {
                    if (a instanceof CompoundDataset) {
                        c = new CompoundByteDataset(a);
                        break;
                    }
                    c = new ByteDataset(a);
                    break;
                }
                case 2: {
                    if (a instanceof CompoundDataset) {
                        c = new CompoundShortDataset(a);
                        break;
                    }
                    c = new ShortDataset(a);
                    break;
                }
                case 3: {
                    if (a instanceof CompoundDataset) {
                        c = new CompoundIntegerDataset(a);
                        break;
                    }
                    c = new IntegerDataset(a);
                    break;
                }
                case 4: {
                    if (a instanceof CompoundDataset) {
                        c = new CompoundLongDataset(a);
                        break;
                    }
                    c = new LongDataset(a);
                    break;
                }
                case 100: {
                    if (a instanceof CompoundDataset) {
                        c = new CompoundByteDataset((CompoundDataset)a);
                        break;
                    }
                    c = new CompoundByteDataset(a);
                    break;
                }
                case 200: {
                    if (a instanceof CompoundDataset) {
                        c = new CompoundShortDataset((CompoundDataset)a);
                        break;
                    }
                    c = new CompoundShortDataset(a);
                    break;
                }
                case 300: {
                    if (a instanceof CompoundDataset) {
                        c = new CompoundIntegerDataset((CompoundDataset)a);
                        break;
                    }
                    c = new CompoundIntegerDataset(a);
                    break;
                }
                case 400: {
                    if (a instanceof CompoundDataset) {
                        c = new CompoundLongDataset((CompoundDataset)a);
                        break;
                    }
                    c = new CompoundLongDataset(a);
                    break;
                }
                case 5: {
                    c = new FloatDataset(a);
                    break;
                }
                case 6: {
                    c = new DoubleDataset(a);
                    break;
                }
                case 500: {
                    if (a instanceof CompoundDataset) {
                        c = new CompoundFloatDataset((CompoundDataset)a);
                        break;
                    }
                    c = new CompoundFloatDataset(a);
                    break;
                }
                case 600: {
                    if (a instanceof CompoundDataset) {
                        c = new CompoundDoubleDataset((CompoundDataset)a);
                        break;
                    }
                    c = new CompoundDoubleDataset(a);
                    break;
                }
                case 7: {
                    c = new ComplexFloatDataset(a);
                    break;
                }
                case 8: {
                    c = new ComplexDoubleDataset(a);
                    break;
                }
                case 203: {
                    if (a instanceof CompoundDataset) {
                        c = RGBDataset.createFromCompoundDataset((CompoundDataset)a);
                        break;
                    }
                    c = new RGBDataset(a);
                    break;
                }
                default: {
                    utilsLogger.error("Dataset of unknown type!");
                    break;
                }
            }
        }
        catch (OutOfMemoryError outOfMemoryError) {
            utilsLogger.error("Not enough memory available to create dataset");
            throw new OutOfMemoryError("Not enough memory available to create dataset");
        }
        return c;
    }

    public static <T extends Dataset> T copy(Class<T> clazz, IDataset d) {
        return (T)DatasetUtils.copy(d, DTypeUtils.getDType(clazz));
    }

    public static Dataset cast(IDataset d, int dtype) {
        Dataset a = DatasetUtils.convertToDataset(d);
        if (a.getDType() == dtype) {
            return a;
        }
        return DatasetUtils.copy(d, dtype);
    }

    public static <T extends Dataset> T cast(Class<T> clazz, IDataset d) {
        return (T)DatasetUtils.cast(d, DTypeUtils.getDType(clazz));
    }

    public static Dataset cast(IDataset d, boolean repeat, int dtype, int isize) {
        Dataset a = DatasetUtils.convertToDataset(d);
        if (a.getDType() == dtype && a.getElementsPerItem() == isize) {
            return a;
        }
        if (isize <= 0) {
            utilsLogger.error("Item size is invalid (>0)");
            throw new IllegalArgumentException("Item size is invalid (>0)");
        }
        if (isize > 1 && dtype <= 6) {
            utilsLogger.error("Item size is inconsistent with dataset type");
            throw new IllegalArgumentException("Item size is inconsistent with dataset type");
        }
        AbstractDataset c = null;
        try {
            switch (dtype) {
                case 0: {
                    c = new BooleanDataset(a);
                    break;
                }
                case 1: {
                    c = new ByteDataset(a);
                    break;
                }
                case 2: {
                    c = new ShortDataset(a);
                    break;
                }
                case 3: {
                    c = new IntegerDataset(a);
                    break;
                }
                case 4: {
                    c = new LongDataset(a);
                    break;
                }
                case 100: {
                    c = new CompoundByteDataset(isize, repeat, a);
                    break;
                }
                case 200: {
                    c = new CompoundShortDataset(isize, repeat, a);
                    break;
                }
                case 300: {
                    c = new CompoundIntegerDataset(isize, repeat, a);
                    break;
                }
                case 400: {
                    c = new CompoundLongDataset(isize, repeat, a);
                    break;
                }
                case 5: {
                    c = new FloatDataset(a);
                    break;
                }
                case 6: {
                    c = new DoubleDataset(a);
                    break;
                }
                case 500: {
                    c = new CompoundFloatDataset(isize, repeat, a);
                    break;
                }
                case 600: {
                    c = new CompoundDoubleDataset(isize, repeat, a);
                    break;
                }
                case 7: {
                    c = new ComplexFloatDataset(a);
                    break;
                }
                case 8: {
                    c = new ComplexDoubleDataset(a);
                    break;
                }
                default: {
                    utilsLogger.error("Dataset of unknown type!");
                    break;
                }
            }
        }
        catch (OutOfMemoryError outOfMemoryError) {
            utilsLogger.error("Not enough memory available to create dataset");
            throw new OutOfMemoryError("Not enough memory available to create dataset");
        }
        return c;
    }

    public static CompoundDataset cast(Dataset[] a, int dtype) {
        AbstractCompoundDataset c = null;
        switch (dtype) {
            case 1: 
            case 100: {
                c = new CompoundByteDataset(a);
                break;
            }
            case 2: 
            case 200: {
                c = new CompoundShortDataset(a);
                break;
            }
            case 3: 
            case 300: {
                c = new CompoundIntegerDataset(a);
                break;
            }
            case 4: 
            case 400: {
                c = new CompoundLongDataset(a);
                break;
            }
            case 5: 
            case 500: {
                c = new CompoundFloatDataset(a);
                break;
            }
            case 6: 
            case 600: {
                c = new CompoundDoubleDataset(a);
                break;
            }
            case 7: {
                if (a.length != 2) {
                    throw new IllegalArgumentException("Need two datasets for complex dataset type");
                }
                c = new ComplexFloatDataset(a[0], a[1]);
                break;
            }
            case 8: {
                if (a.length != 2) {
                    throw new IllegalArgumentException("Need two datasets for complex dataset type");
                }
                c = new ComplexDoubleDataset(a[0], a[1]);
                break;
            }
            default: {
                utilsLogger.error("Dataset of unsupported type!");
            }
        }
        return c;
    }

    public static Dataset makeUnsigned(IDataset a) {
        return DatasetUtils.makeUnsigned(a, false);
    }

    public static Dataset makeUnsigned(IDataset a, boolean check) {
        Dataset d = DatasetUtils.convertToDataset(a);
        if (d.hasFloatingPointElements()) {
            return d;
        }
        if (check && d.min(true).longValue() >= 0L) {
            return d;
        }
        int dtype = d.getDType();
        switch (dtype) {
            case 3: {
                d = new LongDataset(d);
                DatasetUtils.unwrapUnsigned(d, 32);
                break;
            }
            case 2: {
                d = new IntegerDataset(d);
                DatasetUtils.unwrapUnsigned(d, 16);
                break;
            }
            case 1: {
                d = new ShortDataset(d);
                DatasetUtils.unwrapUnsigned(d, 8);
                break;
            }
            case 300: {
                d = new CompoundLongDataset(d);
                DatasetUtils.unwrapUnsigned(d, 32);
                break;
            }
            case 200: {
                d = new CompoundIntegerDataset(d);
                DatasetUtils.unwrapUnsigned(d, 16);
                break;
            }
            case 100: {
                d = new CompoundShortDataset(d);
                DatasetUtils.unwrapUnsigned(d, 8);
            }
        }
        return d;
    }

    public static void unwrapUnsigned(Dataset a, int bitWidth) {
        int dtype = a.getDType();
        double dv = 1L << bitWidth;
        int isize = a.getElementsPerItem();
        IndexIterator it = a.getIterator();
        switch (dtype) {
            case 0: {
                break;
            }
            case 1: {
                break;
            }
            case 2: {
                ShortDataset sds = (ShortDataset)a;
                short soffset = (short)dv;
                while (it.hasNext()) {
                    short x = sds.getAbs(it.index);
                    if (x >= 0) continue;
                    sds.setAbs(it.index, (short)(x + soffset));
                }
                break;
            }
            case 3: {
                IntegerDataset ids = (IntegerDataset)a;
                int ioffset = (int)dv;
                while (it.hasNext()) {
                    int x = ids.getAbs(it.index);
                    if (x >= 0) continue;
                    ids.setAbs(it.index, x + ioffset);
                }
                break;
            }
            case 4: {
                LongDataset lds = (LongDataset)a;
                long loffset = (long)dv;
                while (it.hasNext()) {
                    long x = lds.getAbs(it.index);
                    if (x >= 0L) continue;
                    lds.setAbs(it.index, x + loffset);
                }
                break;
            }
            case 5: {
                FloatDataset fds = (FloatDataset)a;
                float foffset = (float)dv;
                while (it.hasNext()) {
                    float x = fds.getAbs(it.index);
                    if (!(x < 0.0f)) continue;
                    fds.setAbs(it.index, x + foffset);
                }
                break;
            }
            case 6: {
                DoubleDataset dds = (DoubleDataset)a;
                double doffset = dv;
                while (it.hasNext()) {
                    double x = dds.getAbs(it.index);
                    if (!(x < 0.0)) continue;
                    dds.setAbs(it.index, x + doffset);
                }
                break;
            }
            case 100: {
                break;
            }
            case 200: {
                CompoundShortDataset csds = (CompoundShortDataset)a;
                short csoffset = (short)dv;
                short[] csa = new short[isize];
                while (it.hasNext()) {
                    csds.getAbs(it.index, csa);
                    boolean dirty = false;
                    int i = 0;
                    while (i < isize) {
                        short x = csa[i];
                        if (x < 0) {
                            csa[i] = (short)(x + csoffset);
                            dirty = true;
                        }
                        ++i;
                    }
                    if (!dirty) continue;
                    csds.setAbs(it.index, csa);
                }
                break;
            }
            case 300: {
                CompoundIntegerDataset cids = (CompoundIntegerDataset)a;
                int cioffset = (int)dv;
                int[] cia = new int[isize];
                while (it.hasNext()) {
                    cids.getAbs(it.index, cia);
                    boolean dirty = false;
                    int i = 0;
                    while (i < isize) {
                        int x = cia[i];
                        if (x < 0) {
                            cia[i] = x + cioffset;
                            dirty = true;
                        }
                        ++i;
                    }
                    if (!dirty) continue;
                    cids.setAbs(it.index, cia);
                }
                break;
            }
            case 400: {
                CompoundLongDataset clds = (CompoundLongDataset)a;
                long cloffset = (long)dv;
                long[] cla = new long[isize];
                while (it.hasNext()) {
                    clds.getAbs(it.index, cla);
                    boolean dirty = false;
                    int i = 0;
                    while (i < isize) {
                        long x = cla[i];
                        if (x < 0L) {
                            cla[i] = x + cloffset;
                            dirty = true;
                        }
                        ++i;
                    }
                    if (!dirty) continue;
                    clds.setAbs(it.index, cla);
                }
                break;
            }
            default: {
                utilsLogger.error("Dataset of unsupported type for this method");
            }
        }
    }

    public static Dataset eye(int rows, int cols, int offset, int dtype) {
        int[] shape = new int[]{rows, cols};
        Dataset a = DatasetFactory.zeros(shape, dtype);
        int[] nArray = new int[2];
        nArray[1] = offset;
        int[] pos = nArray;
        while (pos[1] < 0) {
            pos[0] = pos[0] + 1;
            pos[1] = pos[1] + 1;
        }
        while (pos[0] < rows && pos[1] < cols) {
            a.set((Object)1, pos);
            pos[0] = pos[0] + 1;
            pos[1] = pos[1] + 1;
        }
        return a;
    }

    public static <T extends Dataset> T diag(T a, int offset) {
        Dataset result;
        block9: {
            int[] nArray;
            int side;
            int[] shape;
            int is;
            int dtype;
            block8: {
                dtype = a.getDType();
                int rank = a.getRank();
                is = a.getElementsPerItem();
                if (rank == 0 || rank > 2) {
                    utilsLogger.error("Rank of dataset should be one or two");
                    throw new IllegalArgumentException("Rank of dataset should be one or two");
                }
                shape = a.getShapeRef();
                if (rank != 1) break block8;
                int side2 = shape[0] + Math.abs(offset);
                int[] pos = new int[]{side2, side2};
                result = DatasetFactory.zeros(is, pos, dtype);
                if (offset >= 0) {
                    pos[0] = 0;
                    pos[1] = offset;
                } else {
                    pos[0] = -offset;
                    pos[1] = 0;
                }
                int i = 0;
                while (pos[0] < side2 && pos[1] < side2) {
                    result.set(a.getObject(i++), pos);
                    pos[0] = pos[0] + 1;
                    pos[1] = pos[1] + 1;
                }
                break block9;
            }
            int n = side = offset >= 0 ? Math.min(shape[0], shape[1] - offset) : Math.min(shape[0] + offset, shape[1]);
            if (side < 0) {
                side = 0;
            }
            result = DatasetFactory.zeros(is, new int[]{side}, dtype);
            if (side <= 0) break block9;
            if (offset >= 0) {
                int[] nArray2 = new int[2];
                nArray = nArray2;
                nArray2[1] = offset;
            } else {
                int[] nArray3 = new int[2];
                nArray = nArray3;
                nArray3[0] = -offset;
            }
            int[] pos = nArray;
            int i = 0;
            while (pos[0] < shape[0] && pos[1] < shape[1]) {
                result.set(a.getObject(pos), i++);
                pos[0] = pos[0] + 1;
                pos[1] = pos[1] + 1;
            }
        }
        return (T)result;
    }

    public static Dataset sliceAndConvertLazyDataset(ILazyDataset lazy) throws DatasetException {
        if (lazy == null) {
            return null;
        }
        IDataset data = lazy instanceof IDataset ? (IDataset)lazy.getSliceView(new Slice[0]) : lazy.getSlice(new Slice[0]);
        return DatasetUtils.convertToDataset(data);
    }

    public static Dataset convertToDataset(IDataset data) {
        if (data == null) {
            return null;
        }
        if (data instanceof Dataset) {
            return (Dataset)data;
        }
        int dtype = DTypeUtils.getDType(data);
        int isize = data.getElementsPerItem();
        if (isize <= 0) {
            throw new IllegalArgumentException("Datasets with " + isize + " elements per item not supported");
        }
        Dataset result = DatasetFactory.zeros(isize, data.getShape(), dtype);
        result.setName(data.getName());
        IndexIterator it = result.getIterator(true);
        int[] pos = it.getPos();
        switch (dtype) {
            case 0: {
                while (it.hasNext()) {
                    result.setObjectAbs(it.index, data.getBoolean(pos));
                }
                break;
            }
            case 1: {
                while (it.hasNext()) {
                    result.setObjectAbs(it.index, data.getByte(pos));
                }
                break;
            }
            case 2: {
                while (it.hasNext()) {
                    result.setObjectAbs(it.index, data.getShort(pos));
                }
                break;
            }
            case 3: {
                while (it.hasNext()) {
                    result.setObjectAbs(it.index, data.getInt(pos));
                }
                break;
            }
            case 4: {
                while (it.hasNext()) {
                    result.setObjectAbs(it.index, data.getLong(pos));
                }
                break;
            }
            case 5: {
                while (it.hasNext()) {
                    result.setObjectAbs(it.index, Float.valueOf(data.getFloat(pos)));
                }
                break;
            }
            case 6: {
                while (it.hasNext()) {
                    result.setObjectAbs(it.index, data.getDouble(pos));
                }
                break;
            }
            default: {
                while (it.hasNext()) {
                    result.setObjectAbs(it.index, data.getObject(pos));
                }
                break block0;
            }
        }
        result.setErrors(data.getErrors());
        return result;
    }

    public static CompoundDataset createCompoundDataset(Dataset ... datasets) {
        if (datasets == null || datasets.length == 0) {
            return null;
        }
        return DatasetUtils.createCompoundDataset(datasets[0].getDType(), datasets);
    }

    public static CompoundDataset createCompoundDataset(int dtype, Dataset ... datasets) {
        if (datasets == null || datasets.length == 0) {
            return null;
        }
        switch (dtype) {
            case 1: 
            case 100: {
                return new CompoundByteDataset(datasets);
            }
            case 2: 
            case 200: {
                return new CompoundShortDataset(datasets);
            }
            case 3: 
            case 300: {
                return new CompoundIntegerDataset(datasets);
            }
            case 4: 
            case 400: {
                return new CompoundLongDataset(datasets);
            }
            case 5: 
            case 500: {
                return new CompoundFloatDataset(datasets);
            }
            case 6: 
            case 600: {
                return new CompoundDoubleDataset(datasets);
            }
            case 7: 
            case 8: {
                if (datasets.length > 2) {
                    utilsLogger.error("At most two datasets are allowed");
                    throw new IllegalArgumentException("At most two datasets are allowed");
                }
                if (datasets.length == 2) {
                    return dtype == 7 ? new ComplexFloatDataset(datasets[0], datasets[1]) : new ComplexDoubleDataset(datasets[0], datasets[1]);
                }
                return dtype == 7 ? new ComplexFloatDataset(datasets[0]) : new ComplexDoubleDataset(datasets[0]);
            }
            case 203: {
                if (datasets.length == 1) {
                    return new RGBDataset(datasets[0]);
                }
                if (datasets.length == 3) {
                    return new RGBDataset(datasets[0], datasets[1], datasets[2]);
                }
                utilsLogger.error("Only one or three datasets are allowed to create a RGB dataset");
                throw new IllegalArgumentException("Only one or three datasets are allowed to create a RGB dataset");
            }
        }
        utilsLogger.error("Dataset type not supported for this operation");
        throw new UnsupportedOperationException("Dataset type not supported");
    }

    public static <T extends CompoundDataset> T createCompoundDataset(Class<T> clazz, Dataset ... datasets) {
        return (T)DatasetUtils.createCompoundDataset(DTypeUtils.getDType(clazz), datasets);
    }

    public static CompoundDataset createCompoundDataset(Dataset dataset, int itemSize) {
        int[] shape;
        int[] nshape = shape = dataset.getShapeRef();
        if (shape != null && itemSize > 1) {
            int size = ShapeUtils.calcSize(shape);
            if (size % itemSize != 0) {
                throw new IllegalArgumentException("Input dataset has number of items that is not a multiple of itemSize");
            }
            int d = shape.length;
            int l = 1;
            while (--d >= 0) {
                if ((l *= shape[d]) % itemSize == 0) break;
            }
            assert (d >= 0);
            nshape = new int[d + 1];
            int i = 0;
            while (i < d) {
                nshape[i] = shape[i];
                ++i;
            }
            nshape[d] = l / itemSize;
        }
        switch (dataset.getDType()) {
            case 1: {
                return new CompoundByteDataset(itemSize, (byte[])dataset.getBuffer(), nshape);
            }
            case 2: {
                return new CompoundShortDataset(itemSize, (short[])dataset.getBuffer(), nshape);
            }
            case 3: {
                return new CompoundIntegerDataset(itemSize, (int[])dataset.getBuffer(), nshape);
            }
            case 4: {
                return new CompoundLongDataset(itemSize, (long[])dataset.getBuffer(), nshape);
            }
            case 5: {
                return new CompoundFloatDataset(itemSize, (float[])dataset.getBuffer(), nshape);
            }
            case 6: {
                return new CompoundDoubleDataset(itemSize, (double[])dataset.getBuffer(), nshape);
            }
        }
        utilsLogger.error("Dataset type not supported for this operation");
        throw new UnsupportedOperationException("Dataset type not supported");
    }

    public static CompoundDataset createCompoundDatasetFromLastAxis(Dataset a, boolean shareData) {
        switch (a.getDType()) {
            case 1: {
                return CompoundByteDataset.createCompoundDatasetWithLastDimension(a, shareData);
            }
            case 2: {
                return CompoundShortDataset.createCompoundDatasetWithLastDimension(a, shareData);
            }
            case 3: {
                return CompoundIntegerDataset.createCompoundDatasetWithLastDimension(a, shareData);
            }
            case 4: {
                return CompoundLongDataset.createCompoundDatasetWithLastDimension(a, shareData);
            }
            case 5: {
                return CompoundFloatDataset.createCompoundDatasetWithLastDimension(a, shareData);
            }
            case 6: {
                return CompoundDoubleDataset.createCompoundDatasetWithLastDimension(a, shareData);
            }
        }
        utilsLogger.error("Dataset type not supported for this operation");
        throw new UnsupportedOperationException("Dataset type not supported");
    }

    public static Dataset createDatasetFromCompoundDataset(CompoundDataset a, boolean shareData) {
        return a.asNonCompoundDataset(shareData);
    }

    public static Dataset coerce(Dataset a, Object obj) {
        int dt = a.getDType();
        int ot = DTypeUtils.getDTypeFromClass(obj.getClass());
        return DatasetUtils.cast(a.clone(), DTypeUtils.getBestDType(dt, ot));
    }

    public static Dataset norm(Dataset a) {
        double amin = a.min(new boolean[0]).doubleValue();
        double aptp = a.max(new boolean[0]).doubleValue() - amin;
        Dataset temp = Maths.subtract(a, amin);
        temp.idivide(aptp);
        return temp;
    }

    public static CompoundDataset norm(CompoundDataset a, boolean overAllElements) {
        Dataset result;
        double[] amin = a.minItem();
        double[] amax = a.maxItem();
        int is = a.getElementsPerItem();
        if (overAllElements) {
            Arrays.sort(amin);
            Arrays.sort(amax);
            double aptp = amax[0] - amin[0];
            result = Maths.subtract(a, amin[0]);
            result.idivide(aptp);
        } else {
            double[] aptp = new double[is];
            int j = 0;
            while (j < is) {
                aptp[j] = amax[j] - amin[j];
                ++j;
            }
            result = Maths.subtract(a, amin);
            result.idivide(aptp);
        }
        return (CompoundDataset)result;
    }

    public static Dataset lognorm(Dataset a) {
        double amin = a.min(new boolean[0]).doubleValue();
        double aptp = Math.log10(a.max(new boolean[0]).doubleValue() - amin + 1.0);
        Dataset temp = Maths.subtract(a, amin - 1.0);
        temp = Maths.log10(temp);
        temp = Maths.divide(temp, aptp);
        return temp;
    }

    public static Dataset lnnorm(Dataset a) {
        double amin = a.min(new boolean[0]).doubleValue();
        double aptp = Math.log(a.max(new boolean[0]).doubleValue() - amin + 1.0);
        Dataset temp = Maths.subtract(a, amin - 1.0);
        temp = Maths.log(temp);
        temp = Maths.divide(temp, aptp);
        return temp;
    }

    public static List<Dataset> meshGrid(Dataset ... axes) {
        Dataset axis;
        ArrayList<Dataset> result = new ArrayList<Dataset>();
        int rank = axes.length;
        if (rank < 2) {
            utilsLogger.error("Two or more axes datasets are required");
            throw new IllegalArgumentException("Two or more axes datasets are required");
        }
        int[] nshape = new int[rank];
        int i = 0;
        while (i < rank) {
            axis = axes[i];
            if (axis.getRank() != 1) {
                utilsLogger.error("Given axis is not 1D");
                throw new IllegalArgumentException("Given axis is not 1D");
            }
            nshape[i] = axis.getSize();
            ++i;
        }
        i = 0;
        while (i < rank) {
            axis = axes[i];
            Dataset coord = DatasetFactory.zeros(nshape, axis.getDType());
            result.add(coord);
            int alen = axis.getSize();
            int j = 0;
            while (j < alen) {
                Object obj = axis.getObjectAbs(j);
                PositionIterator pi = coord.getPositionIterator(i);
                int[] pos = pi.getPos();
                pos[i] = j;
                while (pi.hasNext()) {
                    coord.set(obj, pos);
                }
                ++j;
            }
            ++i;
        }
        return result;
    }

    public static IntegerDataset indices(int ... shape) {
        int rank = shape.length;
        int[] nshape = new int[rank + 1];
        nshape[0] = rank;
        int i = 0;
        while (i < rank) {
            nshape[i + 1] = shape[i];
            ++i;
        }
        IntegerDataset index = new IntegerDataset(nshape);
        if (rank == 1) {
            int alen = shape[0];
            int[] pos = new int[2];
            int j = 0;
            while (j < alen) {
                pos[1] = j;
                index.set((Object)j, pos);
                ++j;
            }
        } else {
            int i2 = 1;
            while (i2 <= rank) {
                int alen = nshape[i2];
                int j = 0;
                while (j < alen) {
                    PositionIterator pi = index.getPositionIterator(0, i2);
                    int[] pos = pi.getPos();
                    pos[0] = i2 - 1;
                    pos[i2] = j;
                    while (pi.hasNext()) {
                        index.set((Object)j, pos);
                    }
                    ++j;
                }
                ++i2;
            }
        }
        return index;
    }

    public static double[] centroid(Dataset a, Dataset ... bases) {
        int rank = a.getRank();
        if (bases.length > 0 && bases.length != rank) {
            throw new IllegalArgumentException("Number of bases must be zero or match rank of dataset");
        }
        int[] shape = a.getShapeRef();
        if (bases.length == rank) {
            int i = 0;
            while (i < rank) {
                Dataset b = bases[i];
                if (b.getRank() != 1 && b.getSize() != shape[i]) {
                    throw new IllegalArgumentException("A base does not have shape to match given dataset");
                }
                ++i;
            }
        }
        double[] dc = new double[rank];
        if (rank == 0) {
            return dc;
        }
        PositionIterator iter = new PositionIterator(shape);
        int[] pos = iter.getPos();
        double tsum = 0.0;
        while (iter.hasNext()) {
            double val = a.getDouble(pos);
            tsum += val;
            int d = 0;
            while (d < rank) {
                Dataset b;
                Dataset dataset = b = bases.length == 0 ? null : bases[d];
                if (b == null) {
                    int n = d;
                    dc[n] = dc[n] + ((double)pos[d] + 0.5) * val;
                } else {
                    int n = d;
                    dc[n] = dc[n] + b.getElementDoubleAbs(pos[d]) * val;
                }
                ++d;
            }
        }
        int d = 0;
        while (d < rank) {
            int n = d++;
            dc[n] = dc[n] / tsum;
        }
        return dc;
    }

    public static List<Double> crossings(Dataset d, double value) {
        if (d.getRank() != 1) {
            utilsLogger.error("Only 1d datasets supported");
            throw new UnsupportedOperationException("Only 1d datasets supported");
        }
        ArrayList<Double> results = new ArrayList<Double>();
        IndexIterator it = d.getIterator();
        double y2 = it.hasNext() ? d.getElementDoubleAbs(it.index) : 0.0;
        double x = 1.0;
        while (it.hasNext()) {
            double y1 = y2;
            y2 = d.getElementDoubleAbs(it.index);
            if (y1 <= value && value < y2 || y1 > value && y2 <= value) {
                double f = (value - y2) / (y2 - y1);
                results.add(x + f);
            }
            x += 1.0;
        }
        if (y2 == value) {
            results.add(x);
        }
        return results;
    }

    public static List<Double> crossings(Dataset xAxis, Dataset yAxis, double yValue) {
        if (xAxis.getSize() > yAxis.getSize()) {
            throw new IllegalArgumentException("Number of values of yAxis must as least be equal to the number of values of xAxis");
        }
        ArrayList<Double> results = new ArrayList<Double>();
        List<Double> indices = DatasetUtils.crossings(yAxis, yValue);
        for (double xi : indices) {
            results.add(Maths.interpolate(xAxis, xi));
        }
        return results;
    }

    public static List<Double> crossings(Dataset xAxis, Dataset yAxis, double yValue, double xRangeProportion) {
        List<Double> vals = DatasetUtils.crossings(xAxis, yAxis, yValue);
        double error = xRangeProportion * xAxis.peakToPeak(new boolean[0]).doubleValue();
        int i = 0;
        while (i <= vals.size() - 3) {
            double v1 = Math.abs(vals.get(i) - vals.get(i + 2));
            if (v1 < error) {
                vals.set(i + 2, (vals.get(i) + vals.get(i + 1) + vals.get(i + 2)) / 3.0);
                vals.remove(i);
                vals.remove(i);
                continue;
            }
            ++i;
        }
        return vals;
    }

    private static void setRow(Object row, Dataset a, int ... pos) {
        int l = Array.getLength(row);
        int rank = pos.length;
        int[] npos = Arrays.copyOf(pos, rank + 1);
        if (rank + 1 < a.getRank()) {
            int i = 0;
            while (i < l) {
                npos[rank] = i;
                Object r = Array.get(row, i);
                DatasetUtils.setRow(r, a, npos);
                ++i;
            }
        } else {
            int i = 0;
            while (i < l) {
                npos[rank] = i;
                Object r = a.getObject(npos);
                Array.set(row, i, r);
                ++i;
            }
        }
    }

    public static Object createJavaArray(Dataset a) {
        Object matrix;
        if (a.getElementsPerItem() > 1) {
            a = DatasetUtils.createDatasetFromCompoundDataset((CompoundDataset)a, true);
        }
        switch (a.getDType()) {
            case 0: {
                matrix = Array.newInstance(Boolean.TYPE, a.getShape());
                break;
            }
            case 1: {
                matrix = Array.newInstance(Byte.TYPE, a.getShape());
                break;
            }
            case 2: {
                matrix = Array.newInstance(Short.TYPE, a.getShape());
                break;
            }
            case 3: {
                matrix = Array.newInstance(Integer.TYPE, a.getShape());
                break;
            }
            case 4: {
                matrix = Array.newInstance(Long.TYPE, a.getShape());
                break;
            }
            case 5: {
                matrix = Array.newInstance(Float.TYPE, a.getShape());
                break;
            }
            case 6: {
                matrix = Array.newInstance(Double.TYPE, a.getShape());
                break;
            }
            default: {
                utilsLogger.error("Dataset type not supported");
                throw new IllegalArgumentException("Dataset type not supported");
            }
        }
        DatasetUtils.setRow(matrix, a, new int[0]);
        return matrix;
    }

    public static void removeNansAndInfinities(Dataset a, Number value) {
        block9: {
            block11: {
                block10: {
                    block8: {
                        if (!(a instanceof DoubleDataset)) break block8;
                        double dvalue = DTypeUtils.toReal(value);
                        DoubleDataset set = (DoubleDataset)a;
                        IndexIterator it = set.getIterator();
                        double[] data = set.getData();
                        while (it.hasNext()) {
                            double x = data[it.index];
                            if (!Double.isNaN(x) && !Double.isInfinite(x)) continue;
                            data[it.index] = dvalue;
                        }
                        break block9;
                    }
                    if (!(a instanceof FloatDataset)) break block10;
                    float fvalue = (float)DTypeUtils.toReal(value);
                    FloatDataset set = (FloatDataset)a;
                    IndexIterator it = set.getIterator();
                    float[] data = set.getData();
                    while (it.hasNext()) {
                        float x = data[it.index];
                        if (!Float.isNaN(x) && !Float.isInfinite(x)) continue;
                        data[it.index] = fvalue;
                    }
                    break block9;
                }
                if (!(a instanceof CompoundDoubleDataset)) break block11;
                double dvalue = DTypeUtils.toReal(value);
                CompoundDoubleDataset set = (CompoundDoubleDataset)a;
                int is = set.getElementsPerItem();
                IndexIterator it = set.getIterator();
                double[] data = set.getData();
                while (it.hasNext()) {
                    int j = 0;
                    while (j < is) {
                        double x = data[it.index + j];
                        if (Double.isNaN(x) || Double.isInfinite(x)) {
                            data[it.index + j] = dvalue;
                        }
                        ++j;
                    }
                }
                break block9;
            }
            if (!(a instanceof CompoundFloatDataset)) break block9;
            float fvalue = (float)DTypeUtils.toReal(value);
            CompoundFloatDataset set = (CompoundFloatDataset)a;
            int is = set.getElementsPerItem();
            IndexIterator it = set.getIterator();
            float[] data = set.getData();
            while (it.hasNext()) {
                int j = 0;
                while (j < is) {
                    float x = data[it.index + j];
                    if (Float.isNaN(x) || Float.isInfinite(x)) {
                        data[it.index + j] = fvalue;
                    }
                    ++j;
                }
            }
        }
    }

    public static void makeFinite(Dataset a) {
        block15: {
            block17: {
                block16: {
                    block14: {
                        if (!(a instanceof DoubleDataset)) break block14;
                        DoubleDataset set = (DoubleDataset)a;
                        IndexIterator it = set.getIterator();
                        double[] data = set.getData();
                        while (it.hasNext()) {
                            double x = data[it.index];
                            if (Double.isNaN(x)) {
                                data[it.index] = 0.0;
                                continue;
                            }
                            if (!Double.isInfinite(x)) continue;
                            double d = data[it.index] = x > 0.0 ? Double.MAX_VALUE : -1.7976931348623157E308;
                        }
                        break block15;
                    }
                    if (!(a instanceof FloatDataset)) break block16;
                    FloatDataset set = (FloatDataset)a;
                    IndexIterator it = set.getIterator();
                    float[] data = set.getData();
                    while (it.hasNext()) {
                        float x = data[it.index];
                        if (Float.isNaN(x)) {
                            data[it.index] = 0.0f;
                            continue;
                        }
                        if (!Float.isInfinite(x)) continue;
                        float f = data[it.index] = x > 0.0f ? Float.MAX_VALUE : -3.4028235E38f;
                    }
                    break block15;
                }
                if (!(a instanceof CompoundDoubleDataset)) break block17;
                CompoundDoubleDataset set = (CompoundDoubleDataset)a;
                int is = set.getElementsPerItem();
                IndexIterator it = set.getIterator();
                double[] data = set.getData();
                while (it.hasNext()) {
                    int j = 0;
                    while (j < is) {
                        double x = data[it.index + j];
                        if (Double.isNaN(x)) {
                            data[it.index + j] = 0.0;
                        } else if (Double.isInfinite(x)) {
                            data[it.index + j] = x > 0.0 ? Double.MAX_VALUE : -1.7976931348623157E308;
                        }
                        ++j;
                    }
                }
                break block15;
            }
            if (!(a instanceof CompoundFloatDataset)) break block15;
            CompoundFloatDataset set = (CompoundFloatDataset)a;
            int is = set.getElementsPerItem();
            IndexIterator it = set.getIterator();
            float[] data = set.getData();
            while (it.hasNext()) {
                int j = 0;
                while (j < is) {
                    float x = data[it.index + j];
                    if (Float.isNaN(x)) {
                        data[it.index + j] = 0.0f;
                    } else if (Float.isInfinite(x)) {
                        data[it.index + j] = x > 0.0f ? Float.MAX_VALUE : -3.4028235E38f;
                    }
                    ++j;
                }
            }
        }
    }

    public static int findIndexEqualTo(Dataset a, double n) {
        IndexIterator iter = a.getIterator();
        while (iter.hasNext()) {
            if (a.getElementDoubleAbs(iter.index) == n) break;
        }
        return iter.index;
    }

    public static int findIndexGreaterThan(Dataset a, double n) {
        IndexIterator iter = a.getIterator();
        while (iter.hasNext()) {
            if (a.getElementDoubleAbs(iter.index) > n) break;
        }
        return iter.index;
    }

    public static int findIndexGreaterThanOrEqualTo(Dataset a, double n) {
        IndexIterator iter = a.getIterator();
        while (iter.hasNext()) {
            if (a.getElementDoubleAbs(iter.index) >= n) break;
        }
        return iter.index;
    }

    public static int findIndexLessThan(Dataset a, double n) {
        IndexIterator iter = a.getIterator();
        while (iter.hasNext()) {
            if (a.getElementDoubleAbs(iter.index) < n) break;
        }
        return iter.index;
    }

    public static int findIndexLessThanOrEqualTo(Dataset a, double n) {
        IndexIterator iter = a.getIterator();
        while (iter.hasNext()) {
            if (a.getElementDoubleAbs(iter.index) <= n) break;
        }
        return iter.index;
    }

    /*
     * Unable to fully structure code
     */
    public static IntegerDataset findFirstOccurrences(Dataset a, Dataset values) {
        block11: {
            if (values.getRank() != 1) {
                throw new IllegalArgumentException("Values dataset must be 1D");
            }
            indexes = new IntegerDataset(new int[]{values.getSize()});
            indexes.fill(-1);
            it = a.getIterator();
            n = values.getSize();
            if (values.getDType() != 4) ** GOTO lbl58
            block0: while (it.hasNext()) {
                x = a.getElementLongAbs(it.index);
                if (x <= (vl = values.getLong(l = 0))) {
                    if (x != vl || indexes.getAbs(l) >= 0) continue;
                    indexes.setAbs(l, it.index);
                    continue;
                }
                h = n - 1;
                vh = values.getLong(h);
                if (x < vh) ** GOTO lbl32
                if (x != vh || indexes.getAbs(h) >= 0) continue;
                indexes.setAbs(h, it.index);
                continue;
lbl-1000:
                // 1 sources

                {
                    m = (l + h) / 2;
                    vm = values.getLong(m);
                    if (x < vm) {
                        h = m;
                        continue;
                    }
                    if (x > vm) {
                        l = m;
                        continue;
                    }
                    if (indexes.getAbs(m) >= 0) continue block0;
                    indexes.setAbs(m, it.index);
                    continue block0;
lbl32:
                    // 3 sources

                    ** while (h - l > 1)
                }
lbl33:
                // 1 sources

            }
            break block11;
lbl-1000:
            // 1 sources

            {
                x = a.getElementDoubleAbs(it.index);
                if (x <= (vl = values.getDouble(l = 0))) {
                    if (x != vl || indexes.getAbs(l) >= 0) continue;
                    indexes.setAbs(l, it.index);
                    continue;
                }
                h = n - 1;
                vh = values.getDouble(h);
                if (!(x >= vh)) ** GOTO lbl57
                if (x != vh || indexes.getAbs(h) >= 0) continue;
                indexes.setAbs(h, it.index);
                continue;
lbl-1000:
                // 1 sources

                {
                    m = (l + h) / 2;
                    vm = values.getDouble(m);
                    if (x < vm) {
                        h = m;
                        continue;
                    }
                    if (x > vm) {
                        l = m;
                        continue;
                    }
                    if (indexes.getAbs(m) >= 0) continue block2;
                    indexes.setAbs(m, it.index);
                    continue block2;
lbl57:
                    // 3 sources

                    ** while (h - l > 1)
                }
lbl58:
                // 8 sources

                ** while (it.hasNext())
            }
        }
        return indexes;
    }

    /*
     * Unable to fully structure code
     */
    public static IntegerDataset findIndexesForValues(Dataset a, Dataset values) {
        block11: {
            if (values.getRank() != 1) {
                throw new IllegalArgumentException("Values dataset must be 1D");
            }
            indexes = new IntegerDataset(new int[]{a.getSize()});
            indexes.fill(-1);
            it = a.getIterator();
            i = -1;
            n = values.getSize();
            if (values.getDType() != 4) ** GOTO lbl59
            block0: while (it.hasNext()) {
                ++i;
                x = a.getElementLongAbs(it.index);
                if (x <= (vl = values.getLong(l = 0))) {
                    if (x != vl) continue;
                    indexes.setAbs(i, l);
                    continue;
                }
                h = n - 1;
                vh = values.getLong(h);
                if (x < vh) ** GOTO lbl33
                if (x != vh) continue;
                indexes.setAbs(i, h);
                continue;
lbl-1000:
                // 1 sources

                {
                    m = (l + h) / 2;
                    vm = values.getLong(m);
                    if (x < vm) {
                        h = m;
                        continue;
                    }
                    if (x > vm) {
                        l = m;
                        continue;
                    }
                    indexes.setAbs(i, m);
                    continue block0;
lbl33:
                    // 3 sources

                    ** while (h - l > 1)
                }
lbl34:
                // 1 sources

            }
            break block11;
lbl-1000:
            // 1 sources

            {
                ++i;
                x = a.getElementDoubleAbs(it.index);
                if (x <= (vl = values.getDouble(l = 0))) {
                    if (x != vl) continue;
                    indexes.setAbs(i, l);
                    continue;
                }
                h = n - 1;
                vh = values.getDouble(h);
                if (!(x >= vh)) ** GOTO lbl58
                if (x != vh) continue;
                indexes.setAbs(i, h);
                continue;
lbl-1000:
                // 1 sources

                {
                    m = (l + h) / 2;
                    vm = values.getDouble(m);
                    if (x < vm) {
                        h = m;
                        continue;
                    }
                    if (x > vm) {
                        l = m;
                        continue;
                    }
                    indexes.setAbs(i, m);
                    continue block2;
lbl58:
                    // 3 sources

                    ** while (h - l > 1)
                }
lbl59:
                // 7 sources

                ** while (it.hasNext())
            }
        }
        return indexes;
    }

    public static <T extends Dataset> T roll(T a, int shift, Integer axis) {
        T r = DatasetFactory.zeros(a);
        int is = a.getElementsPerItem();
        if (axis == null) {
            IndexIterator it = a.getIterator();
            int s = r.getSize();
            int i = shift % s;
            if (i < 0) {
                i += s;
            }
            while (it.hasNext()) {
                r.setObjectAbs(i, a.getObjectAbs(it.index));
                if ((i += is) < s) continue;
                i %= s;
            }
        } else {
            axis = a.checkAxis(axis);
            PositionIterator pi = a.getPositionIterator(axis);
            int s = a.getShapeRef()[axis];
            Object u = DatasetFactory.zeros(is, a.getClass(), s);
            Object v = DatasetFactory.zeros(u);
            int[] pos = pi.getPos();
            boolean[] hit = pi.getOmit();
            while (pi.hasNext()) {
                a.copyItemsFromAxes(pos, hit, (Dataset)u);
                int i = shift % s;
                if (i < 0) {
                    i += s;
                }
                int j = 0;
                while (j < s) {
                    v.setObjectAbs(i, u.getObjectAbs(j * is));
                    if ((i += is) >= s) {
                        i %= s;
                    }
                    ++j;
                }
                r.setItemsOnAxes(pos, hit, v.getBuffer());
            }
        }
        return r;
    }

    public static <T extends Dataset> T rollAxis(T a, int axis, int start) {
        int r = a.getRank();
        axis = a.checkAxis(axis);
        if (start < 0) {
            start += r;
        }
        if (start < 0 || start > r) {
            throw new IllegalArgumentException("Start is out of range: it should be >= 0 and <= " + r);
        }
        if (axis < start) {
            --start;
        }
        if (axis == start) {
            return a;
        }
        ArrayList<Integer> axes = new ArrayList<Integer>();
        int i = 0;
        while (i < r) {
            if (i != axis) {
                axes.add(i);
            }
            ++i;
        }
        axes.add(start, axis);
        int[] aa = new int[r];
        int i2 = 0;
        while (i2 < r) {
            aa[i2] = (Integer)axes.get(i2);
            ++i2;
        }
        return (T)a.getTransposedView(aa);
    }

    private static SliceND createFlippedSlice(Dataset a, int axis) {
        int[] shape = a.getShapeRef();
        SliceND slice = new SliceND(shape);
        slice.flip(axis);
        return slice;
    }

    public static <T extends Dataset> T flipLeftRight(T a) {
        if (a.getRank() < 2) {
            throw new IllegalArgumentException("Dataset must be at least 2D");
        }
        return (T)a.getSliceView(DatasetUtils.createFlippedSlice(a, 1));
    }

    public static <T extends Dataset> T flipUpDown(T a) {
        return (T)a.getSliceView(DatasetUtils.createFlippedSlice(a, 0));
    }

    public static <T extends Dataset> T rotate90(T a) {
        return DatasetUtils.rotate90(a, 1);
    }

    public static <T extends Dataset> T rotate90(T a, int k) {
        k %= 4;
        while (k < 0) {
            k += 4;
        }
        int r = a.getRank();
        if (r < 2) {
            throw new IllegalArgumentException("Dataset must be at least 2D");
        }
        switch (k) {
            case 1: 
            case 3: {
                int[] axes = new int[r];
                axes[0] = 1;
                axes[1] = 0;
                int i = 2;
                while (i < r) {
                    axes[i] = i;
                    ++i;
                }
                Dataset t = a.getTransposedView(axes);
                return (T)t.getSliceView(DatasetUtils.createFlippedSlice(t, k == 1 ? 0 : 1));
            }
            case 2: {
                SliceND s = DatasetUtils.createFlippedSlice(a, 0);
                s.flip(1);
                return (T)a.getSliceView(s);
            }
        }
        return a;
    }

    public static Dataset select(BooleanDataset condition, Object x, Object y) {
        Object[] all = new Object[]{condition, x, y};
        Dataset[] dAll = BroadcastUtils.convertAndBroadcast(all);
        condition = (BooleanDataset)dAll[0];
        Dataset dx = dAll[1];
        Dataset dy = dAll[2];
        int dt = DTypeUtils.getBestDType(dx.getDType(), dy.getDType());
        int ds = Math.max(dx.getElementsPerItem(), dy.getElementsPerItem());
        Dataset r = DatasetFactory.zeros(ds, condition.getShapeRef(), dt);
        IndexIterator iter = condition.getIterator(true);
        int[] pos = iter.getPos();
        int i = 0;
        while (iter.hasNext()) {
            r.setObjectAbs(i++, condition.getElementBooleanAbs(iter.index) ? dx.getObject(pos) : dy.getObject(pos));
        }
        return r;
    }

    public static Dataset select(BooleanDataset[] conditions, Object[] choices, Object def) {
        int n = conditions.length;
        if (choices.length != n) {
            throw new IllegalArgumentException("Choices list is not same length as conditions list");
        }
        Object[] all = new Object[2 * n];
        System.arraycopy(conditions, 0, all, 0, n);
        System.arraycopy(choices, 0, all, n, n);
        Dataset[] dAll = BroadcastUtils.convertAndBroadcast(all);
        conditions = new BooleanDataset[n];
        Dataset[] dChoices = new Dataset[n];
        System.arraycopy(dAll, 0, conditions, 0, n);
        System.arraycopy(dAll, n, dChoices, 0, n);
        int dt = -1;
        int ds = -1;
        int i = 0;
        while (i < n) {
            int s;
            Dataset a = dChoices[i];
            int t = a.getDType();
            if (t > dt) {
                dt = t;
            }
            if ((s = a.getElementsPerItem()) > ds) {
                ds = s;
            }
            ++i;
        }
        if (dt < 0 || ds < 1) {
            throw new IllegalArgumentException("Dataset types of choices are invalid");
        }
        Dataset r = DatasetFactory.zeros(ds, conditions[0].getShapeRef(), dt);
        Dataset d = DatasetFactory.createFromObject(def).getBroadcastView(r.getShapeRef());
        PositionIterator iter = new PositionIterator(r.getShapeRef());
        int[] pos = iter.getPos();
        int i2 = 0;
        while (iter.hasNext()) {
            int j = 0;
            while (j < n) {
                if (conditions[j].get(pos)) {
                    r.setObjectAbs(i2++, dChoices[j].getObject(pos));
                    break;
                }
                ++j;
            }
            if (j != n) continue;
            r.setObjectAbs(i2++, d.getObject(pos));
        }
        return r;
    }

    public static Dataset choose(IntegerDataset index, Object[] choices, boolean throwAIOOBE, boolean clip) {
        int n = choices.length;
        Object[] all = new Object[n + 1];
        System.arraycopy(choices, 0, all, 0, n);
        all[n] = index;
        Dataset[] dChoices = BroadcastUtils.convertAndBroadcast(all);
        int dt = -1;
        int ds = -1;
        int mr = -1;
        int i = 0;
        while (i < n) {
            int s;
            int t;
            Dataset a = dChoices[i];
            int r = a.getRank();
            if (r > mr) {
                mr = r;
            }
            if ((t = a.getDType()) > dt) {
                dt = t;
            }
            if ((s = a.getElementsPerItem()) > ds) {
                ds = s;
            }
            ++i;
        }
        if (dt < 0 || ds < 1) {
            throw new IllegalArgumentException("Dataset types of choices are invalid");
        }
        index = (IntegerDataset)dChoices[n];
        dChoices[n] = null;
        Dataset r = DatasetFactory.zeros(ds, index.getShape(), dt);
        IndexIterator iter = index.getIterator(true);
        int[] pos = iter.getPos();
        int i2 = 0;
        while (iter.hasNext()) {
            int j = index.getAbs(iter.index);
            if (j < 0) {
                if (throwAIOOBE) {
                    throw new ArrayIndexOutOfBoundsException(j);
                }
                if (clip) {
                    j = 0;
                } else {
                    j %= n;
                    j += n;
                }
            }
            if (j >= n) {
                if (throwAIOOBE) {
                    throw new ArrayIndexOutOfBoundsException(j);
                }
                j = clip ? n - 1 : (j %= n);
            }
            Dataset c = dChoices[j];
            r.setObjectAbs(i2++, c.getObject(pos));
        }
        return r;
    }

    public static List<IntegerDataset> calcPositionsFromIndexes(Dataset indices, int[] shape) {
        int rank = shape.length;
        ArrayList<IntegerDataset> posns = new ArrayList<IntegerDataset>();
        int[] iShape = indices.getShapeRef();
        int i = 0;
        while (i < rank) {
            posns.add(new IntegerDataset(iShape));
            ++i;
        }
        IndexIterator it = indices.getIterator(true);
        int[] pos = it.getPos();
        while (it.hasNext()) {
            int n = indices.getInt(pos);
            int[] p = ShapeUtils.getNDPositionFromShape(n, shape);
            int i2 = 0;
            while (i2 < rank) {
                ((IntegerDataset)posns.get(i2)).setItem(p[i2], pos);
                ++i2;
            }
        }
        return posns;
    }

    public static IntegerDataset calcIndexesFromPositions(List<? extends Dataset> positions, int[] shape, int ... mode) {
        int rank = shape.length;
        if (positions.size() != rank) {
            throw new IllegalArgumentException("Number of position datasets must be equal to rank of shape");
        }
        if (mode == null || mode.length == 0) {
            mode = new int[rank];
        } else if (mode.length == 1) {
            int m = mode[0];
            mode = new int[rank];
            Arrays.fill(mode, m);
        } else if (mode.length != rank) {
            throw new IllegalArgumentException("Mode length greater than one must match rank of shape");
        }
        int i = 0;
        while (i < rank) {
            int m = mode[i];
            if (m < 0 || m > 2) {
                throw new IllegalArgumentException("Unknown mode value - it must be 0, 1, or 2");
            }
            ++i;
        }
        Dataset p = positions.get(0);
        IntegerDataset indexes = new IntegerDataset(p.getShapeRef());
        IndexIterator it = p.getIterator(true);
        int[] iPos = it.getPos();
        int[] tPos = new int[rank];
        while (it.hasNext()) {
            int i2 = 0;
            while (i2 < rank) {
                p = positions.get(i2);
                int j = p.getInt(iPos);
                int d = shape[i2];
                if (mode[i2] == 0) {
                    if (j < 0 || j >= d) {
                        throw new ArrayIndexOutOfBoundsException("Position value exceeds dimension in shape");
                    }
                } else if (mode[i2] == 1) {
                    while (j < 0) {
                        j += d;
                    }
                    while (j >= d) {
                        j -= d;
                    }
                } else {
                    if (j < 0) {
                        j = 0;
                    }
                    if (j >= d) {
                        j = d - 1;
                    }
                }
                tPos[i2] = j;
                ++i2;
            }
            indexes.set((Object)ShapeUtils.getFlat1DIndex(shape, tPos), iPos);
        }
        return indexes;
    }

    public static Serializable serializeDataset(IDataset data) {
        Dataset d = DatasetUtils.convertToDataset(data).getView(false);
        d.clearMetadata(null);
        return d.flatten().getBuffer();
    }

    /*
     * Unable to fully structure code
     */
    public static Dataset extract(IDataset data, IDataset condition) {
        block7: {
            block6: {
                a = DatasetUtils.convertToDataset(data.getSliceView(new Slice[0]));
                b = DatasetUtils.cast(condition.getSliceView(new Slice[0]), 0);
                try {
                    return a.getByBoolean(b);
                }
                catch (IllegalArgumentException v0) {
                    length = ((Number)b.sum(new boolean[0])).intValue();
                    it = new BroadcastPairIterator(a, b, null, false);
                    size = ShapeUtils.calcSize(it.getShape());
                    if (length >= size) break block6;
                    ashape = it.getFirstShape();
                    bshape = it.getSecondShape();
                    r = ashape.length;
                    size = length;
                    i = 0;
                    ** while (i < r)
                }
lbl-1000:
                // 1 sources

                {
                    s = ashape[i];
                    if (s > 1 && bshape[i] == 1) {
                        size *= s;
                    }
                    ++i;
                    continue;
                }
            }
            c = DatasetFactory.zeros(new int[]{size}, a.getDType());
            i = 0;
            if (!it.isOutputDouble()) ** GOTO lbl32
            while (it.hasNext()) {
                if (it.bLong == 0L) continue;
                c.setObjectAbs(i++, it.aDouble);
            }
            break block7;
lbl-1000:
            // 1 sources

            {
                if (it.bLong == 0L) continue;
                c.setObjectAbs(i++, it.aLong);
lbl32:
                // 3 sources

                ** while (it.hasNext())
            }
        }
        return c;
    }

    public static void setShapeToOriginalRank(ILazyDataset a, int[] originalShape, int ... axes) {
        a.setShape(ShapeUtils.getReducedShapeKeepRank(originalShape, axes));
    }
}

