/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.brooklyn.api.entity;

import java.util.Set;

import org.apache.brooklyn.api.effector.Effector;
import org.apache.brooklyn.api.objs.BrooklynType;
import org.apache.brooklyn.api.sensor.Sensor;
import org.apache.brooklyn.util.guava.Maybe;

/**
 * Gives type information for an {@link Entity}. It is an immutable snapshot.
 * 
 * It reflects a given entity at the time the snapshot was created: if sensors
 * were added or removed on-the-fly then those changes will be included in subsequent
 * snapshots. Therefore instances of a given class of entity could have different 
 * EntityTypes.
 */
public interface EntityType extends BrooklynType {

    /**
     * Sensors available on this entity.
     */
    Set<Sensor<?>> getSensors();
    
    /**
     * Effectors available on this entity.
     */
    Set<Effector<?>> getEffectors();

    /** @return an effector with the given name, if it exists.
     */
    public Maybe<Effector<?>> getEffectorByName(String name);
        
    /**
     * The Sensor with the given name, or null if not found.
     */
    Sensor<?> getSensor(String name);
    
    /**
     * @return True if has the sensor with the given name; false otherwise.
     */
    boolean hasSensor(String name);
}
