/*
 * ProxyHandler.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: cstevens.
 * Portions created by cstevens are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, drach, suhler.
 *
 * Version:  2.3
 * Created by cstevens on 99/09/15
 * Last modified by suhler on 04/11/30 15:19:40
 */

package sunlabs.brazil.proxy;

import sunlabs.brazil.server.Handler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;

import sunlabs.brazil.util.http.HttpInputStream;
import sunlabs.brazil.util.http.HttpRequest;
import sunlabs.brazil.util.http.MimeHeaders;
/**/
import java.io.EOFException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.net.UnknownHostException;
import java.net.ConnectException;
import java.util.Properties;

/**
 * Handler for implementing a web proxy.
 * By default, this is a dumb proxy.  It can be combined with other
 * handlers to generate side effects, such as content rewriting.
 * <p>
 * Properties:
 * <dl class=props>
 * <dt>useproxy	<dd>The name of the SocketFactory class to use for
 * this handler.  If additional properties are required to set up
 * the SocketFactory, it should be configured as a handler instead.
 * This is here for convenience only.
 * <dt>auth	<dd>The value of the proxy-authenticate header (if any) sent to the upstream proxy
 * <dt>proxyHost<dd>If specified, the name of the upstream proxy
 * <dt>proxyPort<dd>The up stream proxys port, if a proxyHost is specified (defaults to 80)
 * </dl>
 *
 * @author      Stephen Uhler
 * @version		2.3
 */
public class ProxyHandler
    implements Handler
{
    public static final String PROXY_HOST = "proxyHost";
    public static final String PROXY_PORT = "proxyPort";
    public static final String AUTH = "auth";
    public static final String USE_PROXY = "useproxy";

    String via;

    /**
     * The proxy server.
     */
    public String proxyHost;

    /**
     * The proxy server's port.  Default is 80.
     */
    public int proxyPort = 80;

    /**
     * The string to send as the value for the "Proxy-Authorization"
     * HTTP header (if needed).
     */
    public String auth;

    UseProxy proxyTester;

    /**
     * Do one-time setup.
     * get and process the properties file options, and make sure
     */
    public boolean
    init(Server server, String prefix)
    {
	String str;
	Properties props = server.props;

	proxyHost = props.getProperty(prefix + PROXY_HOST);

	str = props.getProperty(prefix + PROXY_PORT);
	try {
	    proxyPort = Integer.decode(str).intValue();
	} catch (Exception e) {};
	
	auth = props.getProperty(prefix + AUTH);

	/*
	 * Set a proxy.  If more sophisicated initialization is required than newinstance(), 
	 * set up the proxy in a separate handler
	 */

	String useproxy = props.getProperty(prefix + USE_PROXY);
	if (useproxy != null) {
	    try {
		Class type = Class.forName(useproxy.trim());
		proxyTester = (UseProxy) type.newInstance();
	    } catch (Exception e) {
		server.log(Server.LOG_WARNING, prefix,
			"Proxy installation error : " + e);
	    }
	}
	if (proxyTester == null) {
	    proxyTester = new UseProxy() {
		public boolean useProxy(String host, int port) { return true; }
	    };
	}
		
	via = " " + server.hostName + ":" + server.listen.getLocalPort()
		+ " (" + server.name + ")";

	return true;
    }

    /**
     * @see Handler#respond
     */
    public boolean
    respond(Request client)
	throws IOException
    {
	String url = client.url;

	if (url.startsWith("http:") == false) {
	    return false;
	}
	if ((client.query != null) && (client.query.length() > 0)) {
	    url += "?" + client.query;
	}

	MimeHeaders clientHeaders = client.headers;

	/*
	 * "Proxy-Connection" may be used (instead of just "Connection")
	 * to keep alive a connection between a client and this proxy.
	 */
	String pc = clientHeaders.get("Proxy-Connection");
	if (pc != null) {
	    client.connectionHeader = "Proxy-Connection";
	    client.keepAlive = pc.equalsIgnoreCase("Keep-Alive");
	}

	HttpRequest.removePointToPointHeaders(clientHeaders, false);

        HttpRequest target = new HttpRequest(url);
	try {
	    MimeHeaders targetHeaders = target.requestHeaders;

	    target.setMethod(client.method);
	    clientHeaders.copyTo(targetHeaders);
/*	    targetHeaders.add("Via", client.protocol + via);*/
	    
	    /*
	     * We might need to authenticate to a target proxy.
	     */

	    if ((proxyHost != null)
		    && proxyTester.useProxy(target.host, target.port)) {
		target.setProxy(proxyHost, proxyPort);
		if (auth != null) {
		    targetHeaders.add("Proxy-Authorization", auth);
		}
	    }

	    if (client.postData != null) {
		OutputStream out = target.getOutputStream();
		out.write(client.postData);
		out.close();
	    }

	    target.connect();

	    targetHeaders = target.responseHeaders;
	    HttpRequest.removePointToPointHeaders(targetHeaders, true);

	    clientHeaders = client.responseHeaders;
	    targetHeaders.copyTo(clientHeaders);
	    try {
		clientHeaders.add("Via",
			target.status.substring(0, 8) + via);
	    } catch (StringIndexOutOfBoundsException e) {
		clientHeaders.add("Via", via);
	    }

	    client.sendResponse(target.getInputStream(),
		    target.getContentLength(), null, target.getResponseCode());
	} catch (InterruptedIOException e) {
	    /*
	     * Read timeout while reading from the remote side.  We use a
	     * read timeout in case the target never responds.  
	     */
	    client.sendError(408, "Timeout / No response");
	} catch (EOFException e) {
	    client.sendError(500, "No response");
	} catch (UnknownHostException e) {
	    client.sendError(500, "Unknown host");
	} catch (ConnectException e) {
	    client.sendError(500, "Connection refused");
	} catch (IOException e) {
	    /*
	     * An IOException will happen if we can't communicate with the
	     * target or the client.  Rather than attempting to discriminate,
	     * just send an error message to the client, and let the send
	     * fail if the client was the one that was in error.
	     */

	    String msg = "Error from proxy";
	    if (e.getMessage() != null) {
		msg += ": " + e.getMessage();
	    }
	    client.sendError(500, msg);
    	} finally {
	    target.close();
	}
	return true;
    }
}
