#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include "siod.h"
#include "glib/glib.h"
#include "interp.h"


typedef struct _MarshallData  MarshallData;

struct _MarshallData
{
  char *name;
  Marshaller marshaller;
  Func func;
  ParamType return_type;
  ParamType *args;
  int nargs;
};


static void do_prompt           (void);
static void do_read             (gpointer           data,
				 gint               source,
				 GdkInputCondition  condition);
static void init_finish         (void);
static LISP interp_func         (char              *name,
				 LISP               params);
static LISP glue_signal_connect (LISP               params);
static LISP glue_idle_add       (LISP               params);
static LISP glue_idle_remove    (LISP               params);
static LISP glue_timeout_add    (LISP               params);
static LISP glue_timeout_remove (LISP               params);
static void glue_signal_marshal (GtkObject         *object,
				 gpointer           data,
				 gint               nparams,
				 GtkSignalParam    *params,
				 GtkParamType      *param_types,
				 GtkParamType       return_type);
static void glue_signal_destroy (gpointer           data);
static gint glue_callback       (gpointer           data);


static char *siod_argv[] =
{
  "siod",
  "-h100000:10",
  "-g0",
  "-o1000",
  "-s200000",
  "-n2048",
  "-v2"
};
static int nsiod_argv = sizeof (siod_argv) / sizeof (siod_argv[0]);

static GHashTable *marshall_ht = NULL;
static GHashTable *data_ht = NULL;
static GHashTable *id_ht = NULL;


int
main (int   argc,
      char *argv[])
{
  LISP l;
  int i;

  process_cla (nsiod_argv, siod_argv, 1);

  init_storage ();
  init_subrs ();
  init_trace ();
  init_interp ();
  init_finish ();

  gtk_init (&argc, &argv);
  gtk_signal_set_funcs (glue_signal_marshal,
			glue_signal_destroy);

  fcntl (STDIN_FILENO, F_SETFL, O_NONBLOCK);
  gdk_input_add (STDIN_FILENO, GDK_INPUT_READ, do_read, NULL);
  do_prompt ();

  for (l = NIL, i = 0; i < argc; i++)
    l = cons (strcons (strlen (argv[i]), argv[i]), l);
  setvar (cintern ("*args*"), nreverse (l), NIL);
  l = NIL;

  for (i = 1; i < argc; i++)
    vload (argv[i], 0, 1);

  gtk_main ();
  return 0;
}

void
add_func (char       *name,
	  Marshaller  marshaller,
	  Func        func,
	  ParamType   return_val,
	  int         nargs,
	  ...)
{
  MarshallData *data;
  va_list args;
  int i;

  init_nsubr (name, interp_func);

  if (!marshall_ht)
    marshall_ht = g_hash_table_new (g_string_hash,
				    g_string_equal);

  data = g_new (MarshallData, 1);
  data->name = name;
  data->marshaller = marshaller;
  data->func = func;
  data->return_type = return_val;
  data->nargs = nargs;
  data->args = g_new (ParamType, nargs);

  if (nargs > 0)
    {
      va_start (args, nargs);

      for (i = 0; i < nargs; i++)
	data->args[i] = va_arg (args, ParamType);

      va_end (args);
    }

  g_hash_table_insert (marshall_ht, name, data);
}

void
add_const (char *name,
	   int   value)
{
  setvar (cintern (name), flocons (value), NIL);
}

static gint
check_sexp (char *s)
{
  int paren_level;
  int inside_quote;

  paren_level = 0;
  inside_quote = FALSE;

  while (*s)
    {
      if (!inside_quote)
	{
	  if (*s == '(')
	    paren_level++;
	  else if (*s == ')')
	    paren_level--;
	  else if (*s == '"')
	    inside_quote = TRUE;
	}
      else if (*s == '"')
	inside_quote = FALSE;

      s++;
    }

  return (paren_level == 0);
}

static void
do_prompt ()
{
  fprintf (stdout, "> ");
  fflush (stdout);
}

static void
do_read (gpointer          data,
	 gint              source,
	 GdkInputCondition condition)
{
  static GString *string;
  int nread, done;
  char c;

  if (condition & GDK_INPUT_READ)
    {
      do {
	nread = read (source, &c, sizeof (char));
      } while ((nread == -1) && (errno == EAGAIN));

      if (!string)
	string = g_string_new ("");

      done = FALSE;
      if (nread == sizeof (char))
        {
          done = ((c == '\r') || (c == '\n'));

	  if (!done)
	    g_string_append_c (string, c);

	  if (done && (string->len > 0))
	    done = check_sexp (string->str);
        }
      else if ((nread == 0) && (string->len == 0))
        {
          repl_c_string ("(quit)", 0, 0, 1);
	  gtk_exit (0);
        }

      if (done)
	{
	  if (string->len > 0)
	    {
	      repl_c_string (string->str, 0, 0, 1);
	      g_string_truncate (string, 0);
	    }
	  do_prompt ();
	}
    }
}

static void
init_finish ()
{
  init_lsubr ("gtk-signal-connect", glue_signal_connect);
  init_lsubr ("gtk-idle-add", glue_idle_add);
  init_lsubr ("gtk-idle-remove", glue_idle_remove);
  init_lsubr ("gtk-timeout-add", glue_timeout_add);
  init_lsubr ("gtk-timeout-remove", glue_timeout_remove);
}

static guint
data_hash (gpointer a)
{
  return (guint) a;
}

static gint
data_cmp (gpointer a, gpointer b)
{
  return (a == b);
}

static guint
id_hash (long *a)
{
  return *a;
}

static gint
id_cmp (long *a, long *b)
{
  return (*a == *b);
}

static long
pointer_insert (gpointer p)
{
  static long next_id = 1;
  static long *id_chunk = NULL;
  static int id_chunk_index = 0;
  long *id;

  if (!data_ht)
    {
      data_ht = g_hash_table_new ((GHashFunc) data_hash,
				  (GCompareFunc) data_cmp);
      id_ht = g_hash_table_new ((GHashFunc) id_hash,
				(GCompareFunc) id_cmp);
    }

  id = g_hash_table_lookup (data_ht, p);
  if (id)
    return *id;

  if (!id_chunk)
    {
      id_chunk = g_new (long, 1024);
      id_chunk_index = 0;
    }

  id = &id_chunk[id_chunk_index++];
  if (id_chunk_index >= 1024)
    {
      id_chunk = NULL;
      id_chunk_index = 0;
    }

  *id = next_id++;

  g_hash_table_insert (id_ht, id, p);
  g_hash_table_insert (data_ht, p, id);

  return *id;
}

static gpointer
pointer_lookup (long l)
{
  gpointer p;

  if (id_ht)
    {
      p = g_hash_table_lookup (id_ht, &l);
      return p;
    }

  return NULL;
}

static LISP
interp_func (char *name,
	     LISP  params)
{
  LISP return_val;
  MarshallData *data;
  Param args[32];
  Param return_param;
  int i;

  return_val = NIL;

  if (marshall_ht)
    {
      data = g_hash_table_lookup (marshall_ht, name);
      if (data)
	{
	  for (i = 0; i < data->nargs; i++)
	    {
	      if (NULLP (params))
		err ("not enough params", NIL);

	      switch (data->args[i])
		{
		case PARAM_UNKNOWN:
		  err ("unexpected param type", NIL);
		  break;
		case PARAM_VOID:
		  err ("unexpected param type", NIL);
		  break;
		case PARAM_CHAR:
		  args[i].c = (char) get_c_long (car (params));
		  break;
		case PARAM_SHORT:
		  args[i].s = (short) get_c_long (car (params));
		  break;
		case PARAM_INT:
		  args[i].i = (int) get_c_long (car (params));
		  break;
		case PARAM_UNSIGNED:
		  args[i].u = (unsigned) get_c_long (car (params));
		  break;
		case PARAM_FLOAT:
		  args[i].f = (float) get_c_double (car (params));
		  break;
		case PARAM_DOUBLE:
		  args[i].d = (double) get_c_double (car (params));
		  break;
		case PARAM_STRING:
		  args[i].str = (char*) get_c_string (car (params));
		  break;
		case PARAM_POINTER:
		  args[i].p = pointer_lookup (get_c_long (car (params)));
		  break;
		default:
		  err ("function type not supported as a parameter type", NIL);
		  break;
		}

	      params = cdr (params);
	    }

	  return_param = (* data->marshaller) (data->func, args);

	  switch (data->return_type)
	    {
	    case PARAM_UNKNOWN:
	      err ("unexpected return type", NIL);
	      break;
	    case PARAM_VOID:
	      break;
	    case PARAM_CHAR:
	      return_val = flocons (return_param.c);
	      break;
	    case PARAM_SHORT:
	      return_val = flocons (return_param.s);
	      break;
	    case PARAM_INT:
	      return_val = flocons (return_param.i);
	      break;
	    case PARAM_UNSIGNED:
	      return_val = flocons (return_param.u);
	      break;
	    case PARAM_FLOAT:
	      return_val = flocons (return_param.f);
	      break;
	    case PARAM_DOUBLE:
	      return_val = flocons (return_param.d);
	      break;
	    case PARAM_STRING:
	      return_val = strcons (strlen (return_param.str), return_param.str);
	      break;
	    case PARAM_POINTER:
	      return_val = flocons (pointer_insert (return_param.p));
	      break;
	    default:
	      err ("function type not supported as return value", NIL);
	      break;
	    }
	}
    }

  return return_val;
}

static LISP
glue_signal_connect (LISP params)
{
  LISP closure;
  GtkObject *object;
  char *name;
  int id;

  if (NULLP (params))
    err ("not enough params", NIL);

  object = pointer_lookup (get_c_long (car (params)));

  params = cdr (params);
  if (NULLP (params))
    err ("not enough params", NIL);

  name = get_c_string (car (params));

  params = cdr (params);
  if (NULLP (params))
    err ("not enough params", NIL);

  closure = car (params);
  gc_protect (&closure);

  id = gtk_signal_connect (object, name, NULL, closure);

  return flocons (id);
}

static LISP
glue_idle_add (LISP params)
{
  LISP closure;
  int id;

  if (NULLP (params))
    err ("not enough params", NIL);

  closure = car (params);
  gc_protect (&closure);

  id = gtk_idle_add (glue_callback, closure);

  return flocons (id);
}

static LISP
glue_idle_remove (LISP params)
{
  int id;

  if (NULLP (params))
    err ("not enough params", NIL);

  id = get_c_long (car (params));

  gtk_idle_remove (id);

  return NIL;
}

static LISP
glue_timeout_add (LISP params)
{
  LISP closure;
  guint32 interval;
  int id;

  if (NULLP (params))
    err ("not enough params", NIL);

  interval = get_c_long (car (params));

  params = cdr (params);
  if (NULLP (params))
    err ("not enough params", NIL);

  closure = car (params);
  gc_protect (&closure);

  id = gtk_timeout_add (interval, glue_callback, closure);

  return flocons (id);
}

static LISP
glue_timeout_remove (LISP params)
{
  int id;

  if (NULLP (params))
    err ("not enough params", NIL);

  id = get_c_long (car (params));

  gtk_timeout_remove (id);

  return NIL;
}

static void
glue_signal_marshal (GtkObject      *object,
		     gpointer        data,
		     gint            nparams,
		     GtkSignalParam *params,
		     GtkParamType   *param_types,
		     GtkParamType    return_type)
{
  LISP sexp;
  LISP closure;
  int i;

  sexp = NIL;

  for (i = nparams - 1; i >= 0; i--)
    {
      switch (param_types[i])
	{
	case GTK_PARAM_NONE:
	  err ("GTK_PARAM_NONE is an invalid parameter type", NIL);
	  break;
	case GTK_PARAM_CHAR:
	  sexp = cons (flocons (params[i].c), sexp);
	  break;
	case GTK_PARAM_SHORT:
	  sexp = cons (flocons (params[i].s), sexp);
	  break;
	case GTK_PARAM_INT:
	  sexp = cons (flocons (params[i].i), sexp);
	  break;
	case GTK_PARAM_LONG:
	  sexp = cons (flocons (params[i].l), sexp);
	  break;
	case GTK_PARAM_POINTER:
	  sexp = cons (flocons (pointer_insert (params[i].p)), sexp);
	  break;
	}
    }

  sexp = cons (flocons (pointer_insert (object)), sexp);

  closure = data;
  sexp = cons (closure, sexp);
  sexp = leval (sexp, NIL);

  switch (return_type)
    {
    case GTK_PARAM_NONE:
      break;
    case GTK_PARAM_CHAR:
      {
	char *return_val;
	return_val = ((char*) params[nparams].p);
	if (NULLP (sexp))
	  *return_val = 0;
	else
	  *return_val = (char) get_c_long (sexp);
      }
      break;
    case GTK_PARAM_SHORT:
      {
	short *return_val;
	return_val = ((short*) params[nparams].p);
	if (NULLP (sexp))
	  *return_val = 0;
	else
	  *return_val = (short) get_c_long (sexp);
      }
      break;
    case GTK_PARAM_INT:
      {
	int *return_val;
	return_val = ((int*) params[nparams].p);
	if (NULLP (sexp))
	  *return_val = 0;
	else
	  *return_val = (int) get_c_long (sexp);
      }
      break;
    case GTK_PARAM_LONG:
      {
	long *return_val;
	return_val = ((long*) params[nparams].p);
	if (NULLP (sexp))
	  *return_val = 0;
	else
	  *return_val = (long) get_c_long (sexp);
      }
      break;
    case GTK_PARAM_POINTER:
      {
	gpointer *return_val;
	return_val = ((gpointer*) params[nparams].p);
	if (NULLP (sexp))
	  *return_val = NULL;
	else
	  *return_val = pointer_lookup (get_c_long (sexp));
      }
      break;
    }
}

static void
glue_signal_destroy (gpointer data)
{
  LISP closure;

  closure = data;
  gc_unprotect (&closure);
}

static gint
glue_callback (gpointer data)
{
  LISP closure;
  LISP sexp;
  gint return_val;

  closure = data;
  sexp = cons (closure, NIL);
  sexp = leval (sexp, NIL);

  if (NULLP (sexp))
    return_val = 0;
  else
    return_val = get_c_long (sexp);

  if (!return_val)
    gc_unprotect (&closure);

  return return_val;
}
