/* GnomENIUS Calculator
 * Copyright (C) 1997 George Lebl.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <gmp.h>
#include <glib.h>
#include "eval.h"
#include "calc.h"
#include "dict.h"
#include "util.h"
#include "funclib.h"
#include "mymath.h"

extern error_t error_num;
extern calcstate_t calcstate;

extern void (*errorout)(char *);

/*returns 10 01 or 11 depending if the operation has left, right or both
branches*/
int
branches(int op)
{
	switch(op) {
		case E_EQUALS: return 3;
		case E_PLUS: return 3;
		case E_MINUS: return 3;
		case E_MUL: return 3;
		case E_DIV: return 3;
		case E_MOD: return 3;
		case E_NEG: return 2;
		case E_EXP: return 3;
		case E_FACT: return 1;
	}
	return 0;
}

/*sets s to the string representation of the primitive, s has to be big
enough! */
void
primstr(char *s, int op)
{
	switch(op) {
		case E_EQUALS: strcpy(s,"="); return;
		case E_PLUS: strcpy(s,"+"); return;
		case E_MINUS: strcpy(s,"-"); return;
		case E_MUL: strcpy(s,"*"); return;
		case E_DIV: strcpy(s,"/"); return;
		case E_MOD: strcpy(s,"%"); return;
		case E_NEG: strcpy(s,"~"); return;
		case E_EXP: strcpy(s,"^"); return;
		case E_FACT: strcpy(s,"!"); return;
	}
	strcpy(s,"<?>");
}

tree_t *
makenum_z_ui(unsigned long num)
{
	tree_t *n;
	n=(tree_t *)g_malloc(sizeof(tree_t));
	n->type=NUMBER_NODE;
	n->data.number.type=INTEGER_TYPE;
	mpz_init(n->data.number.data.ival);
	mpz_set_ui(n->data.number.data.ival,num);
	n->left=NULL;
	n->right=NULL;
	return n;
}

tree_t *
makenum_z(mpz_t num)
{
	tree_t *n;
	n=(tree_t *)g_malloc(sizeof(tree_t));
	n->type=NUMBER_NODE;
	n->data.number.type=INTEGER_TYPE;
	mpz_init_set(n->data.number.data.ival,num);
	n->left=NULL;
	n->right=NULL;
	return n;
}

tree_t *
makenum_q(mpq_t num)
{
	tree_t *n;
	n=(tree_t *)g_malloc(sizeof(tree_t));
	n->type=NUMBER_NODE;
	n->data.number.type=RATIONAL_TYPE;
	mpq_init(n->data.number.data.rval);
	mpq_set(n->data.number.data.rval,num);
	n->left=NULL;
	n->right=NULL;
	makeint(n); /*convert to int if it's possible (e.g. 44/1 is 44)*/
	return n;
}

tree_t *
makenum_f(mpf_t num)
{
	tree_t *n;
	n=(tree_t *)g_malloc(sizeof(tree_t));
	n->type=NUMBER_NODE;
	n->data.number.type=FLOAT_TYPE;
	mpf_init_set(n->data.number.data.fval,num);
	n->left=NULL;
	n->right=NULL;
	makeint(n); /*convert to int if it's possible (e.g. 44.0 is 44)*/
	return n;
}

tree_t *
makefuncb(int func,evalstack_t * stack)
{
	tree_t *n;
	n=(tree_t *)g_malloc(sizeof(tree_t));
	n->type=ACTION_NODE;
	n->data.action.type=PRIMITIVE_TYPE;
	n->data.action.data.primitive=func;
	if(branches(func)&2)
		n->right=t_pop(stack);
	else
		n->right=NULL;
	if(branches(func)&1)
		n->left=t_pop(stack);
	else
		n->left=NULL;
	return n;
}

tree_t *
makefuncd(func_t * func,evalstack_t * stack)
{
	tree_t *n;
	int i;

	n=(tree_t *)g_malloc(sizeof(tree_t));
	n->type=ACTION_NODE;
	n->data.action.type=FUNCTION_TYPE;
	n->data.action.data.function.func=func;
	n->data.action.data.function.args=
		(tree_t * *)g_malloc(sizeof(tree_t *)*func->args);
	for(i=func->args-1;i>=0;i--)
		n->data.action.data.function.args[i]=t_pop(stack);
	n->right=NULL;
	n->left=NULL;
	return n;
}

int
t_push(tree_t *n,evalstack_t * stack)
{
	if((stack->top+1)>=stack->size) {
		/*make the stack larger*/
		stack->stack=(tree_t * *)my_realloc(
			stack->stack,sizeof(tree_t *)*stack->size,
			sizeof(tree_t *)*(stack->size+256));
		if(stack->stack==NULL) /*out of mem right here!*/
			return FALSE;
		stack->size+=256;
	}
	stack->stack[++stack->top]=n;
	return TRUE;
}

tree_t *
t_pop(evalstack_t * stack)
{
	if(stack->top==-1)
		return NULL;
	else
		return stack->stack[stack->top--];
}

void
freenode(tree_t *n)
{
	if(!n)
		return;
	if(n->type==NUMBER_NODE) {
		if(n->data.number.type==INTEGER_TYPE)
			mpz_clear(n->data.number.data.ival);
		else if(n->data.number.type==FLOAT_TYPE)
			mpf_clear(n->data.number.data.fval);
		else /*RATIONAL_TYPE*/
			mpq_clear(n->data.number.data.rval);
	}
}

/*free arguments to a dictionary function*/
void
freeargs(tree_t *n)
{
	int i;

	if(n->type!=ACTION_NODE ||
		n->data.action.type!=FUNCTION_TYPE)
		return;

	if(!n->data.action.data.function.args)
		return;

	for(i=0;i<n->data.action.data.function.func->args;i++)
		if(n->data.action.data.function.args[i])
			freetree(n->data.action.data.function.args[i]);
}

void
freetree(tree_t *n)
{
	if(!n)
		return;
	if(n->left)
		freetree(n->left);
	if(n->right)
		freetree(n->right);
	freeargs(n);
	freenode(n);
}

tree_t *
copynode(tree_t *o)
{
	tree_t *n;
	int i;

	if(!o)
		return NULL;

	n=(tree_t *)g_malloc(sizeof(tree_t));
	n->left=copynode(o->left);
	n->right=copynode(o->right);

	n->type=o->type;
	if(o->type==NUMBER_NODE) {
		n->data.number.type=o->data.number.type;
		if(o->data.number.type==INTEGER_TYPE) {
			mpz_init_set(n->data.number.data.ival,
				o->data.number.data.ival);
		} else if(o->data.number.type==FLOAT_TYPE) {
			mpf_init_set(n->data.number.data.fval,
				o->data.number.data.fval);
		} else if(o->data.number.type==RATIONAL_TYPE) {
			mpq_init(n->data.number.data.rval);
			mpq_set(n->data.number.data.rval,
				o->data.number.data.rval);
		}
	} else { /*type==ACTION_NODE*/
		n->data.action.type=o->data.action.type;
		if(o->data.action.type==PRIMITIVE_TYPE) {
			n->data.action.data.primitive=
				o->data.action.data.primitive;
		} else if(o->data.number.type==FUNCTION_TYPE) {
			n->data.action.data.function.func=
				o->data.action.data.function.func;
			for(i=0;i<o->data.action.data.function.func->args;i++)
				n->data.action.data.function.args[i]=copynode(
					o->data.action.data.function.args[i]);
		}
	}

	return n;
}

/*evaluate a treenode, the tree node will become a number node*/
/*or at least all the calculatable parts will be calculated so*/
/*it will be a reduced tree*/
/*the tree will be freed*/
tree_t *
evalnode(tree_t *n)
{
	tree_t *r=NULL;
	int i;

	if(n==NULL)
		return NULL;
	if(n->type==NUMBER_NODE)
		return n;

	if(n->data.action.type==FUNCTION_TYPE) {
		for(i=0;i<n->data.action.data.function.func->args;i++)
			n->data.action.data.function.args[i]=
				evalnode(n->data.action.data.function.args[i]);
		
		if(n->data.action.data.function.func->type==USER_FUNC) {
			r=copynode(
				n->data.action.data.function.func->data.value);
			r=evalnode(r);
		} else {
			r=(*n->data.action.data.function.func->data.func)
				(n->data.action.data.function.args);
		}
		if(r!=NULL) {
			freetree(n);
			return r;
		} else if(error_num!=NO_ERROR) {
			/*something errored out*/
			freetree(n);
			return NULL;
		} /*r==NULL && error_num==NO_ERROR*/
		/*this error is not too serious, just this action can't
		  be computed*/
		return n;
	}

	if(n->right) {
		if((n->right=evalnode(n->right))==NULL) {
			freetree(n);
			return NULL; /*an error occured*/
		}
	}
	/*don't evaluate left side if this is an equals operation!*/
	if(n->data.action.data.primitive!=E_EQUALS && n->left) {
		if((n->left=evalnode(n->left))==NULL) {
			freetree(n);
			return NULL; /*an error occured*/
		}
		/*can't evaluate this!*/
		if(n->left->type!=NUMBER_NODE)
			return n;
	}
	/*can't evaluate this!*/
	if(n->right!=NULL && n->right->type!=NUMBER_NODE)
		return n;
	switch(n->data.action.data.primitive) {
		case E_EQUALS: r=equalsop(n->left,n->right); break;
		case E_PLUS: r=plusop(n->left,n->right); break;
		case E_MINUS: r=minusop(n->left,n->right); break;
		case E_MUL: r=mulop(n->left,n->right); break;
		case E_DIV: r=divop(n->left,n->right); break;
		case E_MOD: r=modop(n->left,n->right); break;
		case E_NEG: r=negop(n->right); break;
		case E_EXP: r=expop(n->left,n->right); break;
		case E_FACT: r=factop(n->left); break;
	}
	if(r!=NULL) {
		freetree(n);
		return r;
	} else if(error_num!=NO_ERROR) {
		/*something errored out*/
		freetree(n);
		return NULL;
	} /*r==NULL && error_num==NO_ERROR*/
	/*this error is not too serious, just this action can't be computed*/
	return n;
}

/*make node float*/
void
makefloat(tree_t *n)
{
	mpf_t fr;

	if(n->type!=NUMBER_NODE || n->data.number.type==FLOAT_TYPE)
		return;
	n->data.number.type=FLOAT_TYPE;
	if(n->data.number.type==RATIONAL_TYPE) {
		mpf_init(fr);
		mpf_set_q(fr,n->data.number.data.rval);
		mpq_clear(n->data.number.data.rval);
		mpf_init(n->data.number.data.fval);
		mpf_set(n->data.number.data.fval,fr);
		mpf_clear(fr);
	} else { /*type==INTEGER_TYPE*/
		mpf_init(fr);
		mpf_set_z(fr,n->data.number.data.ival);
		mpz_clear(n->data.number.data.ival);
		mpf_init_set(n->data.number.data.fval,fr);
		mpf_clear(fr);
	}
}

/*make float node rational*/
void
makefloatrational(tree_t *n)
{
	char *s;
	long int e;

	if(n->type!=NUMBER_NODE || n->data.number.type!=FLOAT_TYPE)
		return;

	s=mpf_get_str(NULL,&e,10,0,n->data.number.data.fval);
	e-=strlen(s);
	if(e>0) {
		s=my_realloc(s,strlen(s)+1,strlen(s)+e+1);
		for(;e>0;e--)
			strcat(s,"0");
	}
	mpf_clear(n->data.number.data.fval);
	mpq_init(n->data.number.data.rval);
	mpz_set_str(mpq_numref(n->data.number.data.rval),s,10);
	mpz_set_ui(mpq_denref(n->data.number.data.rval),10);
	mpz_pow_ui(mpq_denref(n->data.number.data.rval),
		mpq_denref(n->data.number.data.rval),-e);

	g_free(s);

	n->data.number.type=RATIONAL_TYPE;
	mpq_canonicalize(n->data.number.data.rval);
}


/*make node int if at all possible*/
void
makeint(tree_t *n)
{
	mpz_t ir;
	mpf_t fr;

	if(n->type!=NUMBER_NODE || n->data.number.type==INTEGER_TYPE)
		return;
	if(n->data.number.type==RATIONAL_TYPE) {
		mpq_canonicalize(n->data.number.data.rval);
		if(mpz_cmp_ui(mpq_denref(n->data.number.data.rval),1)==0) {
			mpz_init_set(ir,mpq_numref(n->data.number.data.rval));
			mpq_clear(n->data.number.data.rval);
			mpz_init_set(n->data.number.data.ival,ir);
			mpz_clear(ir);
			n->data.number.type=INTEGER_TYPE;
		}
	} else { /*type==FLOAT_TYPE*/
		if(!calcstate.make_floats_ints) /*are we allowed to do this?*/
			return;
		/*gotta find a better way of doing this!*/
		mpz_init(ir);
		mpz_set_f(ir,n->data.number.data.fval);
		mpf_init(fr);
		mpf_set_z(fr,ir);
		if(mpf_cmp(fr,n->data.number.data.fval)==0) {
			n->data.number.type=INTEGER_TYPE;
			mpf_clear(n->data.number.data.fval);
			mpz_init_set(n->data.number.data.ival,ir);
		}
		mpf_clear(fr);
		mpz_clear(ir);
	}
}

/*make both number nodes the same type*/
void
makesame(tree_t *l,tree_t *r)
{
	mpf_t fr;
	mpq_t rr;

	if(l->type!=NUMBER_NODE || r->type!=NUMBER_NODE)
		return;
	if(l->data.number.type==r->data.number.type)
		return;

	if(r->data.number.type==FLOAT_TYPE) {
		mpf_init(fr);
		if(l->data.number.type==INTEGER_TYPE) {
			mpf_set_z(fr,l->data.number.data.ival);
			mpz_clear(l->data.number.data.ival);
		} else if(l->data.number.type==RATIONAL_TYPE) {
			mpf_set_q(fr,l->data.number.data.rval);
			mpq_clear(l->data.number.data.rval);
		}
		mpf_init_set(l->data.number.data.fval,fr);
		l->data.number.type=FLOAT_TYPE;
		mpf_clear(fr);
	} else if(l->data.number.type==FLOAT_TYPE) {
		mpf_init(fr);
		if(r->data.number.type==INTEGER_TYPE) {
			mpf_set_z(fr,r->data.number.data.ival);
			mpz_clear(r->data.number.data.ival);
		} else if(r->data.number.type==RATIONAL_TYPE) {
			mpf_set_q(fr,r->data.number.data.rval);
			mpq_clear(r->data.number.data.rval);
		}
		mpf_init_set(r->data.number.data.fval,fr);
		r->data.number.type=FLOAT_TYPE;
		mpf_clear(fr);
	} else if(r->data.number.type==RATIONAL_TYPE) {
		mpq_init(rr); /*l is for sure integer!*/
		mpq_set_z(rr,l->data.number.data.ival);
		mpz_clear(l->data.number.data.ival);
		mpq_init(l->data.number.data.rval);
		mpq_set(l->data.number.data.rval,rr);
		l->data.number.type=RATIONAL_TYPE;
		mpq_clear(rr);
	} else if(l->data.number.type==RATIONAL_TYPE) {
		mpq_init(rr); /*r is for sure integer!*/
		mpq_set_z(rr,r->data.number.data.ival);
		mpz_clear(r->data.number.data.ival);
		mpq_init(r->data.number.data.rval);
		mpq_set(r->data.number.data.rval,rr);
		r->data.number.type=RATIONAL_TYPE;
		mpq_clear(rr);
	}
}

tree_t *
plusop(tree_t *l,tree_t *r)
{
	tree_t *n=NULL;

	makesame(l,r);

	/*if l is something so is r!*/
	switch(l->data.number.type) {
		case INTEGER_TYPE:
			mpz_add(l->data.number.data.ival,
				l->data.number.data.ival,
				r->data.number.data.ival);
			n=makenum_z(l->data.number.data.ival);
			break;
		case FLOAT_TYPE:
			mpf_add(l->data.number.data.fval,
				l->data.number.data.fval,
				r->data.number.data.fval);
			n=makenum_f(l->data.number.data.fval);
			break;
		case RATIONAL_TYPE:
			mpq_add(l->data.number.data.rval,
				l->data.number.data.rval,
				r->data.number.data.rval);
			n=makenum_q(l->data.number.data.rval);
			break;
	}
	return n;
}

tree_t *
minusop(tree_t *l,tree_t *r)
{
	tree_t *n=NULL;

	makesame(l,r);

	/*if l is something so is r!*/
	switch(l->data.number.type) {
		case INTEGER_TYPE:
			mpz_sub(l->data.number.data.ival,
				l->data.number.data.ival,
				r->data.number.data.ival);
			n=makenum_z(l->data.number.data.ival);
			break;
		case FLOAT_TYPE:
			mpf_sub(l->data.number.data.fval,
				l->data.number.data.fval,
				r->data.number.data.fval);
			n=makenum_f(l->data.number.data.fval);
			break;
		case RATIONAL_TYPE:
			mpq_sub(l->data.number.data.rval,
				l->data.number.data.rval,
				r->data.number.data.rval);
			n=makenum_q(l->data.number.data.rval);
			break;
	}
	return n;
}

tree_t *
mulop(tree_t *l,tree_t *r)
{
	tree_t *n=NULL;

	makesame(l,r);

	/*if l is something so is r!*/
	switch(l->data.number.type) {
		case INTEGER_TYPE:
			mpz_mul(l->data.number.data.ival,
				l->data.number.data.ival,
				r->data.number.data.ival);
			n=makenum_z(l->data.number.data.ival);
			break;
		case FLOAT_TYPE:
			mpf_mul(l->data.number.data.fval,
				l->data.number.data.fval,
				r->data.number.data.fval);
			n=makenum_f(l->data.number.data.fval);
			break;
		case RATIONAL_TYPE:
			mpq_mul(l->data.number.data.rval,
				l->data.number.data.rval,
				r->data.number.data.rval);
			n=makenum_q(l->data.number.data.rval);
			break;
	}
	return n;
}

tree_t *
divop(tree_t *l,tree_t *r)
{
	mpq_t rr;
	tree_t *n=NULL;

	makesame(l,r);

	/*if l is something so is r!*/
	switch(l->data.number.type) {
		case INTEGER_TYPE:
			if(mpz_sgn(r->data.number.data.ival)==0) {
				(*errorout)("Division by zero! (ignoring '/')");
				return NULL;
			}
			mpq_init(rr);
			mpq_set_z(rr,l->data.number.data.ival);
			mpz_set(mpq_denref(rr),r->data.number.data.ival);
			n=makenum_q(rr);
			mpq_clear(rr);
			break;
		case FLOAT_TYPE:
			if(mpf_sgn(r->data.number.data.fval)==0) {
				(*errorout)("Division by zero! (ignoring '/')");
				return NULL;
			}
			mpf_div(l->data.number.data.fval,
				l->data.number.data.fval,
				r->data.number.data.fval);
			n=makenum_f(l->data.number.data.fval);
			break;
		case RATIONAL_TYPE:
			if(mpq_sgn(r->data.number.data.rval)==0) {
				(*errorout)("Division by zero! (ignoring '/')");
				return NULL;
			}
			mpq_div(l->data.number.data.rval,
				l->data.number.data.rval,
				r->data.number.data.rval);
			n=makenum_q(l->data.number.data.rval);
			break;
	}
	return n;
}

tree_t *
modop(tree_t *l,tree_t *r)
{
	tree_t *n=NULL;

	makesame(l,r);

	/*if l is something so is r!*/
	if(l->data.number.type==INTEGER_TYPE) {
		if(mpz_sgn(r->data.number.data.ival)==0) {
			(*errorout)("Division by zero! (ignoring '%')");
			return NULL;
		}
		mpz_mod(l->data.number.data.ival,l->data.number.data.ival,
			r->data.number.data.ival);
		n=makenum_z(l->data.number.data.ival);
	} else {
		(*errorout)("Can't do modulo of floats or rationals!"
			" (ignoring '%')");
		return NULL;
	}
	return n;
}

tree_t *
negop(tree_t *r)
{
	tree_t *n=NULL;

	switch(r->data.number.type) {
		case INTEGER_TYPE:
			mpz_neg(r->data.number.data.ival,
				r->data.number.data.ival);
			n=makenum_z(r->data.number.data.ival);
			break;
		case FLOAT_TYPE:
			mpf_neg(r->data.number.data.fval,
				r->data.number.data.fval);
			n=makenum_f(r->data.number.data.fval);
			break;
		case RATIONAL_TYPE:
			mpq_neg(r->data.number.data.rval,
				r->data.number.data.rval);
			n=makenum_q(r->data.number.data.rval);
			break;
	}
	return n;
}

/*this can do rational exponantiation*/
/*this is hardly an optimized calculaton!!!!!!!*/
tree_t *
expoprat(tree_t *l, tree_t *r)
{
	mpf_t fr;
	mpf_t fr2;
	mpf_t frt;
	unsigned long ne;
	unsigned long de;
	tree_t *r1;
	tree_t *r2;
	tree_t *n;

	if((mpz_cmp_ui(mpq_numref(r->data.number.data.rval),ULONG_MAX)>0) ||
		(mpz_cmp_ui(mpq_denref(r->data.number.data.rval),ULONG_MAX)>0))
	{
		(*errorout)("Exponent numerator/quotient too big!"
			" (ignoring '^')");
		return NULL;
	}
	if(mpz_sgn(mpq_numref(r->data.number.data.rval))==0)
		/*exponent is 0 return 1*/
		return makenum_z_ui(1);

	ne=mpz_get_ui(mpq_numref(r->data.number.data.rval));
	de=mpz_get_ui(mpq_denref(r->data.number.data.rval));
				/*this will make it allways
					positive!*/

	makefloat(l);
	

	/*
	 * Newton's method: Xn+1 = Xn - f(Xn)/f'(Xn)
	 */
	
	mpf_init(fr);
	mpf_init(fr2);
	mpf_init(frt);
	mpf_div_ui(fr,l->data.number.data.fval,2); /*use half the value
						     as an initial guess*/
	for(;;) {
		mympf_pow_ui(fr2,fr,de);
		mpf_sub(fr2,fr2,l->data.number.data.fval);

		mympf_pow_ui(frt,fr,de-1);
		mpf_mul_ui(frt,frt,de);
		mpf_div(fr2,fr2,frt);
		mpf_neg(fr2,fr2);
		mpf_add(fr2,fr2,fr);

		
		if(mpf_cmp(fr2,fr)==0)
			break;
		mpf_set(fr,fr2);
	}
	mpf_clear(fr2);
	mpf_clear(frt);

	r1=makenum_f(fr);
	mpf_clear(fr);

	r2=makenum_z(mpq_numref(r->data.number.data.rval));

	n=expopint(r1,r2);
	freetree(r1);
	freetree(r2);
	return n;
}
/*this can do integer exponantiation with all other types*/
tree_t *
expopint(tree_t *l, tree_t *r)
{
	mpq_t rr;
	unsigned long e;
	int reverse=FALSE;
	tree_t *n=NULL;

	if(mpz_cmp_ui(r->data.number.data.ival,ULONG_MAX)>0) {
		(*errorout)("Exponent too big! (ignoring '^')");
		return NULL;
	}
	if(mpz_sgn(r->data.number.data.ival)==0) /*exponent is 0 return 1*/
		return makenum_z_ui(1);

	e=mpz_get_ui(r->data.number.data.ival); /*this will make it allways
						positive!*/
	if(mpz_sgn(r->data.number.data.ival)<0)
		reverse=TRUE;

	switch(l->data.number.type) {
		case RATIONAL_TYPE:
			mpq_init(rr);
			mpz_pow_ui(mpq_numref(rr),
				mpq_numref(l->data.number.data.rval),e);
			mpz_pow_ui(mpq_denref(rr),
				mpq_denref(l->data.number.data.rval),e);
			/*the exponent was negative! reverse the result!*/
			if(reverse)
				mpq_inv(rr,rr);
			n=makenum_q(rr);
			mpq_clear(rr);
			break;
		case INTEGER_TYPE:
			/*start calculation as if e was negative*/
			mpq_init(rr);
			mpz_set_ui(mpq_numref(rr),1);
			mpz_pow_ui(mpq_denref(rr),l->data.number.data.ival,
				e);
			if(!reverse) {
				n=makenum_z(mpq_denref(rr));
			} else
				n=makenum_q(rr);
			mpq_clear(rr);
			break;
		case FLOAT_TYPE:
			mympf_pow_ui(l->data.number.data.fval,
				l->data.number.data.fval,e);

			if(reverse)
				mpf_ui_div(l->data.number.data.fval,1,
					l->data.number.data.fval);
			n=makenum_f(l->data.number.data.fval);
			break;
	}
	return n;
}

tree_t *
expop(tree_t *l, tree_t *r)
{
	if(r->data.number.type==INTEGER_TYPE)
		return expopint(l,r);
	else if(r->data.number.type==RATIONAL_TYPE)
		return expoprat(l,r);

	/*FIXME: this is ugly and slow and has gotta go*/
	makefloatrational(r);
	return expoprat(l,r);
}

tree_t *
factop(tree_t *l)
{
	tree_t *n;

	if(l->data.number.type==INTEGER_TYPE) {
		if(mpz_sgn(l->data.number.data.ival)==-1) {
			(*errorout)("Can't do factorials of negative numbers!"
				" (ignoring '!')");
			return NULL;
		}
		if(mpz_cmp_ui(l->data.number.data.ival,ULONG_MAX)>0) {
			(*errorout)("Number too large to compute factorial!"
				" (ignoring '!')");
			return NULL;
		}
		mpz_fac_ui(l->data.number.data.ival,
			mpz_get_ui(l->data.number.data.ival));
		n=makenum_z(l->data.number.data.ival);
	} else {
		(*errorout)("Can't do factorials of rationals or floats!"
				" (ignoring '!')");
		return NULL;
	}
	return n;
}

tree_t *
equalsop(tree_t *l, tree_t *r)
{
	if(l->type!=ACTION_NODE ||
		l->data.action.type!=FUNCTION_TYPE) {
		(*errorout)("Left value is NOT a function! (ignoring '=')");
		return NULL;
	}
	replacefunc(l->data.action.data.function.func,
		d_makeufunc(l->data.action.data.function.func->id,
		copynode(r),0));

	return copynode(r);
}
