/* GnomENIUS Calculator
 * Copyright (C) 1997 George Lebl.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <glib.h>
#include <gmp.h>
#include "eval.h"
#include "dict.h"
#include "funclib.h"
#include "mymath.h"

/*sin function*/
tree_t *
sin_op(tree_t * * a)
{
	mpf_t fr;
	mpf_t pitmp;

	tree_t *n;

	if(a[0]->type!=NUMBER_NODE)
		return NULL;

	mpf_init(fr);
	makefloat(a[0]);
	mpf_set(fr,a[0]->data.number.data.fval);

	mpf_init(pitmp);
	mympf_getpi(pitmp);
	mpf_mul_ui(pitmp,pitmp,2);

 	/*mod it down to -pi*2<fr<pi*2 */
 	while(mpf_cmp(fr,pitmp)>0)
 		mpf_sub(fr,fr,pitmp);
 	mpf_neg(pitmp,pitmp);
 	while(mpf_cmp(fr,pitmp)<0)
 		mpf_sub(fr,fr,pitmp);
 	mpf_clear(pitmp);

	mympf_sin(fr,fr);

	n=makenum_f(fr);
	mpf_clear(fr);
	return n;
}

/*cos function*/
tree_t *
cos_op(tree_t * * a)
{
	mpf_t fr;
	mpf_t pitmp;

	tree_t *n;

	if(a[0]->type!=NUMBER_NODE)
		return NULL;

	mpf_init(fr);
	makefloat(a[0]);
	mpf_set(fr,a[0]->data.number.data.fval);


	mpf_init(pitmp);
	mympf_getpi(pitmp);
	mpf_mul_ui(pitmp,pitmp,2);

 	/*get it down to -pi*2<fr<pi*2 */
 	while(mpf_cmp(fr,pitmp)>0)
 		mpf_sub(fr,fr,pitmp);
 	mpf_neg(pitmp,pitmp);
 	while(mpf_cmp(fr,pitmp)<0)
 		mpf_sub(fr,fr,pitmp);
 	mpf_clear(pitmp);


	mympf_cos(fr,fr);

	n=makenum_f(fr);
	mpf_clear(fr);
	return n;
}

/*tan function*/
tree_t *
tan_op(tree_t * * a)
{
	mpf_t fr;
	mpf_t fr2;
	mpf_t pitmp;

	tree_t *n;

	if(a[0]->type!=NUMBER_NODE)
		return NULL;

	mpf_init(fr);
	makefloat(a[0]);
	mpf_set(fr,a[0]->data.number.data.fval);


	mpf_init(pitmp);
	mympf_getpi(pitmp);
	mpf_mul_ui(pitmp,pitmp,2);

 	/*get it down to -pi*2<fr<pi*2 */
 	while(mpf_cmp(fr,pitmp)>0)
 		mpf_sub(fr,fr,pitmp);
 	mpf_neg(pitmp,pitmp);
 	while(mpf_cmp(fr,pitmp)<0)
 		mpf_sub(fr,fr,pitmp);
 	mpf_clear(pitmp);

	/*is this algorithm allways precise??? sin/cos*/
	mpf_init(fr2);
	mympf_cos(fr2,fr);
	mympf_sin(fr,fr);
	mpf_div(fr,fr,fr2);
	mpf_clear(fr2);

	n=makenum_f(fr);
	mpf_clear(fr);
	return n;
}

/*e function (or e variable actually)*/
tree_t *
e_op(tree_t * * a)
{
	mpf_t fr;

	tree_t *n;

	mpf_init(fr);
	mympf_gete(fr);

	n=makenum_f(fr);
	mpf_clear(fr);
	return n;
}

/*pi function (or pi variable or whatever)*/
tree_t *
pi_op(tree_t * * a)
{
	mpf_t fr;

	tree_t *n;

	mpf_init(fr);
	mympf_getpi(fr);

	n=makenum_f(fr);
	mpf_clear(fr);
	return n;
}

/*add the routines to the dictionary*/
void
funclib_addall(void)
{
	d_addfunc(d_makebifunc("sin",sin_op,1));
	d_addfunc(d_makebifunc("cos",cos_op,1));
	d_addfunc(d_makebifunc("tan",tan_op,1));
	d_addfunc(d_makebifunc("pi",pi_op,0));
	d_addfunc(d_makebifunc("e",e_op,0));
}
