#include <libintl.h>
#include <string.h>
#include "applet.h"
#include "gdkextra.h"

#define _(String) gettext(String)


#define APPLET_CALLBACK "panel_applet_callback"
#define APPLET_ID       "panel_applet_id"
#define APPLET_PARAMS   "panel_applet_params"
#define APPLET_FLAGS    "panel_applet_flags"

#define DEFAULT_STEP_SIZE 6
#define DEFAULT_DELAY     0
#define DEFAULT_HEIGHT    48


typedef struct {
	int x, y;
} Point;


static GtkWidget *applet_menu;
static GtkWidget *applet_menu_prop_separator;
static GtkWidget *applet_menu_prop_item;


Panel *the_panel = NULL;


static void
move_window(GtkWindow *window, int x, int y)
{
	GtkWidget *widget;

	widget = GTK_WIDGET(window);
	
	gdk_window_set_hints(widget->window, x, y, 0, 0, 0, 0, GDK_HINT_POS);
	gdk_window_move(widget->window, x, y);
	gtk_widget_draw(widget, NULL); /* FIXME: this should draw only the newly exposed area! */
}


static void
move_horiz(int src_x, int dest_x)
{
	int x;

	if (the_panel->step_size != 0)
		if (src_x < dest_x)
			for (x = src_x; x < dest_x; x += the_panel->step_size) {
				move_window(GTK_WINDOW(the_panel->window), x, 0); 
				/* FIXME: do delay */
			}
		else
			for (x = src_x; x > dest_x; x -= the_panel->step_size) {
				move_window(GTK_WINDOW(the_panel->window), x, 0); 
				/* FIXME: do delay */
			}
	
	move_window(GTK_WINDOW(the_panel->window), dest_x, 0);
}

static void
move_vert(int src_y, int dest_y)
{
	int y;

	if (the_panel->step_size != 0)
		if (src_y < dest_y)
			for (y = src_y; y < dest_y; y += the_panel->step_size) {
				move_window(GTK_WINDOW(the_panel->window), 0, y);
				/* FIXME: do delay */
			}
		else
			for (y = src_y; y > dest_y; y -= the_panel->step_size) {
				move_window(GTK_WINDOW(the_panel->window), 0, y);
				/* FIXME: do delay */
			}

	move_window(GTK_WINDOW(the_panel->window), 0, dest_y);
}


static void
pop_up(void)
{
	int width, height;
	int swidth, sheight;
	
	if (the_panel->state == PANEL_SHOWN)
		return;

	width   = the_panel->window->allocation.width;
	height  = the_panel->window->allocation.height;
	swidth  = gdk_screen_width();
	sheight = gdk_screen_height();

	switch (the_panel->pos) {
		case PANEL_POS_TOP:
			move_vert(-height + 1, 0);
			break;

		case PANEL_POS_BOTTOM:
			move_vert(sheight - 1, sheight - height);
			break;

		case PANEL_POS_LEFT:
			move_horiz(-width + 1, 0);
			break;

		case PANEL_POS_RIGHT:
			move_horiz(swidth - 1, swidth - width);
			break;
	}

	the_panel->state = PANEL_SHOWN;
}

static void
pop_down(void)
{
	int width, height;
	int swidth, sheight;

	if (the_panel->state == PANEL_HIDDEN)
		return;

	width   = the_panel->window->allocation.width;
	height  = the_panel->window->allocation.height;
	swidth  = gdk_screen_width();
	sheight = gdk_screen_height();

	switch (the_panel->pos) {
		case PANEL_POS_TOP:
			move_vert(0, -height + 1);
			break;

		case PANEL_POS_BOTTOM:
			move_vert(sheight - height, sheight - 1);
			break;

		case PANEL_POS_LEFT:
			move_horiz(0, -width + 1);
			break;

		case PANEL_POS_RIGHT:
			move_horiz(swidth - width, swidth - 1);
			break;
	}

	the_panel->state = PANEL_HIDDEN;
}


static void
panel_block_signals(void)
{
	gtk_signal_handler_block(GTK_OBJECT(the_panel->window), the_panel->enter_notify_id);
	gtk_signal_handler_block(GTK_OBJECT(the_panel->window), the_panel->leave_notify_id);
}


static void
panel_unblock_signals(void)
{
	gtk_signal_handler_unblock(GTK_OBJECT(the_panel->window), the_panel->leave_notify_id);
	gtk_signal_handler_unblock(GTK_OBJECT(the_panel->window), the_panel->enter_notify_id);
}


static gint
panel_enter_notify(GtkWidget *widget, GdkEventCrossing *event, gpointer data)
{
	if ((the_panel->mode == PANEL_STAYS_PUT) ||
	    (event->detail == GDK_NOTIFY_INFERIOR))
		return FALSE;

	panel_block_signals();
	pop_up();
	panel_unblock_signals();
	
	return FALSE;
}


static gint
panel_leave_notify(GtkWidget *widget, GdkEventCrossing *event, gpointer data)
{
	if ((the_panel->mode == PANEL_STAYS_PUT) ||
	    (event->detail == GDK_NOTIFY_INFERIOR))
		return FALSE;

	panel_block_signals();
	pop_down();
	panel_unblock_signals();

	return FALSE;
}

static void
max_pos(GtkWidget *widget, gpointer data)
{
	Point *point;

	point = data;

	point->x = MAX(point->x, widget->allocation.x + widget->allocation.width);
	point->y = MAX(point->y, widget->allocation.y + widget->allocation.height);
}


static void
find_applet_placement(int *x, int *y)
{
	Point point;

	point.x = 0;
	point.y = 0;

	*x = 0;
	*y = 0;
	return; /* FIXME: this is for now; fix to calculate appropriate position */
	
	gtk_container_foreach(GTK_CONTAINER(the_panel->fixed),
			      (GtkCallback) max_pos,
			      &point);

	switch (the_panel->pos) {
		case PANEL_POS_TOP:
		case PANEL_POS_BOTTOM:
			*x = point.x;
			*y = 0;
			break;

		case PANEL_POS_LEFT:
		case PANEL_POS_RIGHT:
			*x = 0;
			*y = point.y;
			break;
	}
}


static void
change_window_cursor(GdkWindow *window, GdkCursorType cursor_type)
{
	GdkCursor *cursor;

	cursor = gdk_cursor_new(cursor_type);
	gdk_window_set_cursor(window, cursor);
	gdk_cursor_destroy(cursor);
}


static void
realize_change_cursor(GtkWidget *widget, gpointer data)
{
	change_window_cursor(widget->window, GDK_ARROW);
}


static AppletCallback
get_applet_callback(GtkWidget *applet)
{
	return gtk_object_get_data(GTK_OBJECT(applet), APPLET_CALLBACK);
}


static char *
get_applet_id(GtkWidget *applet)
{
	return gtk_object_get_data(GTK_OBJECT(applet), APPLET_ID);
}


static char *
get_applet_params(GtkWidget *applet)
{
	return gtk_object_get_data(GTK_OBJECT(applet), APPLET_PARAMS);
}


static long
get_applet_flags(GtkWidget *applet)
{
	return (long) gtk_object_get_data(GTK_OBJECT(applet), APPLET_FLAGS);
}


static void
get_applet_geometry(GtkWidget *applet, int *x, int *y, int *width, int *height)
{
	if (x)
		*x = applet->allocation.x;

	if (y)
		*y = applet->allocation.y;

	if (width)
		*width = applet->allocation.width;

	if (height)
		*height = applet->allocation.height;
}


static gpointer
call_applet(GtkWidget *applet, AppletCommand *cmd)
{
	AppletCallback callback;
	
	cmd->panel  = the_panel;
	cmd->applet = GTK_BIN(applet)->child;

	callback = get_applet_callback(applet);

	if (callback)
		return (*callback) (cmd);
	else
		return NULL;
}


static void
save_applet_configuration(GtkWidget *widget, gpointer data)
{
	static int num = 0;

	int   xpos, ypos;
	char *id;
	char *params;
	char *path;
	char  buf[256];

	id     = get_applet_id(widget);
	params = get_applet_params(widget);
	get_applet_geometry(widget, &xpos, &ypos, NULL, NULL);

	/* XXX: The increasing number is sort of a hack to guarantee unique keys */

	sprintf(buf, "%d,", num++);
	path = g_copy_strings("/panel/Applets/", buf, id, ",", params, NULL);
	sprintf(buf, "%d %d", xpos, ypos);
	gnome_config_set_string(path, buf);
	g_free(path);
}


static void
panel_quit(void)
{
	gnome_config_clean_section("/panel/Applets");
	gtk_container_foreach(GTK_CONTAINER(the_panel->fixed), save_applet_configuration, NULL);
	gnome_config_sync();
	gtk_widget_destroy(the_panel->window);
	applets_destroy();
}


static void
applet_drag_start(GtkWidget *applet, int warp)
{
	the_panel->applet_being_dragged = applet;

	if (warp)
		gdk_pointer_warp(NULL, applet->window,
				 0, 0, 0, 0,
				 applet->allocation.width / 2,
				 applet->allocation.height / 2);
	
	gtk_widget_get_pointer(the_panel->window,
			       &the_panel->applet_drag_click_x,
			       &the_panel->applet_drag_click_y);
	get_applet_geometry(applet,
			    &the_panel->applet_drag_orig_x,
			    &the_panel->applet_drag_orig_y,
			    NULL,
			    NULL);

	gtk_grab_add(applet);
}


static void
applet_drag_end(GtkWidget *applet)
{
	the_panel->applet_being_dragged = NULL;
	gtk_grab_remove(applet);
}


static void
move_applet_callback(GtkWidget *widget, gpointer data)
{
	GtkWidget      *applet;

	/* FIXME: when the mouse moves outside of the applet, no
	 * motion events are sent to it, even when the applet has a
	 * gtk_grab to it.  However, when the drag was started by the
	 * mouse instead of the menu, it works.  I don't know why this
	 * happens.
	 */
	
	applet = gtk_object_get_user_data(GTK_OBJECT(applet_menu));
	applet_drag_start(applet, TRUE);
}


static void
remove_applet_callback(GtkWidget *widget, gpointer data)
{
	GtkWidget *applet;

	applet = gtk_object_get_user_data(GTK_OBJECT(applet_menu));
	gtk_widget_destroy(applet);
}


static void
applet_properties_callback(GtkWidget *widget, gpointer data)
{
	GtkWidget     *applet;
	AppletCommand  cmd;

	applet = gtk_object_get_user_data(GTK_OBJECT(applet_menu));

	cmd.cmd = APPLET_CMD_PROPERTIES;

	call_applet(applet, &cmd);
}


static void
create_applet_menu(void)
{
	GtkWidget *menuitem;
	
	applet_menu = gtk_menu_new();

	menuitem = gtk_menu_item_new_with_label(_("Move applet"));
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
			   (GtkSignalFunc) move_applet_callback,
			   NULL);
	gtk_menu_append(GTK_MENU(applet_menu), menuitem);
	gtk_widget_show(menuitem);

	menuitem = gtk_menu_item_new_with_label(_("Remove from panel"));
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
			   (GtkSignalFunc) remove_applet_callback,
			   NULL);
	gtk_menu_append(GTK_MENU(applet_menu), menuitem);
	gtk_widget_show(menuitem);

	menuitem = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(applet_menu), menuitem);
	gtk_widget_show(menuitem);
	applet_menu_prop_separator = menuitem;
	
	menuitem = gtk_menu_item_new_with_label(_("Applet properties..."));
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
			   (GtkSignalFunc) applet_properties_callback,
			   NULL);
	gtk_menu_append(GTK_MENU(applet_menu), menuitem);
	gtk_widget_show(menuitem);
	applet_menu_prop_item = menuitem;

	gtk_signal_connect_after(GTK_OBJECT(applet_menu), "deactivate",
				 (GtkSignalFunc) panel_unblock_signals,
				 NULL);
}


static void
show_applet_menu(GtkWidget *applet)
{
	long flags;

	flags = get_applet_flags(applet);

	if (flags & APPLET_HAS_PROPERTIES) {
		gtk_widget_show(applet_menu_prop_separator);
		gtk_widget_show(applet_menu_prop_item);
	} else {
		gtk_widget_hide(applet_menu_prop_separator);
		gtk_widget_hide(applet_menu_prop_item);
	}

	gtk_object_set_user_data(GTK_OBJECT(applet_menu), applet);

	panel_block_signals();
	gtk_menu_popup(GTK_MENU(applet_menu), NULL, NULL, NULL, NULL, 3, 0);
}


static gint
panel_applet_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	GdkEventButton *bevent;
	gint            x, y;
	int             x1, y1, x2, y2;
	int             dx, dy;
	int             xpos, ypos;
	int             width, height;

	get_applet_geometry(widget, &xpos, &ypos, &width, &height);

	switch (event->type) {
		case GDK_BUTTON_PRESS:
			bevent = (GdkEventButton *) event;

			if (the_panel->applet_being_dragged)
				return TRUE;

			switch (bevent->button) {
				case 2: /* Start drag */
					if (the_panel->applet_being_dragged)
						break;

					applet_drag_start(widget, FALSE);
					return TRUE;

				case 3: /* Applet menu */
					show_applet_menu(widget);
					return TRUE;
			}

			break;

		case GDK_BUTTON_RELEASE:
			bevent = (GdkEventButton *) event;
			
			if (the_panel->applet_being_dragged == widget) {
				applet_drag_end(widget);
				return TRUE;
			}

			break;

		case GDK_MOTION_NOTIFY:
			if (the_panel->applet_being_dragged == widget) {
				gtk_widget_get_pointer(the_panel->window, &x, &y);

				dx = x - the_panel->applet_drag_click_x;
				dy = y - the_panel->applet_drag_click_y;

				x1 = the_panel->applet_drag_orig_x + dx;
				y1 = the_panel->applet_drag_orig_y + dy;
				x2 = x1 + width;
				y2 = y1 + height;

				if (x2 > the_panel->window->allocation.width)
					x1 = the_panel->window->allocation.width - width;

				if (y2 > the_panel->window->allocation.height)
					y1 = the_panel->window->allocation.height - height;

				if (x1 < 0)
					x1 = 0;

				if (y1 < 0)
					y1 = 0;

				if ((x1 != xpos) || (y1 != ypos))
					gtk_fixed_move(GTK_FIXED(the_panel->fixed), widget, x1, y1);

				return TRUE;
			}

			break;

		default:
			break;
	}

	return FALSE;
}

static GtkWidget *
listening_parent(GtkWidget *widget)
{
	if (GTK_WIDGET_NO_WINDOW(widget))
		return listening_parent(widget->parent);

	return widget;
}


static gint
panel_sub_event_handler (GtkWidget *widget, GdkEvent *event, gpointer data)
{
	GdkEventButton *bevent;

	switch (event->type) {
		case GDK_BUTTON_PRESS:
			bevent = (GdkEventButton *) event;

			switch (bevent->button) {
				case 2:
				case 3:
					return gtk_widget_event(listening_parent(widget->parent), event);
			}

			break;

		case GDK_BUTTON_RELEASE:
			bevent = (GdkEventButton *) event;
			
			if (the_panel->applet_being_dragged)
				return gtk_widget_event(listening_parent(widget->parent), event);
			break;

		case GDK_MOTION_NOTIFY:
			if (the_panel->applet_being_dragged)
				return gtk_widget_event(listening_parent(widget->parent), event);
			break;

		default:
			break;
	}

	return FALSE;
}


static void
get_applet_type(gpointer key, gpointer value, gpointer user_data)
{
	GList **list = user_data;

	*list = g_list_prepend(*list, g_strdup(key));
}


GList *
get_applet_types(void)
{
	GList *list = NULL;

	g_hash_table_foreach(applet_files_ht, get_applet_type, &list);
	return list;
}


void
panel_init(void)
{
	the_panel = g_new(Panel, 1);

	the_panel->window = gtk_window_new(GTK_WINDOW_POPUP);

	the_panel->fixed = gtk_fixed_new();
	gtk_container_add(GTK_CONTAINER(the_panel->window), the_panel->fixed);
	gtk_widget_show(the_panel->fixed);

	the_panel->pos   = PANEL_POS_BOTTOM;
	the_panel->state = PANEL_SHOWN;
	the_panel->mode  = PANEL_GETS_HIDDEN;
	gtk_widget_set_usize(the_panel->window, gdk_screen_width(), DEFAULT_HEIGHT);
	gtk_widget_set_uposition(the_panel->window, 0, gdk_screen_height() - DEFAULT_HEIGHT);

#if 0
	the_panel->pos   = PANEL_POS_TOP;
	the_panel->state = PANEL_SHOWN;
	the_panel->mode  = PANEL_GETS_HIDDEN;
	gtk_widget_set_usize(the_panel->window, gdk_screen_width(), DEFAULT_HEIGHT);
	gtk_widget_set_uposition(the_panel->window, 0, 0);
#endif
#if 0
	the_panel->pos   = PANEL_POS_RIGHT;
	the_panel->state = PANEL_SHOWN;
	the_panel->mode  = PANEL_GETS_HIDDEN;
	gtk_widget_set_usize(the_panel->window, DEFAULT_HEIGHT, gdk_screen_height());
	gtk_widget_set_uposition(the_panel->window, gdk_screen_width() - DEFAULT_HEIGHT, 0);
#endif
#if 0
	the_panel->pos   = PANEL_POS_LEFT;
	the_panel->state = PANEL_SHOWN;
	the_panel->mode  = PANEL_GETS_HIDDEN;
	gtk_widget_set_usize(the_panel->window, DEFAULT_HEIGHT, gdk_screen_height());
	gtk_widget_set_uposition(the_panel->window, 0, 0);
#endif
	the_panel->step_size            = DEFAULT_STEP_SIZE;
	the_panel->delay                = DEFAULT_DELAY;
	the_panel->applet_being_dragged = NULL;

	the_panel->enter_notify_id = gtk_signal_connect(GTK_OBJECT(the_panel->window), "enter_notify_event",
							(GtkSignalFunc) panel_enter_notify,
							the_panel);
	the_panel->leave_notify_id = gtk_signal_connect(GTK_OBJECT(the_panel->window), "leave_notify_event",
							(GtkSignalFunc) panel_leave_notify,
							the_panel);
	gtk_signal_connect_after(GTK_OBJECT(the_panel->window), "realize",
				 (GtkSignalFunc) realize_change_cursor,
				 NULL);

	create_applet_menu();
}


void
panel_bind_events (GtkWidget *widget, void *data)
{
	gint       events;

	if (!GTK_WIDGET_NO_WINDOW (widget)) {
		events = (gtk_widget_get_events(widget) |
			  GDK_BUTTON_PRESS_MASK |
			  GDK_BUTTON_RELEASE_MASK |
			  GDK_POINTER_MOTION_MASK |
			  GDK_POINTER_MOTION_HINT_MASK);
		
		gtk_widget_set_events(widget, events);
		
		gtk_signal_connect(GTK_OBJECT(widget), "event",
				   (GtkSignalFunc) panel_sub_event_handler, the_panel);
	}
	
	if (GTK_IS_CONTAINER (widget))
		gtk_container_foreach (GTK_CONTAINER (widget), panel_bind_events, 0);
}


void
panel_register_toy(GtkWidget *applet, char *id, char *params, int xpos, int ypos, long flags)
{
	GtkWidget *eventbox;
	gint      events;
	
	g_assert(applet != NULL);

	/* We wrap the applet in a GtkEventBox so that we can capture events over it */

	eventbox = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(eventbox), applet);

	events = (gtk_widget_get_events(eventbox) |
		  GDK_BUTTON_PRESS_MASK |
		  GDK_BUTTON_RELEASE_MASK |
		  GDK_POINTER_MOTION_MASK |
		  GDK_POINTER_MOTION_HINT_MASK);
	
	gtk_widget_set_events(eventbox, events);
	
	gtk_signal_connect(GTK_OBJECT(eventbox), "event", (GtkSignalFunc) panel_applet_event, the_panel);
	panel_bind_events (applet, 0);
	
	/* Attach our private data to the applet */

	gtk_object_set_data(GTK_OBJECT(eventbox), APPLET_ID, id);
	gtk_object_set_data(GTK_OBJECT(eventbox), APPLET_PARAMS, params);
	gtk_object_set_data(GTK_OBJECT(eventbox), APPLET_FLAGS, (gpointer) flags);

	gtk_fixed_put(GTK_FIXED(the_panel->fixed), eventbox, xpos, ypos);
	gtk_widget_show(eventbox);
	gtk_widget_show(applet);
}


gpointer
panel_command(PanelCommand *cmd)
{
	int xpos, ypos;
	
	g_assert(cmd != NULL);
	
	switch (cmd->cmd) {
		case PANEL_CMD_REGISTER_TOY:
			panel_register_toy(cmd->params.register_toy.applet,
					   cmd->params.register_toy.id,
					   cmd->params.register_toy.params,
					   cmd->params.register_toy.xpos,
					   cmd->params.register_toy.ypos,
					   cmd->params.register_toy.flags);
			return NULL;

		case PANEL_CMD_QUIT:
			panel_quit();
			gtk_main_quit();
			return NULL;

		case PANEL_CMD_NEW_APPLET:
			find_applet_placement(&xpos, &ypos);
			applets_init_applet(cmd->params.new_applet.applet,
					    cmd->params.new_applet.params,
					    xpos, ypos);
			return NULL;

		case PANEL_CMD_GET_APPLET_TYPES:
			return get_applet_types();
	}

	return NULL;
}
