/* This is -*- C -*- */
/* $Id: guppi-linegraph-view.c,v 1.4 2000/12/19 04:53:44 trow Exp $ */

/*
 * guppi-linegraph-view.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <math.h>
#include <guppi-2d.h>
#include <guppi-memory.h>
#include <guppi-date-series.h>
#include <guppi-curve.h>
#include "guppi-linegraph-view.h"
#include "guppi-linegraph-state.h"
#include "guppi-linegraph-item.h"
#include "guppi-linegraph-print.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_linegraph_view_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_linegraph_view_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_linegraph_view_finalize (GtkObject * obj)
{
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/*****************************************************************************/

static void
state_changed (GuppiElementView * view)
{

}

/*****************************************************************************/

static void
guppi_linegraph_view_class_init (GuppiLinegraphViewClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiElementViewClass *view_class = GUPPI_ELEMENT_VIEW_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_ELEMENT_VIEW);

  object_class->get_arg = guppi_linegraph_view_get_arg;
  object_class->set_arg = guppi_linegraph_view_set_arg;
  object_class->finalize = guppi_linegraph_view_finalize;

  view_class->canvas_item_type = GUPPI_TYPE_LINEGRAPH_ITEM;
  view_class->print_type = GUPPI_TYPE_LINEGRAPH_PRINT;
  view_class->state_changed = state_changed;

}

static void
guppi_linegraph_view_init (GuppiLinegraphView * obj)
{

}

GtkType guppi_linegraph_view_get_type (void)
{
  static GtkType guppi_linegraph_view_type = 0;
  if (!guppi_linegraph_view_type) {
    static const GtkTypeInfo guppi_linegraph_view_info = {
      "GuppiLinegraphView",
      sizeof (GuppiLinegraphView),
      sizeof (GuppiLinegraphViewClass),
      (GtkClassInitFunc) guppi_linegraph_view_class_init,
      (GtkObjectInitFunc) guppi_linegraph_view_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_linegraph_view_type =
      gtk_type_unique (GUPPI_TYPE_ELEMENT_VIEW, &guppi_linegraph_view_info);
  }
  return guppi_linegraph_view_type;
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

static ArtVpath *
build_2seqscalar (GuppiLinegraphView *view,
		  GuppiSeqScalar *x_data,
		  GuppiSeqScalar *y_data)
{
  return guppi_2d_make_path (x_data, y_data);
}

static ArtVpath *
build_date_series (GuppiLinegraphView *view, GuppiDateSeries *ser)
{
  double x0, x1;
  GDate sd, ed;
  gint N, j;
  double *tbuf;
  double *buf;
  ArtVpath *path = NULL;

  if (guppi_date_indexed_empty (GUPPI_DATE_INDEXED (ser)))
    return NULL;

  guppi_element_view_get_bbox_vp (GUPPI_ELEMENT_VIEW (view),
				  &x0, NULL, &x1, NULL);
  
  g_date_set_julian (&sd, (gint)floor (x0));
  g_date_set_julian (&ed, (gint)ceil (x1));

  guppi_date_indexed_decr (GUPPI_DATE_INDEXED (ser), &sd);
  guppi_date_indexed_incr (GUPPI_DATE_INDEXED (ser), &ed);
  
  N = (gint)g_date_julian (&ed) - (gint)g_date_julian (&sd) + 1;
  
  tbuf = guppi_new (double, N);
  buf = guppi_new (double, N);

  N = guppi_date_series_get_range_timecoded (ser, &sd, &ed, tbuf, buf, N);

  if (N > 0) {
    
    path = guppi_new (ArtVpath, N + 1);

    for (j=0; j<N; ++j) {
      path[j].code = ART_LINETO;
      path[j].x = tbuf[j];
      path[j].y = buf[j];
    }
    path[0].code = ART_MOVETO_OPEN;
    path[N].code = ART_END;
  }
  
  guppi_free (tbuf);
  guppi_free (buf);

  return path;
}

static ArtVpath *
build_curve (GuppiLinegraphView *view, GuppiCurve *curve)
{
  gsize N = 80; /* The number of samples should not be constant. */
  double t0, t1, x0, y0, x1, y1;
  ArtVpath *path;

  guppi_curve_parameter_bounds (curve, &t0, &t1);

  guppi_element_view_get_bbox_vp (GUPPI_ELEMENT_VIEW (view),
				  &x0, &y0, &x1, &y1);

  guppi_curve_clamp_to_bbox (curve, &t0, &t1, x0, y0, x1, y1);

  path = guppi_new (ArtVpath, N + 1);
  
  /* ...and the sampling shouldn't be uniform! */
  guppi_curve_sample_uniformly_to_path (curve, t0, t1, N, path);

  return path;
}


ArtVpath *
guppi_linegraph_view_build_path (GuppiLinegraphView *view)
{
  GuppiLinegraphState *state;
  GuppiData *data;
  GuppiSeqScalar *x_data;
  GuppiSeqScalar *y_data;

  g_return_val_if_fail (view && GUPPI_IS_LINEGRAPH_VIEW (view), NULL);

  state = GUPPI_LINEGRAPH_STATE (guppi_element_view_state (GUPPI_ELEMENT_VIEW (view)));

  data = guppi_linegraph_state_data (state);
  x_data = guppi_linegraph_state_x_data (state);
  y_data = guppi_linegraph_state_y_data (state);

  if (x_data && y_data) {

    return build_2seqscalar (view, x_data, y_data);

  } else if (data) {

    if (GUPPI_IS_DATE_SERIES (data)) 
      return build_date_series (view, GUPPI_DATE_SERIES (data));
    else if (GUPPI_IS_CURVE (data))
      return build_curve (view, GUPPI_CURVE (data));
    else
      return NULL;

  } 

  return NULL;
}

/* $Id: guppi-linegraph-view.c,v 1.4 2000/12/19 04:53:44 trow Exp $ */
