/* This is -*- C -*- */
/* $Id: guppi-seq-integer.c,v 1.10 2000/12/14 20:23:04 trow Exp $ */

/*
 * guppi-seq-integer.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <gnome.h>
#include <guppi-convenient.h>
#include <guppi-string.h>
#include "guppi-seq-integer.h"
#include "guppi-seq-integer-impl.h"


static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_seq_integer_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_integer_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_integer_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_seq_integer_finalize (GtkObject * obj)
{
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
write_xml_element (GuppiSeq *seq, gint i, GuppiOutputXML *out)
{
  gint j = guppi_seq_integer_get (GUPPI_SEQ_INTEGER (seq), i);
  guppi_output_xml_elementf (out, "int", "%d", j);
}

static GuppiData *
create_xml_object (GuppiAttributesXML *ax)
{
  return guppi_seq_integer_new ();
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

typedef struct _IntContext IntContext;
struct _IntContext {
  gboolean in_int;
  GuppiSeqInteger *seq;
};

static GuppiContextXML *
begin_element (const CHAR *name, const CHAR **attr, gpointer ud)
{
  IntContext *ic = (IntContext *)ud;

  g_return_val_if_fail (ic, NULL);

  if (! strcmp (name, "gpi:int")) {
    if (ic->in_int) {
      g_warning ("Nested int elements not allowed.");
      return NULL;
    }
    ic->in_int = TRUE;
  } else if (! strcmp (name, "gpi:missing")) {
    if (ic->in_int) {
      g_warning ("missing tag not allowed inside of int");
      return NULL;
    }
    guppi_seq_append_missing (GUPPI_SEQ (ic->seq));
  } else {
    g_warning ("Unknown tag: %s", name);
  }

  return NULL;
}

static void
end_element (const CHAR *name, gpointer ud)
{
  IntContext *ic = (IntContext *)ud;

  g_return_if_fail (ic);

  if (!strcmp (name, "gpi:int"))
    ic->in_int = FALSE;
}

static void
characters (const CHAR *s, gint len, gpointer ud)
{
  IntContext *ic = (IntContext *)ud;

  g_return_if_fail (ic);

  if (ic->in_int) {
    gchar *str = guppi_strndup (s, len);

    if (guppi_string_is_int (str)) {
      double n = atoi (str);
      guppi_seq_integer_append (ic->seq, n);
    } else {
      g_warning ("Ill-formed integer: %s", str);
    }
    guppi_free(str);
  } else {
    g_warning ("Illegal characters.");
  }
}

static GuppiContextXML *
content_xml_context (GuppiData *d, GuppiAttributesXML *ax)
{
  GuppiContextXML *ctx;
  IntContext *ic;

  ic = guppi_new0 (IntContext, 1);
  ic->in_int = FALSE;
  ic->seq = GUPPI_SEQ_INTEGER (d);

  ctx = guppi_context_xml_new ();
  guppi_context_xml_set_begin_element_fn (ctx, begin_element);
  guppi_context_xml_set_end_element_fn (ctx, end_element);
  guppi_context_xml_set_characters_fn (ctx, characters);
  guppi_context_xml_set_user_data_free (ctx, ic);

  return ctx;
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

static void
guppi_seq_integer_class_init (GuppiSeqIntegerClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiDataClass *data_class = GUPPI_DATA_CLASS (klass);
  GuppiSeqClass *seq_class = GUPPI_SEQ_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_SEQ);

  data_class->type_name = _("Integer Sequence");
  data_class->impl_type = GUPPI_TYPE_SEQ_INTEGER_IMPL;
  data_class->create_xml_object = create_xml_object;
  data_class->content_xml_context = content_xml_context;
  
  seq_class->write_xml_element = write_xml_element;

  object_class->get_arg = guppi_seq_integer_get_arg;
  object_class->set_arg = guppi_seq_integer_set_arg;
  object_class->destroy = guppi_seq_integer_destroy;
  object_class->finalize = guppi_seq_integer_finalize;

}

static void
guppi_seq_integer_init (GuppiSeqInteger * obj)
{

}

GtkType guppi_seq_integer_get_type (void)
{
  static GtkType guppi_seq_integer_type = 0;
  if (!guppi_seq_integer_type) {
    static const GtkTypeInfo guppi_seq_integer_info = {
      "GuppiSeqInteger",
      sizeof (GuppiSeqInteger),
      sizeof (GuppiSeqIntegerClass),
      (GtkClassInitFunc) guppi_seq_integer_class_init,
      (GtkObjectInitFunc) guppi_seq_integer_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_seq_integer_type =
      gtk_type_unique (GUPPI_TYPE_SEQ, &guppi_seq_integer_info);
  }
  return guppi_seq_integer_type;
}

GuppiData *
guppi_seq_integer_new (void)
{
  return guppi_data_newv (GUPPI_TYPE_SEQ_INTEGER, NULL, 0, NULL);
}

/*****************************************************************************/

/* Sequence Operation Stuff */

typedef struct _GuppiDataOp_Int GuppiDataOp_Int;
struct _GuppiDataOp_Int {
  GuppiDataOp op;
  gint i;
  gsize N;
  gint x;
  const gint *in_ptr;
};

static void
op_set (GuppiData * d, GuppiDataOp * op)
{
  GuppiDataOp_Int *int_op = (GuppiDataOp_Int *) op;

  GuppiSeqIntegerImpl *impl;
  GuppiSeqIntegerImplClass *impl_class;

  impl = GUPPI_SEQ_INTEGER_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_INTEGER_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->set);
  impl_class->set (impl, int_op->i, int_op->x);
}

static void
op_set_many (GuppiData * d, GuppiDataOp * op)
{
  GuppiDataOp_Int *int_op = (GuppiDataOp_Int *) op;

  GuppiSeqIntegerImpl *impl;
  GuppiSeqIntegerImplClass *impl_class;

  gint k;

  impl = GUPPI_SEQ_INTEGER_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_INTEGER_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->set);

  /* Needs optimization */
  for (k = 0; k < int_op->N; ++k)
    impl_class->set (impl, int_op->i + k, int_op->x);
}

static void
op_insert (GuppiData * d, GuppiDataOp * op)
{
  GuppiDataOp_Int *int_op = (GuppiDataOp_Int *) op;

  GuppiSeqIntegerImpl *impl;
  GuppiSeqIntegerImplClass *impl_class;

  impl = GUPPI_SEQ_INTEGER_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_INTEGER_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->insert);
  impl_class->insert (impl, int_op->i, int_op->in_ptr, int_op->N);
}

/*****************************************************************************/

gint guppi_seq_integer_get (const GuppiSeqInteger * gsi, gint i)
{
  const GuppiSeqIntegerImpl *impl;
  GuppiSeqIntegerImplClass *impl_class;

  g_return_val_if_fail (gsi != NULL, 0);
  g_return_val_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (gsi), i), 0);

  impl = GUPPI_SEQ_INTEGER_IMPL (guppi_data_impl (GUPPI_DATA (gsi)));
  impl_class = GUPPI_SEQ_INTEGER_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->get);
  return (impl_class->get) (impl, i);
}

void
guppi_seq_integer_get_many (const GuppiSeqInteger * gsi, gint first,
			    gint last, gint * dest)
{
  gint i;

  g_return_if_fail (gsi != NULL);
  g_return_if_fail (dest != NULL);
  g_return_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (gsi), first));
  g_return_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (gsi), last));

  /* This can be optimized later. */
  for (i = first; i <= last; ++i)
    dest[i - first] = guppi_seq_integer_get (gsi, i);
}

void
guppi_seq_integer_set (GuppiSeqInteger * gsi, gint i, gint val)
{
  GuppiDataOp_Int op;

  g_return_if_fail (gsi != NULL);
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (gsi)));
  g_return_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (gsi), i));

  if (guppi_seq_missing (GUPPI_SEQ (gsi), i) ||
      guppi_seq_integer_get (gsi, i) != val) {

    op.op.op = op_set;
    op.i = i;
    op.x = val;

    guppi_seq_changed_set (GUPPI_SEQ (gsi), i, i, (GuppiDataOp *) & op);
  }
}



void
guppi_seq_integer_set_many_dup (GuppiSeqInteger * gsi, gint first, gint last,
				gint val)
{
  GuppiDataOp_Int op;

  g_return_if_fail (gsi != NULL);
  g_return_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (gsi), first));
  g_return_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (gsi), last));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (gsi)));

  guppi_2sort_i (&first, &last);

  op.op.op = op_set_many;
  op.i = first;
  op.N = last - first + 1;
  op.x = val;

  guppi_seq_changed_set (GUPPI_SEQ (gsi), first, last, (GuppiDataOp *) & op);
}

void
guppi_seq_integer_set_all (GuppiSeqInteger * gsi, gint val)
{
  gint i0, i1;
  g_return_if_fail (gsi != NULL);
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (gsi)));

  guppi_seq_indices (GUPPI_SEQ (gsi), &i0, &i1);
  guppi_seq_integer_set_many_dup (gsi, i0, i1, val);
}

void
guppi_seq_integer_prepend (GuppiSeqInteger * gsi, gint val)
{
  gint first;
  g_return_if_fail (gsi != NULL);
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (gsi)));

  first = guppi_seq_min_index (GUPPI_SEQ (gsi));
  guppi_seq_integer_insert (gsi, first, val);
}

void
guppi_seq_integer_prepend_many (GuppiSeqInteger * gsi, const gint * ptr,
				gsize N)
{
  gint first;
  g_return_if_fail (gsi != NULL);
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (gsi)));
  g_return_if_fail (ptr != NULL);

  if (N == 0)
    return;

  first = guppi_seq_min_index (GUPPI_SEQ (gsi));
  guppi_seq_integer_insert_many (gsi, first, ptr, N);
}

void
guppi_seq_integer_append (GuppiSeqInteger * gsi, gint val)
{
  gint last;
  g_return_if_fail (gsi != NULL);
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (gsi)));

  last = guppi_seq_max_index (GUPPI_SEQ (gsi));
  guppi_seq_integer_insert (gsi, last + 1, val);
}

void
guppi_seq_integer_append_many (GuppiSeqInteger * gsi, const gint * ptr,
			       gsize N)
{
  gint last;
  g_return_if_fail (gsi != NULL);
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (gsi)));
  g_return_if_fail (ptr != NULL);

  if (N == 0)
    return;

  last = guppi_seq_max_index (GUPPI_SEQ (gsi));
  guppi_seq_integer_insert_many (gsi, last + 1, ptr, N);
}

void
guppi_seq_integer_insert (GuppiSeqInteger * gsi, gint i, gint val)
{
  g_return_if_fail (gsi != NULL);
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (gsi)));

  guppi_seq_integer_insert_many (gsi, i, &val, 1);
}

void
guppi_seq_integer_insert_many (GuppiSeqInteger * gsi, gint i,
			       const gint * ptr, gsize N)
{
  GuppiDataOp_Int op;

  g_return_if_fail (gsi != NULL && GUPPI_IS_SEQ_INTEGER (gsi));
  g_return_if_fail (ptr != NULL);

  if (N == 0)
    return;

  op.op.op = op_insert;
  op.i = i;
  op.N = N;
  op.in_ptr = ptr;

  guppi_seq_changed_insert (GUPPI_SEQ (gsi), i, N, (GuppiDataOp *) & op);
}

gint guppi_seq_integer_min (const GuppiSeqInteger * gsi)
{
  gint i, i0, i1, x = 0, a;
  const GuppiSeqIntegerImpl *impl;
  GuppiSeqIntegerImplClass *impl_class;

  g_return_val_if_fail (gsi != NULL, 0);

  impl = GUPPI_SEQ_INTEGER_IMPL (guppi_data_impl (GUPPI_DATA (gsi)));
  impl_class = GUPPI_SEQ_INTEGER_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->range) {
    (impl_class->range) (impl, &x, NULL);
  } else if (!guppi_seq_empty (GUPPI_SEQ (gsi))) {
    guppi_seq_indices (GUPPI_SEQ (gsi), &i0, &i1);
    x = guppi_seq_integer_get (gsi, i0);
    for (i = i0 + 1; i <= i1; ++i) {
      a = guppi_seq_integer_get (gsi, i);
      if (a < x)
	x = a;
    }
  }

  return x;
}

gint guppi_seq_integer_max (const GuppiSeqInteger * gsi)
{
  gint i, i0, i1, x = 0, a;
  const GuppiSeqIntegerImpl *impl;
  GuppiSeqIntegerImplClass *impl_class;

  g_return_val_if_fail (gsi != NULL, 0);

  impl = GUPPI_SEQ_INTEGER_IMPL (guppi_data_impl (GUPPI_DATA (gsi)));
  impl_class = GUPPI_SEQ_INTEGER_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->range) {
    (impl_class->range) (impl, NULL, &x);
  } else if (!guppi_seq_empty (GUPPI_SEQ (gsi))) {
    guppi_seq_indices (GUPPI_SEQ (gsi), &i0, &i1);
    x = guppi_seq_integer_get (gsi, i0);
    for (i = i0 + 1; i <= i1; ++i) {
      a = guppi_seq_integer_get (gsi, i);
      if (a > x)
	x = a;
    }
  }

  return x;
}

gint guppi_seq_integer_frequency (const GuppiSeqInteger * gsi, gint k)
{
  gint min, max, i, i0, i1, count = 0;
  const GuppiSeqIntegerImpl *impl;
  GuppiSeqIntegerImplClass *impl_class;

  g_return_val_if_fail (gsi != NULL, 0);

  impl = GUPPI_SEQ_INTEGER_IMPL (guppi_data_impl (GUPPI_DATA (gsi)));
  impl_class = GUPPI_SEQ_INTEGER_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  min = guppi_seq_integer_min (gsi);
  max = guppi_seq_integer_max (gsi);

  if (k < min || k > max)
    return 0;

  if (impl_class->frequency) {
    return (impl_class->frequency) (impl, k);
  } else {
    guppi_seq_indices (GUPPI_SEQ (gsi), &i0, &i1);
    for (i = i0; i <= i1; ++i)
      if (guppi_seq_integer_get (gsi, i) == k)
	++count;
    return count;
  }
}



/* $Id: guppi-seq-integer.c,v 1.10 2000/12/14 20:23:04 trow Exp $ */
