/* This is -*- C -*- */
/* vim: set sw=2: */
/* $Id: poly.c,v 1.2 2000/12/14 20:22:50 trow Exp $ */

/*
 * guppi-curve-poly-impl.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <math.h>
#include <guppi-memory.h>
#include <guppi-data-impl-plug-in.h>
#include "scm-curve-poly.h"
#include "poly.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0,
  ARG_C0,
  ARG_C1,
  ARG_C2,
  ARG_C3,
  ARG_C4,
  ARG_C5,
  ARG_C6,
  ARG_C7,
  ARG_C8,
  ARG_C9
};

static void
guppi_curve_poly_impl_get_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_curve_poly_impl_set_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  GuppiCurvePolyImpl *poly = GUPPI_CURVE_POLY_IMPL (obj);

  switch (arg_id) {

  case ARG_C0:
    guppi_curve_poly_impl_set_coefficient (poly, 0, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_C1:
    guppi_curve_poly_impl_set_coefficient (poly, 1, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_C2:
    guppi_curve_poly_impl_set_coefficient (poly, 2, GTK_VALUE_DOUBLE (*arg));
    break;


  case ARG_C3:
    guppi_curve_poly_impl_set_coefficient (poly, 3, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_C4:
    guppi_curve_poly_impl_set_coefficient (poly, 4, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_C5:
    guppi_curve_poly_impl_set_coefficient (poly, 5, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_C6:
    guppi_curve_poly_impl_set_coefficient (poly, 6, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_C7:
    guppi_curve_poly_impl_set_coefficient (poly, 7, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_C8:
    guppi_curve_poly_impl_set_coefficient (poly, 8, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_C9:
    guppi_curve_poly_impl_set_coefficient (poly, 9, GTK_VALUE_DOUBLE (*arg));
    break;

  default:
    break;
  };
}

static void
guppi_curve_poly_impl_finalize (GtkObject *obj)
{
  GuppiCurvePolyImpl *x = GUPPI_CURVE_POLY_IMPL(obj);

  guppi_free (x->c);
  x->c = NULL;

  guppi_free (x->ext_pts);
  x->ext_pts = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

#define C0(x) ((x)->c[0])
#define C1(x) ((x)->c[1])
#define C2(x) ((x)->c[2])
#define C3(x) ((x)->c[3])
#define C4(x) ((x)->c[4])

static void
find_ext_points (GuppiCurvePolyImpl *poly)
{
  double c0, c1, c2;

  if (poly->ext_pts != NULL || poly->ext_end_pts_only)
    return;

  if (poly->degree <= 1) {
    poly->ext_end_pts_only = TRUE;
    return;
  }

  if (poly->degree == 2) {
    c0 = C1 (poly);
    c1 = 2 * C2 (poly);
    poly->ext_pts = guppi_new (double, 1);
    poly->ext_pts[0] = - c0 / c1;
    return;
  }

  if (poly->degree == 3) {
    double b2, dd, r1, r2;

    c0 = C1 (poly);
    c1 = 2 * C2 (poly);
    c2 = 3 * C3 (poly);
    b2 = c1 / 2;
    dd = b2*b2 - c0 * c2;

    if (dd >= 0) {
      double d = sqrt (dd);
      double r1a = -c1 + d;
      double r1b = -c1 - d;

      r1 = (fabs (r1a) > fabs (r1b)) ? r1a : r1b;
      r1 /= 2 * c2;
      r2 = c0 / (c2 * r1);

      poly->ext_pts = guppi_new (double, 2);
      poly->ext_pts[0] = r1;
      poly->ext_pts[1] = r2;
      
    } else {
      poly->ext_end_pts_only = TRUE;
    }

    return;
  }

  g_assert_not_reached ();
}


static void
bounds (GuppiCurveImpl *impl, double *a, double *b)
{
  GuppiCurvePolyImpl *poly = GUPPI_CURVE_POLY_IMPL (impl);

  if (a) *a = poly->min;
  if (b) *b = poly->max;
}

static void
get (GuppiCurveImpl *impl, double t, double *x, double *y)
{
  GuppiCurvePolyImpl *poly = GUPPI_CURVE_POLY_IMPL (impl);
  gint i;
  double run;
  
  if (x) *x = t;

  if (y) {
    run = 0;
    i = poly->degree;
    if (i >= 0) {
      run = poly->c[i];
      g_message ("%d %g (%g)", i, run, t);
      --i;
    }
    while (i >= 0) {
      run = run * t + poly->c[i];
      g_message ("%d %g %g", i, poly->c[i], run);
      --i;
    }
    *y = run;
  }
}

static void
bbox (GuppiCurveImpl *impl, double a, double b, 
      double *x0, double *y0, double *x1, double *y1)
{
  GuppiCurvePolyImpl *poly = GUPPI_CURVE_POLY_IMPL (impl);
  double ay, by, ymin, ymax, x, y;

  if (x0) *x0 = MIN (a, b);
  if (x1) *x1 = MAX (a, b);

  if (! (y0 || y1))
    return;

  get (impl, a, NULL, &ay);
  get (impl, b, NULL, &by);
  ymin = MIN (ay, by);
  ymax = MAX (ay, by);

  if (!poly->ext_end_pts_only) {
    if (poly->ext_pts == NULL)
      find_ext_points (poly);
    
    if (poly->ext_pts) {
      gint i;
      for (i=0; i<poly->degree; ++i) {
	x = poly->ext_pts[i];
	if (a <= x && x <= b) {
	  get (impl, x, NULL, &y);
	  if (y < ymin) ymin = y;
	  if (y > ymax) ymax = y;
	}
      }
    }
  }

  if (y0) *y0 = ymin;
  if (y1) *y1 = ymax;
}

static gboolean
clamp (GuppiCurveImpl *impl, double *a, double *b,
       double x0, double y0, double x1, double y1)
{
  /*GuppiCurvePolyImpl *poly = GUPPI_CURVE_POLY_IMPL (impl);*/

  if (a) *a = x0;
  if (b) *b = x1;

  return TRUE;
}


static void
sample (GuppiCurveImpl *impl, gconstpointer t_vec, gint t_stride, gsize N,
	gpointer x_vec, gint x_stride,
	gpointer y_vec, gint y_stride)
{
  GuppiCurvePolyImpl *poly = GUPPI_CURVE_POLY_IMPL (impl);
  gsize j;
  gint i;
  double run, x, cc;
  const gchar *tp = (const gchar *)t_vec;
  gchar *xp = (gchar *)x_vec;
  gchar *yp = (gchar *)y_vec;

  if (poly->degree <= 0) {
    /* Deal with the pathological case separately, so that we can eliminate
       a check from inside the loop below. */
    cc = poly->degree == 0 ? poly->c[poly->degree] : 0;
    for (j=0; j<N; ++j) {
      x = *(const double *)tp;
      if (xp) {
	*(double *)xp = x;
	xp += x_stride;
      }
      if (yp) {
	*(double *)yp = cc;
	yp += y_stride;
      }

      tp += t_stride;
    }
    return;
  }

  for (j=0; j<N; ++j) {

    x = *(const double *)tp;
    tp += t_stride;


    if (xp) {
      *(double *)xp = x;
      xp += x_stride;
    }

    if (yp) {
      i = poly->degree;
      run = poly->c[i];
      --i;
      while (i >= 0) {
	run += run * x + poly->c[i];
	--i;
      }
      *(double *)yp = run;
      yp += y_stride;
    }

  }
}

static void
sample_uniformly (GuppiCurveImpl *impl,
		  double t0, double t1, gsize N,
		  double *x_vec, gint x_stride,
		  double *y_vec, gint y_stride)
{
  GuppiCurvePolyImpl *poly = GUPPI_CURVE_POLY_IMPL (impl);
  gint j;
  gint i;
  double run, x;
  gchar *xp = (gchar *)x_vec;
  gchar *yp = (gchar *)y_vec;

  if (poly->degree <= 0) {
    /* Again, we break out the pathological case. */
    double cc = poly->degree == 0 ? poly->c[poly->degree] : 0;
    for (j = 0; j<N; ++j) {

      x = t0 + j*(t1-t0)/(N-1);

      if (xp) {
	*(double *)xp = x;
	xp += x_stride;
      }

      if (yp) {
	*(double *)yp = cc;
	yp += y_stride;
      }

    }
    return;
  }

  for (j=0; j<N; ++j) {

    x = t0 + j*(t1-t0)/(N-1);

    if (xp) {
      *(double *)xp = x;
      xp += x_stride;
    }
    
    if (yp) {
      i = poly->degree;
      run = poly->c[i];
      --i;
      while (i >= 0) {
	run += run * x + poly->c[i];
	--i;
      }
      *(double *)yp = run;
      yp += y_stride;
    }

  }
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

#define add_arg(str, t, symb) \
gtk_object_add_arg_type ("GuppiCurvePolyImpl::" str, t, GTK_ARG_WRITABLE, symb)

static void
guppi_curve_poly_impl_class_init (GuppiCurvePolyImplClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *)klass;
  GuppiCurveImplClass *curve_class = GUPPI_CURVE_IMPL_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_CURVE_IMPL);

  curve_class->bounds = bounds;
  curve_class->get = get;
  curve_class->bbox = bbox;
  curve_class->clamp = clamp;
  curve_class->sample = sample;
  curve_class->sample_uniformly = sample_uniformly;

  object_class->get_arg = guppi_curve_poly_impl_get_arg;
  object_class->set_arg = guppi_curve_poly_impl_set_arg;
  object_class->finalize = guppi_curve_poly_impl_finalize;

  add_arg ("c0", GTK_TYPE_DOUBLE, ARG_C0);
  add_arg ("c1", GTK_TYPE_DOUBLE, ARG_C1);
  add_arg ("c2", GTK_TYPE_DOUBLE, ARG_C2);
  add_arg ("c3", GTK_TYPE_DOUBLE, ARG_C3);
  add_arg ("c4", GTK_TYPE_DOUBLE, ARG_C4);
  add_arg ("c5", GTK_TYPE_DOUBLE, ARG_C5);
  add_arg ("c6", GTK_TYPE_DOUBLE, ARG_C6);
  add_arg ("c7", GTK_TYPE_DOUBLE, ARG_C7);
  add_arg ("c8", GTK_TYPE_DOUBLE, ARG_C8);
  add_arg ("c9", GTK_TYPE_DOUBLE, ARG_C9);
}

static void
guppi_curve_poly_impl_init (GuppiCurvePolyImpl *obj)
{
  obj->min = -1000;
  obj->max = 1000;

  obj->degree = -1;
}

GtkType
guppi_curve_poly_impl_get_type (void)
{
  static GtkType guppi_curve_poly_impl_type = 0;
  if (!guppi_curve_poly_impl_type) {
    static const GtkTypeInfo guppi_curve_poly_impl_info = {
      "GuppiCurvePolyImpl",
      sizeof (GuppiCurvePolyImpl),
      sizeof (GuppiCurvePolyImplClass),
      (GtkClassInitFunc)guppi_curve_poly_impl_class_init,
      (GtkObjectInitFunc)guppi_curve_poly_impl_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_curve_poly_impl_type = gtk_type_unique (GUPPI_TYPE_CURVE_IMPL, &guppi_curve_poly_impl_info);
  }
  return guppi_curve_poly_impl_type;
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

gint
guppi_curve_poly_impl_degree (GuppiCurvePolyImpl *poly)
{
  g_return_val_if_fail (poly && GUPPI_IS_CURVE_POLY_IMPL (poly), -1);

  return poly->degree;
}

double
guppi_curve_poly_impl_coefficient (GuppiCurvePolyImpl *poly, gint i)
{
  g_return_val_if_fail (poly && GUPPI_IS_CURVE_POLY_IMPL (poly), 0);
  
  if (i < 0 || i > poly->degree) 
    return 0;

  return poly->c[i];
}

void
guppi_curve_poly_impl_set_coefficient (GuppiCurvePolyImpl *poly,
				       gint i, double a)
{
  double *nc;
  gint j;

  g_return_if_fail (poly && GUPPI_IS_CURVE_POLY_IMPL (poly));
  g_return_if_fail (i >= 0);

  if (i > poly->degree && a == 0)
    return;

  if (i > poly->degree) {
    nc = guppi_new0 (double, i+1);
    for (j=0; j<=poly->degree; ++j)
      nc[j] = poly->c[j];
    guppi_free (poly->c);
    poly->c = nc;
    poly->degree = i;
  }

  poly->c[i] = a;

  while (poly->degree >= 0 && poly->c[poly->degree] == 0)
    --poly->degree;
}


/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

GuppiData *
guppi_curve_new_polynomial2 (double c0, double c1, double c2)
{
  return guppi_data_new_by_type (GUPPI_TYPE_CURVE,
				 GUPPI_TYPE_CURVE_POLY_IMPL,
				 "c0", c0,
				 "c1", c1,
				 "c2", c2,
				 NULL);
}

GuppiData *
guppi_curve_new_polynomial3 (double c0, double c1, double c2, double c3)
{
  return guppi_data_new_by_type (GUPPI_TYPE_CURVE,
				 GUPPI_TYPE_CURVE_POLY_IMPL,
				 "c0", c0,
				 "c1", c1,
				 "c2", c2,
				 "c3", c3,
				 NULL);
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

static GuppiDataImpl *
make_impl (void)
{
  return GUPPI_DATA_IMPL (guppi_type_new (guppi_curve_poly_impl_get_type ()));
}

GuppiPlugIn *guppi_plug_in (void);

GuppiPlugIn *
guppi_plug_in (void)
{
  GuppiPlugIn *pi;
  GuppiDataImplPlugIn *dimpi;

  pi = guppi_data_impl_plug_in_new ();
  dimpi = GUPPI_DATA_IMPL_PLUG_IN (pi);

  pi->magic_number = GUPPI_PLUG_IN_MAGIC_NUMBER;
  dimpi->impl_constructor = make_impl;

  scm_curve_poly_init ();

  return pi;
}
