/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-date-series-core-impl.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include <config.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <math.h>
#include <guppi-convenient.h>
#include <gnan.h>
#include <guppi-data-impl-plug-in.h>
#include "core.h"
#include "scm-date-series-core.h"
#ifdef HAVE_PYTHON
#include "python-date-series-core.h"
#endif

static GtkObjectClass * parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_date_series_core_impl_get_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_date_series_core_impl_set_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_date_series_core_impl_finalize (GtkObject *obj)
{
  GuppiDateSeriesCoreImpl *core = GUPPI_DATE_SERIES_CORE_IMPL (obj);

  guppi_unref0 (core->garray);

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

static void
ensure_contains (GuppiDateSeriesCoreImpl *core, GDate *dt)
{
  double *data;
  gint i;

  if (core->size == 0) {

    guppi_garray_set_size (core->garray, 186);
    core->size = 1;
    core->start_date = *dt;
    core->end_date = *dt;
    data = (double *)guppi_garray_data (core->garray);
    data[0] = NAN;
    return;
  }

  i = (gint)g_date_julian (dt) - (gint)g_date_julian (&core->start_date);

  if (0 <= i && i < core->size)
    return;

  if (i < 0) {
    gint new_size, j;

    new_size = core->size - i;
    
    if (guppi_garray_size (core->garray) < new_size)
      guppi_garray_set_size (core->garray, new_size);

    data = (double *)guppi_garray_data (core->garray);

    for (j = core->size-1; j >= 0; --j)
      data[j-i] = data[j];
    for (j = 0; j<-i; ++j)
      data[j] = NAN;

    core->size = new_size;
    core->start_date = *dt;

    return;
  }

  if (i >= core->size) {
    gint new_size, j;

    new_size = i+1;

    if (guppi_garray_size (core->garray) < new_size)
      guppi_garray_set_size (core->garray, new_size);

    data = (double *)guppi_garray_data (core->garray);

    for (j = core->size; j < new_size; ++j)
      data[j] = NAN;

    core->size = new_size;
    core->end_date = *dt;

    return;
  }

  g_assert_not_reached ();
}

static GuppiDataImpl *
v_data_copy (GuppiDataImpl *impl)
{
  g_assert_not_reached ();
  return NULL;
}

static gint
v_data_get_size_in_bytes (GuppiDataImpl *impl)
{
  return -1;
}

static void
v_di_bounds (GuppiDateIndexedImpl *impl, GDate *start, GDate *end)
{
  GuppiDateSeriesCoreImpl *core = GUPPI_DATE_SERIES_CORE_IMPL (impl);

  if (start)
    *start = core->start_date;
  if (end)
    *end = core->end_date;
}

static gboolean
v_di_valid (GuppiDateIndexedImpl *impl, const GDate *dt)
{
  GuppiDateSeriesCoreImpl *core = GUPPI_DATE_SERIES_CORE_IMPL (impl);
  gint i;
  double *data;

  data = (double *)guppi_garray_data (core->garray);
  i = (gint)g_date_julian ((GDate *)dt) - (gint)g_date_julian(&core->start_date);

  return !g_isnan (data[i]);
}

static gboolean
v_di_step (GuppiDateIndexedImpl *impl, const GDate *dt,
	   gint delta, GDate *modified)
{
  GuppiDateSeriesCoreImpl *core = GUPPI_DATE_SERIES_CORE_IMPL (impl);
  gint i, N;
  double *data;

  data = (double *)guppi_garray_data (core->garray);
  i = (gint)g_date_julian ((GDate *)dt) - (gint)g_date_julian(&core->start_date);
  N = (gint)g_date_julian (&core->end_date) - (gint)g_date_julian(&core->start_date);

  while (delta && 0 <= i && i <= N) {
    
    if (!g_isnan (data[i])) {
      if (delta > 0)
	--delta;
      else 
	++delta;
    }
    
    i += (delta > 0 ? 1 : -1);
  }

  return delta == 0;
}

static gint
v_di_size (GuppiDateIndexedImpl *impl)
{
  GuppiDateSeriesCoreImpl *core = GUPPI_DATE_SERIES_CORE_IMPL (impl);
  gint i, N, count;
  double *data;

  data = (double *)guppi_garray_data (core->garray);
  N = (gint)g_date_julian (&core->end_date) - (gint)g_date_julian(&core->start_date);
  count = 0;

  for (i=0; i<=N; ++i)
    if (!g_isnan (data[i]))
      ++count;

  return count;
}

static void
v_di_bounds_hint (GuppiDateIndexedImpl *impl,
		  const GDate *start, const GDate *end)
{
  GuppiDateSeriesCoreImpl *core = GUPPI_DATE_SERIES_CORE_IMPL (impl);

  ensure_contains (core, (GDate *)start);
  ensure_contains (core, (GDate *)end);
}

static double
v_ds_get (GuppiDateSeriesImpl *impl, GDate *dt)
{
  GuppiDateSeriesCoreImpl *core = GUPPI_DATE_SERIES_CORE_IMPL (impl);
  double *data;
  gint i;

  data = (double *)guppi_garray_data (core->garray);
  i = (gint)g_date_julian (dt) - (gint)g_date_julian (&core->start_date);

  return data[i];
}

static void
v_ds_set (GuppiDateSeriesImpl *impl, GDate *dt, double x)
{
  GuppiDateSeriesCoreImpl *core = GUPPI_DATE_SERIES_CORE_IMPL (impl);
  double *data;
  gint i;

  ensure_contains (core, dt);

  data = (double *)guppi_garray_data (core->garray);
  i = (gint)g_date_julian (dt) - (gint)g_date_julian (&core->start_date);

  data[i] = x;
}

static void
v_ds_unset (GuppiDateSeriesImpl *impl, GDate *dt)
{
  GuppiDateSeriesCoreImpl *core = GUPPI_DATE_SERIES_CORE_IMPL (impl);
  double *data;
  gint i;

  data = (double *)guppi_garray_data (core->garray);
  i = (gint)g_date_julian (dt) - (gint)g_date_julian (&core->start_date);

  if (0 <= i && i < core->size)
    data[i] = NAN;
}

static gboolean
v_ds_get_many (GuppiDateSeriesImpl *impl, GDate *dt, gint count, double *buf,
	       gint *retval)
{
  GuppiDateSeriesCoreImpl *core = GUPPI_DATE_SERIES_CORE_IMPL (impl);
  double *data;
  gint i, j;

  data = (double *)guppi_garray_data (core->garray);
  i = (gint)g_date_julian (dt) - (gint)g_date_julian (&core->start_date);
  j = 0;

  while (count && 0 <= i && i < core->size) {
    if (!g_isnan (data[i])) {
      buf[j] = data[i];
      ++j;
      if (count > 0)
	--count;
      else
	++count;
    }
    
    if (count > 0)
      ++i;
    else 
      --i;
  }

  if (retval)
    *retval = j;

  return TRUE;
}

static gboolean
v_ds_get_range (GuppiDateSeriesImpl *impl, GDate *sd, GDate *ed,
		double *timecode, double *buf, gint bufsize, gint *retval)
{
  GuppiDateSeriesCoreImpl *core = GUPPI_DATE_SERIES_CORE_IMPL (impl);
  double *data;
  gint i, N, j, jul;

  data = (double *)guppi_garray_data (core->garray);
  jul = (gint)g_date_julian (&core->start_date);
  i = (gint)g_date_julian (sd) - jul;
  N = (gint)g_date_julian (ed) - jul;
  j = 0;

  while (j < bufsize && 0 <= i && i <= N) {
    if (!g_isnan (data[i])) {
      if (timecode)
	timecode[j] = jul + i;
      if (buf)
	buf[j] = data[i];
      ++j;
    }
    ++i;
  }

  if (retval)
    *retval = j;

  return TRUE;
}

static gboolean
v_ds_get_bounds (GuppiDateSeriesImpl *impl, GDate *sd, GDate *ed,
		 double *min, double *max, gboolean *retval)
{
  GuppiDateSeriesCoreImpl *core = GUPPI_DATE_SERIES_CORE_IMPL (impl);
  double *data;
  gint i, N;
  double m = 0, M = 0;
  gboolean first = TRUE;

  data = (double *)guppi_garray_data (core->garray);
  i = (gint)g_date_julian (sd) - (gint)g_date_julian (&core->start_date);
  N = (gint)g_date_julian (ed) - (gint)g_date_julian (&core->start_date);

  for (i=0; i<=N; ++i) {
    if (!g_isnan (data[i])) {
      if (first) {
	m = M = data[i];
	first = FALSE;
      } else {
	double x = data[i];
	if (x < m) m = x;
	if (x > M) M = x;
      }
    }
  }

  if (min) *min = m;
  if (max) *max = M;

  if (retval)
    *retval = !first;

  return TRUE;
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

static void
guppi_date_series_core_impl_class_init (GuppiDateSeriesCoreImplClass *klass)
{
  double d;
  GtkObjectClass* object_class = (GtkObjectClass *)klass;
  GuppiDataImplClass *data_class = GUPPI_DATA_IMPL_CLASS (klass);
  GuppiDateIndexedImplClass *di_class = GUPPI_DATE_INDEXED_IMPL_CLASS (klass);
  GuppiDateSeriesImplClass *ds_class = GUPPI_DATE_SERIES_IMPL_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_DATE_SERIES_IMPL);

  object_class->get_arg = guppi_date_series_core_impl_get_arg;
  object_class->set_arg = guppi_date_series_core_impl_set_arg;
  object_class->finalize = guppi_date_series_core_impl_finalize;

  data_class->impl_name = _("Core Date Series");

  data_class->copy = v_data_copy;
  data_class->get_size_in_bytes = v_data_get_size_in_bytes;

  di_class->bounds = v_di_bounds;
  di_class->valid = v_di_valid;
  di_class->step = v_di_step;
  di_class->size = v_di_size;
  di_class->bounds_hint = v_di_bounds_hint;

  ds_class->get = v_ds_get;
  ds_class->set = v_ds_set;
  ds_class->unset = v_ds_unset;
  ds_class->get_many = v_ds_get_many;
  ds_class->get_range = v_ds_get_range;
  ds_class->get_bounds = v_ds_get_bounds;

  d = NAN;
  g_assert (g_isnan (d));
}

static void
guppi_date_series_core_impl_init (GuppiDateSeriesCoreImpl *obj)
{
  g_date_clear (&obj->start_date, 1);
  g_date_clear (&obj->end_date, 1);
  obj->size = 0;
  obj->garray = guppi_garray_new (sizeof (double));
}

GtkType
guppi_date_series_core_impl_get_type (void)
{
  static GtkType guppi_date_series_core_impl_type = 0;
  if (!guppi_date_series_core_impl_type) {
    static const GtkTypeInfo guppi_date_series_core_impl_info = {
      "GuppiDateSeriesCoreImpl",
      sizeof (GuppiDateSeriesCoreImpl),
      sizeof (GuppiDateSeriesCoreImplClass),
      (GtkClassInitFunc)guppi_date_series_core_impl_class_init,
      (GtkObjectInitFunc)guppi_date_series_core_impl_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_date_series_core_impl_type = gtk_type_unique (GUPPI_TYPE_DATE_SERIES_IMPL, &guppi_date_series_core_impl_info);
  }
  return guppi_date_series_core_impl_type;
}

GtkObject *
guppi_date_series_core_impl_new (void)
{
  return GTK_OBJECT (guppi_type_new (guppi_date_series_core_impl_get_type ()));
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

static GuppiDataImpl *
make_impl (void)
{
  return GUPPI_DATA_IMPL (guppi_type_new (guppi_date_series_core_impl_get_type ()));
}

GuppiPlugIn *guppi_plug_in (void);

GuppiPlugIn *
guppi_plug_in (void)
{
  GuppiPlugIn *pi;
  GuppiDataImplPlugIn *dimpi;

  pi = guppi_data_impl_plug_in_new ();
  dimpi = GUPPI_DATA_IMPL_PLUG_IN (pi);

  pi->magic_number = GUPPI_PLUG_IN_MAGIC_NUMBER;
  dimpi->impl_constructor = make_impl;

  scm_date_series_core_init ();

#ifdef HAVE_PYTHON
  python_date_series_core_init ();
#endif

  return pi;
}
