/* This is -*- C -*- */
/* $Id: guppi-pie-state.c,v 1.24 2001/01/16 23:36:03 trow Exp $ */

/*
 * guppi-pie-state.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <libgnomeui/gnome-color-picker.h>

#include <math.h>
#include <glade/glade.h>
#include <guppi-useful.h>
#include <guppi-rgb.h>
#include <guppi-defaults.h>
#include <guppi-data-select.h>
#include <guppi-style.h>
#include <guppi-metrics.h>
#include <guppi-seq-string.h>
#include <guppi-seq-categorical.h>
#include "guppi-pie-state.h"
#include "guppi-pie-view.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0,
  ARG_DATA,
  ARG_DATA_LABELS,
  ARG_DATA_STYLES,
  ARG_RADIUS,
  ARG_RADIUS_LOCK,
  ARG_RADIUS_MAXIMIZE,
  ARG_EDGE_WIDTH,
  ARG_BASE_OFFSET,
  ARG_BASE_ANGLE,
  ARG_DEFAULT_SLICE_COLOR,
  ARG_USE_STOCK_COLORS,
  ARG_EDGE_COLOR,
  ARG_SHOW_PERCENTAGES,
  ARG_LABEL_FONT,
  ARG_LABEL_COLOR,
};

static void
guppi_pie_state_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiPieState *state = GUPPI_PIE_STATE (obj);

  switch (arg_id) {

  case ARG_DATA:
    GTK_VALUE_POINTER (*arg) = guppi_pie_state_data (state);
    break;

  case ARG_DATA_LABELS:
    GTK_VALUE_POINTER (*arg) = guppi_pie_state_labels (state);
    break;

  case ARG_DATA_STYLES:
    GTK_VALUE_POINTER (*arg) = guppi_pie_state_styles (state);
    break;

  case ARG_RADIUS:
    GTK_VALUE_DOUBLE (*arg) = guppi_pie_state_radius (state);
    break;

  case ARG_RADIUS_LOCK:
    GTK_VALUE_BOOL (*arg) = guppi_pie_state_radius_lock (state);
    break;

  case ARG_RADIUS_MAXIMIZE:
    GTK_VALUE_BOOL (*arg) = guppi_pie_state_radius_maximize (state);
    break;

  case ARG_EDGE_WIDTH:
    GTK_VALUE_DOUBLE (*arg) = guppi_pie_state_edge_width (state);
    break;

  case ARG_BASE_OFFSET:
    GTK_VALUE_DOUBLE (*arg) = guppi_pie_state_base_offset (state);
    break;

  case ARG_BASE_ANGLE:
    GTK_VALUE_DOUBLE (*arg) = guppi_pie_state_base_angle (state);
    break;

  case ARG_DEFAULT_SLICE_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_pie_state_default_slice_color (state);
    break;

  case ARG_USE_STOCK_COLORS:
    GTK_VALUE_UINT (*arg) = guppi_pie_state_use_stock_colors (state);
    break;

  case ARG_EDGE_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_pie_state_edge_color (state);
    break;

  case ARG_SHOW_PERCENTAGES:
    GTK_VALUE_BOOL (*arg) = guppi_pie_state_show_percentages (state);
    break;

  case ARG_LABEL_FONT:
    GTK_VALUE_POINTER (*arg) = guppi_pie_state_label_font (state);
    break;

  case ARG_LABEL_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_pie_state_label_color (state);
    break;

  default:
    break;
  };
}

static void
guppi_pie_state_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiPieState *state = GUPPI_PIE_STATE (obj);

  switch (arg_id) {

  case ARG_DATA:
    guppi_pie_state_set_data (state, GUPPI_DATA0 (GTK_VALUE_POINTER (*arg)));
    break;

  case ARG_DATA_LABELS:
    guppi_pie_state_set_labels (state,
				GUPPI_DATA0 (GTK_VALUE_POINTER (*arg)));
    break;

  case ARG_DATA_STYLES:
    guppi_pie_state_set_styles (state,
				GUPPI_SEQ_STYLE0 (GTK_VALUE_POINTER (*arg)));
    break;

  case ARG_RADIUS:
    guppi_pie_state_set_radius (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_RADIUS_LOCK:
    guppi_pie_state_set_radius_lock (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_RADIUS_MAXIMIZE:
    guppi_pie_state_set_radius_maximize (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_EDGE_WIDTH:
    guppi_pie_state_set_edge_width (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_BASE_OFFSET:
    guppi_pie_state_set_base_offset (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_BASE_ANGLE:
    guppi_pie_state_set_base_angle (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_DEFAULT_SLICE_COLOR:
    guppi_pie_state_set_default_slice_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_USE_STOCK_COLORS:
    guppi_pie_state_set_use_stock_colors (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_EDGE_COLOR:
    guppi_pie_state_set_edge_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_SHOW_PERCENTAGES:
    guppi_pie_state_set_show_percentages (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_LABEL_FONT:
    guppi_pie_state_set_label_font (state,
				    GNOME_FONT (GTK_VALUE_POINTER (*arg)));
    break;

  case ARG_LABEL_COLOR:
    guppi_pie_state_set_label_color (state, GTK_VALUE_UINT (*arg));
    break;

  default:
    break;
  };
}

static void
guppi_pie_state_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_pie_state_finalize (GtkObject * obj)
{
  GuppiPieState *state = GUPPI_PIE_STATE (obj);

  guppi_unref0 (state->label_font);
  guppi_unref0 (state->slice_offsets);

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/*****************************************************************************/

/*** Config Widgets ***/

static void
push_state_to_data_widget (GuppiPieState * state, GladeXML * xml)
{
  GtkWidget *w;

  w = glade_xml_get_widget (xml, "values_data");
  guppi_data_select_set_selected_data (GUPPI_DATA_SELECT (w),
				       GUPPI_DATA0 (guppi_pie_state_data
						    (state)));

  w = glade_xml_get_widget (xml, "labels_data");
  guppi_data_select_set_selected_data (GUPPI_DATA_SELECT (w),
				       GUPPI_DATA0 (guppi_pie_state_labels
						    (state)));
  gtk_widget_set_sensitive (w, guppi_pie_state_need_separate_label_data (state));
}

static void
values_data_cb (GuppiDataSelect * picker, GuppiData * data,
		GuppiPieState * state)
{
  guppi_pie_state_set_data (state, data);
}

static void
labels_data_cb (GuppiDataSelect * picker, GuppiData * data,
		GuppiPieState * state)
{
  guppi_pie_state_set_labels (state, data);
}

static void
connect_data_signals (GuppiPieState * state, GladeXML * xml)
{
  GtkWidget *w;

  w = glade_xml_get_widget (xml, "values_data");
  gtk_signal_connect (GTK_OBJECT (w), "selected_data",
		      GTK_SIGNAL_FUNC (values_data_cb), state);

  w = glade_xml_get_widget (xml, "labels_data");
  gtk_signal_connect (GTK_OBJECT (w), "selected_data",
		      GTK_SIGNAL_FUNC (labels_data_cb), state);
}

static void
data_destroy_handler (GtkWidget * w, GladeXML * xml)
{
  GtkObject *state;

  state = GTK_OBJECT (gtk_object_get_user_data (GTK_OBJECT (w)));

  gtk_signal_disconnect_by_func (GTK_OBJECT (state),
				 push_state_to_data_widget, xml);
}

static gboolean
allowed_data_types_fn (GtkType type, gpointer user_data)
{
  return type == GUPPI_TYPE_SEQ_SCALAR || type == GUPPI_TYPE_SEQ_CATEGORICAL;
}

static GtkWidget *
config_data_cb (gpointer user_data)
{
  GuppiPieState *state = GUPPI_PIE_STATE (user_data);
  const gchar *xml_path;
  GladeXML *xml;
  GtkWidget *w;

  xml_path = guppi_glade_path ("guppi-pie-data.glade");
  g_assert (xml_path != NULL);

  xml = glade_xml_new (xml_path, "pie_table");
  g_assert (xml != NULL);

  push_state_to_data_widget (state, xml);	/* Initialize */
  connect_data_signals (state, xml);

  w = glade_xml_get_widget (xml, "values_data");
  guppi_data_select_set_allowed_type_fn (GUPPI_DATA_SELECT (w),
					 allowed_data_types_fn,
					 NULL);

  w = glade_xml_get_widget (xml, "pie_table");

  gtk_signal_connect (GTK_OBJECT (state), "changed",
		      GTK_SIGNAL_FUNC (push_state_to_data_widget), xml);

  gtk_object_set_user_data (GTK_OBJECT (w), state);

  gtk_signal_connect (GTK_OBJECT (w), "destroy",
		      GTK_SIGNAL_FUNC (data_destroy_handler), xml);

  return w;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static void
push_state_to_appearance_widget (GuppiPieState * state, GladeXML * xml)
{
  GtkWidget *w;
  gint r, g, b, a;

  w = glade_xml_get_widget (xml, "radius_max_check");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				guppi_pie_state_radius_maximize (state));

  w = glade_xml_get_widget (xml, "radius_lock_check");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				guppi_pie_state_radius_lock (state));

  w = glade_xml_get_widget (xml, "thickness_spinner");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
			     guppi_pie_state_edge_width (state));

  w = glade_xml_get_widget (xml, "edge_color");
  UINT_TO_RGBA (guppi_pie_state_edge_color (state), &r, &g, &b, &a);
  gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (w), r, g, b, a);
}

static void
radius_max_check_cb (GtkToggleButton * b, gpointer user_data)
{
  guppi_pie_state_set_radius_maximize (GUPPI_PIE_STATE (user_data),
				       gtk_toggle_button_get_active (b));
}

static void
radius_lock_check_cb (GtkToggleButton * b, gpointer user_data)
{
  guppi_pie_state_set_radius_lock (GUPPI_PIE_STATE (user_data),
				   gtk_toggle_button_get_active (b));
}

static void
edge_thickness_spinner_changed_cb (GtkSpinButton * spin, gpointer user_data)
{
  guppi_pie_state_set_edge_width (GUPPI_PIE_STATE (user_data),
				  gtk_spin_button_get_value_as_float (spin));
}

static void
edge_color_set_cb (GnomeColorPicker * pick, guint r, guint g, guint b,
		   guint a, gpointer user_data)
{
  guppi_pie_state_set_edge_color (GUPPI_PIE_STATE (user_data),
				  RGBA_TO_UINT (r >> 8, g >> 8, b >> 8,
						a >> 8));
}

static void
connect_appearance_signals (GuppiPieState * state, GladeXML * xml)
{
  GtkWidget *w;

  w = glade_xml_get_widget (xml, "radius_max_check");
  gtk_signal_connect (GTK_OBJECT (w), "toggled",
		      GTK_SIGNAL_FUNC (radius_max_check_cb), state);

  w = glade_xml_get_widget (xml, "radius_lock_check");
  gtk_signal_connect (GTK_OBJECT (w), "toggled",
		      GTK_SIGNAL_FUNC (radius_lock_check_cb), state);

  w = glade_xml_get_widget (xml, "thickness_spinner");
  gtk_signal_connect (GTK_OBJECT (w), "changed",
		      GTK_SIGNAL_FUNC (edge_thickness_spinner_changed_cb),
		      state);

  w = glade_xml_get_widget (xml, "edge_color");
  gtk_signal_connect (GTK_OBJECT (w), "color_set",
		      GTK_SIGNAL_FUNC (edge_color_set_cb), state);
}

static void
appearance_destroy_handler (GtkWidget * w, GladeXML * xml)
{
  GtkObject *state;

  state = GTK_OBJECT (gtk_object_get_user_data (GTK_OBJECT (w)));

  gtk_signal_disconnect_by_func (GTK_OBJECT (state),
				 push_state_to_appearance_widget, xml);
}


static GtkWidget *
config_appearance_cb (gpointer user_data)
{
  GuppiPieState *state = GUPPI_PIE_STATE (user_data);
  const gchar *xml_path;
  GladeXML *xml;
  GtkWidget *w;

  xml_path = guppi_glade_path ("guppi-pie-looks.glade");
  g_assert (xml_path != NULL);

  xml = glade_xml_new (xml_path, "pie_state");
  g_assert (xml != NULL);

  push_state_to_appearance_widget (state, xml);	/* Initialize */
  connect_appearance_signals (state, xml);

  w = glade_xml_get_widget (xml, "pie_state");

  gtk_signal_connect (GTK_OBJECT (state), "changed",
		      GTK_SIGNAL_FUNC (push_state_to_appearance_widget), xml);

  gtk_object_set_user_data (GTK_OBJECT (w), state);

  gtk_signal_connect (GTK_OBJECT (w), "destroy",
		      GTK_SIGNAL_FUNC (appearance_destroy_handler), xml);

  return w;
}

static GuppiConfigItem *
config_tree (GuppiElementState * state)
{
  GuppiConfigItem *node1;
  GuppiConfigItem *node2;

  node1 = guppi_config_item_new (_("Pie Chart: Data"),
				 _("Data"), config_data_cb, state);
  node2 = guppi_config_item_new (_("Pie Chart: Appearance"),
				 _("Appearance"),
				 config_appearance_cb, state);

  guppi_config_item_append (node1, node2);

  return node1;
}

/*****************************************************************************/

#define add_arg(str, t, symb) \
gtk_object_add_arg_type("GuppiPieState::" str, t, GTK_ARG_READWRITE, symb)

#define add_arg_w(str, t, symb) \
gtk_object_add_arg_type("GuppiPieState::" str, t, GTK_ARG_WRITABLE, symb)

static void
guppi_pie_state_class_init (GuppiPieStateClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiElementStateClass *state_class = GUPPI_ELEMENT_STATE_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_ELEMENT_STATE);

  object_class->get_arg = guppi_pie_state_get_arg;
  object_class->set_arg = guppi_pie_state_set_arg;
  object_class->destroy = guppi_pie_state_destroy;
  object_class->finalize = guppi_pie_state_finalize;

  state_class->name = _("Pie Chart");

  state_class->view_type = GUPPI_TYPE_PIE_VIEW;
  state_class->config_tree = config_tree;

  add_arg ("data", GTK_TYPE_POINTER, ARG_DATA);
  add_arg ("data_labels", GTK_TYPE_POINTER, ARG_DATA_LABELS);
  add_arg ("data_styles", GTK_TYPE_POINTER, ARG_DATA_STYLES);
  add_arg ("radius", GTK_TYPE_DOUBLE, ARG_RADIUS);
  add_arg ("radius_lock", GTK_TYPE_BOOL, ARG_RADIUS_LOCK);
  add_arg ("radius_maximize", GTK_TYPE_BOOL, ARG_RADIUS_MAXIMIZE);
  add_arg ("edge_width", GTK_TYPE_DOUBLE, ARG_EDGE_WIDTH);
  add_arg ("base_offset", GTK_TYPE_DOUBLE, ARG_BASE_OFFSET);
  add_arg ("base_angle", GTK_TYPE_DOUBLE, ARG_BASE_ANGLE);
  add_arg ("default_slice_color_rgba", GTK_TYPE_UINT,
	   ARG_DEFAULT_SLICE_COLOR);
  add_arg ("use_stock_colors", GTK_TYPE_UINT, ARG_USE_STOCK_COLORS);
  add_arg ("edge_color_rgba", GTK_TYPE_UINT, ARG_EDGE_COLOR);
  add_arg ("show_percentages", GTK_TYPE_BOOL, ARG_SHOW_PERCENTAGES);
  add_arg ("label_font", GTK_TYPE_POINTER, ARG_LABEL_FONT);
  add_arg ("label_color_rgba", GTK_TYPE_UINT, ARG_LABEL_COLOR);
}

static void
guppi_pie_state_init (GuppiPieState * obj)
{
  GuppiElementState *state = GUPPI_ELEMENT_STATE (obj);
  GuppiShared *sh;

  sh = guppi_shared_data ();

  guppi_element_state_add_shared (state, SHARED_DATA, sh);

  guppi_element_state_add_shared (state,
				  SHARED_LABEL_DATA, guppi_shared_data ());

  guppi_element_state_add_shared (state,
				  SHARED_STYLE_DATA, guppi_shared_data ());



  obj->radius = 72;
  obj->radius_lock = TRUE;
  obj->edge_width = guppi_in2pt (1.0 / 48);
  obj->base_offset = 0;
  obj->base_angle = 0;
  obj->default_slice_color = RGBA_RED;
  obj->use_stock_colors = 1;
  obj->edge_color = RGBA_BLACK;
  obj->show_percentage = TRUE;

  obj->label_font = guppi_default_font ();
  guppi_ref (obj->label_font);

  obj->label_color = RGBA_BLACK;
}

GtkType guppi_pie_state_get_type (void)
{
  static GtkType guppi_pie_state_type = 0;
  if (!guppi_pie_state_type) {
    static const GtkTypeInfo guppi_pie_state_info = {
      "GuppiPieState",
      sizeof (GuppiPieState),
      sizeof (GuppiPieStateClass),
      (GtkClassInitFunc) guppi_pie_state_class_init,
      (GtkObjectInitFunc) guppi_pie_state_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_pie_state_type =
      gtk_type_unique (GUPPI_TYPE_ELEMENT_STATE, &guppi_pie_state_info);
  }
  return guppi_pie_state_type;
}

GuppiElementState *
guppi_pie_state_new (void)
{
  return GUPPI_ELEMENT_STATE (guppi_type_new (guppi_pie_state_get_type ()));
}

/**************************************************************************/

static GuppiSeqScalar *
get_slice_offsets (GuppiPieState * state)
{
  if (state->slice_offsets == NULL) {
    GuppiSeq *pie_data;
    gint i0, i1, i;

    pie_data = GUPPI_SEQ0 (guppi_pie_state_data (state));

    if (pie_data == NULL)
      return NULL;

    guppi_seq_indices (GUPPI_SEQ (pie_data), &i0, &i1);

    /* sequence creation should work better :-( */
    state->slice_offsets = GUPPI_SEQ_SCALAR (guppi_seq_scalar_new ());
    guppi_seq_size_hint (GUPPI_SEQ (state->slice_offsets), i1 - i0 + 1);
    for (i = i0; i <= i1; ++i)
      guppi_seq_scalar_append (state->slice_offsets, 0);
    guppi_seq_set_min_index (GUPPI_SEQ (state->slice_offsets), i0);

  }

  return state->slice_offsets;
}

static void
soff_insert_cb (GuppiSeq * slice_data, gint i0, gsize N,
		GuppiSeqScalar * soff)
{
  gsize j = 0;
  for (j = 0; j < N; ++j)
    guppi_seq_scalar_insert (soff, i0, 0);
}

static void
soff_delete_cb (GuppiSeq * slice_data, gint i, gsize N, GuppiSeqScalar * soff)
{
  guppi_seq_delete_many (GUPPI_SEQ (soff), i, N);
}

static void
connect_slice_and_offset_data (GuppiPieState * state)
{
  GuppiSeq *pie_data;
  GuppiSeqScalar *soff;

  pie_data = GUPPI_SEQ (guppi_pie_state_data (state));
  soff = get_slice_offsets (state);

  g_return_if_fail (soff != NULL);
  if (pie_data == NULL)
    return;

  gtk_signal_connect_after (GTK_OBJECT (pie_data),
			    "changed_insert",
			    GTK_SIGNAL_FUNC (soff_insert_cb), soff);

  gtk_signal_connect_after (GTK_OBJECT (pie_data),
			    "changed_delete",
			    GTK_SIGNAL_FUNC (soff_delete_cb), soff);
}

static void
disconnect_slice_and_offset_data (GuppiPieState * state)
{
  GuppiSeq *pie_data;
  GuppiSeqScalar *soff;

  pie_data = GUPPI_SEQ (guppi_pie_state_data (state));
  soff = get_slice_offsets (state);

  g_return_if_fail (soff != NULL);
  if (pie_data == NULL)
    return;

  gtk_signal_disconnect_by_data (GTK_OBJECT (pie_data), soff);
}

/**************************************************************************/


GuppiData *
guppi_pie_state_data (GuppiPieState * state)
{
  GtkObject *obj;

  g_return_val_if_fail (state != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_PIE_STATE (state), NULL);

  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (state),
					SHARED_DATA);

  return GUPPI_DATA0 (obj);

}

GuppiData *
guppi_pie_state_labels (GuppiPieState * state)
{
  GtkObject *obj;

  g_return_val_if_fail (state != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_PIE_STATE (state), NULL);

  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (state),
					SHARED_LABEL_DATA);

  return GUPPI_DATA0 (obj);
}

GuppiSeqStyle *
guppi_pie_state_styles (GuppiPieState * state)
{
  GtkObject *obj;

  g_return_val_if_fail (state != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_PIE_STATE (state), NULL);

  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (state),
					SHARED_STYLE_DATA);

  return GUPPI_SEQ_STYLE0 (obj);
}

void
guppi_pie_state_set_data (GuppiPieState * state, GuppiData * sd)
{
  GuppiElementState *es;

  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));
  g_return_if_fail (sd == NULL || GUPPI_IS_DATA (sd));

  es = GUPPI_ELEMENT_STATE (state);

  if (state->slice_offsets) {
    disconnect_slice_and_offset_data (state);
    guppi_unref0 (state->slice_offsets);
  }

  guppi_element_state_set_shared (es, SHARED_DATA, sd);

  connect_slice_and_offset_data (state);

  /* When we have categorical data, it acts as both the value & label data. */
  if (GUPPI_IS_SEQ_CATEGORICAL (sd))
    guppi_pie_state_set_labels (state, sd);
}

void
guppi_pie_state_set_labels (GuppiPieState * state, GuppiData * sd)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));
  g_return_if_fail (sd == NULL || GUPPI_IS_DATA (sd));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (state),
				  SHARED_LABEL_DATA, sd);
}

void
guppi_pie_state_set_styles (GuppiPieState * state, GuppiSeqStyle * sd)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));
  g_return_if_fail (sd == NULL || GUPPI_IS_SEQ_STYLE (sd));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (state),
				  SHARED_STYLE_DATA, sd);
}

void
guppi_pie_state_set_radius (GuppiPieState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->radius != x) {
    state->radius = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_pie_state_set_radius_lock (GuppiPieState * state, gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));

  if (state->radius_lock != x) {
    state->radius_lock = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_pie_state_set_radius_maximize (GuppiPieState * state, gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));

  if (state->radius_maximize != x) {
    state->radius_maximize = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_pie_state_set_edge_width (GuppiPieState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->edge_width != x) {
    state->edge_width = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}


void
guppi_pie_state_set_base_offset (GuppiPieState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));

  if (state->base_offset != x) {
    state->base_offset = MAX (x, 0);
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_pie_state_set_base_angle (GuppiPieState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));

  /* Normalize our angle */
  while (x < 0)
    x += 2 * M_PI;
  while (x >= 2 * M_PI)
    x -= 2 * M_PI;

  if (state->base_angle != x) {
    state->base_angle = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_pie_state_set_default_slice_color (GuppiPieState * state, guint32 c)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));

  if (state->default_slice_color != c) {
    state->default_slice_color = c;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_pie_state_set_use_stock_colors (GuppiPieState * state, guint32 c)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));

  if (state->use_stock_colors != c) {
    state->use_stock_colors = c;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_pie_state_set_edge_color (GuppiPieState * state, guint32 c)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));

  if (state->edge_color != c) {
    state->edge_color = c;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_pie_state_set_show_percentages (GuppiPieState * state, gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));

  if (state->show_percentage != x) {
    state->show_percentage = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_pie_state_set_label_font (GuppiPieState * state, GnomeFont * f)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));
  g_return_if_fail (f != NULL);
  g_return_if_fail (GNOME_IS_FONT (f));

  if (state->label_font != f) {

    guppi_refcounting_assign (state->label_font, f);

    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_pie_state_set_label_color (GuppiPieState * state, guint32 c)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));

  if (state->label_color != c) {
    state->label_color = c;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

/**************************************************************************/

void
guppi_pie_state_slice_bounds (GuppiPieState * state,
			      gint * i0, gint * i1)
{
  GuppiData *data;

  g_return_if_fail (state && GUPPI_IS_PIE_STATE (state));

  data = guppi_pie_state_data (state);
  if (data == NULL)
    return;
  
  if (GUPPI_IS_SEQ_SCALAR (data)) {
    guppi_seq_bounds (GUPPI_SEQ (data), i0, i1);
  } else if (GUPPI_IS_SEQ_CATEGORICAL (data)) {
    GuppiCategory *cat;
    cat = guppi_seq_categorical_category (GUPPI_SEQ_CATEGORICAL (data));
    if (i0) *i0 = (gint)guppi_category_min_code (cat);
    if (i1) *i1 = (gint)guppi_category_max_code (cat);
  } else {
    if (i0) *i0 = 0;
    if (i1) *i1 = -1;
    g_warning ("Unknown data type.");
  }
}

double
guppi_pie_state_slice_percentage (GuppiPieState * state, gint i)
{
  GuppiData *data;
  gint i0, i1;

  g_return_val_if_fail (state && GUPPI_IS_PIE_STATE (state), -1);
  
  guppi_pie_state_slice_bounds (state, &i0, &i1);
  g_return_val_if_fail (i0 <= i && i <= i1, -1);

  data = guppi_pie_state_data (state);
  if (data == NULL)
    return -1;

  if (GUPPI_IS_SEQ_SCALAR (data)) {
    GuppiSeqScalar *ss = GUPPI_SEQ_SCALAR (data);
    double x = fabs (guppi_seq_scalar_get (ss, i));
    double y = guppi_seq_scalar_sum_abs (ss);
    return y > 0 ? x/y : -1;
  } else if (GUPPI_IS_SEQ_CATEGORICAL (data)) {
    GuppiSeqCategorical *seq;
    GuppiCategory *cat;
    const gchar *name;

    seq = GUPPI_SEQ_CATEGORICAL (data);
    cat = guppi_seq_categorical_category (seq);
    name = guppi_category_find_by_code (cat, i);
    return guppi_seq_categorical_percentage (seq, name);
  } else {
    g_warning ("Unknown data type.");
    return -1;
  }
}

gboolean
guppi_pie_state_need_separate_label_data (GuppiPieState * state)
{
  GuppiData *data;
  g_return_val_if_fail (state && GUPPI_IS_PIE_STATE (state), FALSE);

  data = guppi_pie_state_data (state);
  if (data == NULL)
    return FALSE;

  return ! GUPPI_IS_SEQ_CATEGORICAL (data);
}

const gchar *
guppi_pie_state_slice_label (GuppiPieState * state, gint i)
{
  GuppiData *data;


  g_return_val_if_fail (state && GUPPI_IS_PIE_STATE (state), NULL);

  data = guppi_pie_state_data (state);
  if (data == NULL)
    return NULL;

  if (GUPPI_IS_SEQ_SCALAR (data)) {
    GuppiSeqString *ld = GUPPI_SEQ_STRING (guppi_pie_state_labels (state));
    if (ld == NULL)
      return NULL;
    if (!guppi_seq_in_bounds (GUPPI_SEQ (ld), i))
      return NULL;
    return guppi_seq_string_get (ld, i);
  } else if (GUPPI_IS_SEQ_CATEGORICAL (data)) {
    GuppiCategory *cat;
    cat = guppi_seq_categorical_category (GUPPI_SEQ_CATEGORICAL (data));
    return guppi_category_find_by_code (cat, i);
  } else {
    g_warning ("Unknown data type.");
    return NULL;
  }
}

guint32 guppi_pie_state_slice_color (GuppiPieState * state, gint i)
{
  GuppiSeqStyle *styles;
  int stock;

  g_return_val_if_fail (state != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_PIE_STATE (state), 0);

  styles = guppi_pie_state_styles (state);

  if (styles && guppi_seq_in_bounds (GUPPI_SEQ (styles), i)) {
    GuppiStyle *sty = guppi_seq_style_get (styles, i);
    if (sty)
      return guppi_style_color (sty);
  }

  if (0 != (stock = guppi_pie_state_use_stock_colors (state))) {
    GuppiStyle *sty = guppi_style_stock (i, stock);
    if (sty != NULL)
      return guppi_style_color (sty);
  }

  return state->default_slice_color;
}

double
guppi_pie_state_slice_offset (GuppiPieState * state, gint i)
{
  GuppiSeqScalar *soff;
  double offset;

  g_return_val_if_fail (state != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_PIE_STATE (state), 0);

  soff = get_slice_offsets (state);
  offset = guppi_pie_state_base_offset (state);

  if (guppi_seq_in_bounds (GUPPI_SEQ (soff), i)) {

    offset += MAX (guppi_seq_scalar_get (soff, i), 0);

  }

  return MAX (offset, 0);
}

void
guppi_pie_state_set_slice_offset (GuppiPieState * state, gint i, double x)
{
  GuppiSeqScalar *soff;

  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_PIE_STATE (state));

  x -= guppi_pie_state_base_offset (state);

  soff = get_slice_offsets (state);

  if (!guppi_seq_in_bounds (GUPPI_SEQ (soff), i))
    return;

  guppi_seq_scalar_set (soff, i, x);
  guppi_element_state_changed (GUPPI_ELEMENT_STATE (state));
}

/* $Id: guppi-pie-state.c,v 1.24 2001/01/16 23:36:03 trow Exp $ */
