/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-date-series.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <guppi-convenient.h>
#include "guppi-date-series.h"
#include "guppi-date-series-impl.h"

static GtkObjectClass * parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_date_series_get_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_date_series_set_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_date_series_finalize (GtkObject *obj)
{
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
write_xml_element (GuppiDateIndexed *dind, const GDate *dt,
		   GuppiOutputXML *out)
{
  double x;

  x = guppi_date_series_get (GUPPI_DATE_SERIES (dind), (GDate *)dt);

  guppi_output_xml_tagf (out, "scalar", "date='%d-%d-%d'",
			 g_date_year ((GDate *)dt),
			 g_date_month ((GDate *)dt),
			 g_date_day ((GDate *)dt));
  guppi_output_xml_printf (out, "%g", x);
  guppi_output_xml_end_tag_check (out, "scalar");
}

static GuppiData *
create_xml_object (GuppiAttributesXML *ax)
{
  g_assert_not_reached ();
  return NULL;
}

static GuppiContextXML *
content_xml_context (GuppiData *data, GuppiAttributesXML *ax)
{
  g_assert_not_reached ();
  return NULL;
}

static void
guppi_date_series_class_init (GuppiDateSeriesClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *)klass;
  GuppiDataClass *data_class = GUPPI_DATA_CLASS (klass);
  GuppiDateIndexedClass *dind_class = GUPPI_DATE_INDEXED_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_DATE_INDEXED);

  dind_class->write_xml_element = write_xml_element;

  data_class->type_name = _("Date Series");
  data_class->impl_type = GUPPI_TYPE_DATE_SERIES_IMPL;
  data_class->create_xml_object = create_xml_object;
  data_class->content_xml_context = content_xml_context;

  object_class->get_arg = guppi_date_series_get_arg;
  object_class->set_arg = guppi_date_series_set_arg;
  object_class->finalize = guppi_date_series_finalize;

}

static void
guppi_date_series_init (GuppiDateSeries *obj)
{

}

GtkType
guppi_date_series_get_type (void)
{
  static GtkType guppi_date_series_type = 0;
  if (!guppi_date_series_type) {
    static const GtkTypeInfo guppi_date_series_info = {
      "GuppiDateSeries",
      sizeof (GuppiDateSeries),
      sizeof (GuppiDateSeriesClass),
      (GtkClassInitFunc)guppi_date_series_class_init,
      (GtkObjectInitFunc)guppi_date_series_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_date_series_type = gtk_type_unique (GUPPI_TYPE_DATE_INDEXED, &guppi_date_series_info);
  }
  return guppi_date_series_type;
}

GuppiData *
guppi_date_series_new (void)
{
  return guppi_data_newv (GUPPI_TYPE_DATE_SERIES, NULL, 0, NULL); 
}

double
guppi_date_series_get (GuppiDateSeries *ser, GDate *dt)
{
  GuppiDateSeriesImpl *impl;
  GuppiDateSeriesImplClass *impl_class;

  g_return_val_if_fail (GUPPI_IS_DATE_SERIES (ser), 0);
  g_return_val_if_fail (dt && g_date_valid (dt), 0);

  impl = GUPPI_DATE_SERIES_IMPL (guppi_data_impl (GUPPI_DATA (ser)));
  impl_class = GUPPI_DATE_SERIES_IMPL_CLASS (GTK_OBJECT (impl)->klass);
  
  g_assert (impl_class->get);

  return impl_class->get (impl, dt);
}

void
guppi_date_series_set (GuppiDateSeries *ser, GDate *dt, double x)
{
  GuppiDateSeriesImpl *impl;
  GuppiDateSeriesImplClass *impl_class;

  g_return_if_fail (GUPPI_IS_DATE_SERIES (ser));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (ser)));
  g_return_if_fail (dt && g_date_valid (dt));

  impl = GUPPI_DATE_SERIES_IMPL (guppi_data_impl (GUPPI_DATA (ser)));
  impl_class = GUPPI_DATE_SERIES_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->set);

  impl_class->set (impl, dt, x);
}

void
guppi_date_series_unset (GuppiDateSeries *ser, GDate *dt)
{
  GuppiDateSeriesImpl *impl;
  GuppiDateSeriesImplClass *impl_class;

  g_return_if_fail (GUPPI_IS_DATE_SERIES (ser));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (ser)));
  g_return_if_fail (dt && g_date_valid (dt));

  impl = GUPPI_DATE_SERIES_IMPL (guppi_data_impl (GUPPI_DATA (ser)));
  impl_class = GUPPI_DATE_SERIES_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->unset);

  impl_class->unset (impl, dt);
}

gint
guppi_date_series_get_many (GuppiDateSeries *ser,
			    GDate *base_date, gint count,
			    double *buffer)
{
  GuppiDateSeriesImpl *impl;
  GuppiDateSeriesImplClass *impl_class;
  gint retval;
  GDate dt;
  gint j;

  g_return_val_if_fail (ser && GUPPI_IS_DATE_SERIES (ser), 0);
  g_return_val_if_fail (base_date && g_date_valid (base_date), 0);

  if (count == 0)
    return 0;

  g_return_val_if_fail (buffer, 0);

  impl = GUPPI_DATE_SERIES_IMPL (guppi_data_impl (GUPPI_DATA (ser)));
  impl_class = GUPPI_DATE_SERIES_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->get_many
      && impl_class->get_many (impl, base_date, count, buffer, &retval))
    return retval;

  dt = *base_date;
  j = 0;

  while (count
	 && g_date_valid (&dt) 
	 && guppi_date_indexed_in_bounds (GUPPI_DATE_INDEXED (ser), &dt)) {
    
    if (guppi_date_indexed_valid (GUPPI_DATE_INDEXED (ser), &dt)) {
      buffer[j] = guppi_date_series_get (ser, &dt);
      ++j;
      if (count > 0)
	--count;
      else
	++count;
    }
    
    if (count > 0)
      g_date_add_days (&dt, 1);
    else
      g_date_subtract_days (&dt, 1);
  }

  return j;
}

gint
guppi_date_series_get_range (GuppiDateSeries *ser,
			     GDate *start_date, GDate *end_date,
			     double *buffer, gint bufsize)
{
  g_return_val_if_fail (ser && GUPPI_IS_DATE_SERIES (ser), 0);
  g_return_val_if_fail (start_date && g_date_valid (start_date), 0);
  g_return_val_if_fail (end_date && g_date_valid (end_date), 0);

  if (bufsize == 0)
    return 0;

  g_return_val_if_fail (buffer, 0);

  return guppi_date_series_get_range_timecoded (ser, start_date, end_date,
						NULL, buffer, bufsize);
}

gint
guppi_date_series_get_range_timecoded (GuppiDateSeries *ser,
				       GDate *start_date, GDate *end_date,
				       double *timecode, double *buffer,
				       gint bufsize)
{
  GuppiDateSeriesImpl *impl;
  GuppiDateSeriesImplClass *impl_class;
  gint retval;
  GDate dt;
  gint j;

  g_return_val_if_fail (ser && GUPPI_IS_DATE_SERIES (ser), 0);
  g_return_val_if_fail (start_date && g_date_valid (start_date), 0);
  g_return_val_if_fail (end_date && g_date_valid (end_date), 0);

  if (bufsize == 0)
    return 0;

  g_return_val_if_fail (buffer || timecode, 0);

  guppi_date_indexed_clamp (GUPPI_DATE_INDEXED (ser), start_date);
  guppi_date_indexed_clamp (GUPPI_DATE_INDEXED (ser), end_date);
  guppi_2sort_dt (&start_date, &end_date);

  impl = GUPPI_DATE_SERIES_IMPL (guppi_data_impl (GUPPI_DATA (ser)));
  impl_class = GUPPI_DATE_SERIES_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  /* Try to use an implementation-supplied method. */
  if (impl_class->get_range
      && impl_class->get_range (impl, start_date, end_date,
				timecode, buffer, bufsize, &retval))
    return retval;


  dt = *start_date;
  j = 0;

  if (!guppi_date_indexed_valid (GUPPI_DATE_INDEXED (ser), &dt))
    guppi_date_indexed_incr (GUPPI_DATE_INDEXED (ser), &dt);

  while (g_date_lteq (&dt, end_date) && j < bufsize) {
    
    if (timecode)
      timecode[j] = g_date_julian (&dt);
    if (buffer)
      buffer[j] = guppi_date_series_get (ser, &dt);
    ++j;
    
    guppi_date_indexed_incr (GUPPI_DATE_INDEXED (ser), &dt);
  }
  
  return j;

}

gboolean
guppi_date_series_get_bounds (GuppiDateSeries *ser,
			      GDate *start_date, GDate *end_date,
			      double *min, double *max)
{
  GuppiDateSeriesImpl *impl;
  GuppiDateSeriesImplClass *impl_class;
  gboolean retval;
  double *buffer;
  gint i, N;
  double m, M;


  g_return_val_if_fail (ser && GUPPI_IS_DATE_SERIES (ser), 0);
  g_return_val_if_fail (start_date && g_date_valid (start_date), 0);
  g_return_val_if_fail (end_date && g_date_valid (end_date), 0);

  impl = GUPPI_DATE_SERIES_IMPL (guppi_data_impl (GUPPI_DATA (ser)));
  impl_class = GUPPI_DATE_SERIES_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  guppi_date_indexed_clamp (GUPPI_DATE_INDEXED (ser), start_date);
  guppi_date_indexed_clamp (GUPPI_DATE_INDEXED (ser), end_date);
  guppi_2sort_dt (&start_date, &end_date);

  /* Try to use an implementation-supplied method. */
  if (impl_class->get_bounds
      && impl_class->get_bounds (impl, start_date, end_date, min, max,
				 &retval))
    return retval;
    

  N = (gint)g_date_julian (end_date) - (gint)g_date_julian (start_date) + 1;
  buffer = guppi_new (double, N);
  N = guppi_date_series_get_range (ser, start_date, end_date,
				   buffer, N);
  if (N == 0)
    return FALSE;
  
  m = M = buffer[0];
  for (i=1; i<N; ++i) {
    double x = buffer[i];
    if (x < m)
      m = x;
    if (x > M)
      M = x;
  }
  
  if (min)
    *min = m;
  if (max)
    *max = M;
  
  guppi_free (buffer);
  
  return TRUE;
}
