/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-price-series.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <guppi-convenient.h>
#include "guppi-price-series.h"
#include "guppi-price-series-impl.h"

typedef struct _GuppiPriceSeriesPrivate GuppiPriceSeriesPrivate;
struct _GuppiPriceSeriesPrivate {
  gboolean cached_day;
  GDate last_date;
  guint last_valid;
  guint last_get_flag;
  double last_get_value;
};

#define priv(x) ((GuppiPriceSeriesPrivate*)((x)->opaque_internals))

static GtkObjectClass * parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_price_series_get_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_price_series_set_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_price_series_finalize (GtkObject *obj)
{
  GuppiPriceSeries *ser = GUPPI_PRICE_SERIES (obj);

  guppi_free (ser->opaque_internals);
  ser->opaque_internals = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
changed (GuppiData *data)
{
  GuppiPriceSeriesPrivate *p = priv (GUPPI_PRICE_SERIES (data));

  if (GUPPI_DATA_CLASS (parent_class)->changed)
    GUPPI_DATA_CLASS (parent_class)->changed (data);

  p->cached_day = FALSE;
}

static void
write_xml_element (GuppiDateIndexed *ind, const GDate *dt,
		   GuppiOutputXML *out)
{
  GuppiPriceSeries *ser = GUPPI_PRICE_SERIES (ind);
  guint code;
  
  code = guppi_price_series_valid (ser, (GDate *)dt);

  if (!code)
    return;

  guppi_output_xml_tagf (out, "Day",
			 "date='%d-%d-%d'",
			 g_date_year ((GDate *)dt),
			 g_date_month ((GDate *)dt),
			 g_date_day ((GDate *)dt));

  if (code & PRICE_OPEN)
    guppi_output_xml_elementf (out, "open", "%g", 
			       guppi_price_series_open (ser, (GDate *)dt));

  if (code & PRICE_HIGH)
    guppi_output_xml_elementf (out, "high", "%g", 
			       guppi_price_series_high (ser, (GDate *)dt));

  if (code & PRICE_LOW)
    guppi_output_xml_elementf (out, "low", "%g", 
			       guppi_price_series_low (ser, (GDate *)dt));
  
  if (code & PRICE_CLOSE)
    guppi_output_xml_elementf (out, "close", "%g", 
			       guppi_price_series_close (ser, (GDate *)dt));
      
  guppi_output_xml_end_tag_check (out, "Day");
}

static GuppiData *
create_xml_object (GuppiAttributesXML *ax)
{
  return guppi_price_series_new ();
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

typedef struct _PriceParse PriceParse;
struct _PriceParse {
  gboolean in_day;
  GDate date;
  guint32 awaiting_price;
  guint32 seen_prices;
  GuppiPriceSeries *ser;
};

static PriceParse *
pp_new (void)
{
  PriceParse *pp = guppi_new0 (PriceParse, 1);
  g_date_clear (&pp->date, 1);
  return pp;
}

static void
pp_free (gpointer pp)
{
  guppi_free (pp);
}

static GuppiContextXML *
begin_day (const CHAR *name, const CHAR **attr, gpointer ud)
{
  PriceParse *pp = (PriceParse *)ud;
  gint d_d, d_m, d_y;

  g_return_val_if_fail (pp, NULL);

  if (attr == NULL || attr[0] == NULL || strcmp(attr[0], "date")
      || attr[1] == NULL) {
    g_warning ("Date not properly specified with Day.");
    return NULL;
  }

  if (sscanf (attr[1], "%d-%d-%d", &d_y, &d_m, &d_d) != 3) {
    g_warning ("Ill-formed date '%s'.", attr[1]);
    return NULL;
  }

  g_date_set_dmy (&pp->date, d_d, d_m, d_y);
  if (! g_date_valid (&pp->date)) {
    g_warning ("Illegal date '%d-%d-%d'", d_y, d_m, d_d);
    return NULL;
  }

  pp->in_day = TRUE;
  pp->awaiting_price = pp->seen_prices = 0;

  return NULL;
}

static void
end_day (const CHAR *name, gpointer ud)
{
  PriceParse *pp = (PriceParse *)ud;
  g_return_if_fail (pp);

  if (pp->seen_prices == 0)
    g_message ("Empty Day tag.");

  pp->in_day = FALSE;
}

static GuppiContextXML *
begin_misc (const CHAR *name, const CHAR **attr, gpointer ud)
{
  PriceParse *pp = (PriceParse *)ud;
  guint32 flag = 0;

  g_return_val_if_fail (pp, NULL);

  if (!pp->in_day) {
    g_message ("Unknown/misplaced tag: %s", name);
    return NULL;
  }

  if (!strcmp (name, "gpi:open"))
    flag = PRICE_OPEN;
  else if (!strcmp (name, "gpi:high"))
    flag = PRICE_HIGH;
  else if (!strcmp (name, "gpi:low"))
    flag = PRICE_LOW;
  else if (!strcmp (name, "gpi:close"))
    flag = PRICE_CLOSE;

  if (flag == 0) {
    g_message ("Unknown tag '%s'", name);
    return NULL;
  }
    
  pp->awaiting_price = flag;
  if (pp->awaiting_price & pp->seen_prices)
    g_message ("redefining '%s'", name);

  return NULL;
}

static void
end_misc (const CHAR *name, gpointer ud)
{
  PriceParse *pp = (PriceParse *)ud;
  g_return_if_fail (pp);
  pp->awaiting_price = 0;
}

static void
process_chars (const CHAR *s, gint len, gpointer ud)
{
  PriceParse *pp = (PriceParse *)ud;
  gchar *str;
  double x;
  g_return_if_fail (pp);

  if (pp->awaiting_price == 0 || !pp->in_day) {
    g_warning ("Stray characters.");
    return;
  }

  str = guppi_strndup (s, len);
  x = atof (str);
  guppi_free (str);

  guppi_price_series_set (pp->ser, pp->awaiting_price, &pp->date, x);
  pp->seen_prices |= pp->awaiting_price;
}

static GuppiContextXML *
content_xml_context (GuppiData *d, GuppiAttributesXML *ax)
{
  GuppiContextXML *ctx;
  PriceParse *pp;

  pp = pp_new ();
  pp->ser = GUPPI_PRICE_SERIES (d);
  
  ctx = guppi_context_xml_new ();
  guppi_context_xml_set_begin_element_fn (ctx, begin_misc);
  guppi_context_xml_set_end_element_fn (ctx, end_misc);
  guppi_context_xml_add_begin_element_fn (ctx, "gpi:Day", begin_day);
  guppi_context_xml_add_end_element_fn (ctx, "gpi:Day", end_day);
  guppi_context_xml_set_characters_fn (ctx, process_chars);

  guppi_context_xml_set_user_data_destroy_fn (ctx, pp, pp_free);

  return ctx;
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

static void
guppi_price_series_class_init (GuppiPriceSeriesClass *klass)
{
  GtkObjectClass* object_class = (GtkObjectClass *)klass;
  GuppiDataClass *data_class = GUPPI_DATA_CLASS (klass);
  GuppiDateIndexedClass *dind_class = GUPPI_DATE_INDEXED_CLASS (klass);

  parent_class = gtk_type_class(GUPPI_TYPE_DATE_INDEXED);

  object_class->get_arg = guppi_price_series_get_arg;
  object_class->set_arg = guppi_price_series_set_arg;
  object_class->finalize = guppi_price_series_finalize;

  data_class->type_name = _("Price Series");
  data_class->impl_type = GUPPI_TYPE_PRICE_SERIES_IMPL;
  data_class->changed = changed;
  data_class->create_xml_object = create_xml_object;
  data_class->content_xml_context = content_xml_context;

  dind_class->write_xml_element = write_xml_element;

}

static void
guppi_price_series_init (GuppiPriceSeries *obj)
{
  GuppiPriceSeriesPrivate *p = guppi_new (GuppiPriceSeriesPrivate, 1);

  p->cached_day = FALSE;

  obj->opaque_internals = p;
}

GtkType
guppi_price_series_get_type (void)
{
  static GtkType guppi_price_series_type = 0;
  if (!guppi_price_series_type) {
    static const GtkTypeInfo guppi_price_series_info = {
      "GuppiPriceSeries",
      sizeof(GuppiPriceSeries),
      sizeof(GuppiPriceSeriesClass),
      (GtkClassInitFunc)guppi_price_series_class_init,
      (GtkObjectInitFunc)guppi_price_series_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_price_series_type = gtk_type_unique (GUPPI_TYPE_DATE_INDEXED, &guppi_price_series_info);
  }
  return guppi_price_series_type;
}

GuppiData *
guppi_price_series_new (void)
{
  return guppi_data_newv (GUPPI_TYPE_PRICE_SERIES, NULL, 0, NULL);
}

/*****************************************************************************/

typedef struct _GuppiDataOp_PriceSeries GuppiDataOp_PriceSeries;
struct _GuppiDataOp_PriceSeries {
  GuppiDataOp op;

  GDate date;
  guint code;
  double value;
};

static void
op_set (GuppiData *d, GuppiDataOp *op)
{
  GuppiPriceSeries *ser = GUPPI_PRICE_SERIES (d);
  GuppiDataOp_PriceSeries *ps_op = (GuppiDataOp_PriceSeries *)op;

  GuppiPriceSeriesPrivate *p;
  GuppiPriceSeriesImpl *impl;
  GuppiPriceSeriesImplClass *impl_class;

  p = priv (ser);

  impl = GUPPI_PRICE_SERIES_IMPL (guppi_data_impl (GUPPI_DATA(ser)));
  impl_class = GUPPI_PRICE_SERIES_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->set);

  impl_class->set (impl, ps_op->code, &ps_op->date, ps_op->value);

  if (p->cached_day && g_date_compare (&p->last_date, &ps_op->date)) {
    p->last_valid |= ps_op->code;
    if (p->last_get_flag == ps_op->code)
      p->last_get_value = ps_op->value;
  }
}

/*****************************************************************************/

static gboolean
single_bit (guint code)
{
  if (code == 0)
    return FALSE;

  while (code) {
    guint bottom_bit = code & 1;
    code = code>>1;
    if (code && bottom_bit)
      return FALSE;
  }
  return TRUE;
}

static void
cache_valid (GuppiPriceSeriesPrivate *p, GDate *date, guint code)
{
  if (!(p->cached_day && g_date_compare (&p->last_date, date))) {
    p->cached_day = TRUE;
    p->last_date = *date;
    p->last_get_flag = 0;
  }
  p->last_valid = code;
}

static void
cache_value (GuppiPriceSeriesPrivate *p, GDate *date, guint code, double x)
{
  if (p->cached_day && g_date_compare (&p->last_date, date)) {
    p->last_get_flag = code;
    p->last_get_value = x;
  }
}

guint
guppi_price_series_valid (GuppiPriceSeries *ser, GDate *date)
{
  GuppiPriceSeriesPrivate *p;

  GuppiPriceSeriesImpl *impl;
  GuppiPriceSeriesImplClass *impl_class;
  guint code;

  g_return_val_if_fail (ser && GUPPI_IS_PRICE_SERIES (ser), 0);
  g_return_val_if_fail (date && g_date_valid (date), 0);

  p = priv (ser);

  if (p->cached_day && g_date_compare (&p->last_date, date) == 0)
    return p->last_valid;

  if (!guppi_date_indexed_in_bounds (GUPPI_DATE_INDEXED(ser), date))
    return 0;

  impl = GUPPI_PRICE_SERIES_IMPL (guppi_data_impl (GUPPI_DATA(ser)));
  impl_class = GUPPI_PRICE_SERIES_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->valid);

  code = impl_class->valid (impl, date);

  cache_valid (p, date, code);

  return code;
}

double
guppi_price_series_get (GuppiPriceSeries *ser, guint get_code, GDate *date)
{
  GuppiPriceSeriesPrivate *p;
  GuppiPriceSeriesImpl *impl;
  GuppiPriceSeriesImplClass *impl_class;
  gboolean date_match;
  guint valid_code;
  
  g_return_val_if_fail (ser && GUPPI_IS_PRICE_SERIES (ser), 0);
  g_return_val_if_fail (single_bit (get_code), 0);
  g_return_val_if_fail (date && g_date_valid (date), 0);

  p = priv (ser);

  impl = GUPPI_PRICE_SERIES_IMPL (guppi_data_impl (GUPPI_DATA(ser)));
  impl_class = GUPPI_PRICE_SERIES_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  date_match = p->cached_day && (g_date_compare (&p->last_date, date) == 0);

  if (date_match)
    valid_code = p->last_valid;
  else {
    g_assert (impl_class->valid);
    valid_code = impl_class->valid (impl, date);
    cache_valid (p, date, valid_code);
  }
  
  g_return_val_if_fail (valid_code && get_code, 0);

  if (date_match && (p->last_get_flag & get_code)) {

    return p->last_get_value;

  } else {
    double x;

    g_assert (impl_class->get);

    x = impl_class->get (impl, get_code, date);

    cache_value (p, date, get_code, x);

    return x;
  }

  g_assert_not_reached();
  return 0;
}

void
guppi_price_series_set (GuppiPriceSeries *ser, guint code, GDate *date,
			double val)
{
  GuppiDataOp_PriceSeries op;

  g_return_if_fail (ser && GUPPI_IS_PRICE_SERIES (ser));
  g_return_if_fail (single_bit (code));
  g_return_if_fail (date && g_date_valid (date));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (ser)));

  op.op.op = op_set;
  op.date = *date;
  op.code = code;
  op.value = val;

  guppi_data_add_pending_op (GUPPI_DATA (ser), (GuppiDataOp *)&op);
  guppi_data_changed (GUPPI_DATA (ser));
}

double
guppi_price_series_open (GuppiPriceSeries *ser, GDate *date)
{
  return guppi_price_series_get (ser, PRICE_OPEN, date);
}

double
guppi_price_series_high (GuppiPriceSeries *ser, GDate *date)
{
  return guppi_price_series_get (ser, PRICE_HIGH, date);
}

double
guppi_price_series_low (GuppiPriceSeries *ser, GDate *date)
{
  return guppi_price_series_get (ser, PRICE_LOW, date);
}

double
guppi_price_series_close (GuppiPriceSeries *ser, GDate *date)
{
  return guppi_price_series_get (ser, PRICE_CLOSE, date);
}

void
guppi_price_series_set_open (GuppiPriceSeries *ser, GDate *date, double x)
{
  guppi_price_series_set (ser, PRICE_OPEN, date, x);
}

void
guppi_price_series_set_high (GuppiPriceSeries *ser, GDate *date, double x)
{
  guppi_price_series_set (ser, PRICE_HIGH, date, x);
}

void
guppi_price_series_set_low (GuppiPriceSeries *ser, GDate *date, double x)
{
  guppi_price_series_set (ser, PRICE_LOW, date, x);
}

void
guppi_price_series_set_close (GuppiPriceSeries *ser, GDate *date, double x)
{
  guppi_price_series_set (ser, PRICE_CLOSE, date, x);
}

gint
guppi_price_series_get_many (GuppiPriceSeries *ser, guint code,
			     GDate *base_date, gint count,
			     double *buffer)
{
  GuppiPriceSeriesPrivate *p;
  GuppiPriceSeriesImpl *impl;
  GuppiPriceSeriesImplClass *impl_class;

  g_return_val_if_fail (ser && GUPPI_IS_PRICE_SERIES (ser), 0);
  g_return_val_if_fail (single_bit (code), 0);
  g_return_val_if_fail (base_date && g_date_valid (base_date), 0);
  
  if (count == 0)
    return 0;

  g_return_val_if_fail (buffer != NULL, 0);

  p = priv (ser);

  impl = GUPPI_PRICE_SERIES_IMPL (guppi_data_impl (GUPPI_DATA(ser)));
  impl_class = GUPPI_PRICE_SERIES_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->get_many) {

    return impl_class->get_many (impl, code, base_date, count, buffer);

  } else {

    GDate dt = *base_date;
    gint i = 0;

    while (count
	   && guppi_date_indexed_in_bounds (GUPPI_DATE_INDEXED (ser), &dt)) {

      if (guppi_price_series_valid (ser, &dt) & code) {
	buffer[i] = guppi_price_series_get (ser, code, &dt);
	++i;
	if (count > 0)
	  --count;
	else
	  ++count;
      }

      g_date_add_days (&dt, count > 0 ? 1 : -1);
      
    }
    
    return i;
  }

  g_assert_not_reached ();
  return 0;
}

gint
guppi_price_series_get_range (GuppiPriceSeries *ser, guint code,
			      GDate *start_date, GDate *end_date,
			      double *buffer, gint buffer_size)
{
  g_return_val_if_fail (ser && GUPPI_IS_PRICE_SERIES (ser), 0);
  g_return_val_if_fail (single_bit (code), 0);
  g_return_val_if_fail (start_date && g_date_valid (start_date), 0);
  g_return_val_if_fail (end_date && g_date_valid (end_date), 0);

  if (buffer_size == 0) 
    return 0;

  g_return_val_if_fail (buffer != NULL, 0);

  return
    guppi_price_series_get_range_timecoded (ser, code, start_date, end_date,
					    NULL, buffer, buffer_size);
}

gint
guppi_price_series_get_range_timecoded (GuppiPriceSeries *ser, guint code,
					GDate *start_date, GDate *end_date,
					double *timecode, double *buffer,
					gint buffer_size)
{
  GuppiPriceSeriesPrivate *p;
  GuppiPriceSeriesImpl *impl;
  GuppiPriceSeriesImplClass *impl_class;

  g_return_val_if_fail (ser && GUPPI_IS_PRICE_SERIES (ser), 0);
  g_return_val_if_fail (single_bit (code), 0);
  g_return_val_if_fail (start_date && g_date_valid (start_date), 0);
  g_return_val_if_fail (end_date && g_date_valid (end_date), 0);

  if (buffer_size == 0) 
    return 0;

  g_return_val_if_fail (buffer || timecode, 0);

  guppi_2sort_dt (&start_date, &end_date);

  guppi_date_indexed_clamp (GUPPI_DATE_INDEXED (ser), start_date);
  guppi_date_indexed_clamp (GUPPI_DATE_INDEXED (ser), end_date);

  p = priv (ser);

  impl = GUPPI_PRICE_SERIES_IMPL (guppi_data_impl (GUPPI_DATA(ser)));
  impl_class = GUPPI_PRICE_SERIES_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->get_range) {

    return impl_class->get_range (impl, code, start_date, end_date,
				  timecode, buffer, buffer_size);

  } else {

    GDate dt = *start_date;
    gint delta;
    gint i = 0;
    
    delta = - g_date_compare (start_date, end_date);

    do {

      if (guppi_price_series_valid (ser, &dt) & code) {

	if (i >= buffer_size)
	  return -1;
	
	if (timecode)
	  timecode[i] = g_date_julian (&dt);
	if (buffer)
	  buffer[i] = guppi_price_series_get (ser, code, &dt);
	++i;
      }

      g_date_add_days (&dt, delta);

    } while (g_date_compare (&dt, end_date));

    return i;

  }

  return 0;
}

gboolean
guppi_price_series_get_bounds (GuppiPriceSeries *ser, guint code,
			       GDate *start_date, GDate *end_date,
			       double *min, double *max)
{
  GuppiPriceSeriesPrivate *p;
  GuppiPriceSeriesImpl *impl;
  GuppiPriceSeriesImplClass *impl_class;

  g_return_val_if_fail (ser && GUPPI_IS_PRICE_SERIES (ser), TRUE);
  g_return_val_if_fail (single_bit (code), TRUE);
  g_return_val_if_fail (start_date && g_date_valid (start_date), TRUE);
  g_return_val_if_fail (end_date && g_date_valid (end_date), TRUE);

  if (guppi_date_indexed_empty (GUPPI_DATE_INDEXED (ser)))
    return FALSE;

  guppi_2sort_dt (&start_date, &end_date);

  guppi_date_indexed_clamp (GUPPI_DATE_INDEXED (ser), start_date);
  guppi_date_indexed_clamp (GUPPI_DATE_INDEXED (ser), end_date);

  p = priv (ser);

  impl = GUPPI_PRICE_SERIES_IMPL (guppi_data_impl (GUPPI_DATA(ser)));
  impl_class = GUPPI_PRICE_SERIES_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->get_bounds) {

    return impl_class->get_bounds (impl, code, start_date, end_date,
				   min, max);

  } else {

    GDate dt = *start_date;
    double x, m = 0, M = -1;
    gboolean non_empty = FALSE;
    
    do {

      if (guppi_price_series_valid (ser, &dt) & code) {

	x = guppi_price_series_get (ser, code, &dt);
	
	if (!non_empty) {
	  m = x;
	  M = x;
	} else {
	  if (x < m) m = x;
	  if (x > M) M = x;
	}

	non_empty = TRUE;
      }

      g_date_add_days (&dt, 1);

    } while (g_date_lteq (&dt, end_date));

    if (min) *min = m;
    if (max) *max = M;

    return non_empty;
  }

  g_assert_not_reached ();

  return FALSE;
}
