/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-seq-categorical.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
/* #include <gnome.h> */
#include <gtk/gtksignal.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <guppi-convenient.h>
#include "guppi-seq-categorical.h"

typedef struct _GuppiSeqCategoricalPrivate GuppiSeqCategoricalPrivate;
struct _GuppiSeqCategoricalPrivate {
  GuppiCategory *category;

  gboolean auto_add;

  GHashTable *freq_table;
};

#define priv(x) ((GuppiSeqCategoricalPrivate *)((x)->opaque_internals))


static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_seq_categorical_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_categorical_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_categorical_finalize (GtkObject * obj)
{
  GuppiSeqCategorical *seq = GUPPI_SEQ_CATEGORICAL (obj);
  GuppiSeqCategoricalPrivate *p = priv (seq);

  guppi_unref0 (p->category);

  if (p->freq_table)
    g_hash_table_foreach (p->freq_table, guppi_free_hash_val, NULL);
  g_hash_table_destroy (p->freq_table);
  p->freq_table = NULL;

  guppi_free (seq->opaque_internals);
  seq->opaque_internals = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/*****************************************************************************/

/**** Frequency Table Updating **** */

/*
  By doing all of this in signals, we ensure that the tables get updated
  even if we directly manipulate the underlying GuppiSeqInteger object.
  Cool, 'eh?
*/

static void
freq_adjust (GuppiSeqCategorical * seq, gint i0, gint i1, gint adj)
{
  GuppiSeqCategoricalPrivate *p = priv (seq);
  GHashTable *freq = p->freq_table;
  gint i;
  gboolean has_missing;
  code_t last_code = GUPPI_INVALID_CODE;
  gpointer last_val = NULL;

  i0 = MAX (i0, guppi_seq_min_index (GUPPI_SEQ (seq)));
  i1 = MIN (i1, guppi_seq_max_index (GUPPI_SEQ (seq)));

  has_missing = guppi_seq_has_missing (GUPPI_SEQ (seq));

  for (i = i0; i <= i1; ++i) {
    code_t c;
    gpointer val;

    if (!has_missing || guppi_seq_available (GUPPI_SEQ (seq), i)) {

      c = (code_t) guppi_seq_integer_get (GUPPI_SEQ_INTEGER (seq), i);
      if (last_val && (c == last_code)) {
	val = last_val;
      } else {
	gpointer ptr = GUPPI_CATEGORY_CODE_TO_POINTER (c);
	val = g_hash_table_lookup (freq, ptr);

	if (val == NULL) {
	  val = guppi_new0 (gint, 1);
	  g_hash_table_insert (freq, ptr, val);
	}

	last_code = c;
	last_val = val;
      }

      *(gint *) val += adj;
    }
  }
}

static void
set_before_cb (GuppiSeq * seq, gint i0, gint i1, gpointer foo)
{
  freq_adjust (GUPPI_SEQ_CATEGORICAL (seq), i0, i1, -1);
}

static void
set_after_cb (GuppiSeq * seq, gint i0, gint i1, gpointer foo)
{
  freq_adjust (GUPPI_SEQ_CATEGORICAL (seq), i0, i1, 1);
}

static void
delete_before_cb (GuppiSeq * seq, gint i, gsize N, gpointer foo)
{
  freq_adjust (GUPPI_SEQ_CATEGORICAL (seq), i, i - 1 + N, -1);
}

static void
insert_after_cb (GuppiSeq * seq, gint i, gsize N, gpointer foo)
{
  freq_adjust (GUPPI_SEQ_CATEGORICAL (seq), i, i - 1 + N, 1);
}

static void
write_xml_attributes (GuppiData *data, GuppiOutputXML *xml)
{
  GuppiCategory *cat;

  if (GUPPI_DATA_CLASS(parent_class)->write_xml_attributes)
    GUPPI_DATA_CLASS(parent_class)->write_xml_attributes (data, xml);

  cat = guppi_seq_categorical_category (GUPPI_SEQ_CATEGORICAL (data));

  guppi_output_xml_tag (xml, "Attribute");

  guppi_output_xml_element (xml, "name", "category");

  guppi_output_xml_tag (xml, "value");
  guppi_data_write_xml (GUPPI_DATA (cat), xml);
  guppi_output_xml_end_tag_check (xml, "value");

  guppi_output_xml_end_tag_check (xml, "Attribute");
}

static void
data_import_handler (GuppiData *d, gpointer ud)
{
  GuppiAttributesXML *ax = GUPPI_ATTRIBUTES_XML (ud);

  guppi_attributes_xml_set_pointer_unref (ax, "category", d);
}

static void
prepare_xml_attributes (GuppiAttributesXML *ax)
{
  GuppiContextXML *data_ctx;

  if (GUPPI_DATA_CLASS (parent_class)->prepare_xml_attributes)
    GUPPI_DATA_CLASS (parent_class)->prepare_xml_attributes (ax);

  guppi_attributes_xml_declare_pointer (ax, "category");
  data_ctx = guppi_data_read_xml_context (data_import_handler, ax);
  guppi_attributes_xml_set_special_import_context (ax, "category", data_ctx);
}

static GuppiData *
create_xml_object (GuppiAttributesXML *ax)
{
  return guppi_seq_categorical_new ();
}

static void
init_xml_object (GuppiData *d, GuppiAttributesXML *ax)
{
  GuppiSeqCategorical *seq;
  GuppiCategory *cat;

  if (GUPPI_DATA_CLASS (parent_class)->init_xml_object)
    GUPPI_DATA_CLASS (parent_class)->init_xml_object (d, ax);

  seq = GUPPI_SEQ_CATEGORICAL (d);
  cat = GUPPI_CATEGORY (guppi_attributes_xml_get_pointer (ax, "category"));

  guppi_seq_categorical_set_category (seq, cat);
}

static void
guppi_seq_categorical_class_init (GuppiSeqCategoricalClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiDataClass *data_class = GUPPI_DATA_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_SEQ_INTEGER);

  data_class->type_name = _("Categorical Sequence");

  /* We inherit the impl_type from GuppiSeqIntegerClass */
  /* data_class->impl_type = GUPPI_TYPE_SEQ_INTEGER_IMPL; */

  data_class->type_name_for_impl_lookup = "GuppiSeqInteger";
  data_class->write_xml_attributes = write_xml_attributes;
  data_class->prepare_xml_attributes = prepare_xml_attributes;
  data_class->create_xml_object = create_xml_object;
  data_class->init_xml_object = init_xml_object;
  
  object_class->get_arg = guppi_seq_categorical_get_arg;
  object_class->set_arg = guppi_seq_categorical_set_arg;
  object_class->finalize = guppi_seq_categorical_finalize;

}

static void
guppi_seq_categorical_init (GuppiSeqCategorical * obj)
{
  GuppiSeqCategoricalPrivate *p;

  p = guppi_new0 (GuppiSeqCategoricalPrivate, 1);

  p->category = GUPPI_CATEGORY (guppi_category_new ());

  p->auto_add = TRUE;

  p->freq_table = g_hash_table_new (NULL, NULL);

  obj->opaque_internals = p;

  gtk_signal_connect (GTK_OBJECT (obj), "changed_set",
		      GTK_SIGNAL_FUNC (set_before_cb), NULL);
  gtk_signal_connect_after (GTK_OBJECT (obj), "changed_set",
			    GTK_SIGNAL_FUNC (set_after_cb), NULL);

  gtk_signal_connect_after (GTK_OBJECT (obj), "changed_insert",
			    GTK_SIGNAL_FUNC (insert_after_cb), NULL);

  gtk_signal_connect (GTK_OBJECT (obj), "changed_delete",
		      GTK_SIGNAL_FUNC (delete_before_cb), NULL);
}

GtkType guppi_seq_categorical_get_type (void)
{
  static GtkType guppi_seq_categorical_type = 0;
  if (!guppi_seq_categorical_type) {
    static const GtkTypeInfo guppi_seq_categorical_info = {
      "GuppiSeqCategorical",
      sizeof (GuppiSeqCategorical),
      sizeof (GuppiSeqCategoricalClass),
      (GtkClassInitFunc) guppi_seq_categorical_class_init,
      (GtkObjectInitFunc) guppi_seq_categorical_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_seq_categorical_type = gtk_type_unique (GUPPI_TYPE_SEQ_INTEGER,
						  &guppi_seq_categorical_info);
  }
  return guppi_seq_categorical_type;
}

GuppiData *
guppi_seq_categorical_new (void)
{
  return guppi_data_newv (GUPPI_TYPE_SEQ_CATEGORICAL, NULL, 0, NULL);
}

GuppiCategory *
guppi_seq_categorical_category (GuppiSeqCategorical * seq)
{
  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_CATEGORICAL (seq), NULL);

  return priv (seq)->category;
}

void
guppi_seq_categorical_set_category (GuppiSeqCategorical * seq,
				    GuppiCategory * cat)
{
  GuppiSeqCategoricalPrivate *p;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_CATEGORICAL (seq));
  g_return_if_fail (cat != NULL && GUPPI_IS_CATEGORY (cat));

  p = priv (seq);

  guppi_refcounting_assign (p->category, cat);
  p->auto_add = FALSE;
}

gboolean guppi_seq_categorical_auto_add (GuppiSeqCategorical * seq)
{
  GuppiSeqCategoricalPrivate *p;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_CATEGORICAL (seq), FALSE);

  p = priv (seq);

  return p->auto_add;
}

void
guppi_seq_categorical_set_auto_add (GuppiSeqCategorical * seq, gboolean x)
{
  GuppiSeqCategoricalPrivate *p;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_CATEGORICAL (seq));

  p = priv (seq);

  p->auto_add = x;
}

const gchar *
guppi_seq_categorical_get (GuppiSeqCategorical * seq, gint i)
{
  GuppiSeqCategoricalPrivate *p;
  code_t c;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_CATEGORICAL (seq), NULL);
  g_return_val_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (seq), i), NULL);

  p = priv (seq);

  c = (code_t) guppi_seq_integer_get (GUPPI_SEQ_INTEGER (seq), i);

  return guppi_category_find_by_code (p->category, c);
}

gboolean
guppi_seq_categorical_set (GuppiSeqCategorical * seq, gint i,
			   const gchar * str)
{
  GuppiSeqCategoricalPrivate *p;
  code_t c;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_CATEGORICAL (seq), FALSE);
  g_return_val_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (seq), i), FALSE);
  g_return_val_if_fail (guppi_data_can_change (GUPPI_DATA (seq)), FALSE);

  p = priv (seq);
  c = guppi_category_find_by_name (p->category, str);

  if (p->auto_add && c == GUPPI_INVALID_CODE)
    c = guppi_category_add_by_name (p->category, str);

  if (c != GUPPI_INVALID_CODE)
    guppi_seq_integer_set (GUPPI_SEQ_INTEGER (seq), i, c);

  return c != GUPPI_INVALID_CODE;
}

gboolean
guppi_seq_categorical_prepend (GuppiSeqCategorical * seq, const gchar * str)
{
  GuppiSeqCategoricalPrivate *p;
  code_t c;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_CATEGORICAL (seq), FALSE);
  g_return_val_if_fail (guppi_data_can_change (GUPPI_DATA (seq)), FALSE);

  p = priv (seq);
  c = guppi_category_find_by_name (p->category, str);

  if (p->auto_add && c == GUPPI_INVALID_CODE)
    c = guppi_category_add_by_name (p->category, str);

  if (c != GUPPI_INVALID_CODE)
    guppi_seq_integer_prepend (GUPPI_SEQ_INTEGER (seq), c);

  return c != GUPPI_INVALID_CODE;
}

gboolean
guppi_seq_categorical_append (GuppiSeqCategorical * seq, const gchar * str)
{
  GuppiSeqCategoricalPrivate *p;
  code_t c;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_CATEGORICAL (seq), FALSE);
  g_return_val_if_fail (guppi_data_can_change (GUPPI_DATA (seq)), FALSE);

  p = priv (seq);
  c = guppi_category_find_by_name (p->category, str);

  if (p->auto_add && c == GUPPI_INVALID_CODE)
    c = guppi_category_add_by_name (p->category, str);

  if (c != GUPPI_INVALID_CODE)
    guppi_seq_integer_append (GUPPI_SEQ_INTEGER (seq), c);

  return c != GUPPI_INVALID_CODE;
}

gboolean
guppi_seq_categorical_insert (GuppiSeqCategorical * seq, gint i,
			      const gchar * str)
{
  GuppiSeqCategoricalPrivate *p;
  code_t c;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_CATEGORICAL (seq), FALSE);
  g_return_val_if_fail (guppi_data_can_change (GUPPI_DATA (seq)), FALSE);

  p = priv (seq);
  c = guppi_category_find_by_name (p->category, str);

  if (p->auto_add && c == GUPPI_INVALID_CODE)
    c = guppi_category_add_by_name (p->category, str);

  if (c != GUPPI_INVALID_CODE)
    guppi_seq_integer_insert (GUPPI_SEQ_INTEGER (seq), i, c);

  return c != GUPPI_INVALID_CODE;
}

gint
guppi_seq_categorical_frequency (GuppiSeqCategorical * seq, const gchar * str)
{
  GuppiSeqCategoricalPrivate *p;
  code_t c;
  gpointer ptr;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_CATEGORICAL (seq), 0);
  g_return_val_if_fail (str != NULL, 0);

  p = priv (seq);

  c = guppi_category_find_by_name (p->category, str);

  ptr = g_hash_table_lookup (p->freq_table,
			     GUPPI_CATEGORY_CODE_TO_POINTER (c));

  return ptr ? *(gint *) ptr : 0;
}

double
guppi_seq_categorical_percentage (GuppiSeqCategorical * seq,
				  const gchar * str)
{
  gsize N;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ_CATEGORICAL (seq), 0);
  g_return_val_if_fail (str != NULL, 0);

  N = guppi_seq_count (GUPPI_SEQ (seq));

  if (N == 0)
    return -1;

  return guppi_seq_categorical_frequency (seq, str) / (double) N;
}
