/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-custom-chart-type.c
 *
 * Copyright (C) 2000 Helix Code, Inc.
 *
 * Developed by Jody Goldberg <jgoldberg@home.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <gdk-pixbuf/gnome-canvas-pixbuf.h>

#include <guppi-paths.h>
#include <guppi-memory.h>
#include "guppi-chart-type.h"

typedef struct {
  GuppiCustomChartType publik;

  char const *const *xpm_data;
  GnomePixmap *image;
} CustomChartTypeImpl;

typedef struct {
  GuppiMajorChartType publik;

  GList *minor_chart_types;
  char const *const *xpm_data;
  GnomePixmap *image;
} MajorChartTypeImpl;

typedef struct {
  GuppiMinorChartType publik;

  char const *icon_file;
  GdkPixbuf *image;
} MinorChartTypeImpl;

static GList *custom_chart_types = NULL;
static GList *major_chart_types = NULL;

/***************************************************************************/

static gint
guppi_chart_type_compare (gconstpointer a_ptr, gconstpointer b_ptr)
{
  GuppiCustomChartType const *const a = a_ptr;
  GuppiCustomChartType const *const b = b_ptr;

  return g_strcasecmp (a->name, b->name);
}

GuppiCustomChartType *
guppi_custom_chart_type_register (char const *name,
				  char const *const *xpm_data,
				  char const *description,
				  GuppiCustomChartFactory factory,
				  gconstpointer user_data)
{
  CustomChartTypeImpl *record = guppi_new (CustomChartTypeImpl, 1);

  record->publik.name = name;
  record->publik.description = description;
  record->publik.factory = factory;
  record->publik.user_data = user_data;

  record->xpm_data = xpm_data;
  record->image = NULL;		/* Do not init till we need it */

  custom_chart_types = g_list_insert_sorted (custom_chart_types,
					     (gpointer) record,
					     &guppi_chart_type_compare);

  return &record->publik;
}

/***************************************************************************/

static gint
guppi_major_chart_type_compare (gconstpointer a_ptr, gconstpointer b_ptr)
{
  GuppiMajorChartType const *const a = a_ptr;
  GuppiMajorChartType const *const b = b_ptr;

  return g_strcasecmp (a->name, b->name);
}

GuppiMajorChartType *
guppi_major_chart_type_register (char const *name,
				 char const *const *xpm_data,
				 gconstpointer user_data)
{
  MajorChartTypeImpl *record = guppi_new (MajorChartTypeImpl, 1);

  record->publik.name = name;
  record->publik.user_data = user_data;

  record->xpm_data = xpm_data;
  record->image = NULL;		/* Do not init till we need it */
  record->minor_chart_types = NULL;

  major_chart_types = g_list_insert_sorted (major_chart_types,
					    (gpointer) record,
					    &guppi_major_chart_type_compare);

  return &record->publik;
}

static gint
guppi_minor_chart_type_compare (gconstpointer a_ptr, gconstpointer b_ptr)
{
  GuppiMinorChartType const *const a = a_ptr;
  GuppiMinorChartType const *const b = b_ptr;

  return g_strcasecmp (a->name, b->name);
}

GuppiMinorChartType *
guppi_minor_chart_type_register (GuppiMajorChartType * mjr,
				 char const *name,
				 char const *icon_file,
				 char const *description,
				 int row, int col,
				 GuppiMinorChartFactory factory,
				 gconstpointer user_data)
{
  MinorChartTypeImpl *record = guppi_new (MinorChartTypeImpl, 1);
  MajorChartTypeImpl *major = (MajorChartTypeImpl *) mjr;

  record->publik.name = name;
  record->publik.description = description;
  record->publik.row = row;
  record->publik.col = col;
  record->publik.factory = factory;
  record->publik.user_data = user_data;

  record->icon_file = icon_file;
  record->image = NULL;		/* Do not init till we need it */

  major->minor_chart_types = g_list_insert_sorted (major->minor_chart_types,
						   (gpointer) record,
						   &guppi_minor_chart_type_compare);

  return &record->publik;
}

GnomePixmap *
guppi_custom_chart_type_image (GuppiCustomChartType const *t)
{
  CustomChartTypeImpl *custom;

  g_return_val_if_fail (t != NULL, NULL);

  custom = (CustomChartTypeImpl *) t;
  if (custom->image == NULL) {
    GtkWidget *tmp = gnome_pixmap_new_from_xpm_d ((char **) custom->xpm_data);
    custom->image = GNOME_PIXMAP (tmp);
  }

  return custom->image;
}

GnomePixmap *
guppi_major_chart_type_image (GuppiMajorChartType const *t)
{
  MajorChartTypeImpl *major;

  g_return_val_if_fail (t != NULL, NULL);

  major = (MajorChartTypeImpl *) t;
  if (major->image == NULL) {
    GtkWidget *tmp = gnome_pixmap_new_from_xpm_d ((char **) major->xpm_data);
    major->image = GNOME_PIXMAP (tmp);
  }

  return major->image;
}

GdkPixbuf *
guppi_minor_chart_type_image (GuppiMinorChartType const *t)
{
  MinorChartTypeImpl *minor;

  g_return_val_if_fail (t != NULL, NULL);

  minor = (MinorChartTypeImpl *) t;
  if (minor->image == NULL) {
    char *path = guppi_find_pixmap (minor->icon_file);

    g_return_val_if_fail (path != NULL, NULL);

    minor->image = gdk_pixbuf_new_from_file (path);
    guppi_free (path);
  }

  return minor->image;
}

gboolean
guppi_chart_type_foreach_major (MajorChartTypeFunc handler,
				gpointer user_data)
{
  GList *ptr = major_chart_types;

  while (ptr != NULL && (*handler) (ptr->data, user_data))
    ptr = ptr->next;

  return ptr == NULL;
}

gboolean
guppi_major_chart_type_foreach_minor (GuppiMajorChartType const *t,
				      MinorChartTypeFunc handler,
				      gpointer user_data)
{
  MajorChartTypeImpl const *major = (MajorChartTypeImpl const *) t;
  GList *ptr = major->minor_chart_types;

  while (ptr != NULL && (*handler) (t, ptr->data, user_data))
    ptr = ptr->next;

  return ptr == NULL;
}
