/* This is -*- C -*- */
/* $Id: guppi-element-state.c,v 1.21 2001/01/16 23:36:26 trow Exp $ */

/*
 * guppi-element-state.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <guppi-useful.h>
#include <guppi-unique-id.h>
#include "guppi-plot-plug-in.h"
#include "guppi-element-state.h"
#include "guppi-element-view.h"
#include "guppi-shared-hash.h"
#include "guppi-style.h"

typedef struct _GuppiElementStatePrivate GuppiElementStatePrivate;
struct _GuppiElementStatePrivate {
  guppi_uniq_t id;
  gchar *label;
  GuppiSharedHash *hash;
  guint pending_change;
};

#define priv(x) ((GuppiElementStatePrivate*)((x)->opaque_internals))


static GtkObjectClass *parent_class = NULL;

enum {
  CHANGED,
  CHANGED_SIZE,
  LAST_SIGNAL
};

static guint ges_signals[LAST_SIGNAL] = { 0 };

static void
guppi_element_state_finalize (GtkObject * obj)
{
  GuppiElementState *ges = GUPPI_ELEMENT_STATE (obj);
  GuppiElementStatePrivate *p = priv (ges);

  guppi_finalized (obj);

  guppi_free (p->label);

  guppi_unref0 (p->hash);
  p->hash = NULL;

  if (p->pending_change) {
    gtk_idle_remove (p->pending_change);
    p->pending_change = 0;
  }

  guppi_free (ges->opaque_internals);
  ges->opaque_internals = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
guppi_element_state_class_init (GuppiElementStateClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  parent_class = gtk_type_class (GTK_TYPE_OBJECT);

  object_class->finalize = guppi_element_state_finalize;

  ges_signals[CHANGED] =
    gtk_signal_new ("changed",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiElementStateClass, changed),
		    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

  ges_signals[CHANGED_SIZE] =
    gtk_signal_new ("changed_size",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiElementStateClass, changed_size),
		    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals (object_class, ges_signals, LAST_SIGNAL);
}

static void
guppi_element_state_init (GuppiElementState * obj)
{
  GuppiElementStatePrivate *p = guppi_new0 (GuppiElementStatePrivate, 1);
  obj->opaque_internals = p;

  p->id = guppi_unique_id ();
  p->hash = guppi_shared_hash_new ();
}

GtkType guppi_element_state_get_type (void)
{
  static GtkType guppi_element_state_type = 0;
  if (!guppi_element_state_type) {
    static const GtkTypeInfo guppi_element_state_info = {
      "GuppiElementState",
      sizeof (GuppiElementState),
      sizeof (GuppiElementStateClass),
      (GtkClassInitFunc) guppi_element_state_class_init,
      (GtkObjectInitFunc) guppi_element_state_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_element_state_type =
      gtk_type_unique (GTK_TYPE_OBJECT, &guppi_element_state_info);
  }
  return guppi_element_state_type;
}

/****************************************************************************/

GuppiElementState *
guppi_element_state_new (const gchar * type, const gchar * first_arg_name,
			 ...)
{
  GuppiElementState *state;
  GSList *arg_list;
  GSList *info_list;
  gchar *error;
  va_list var_args;

  g_return_val_if_fail (type != NULL, NULL);

  state = guppi_element_state_newv (type, 0, NULL);
  g_return_val_if_fail (state != NULL, NULL);

  va_start (var_args, first_arg_name);
  error = gtk_object_args_collect (GTK_OBJECT_TYPE (state),
				   &arg_list,
				   &info_list, first_arg_name, var_args);

  if (error) {
    g_warning ("guppi_element_state_new(): %s", error);
    guppi_free (error);
  } else {
    GSList *slist_arg;
    GSList *slist_info;

    slist_arg = arg_list;
    slist_info = info_list;
    while (slist_arg) {
      gtk_object_arg_set (GTK_OBJECT (state), slist_arg->data,
			  slist_info->data);
      slist_arg = g_slist_next (slist_arg);
      slist_info = g_slist_next (slist_info);
    }
    gtk_args_collect_cleanup (arg_list, info_list);
  }

  return state;
}

GuppiElementState *
guppi_element_state_newv (const gchar * type, guint nargs, GtkArg * args)
{
  GuppiPlugIn *plug_in;
  GuppiPlotPlugIn *plot_plug_in;
  GuppiElementState *state;
  GuppiElementStateClass *klass;

  g_return_val_if_fail (type != NULL, NULL);

  /* Find our plug-in. */
  plug_in = guppi_plug_in_lookup ("plot", type);
  if (plug_in == NULL) {
    g_warning ("Unknown plot plug-in: \"%s\"", type);
    return NULL;
  }
  g_return_val_if_fail (GUPPI_IS_PLOT_PLUG_IN (plug_in), NULL);
  plot_plug_in = GUPPI_PLOT_PLUG_IN (plug_in);

  /* Build our plot element. */
  g_return_val_if_fail (plot_plug_in->element_constructor != NULL, NULL);
  state = (plot_plug_in->element_constructor) ();
  g_return_val_if_fail (state != NULL, NULL);

  /* Pass in our args. */
  gtk_object_setv (GTK_OBJECT (state), nargs, args);

  /* Remember the plug in that created us, as necessary... */
  klass = GUPPI_ELEMENT_STATE_CLASS (GTK_OBJECT (state)->klass);
  if (klass->plug_in_code == NULL) {
    klass->plug_in_code = guppi_strdup (guppi_plug_in_code (plug_in));
    guppi_permanent_alloc (klass->plug_in_code);
  }

  return state;
}

/****************************************************************************/

guppi_uniq_t
guppi_element_state_unique_id (GuppiElementState *state)
{
  g_return_val_if_fail (state && GUPPI_IS_ELEMENT_STATE (state), 0);

  return priv (state)->id;
}

const gchar *
guppi_element_state_label (GuppiElementState * state)
{
  GuppiElementStatePrivate *p;

  g_return_val_if_fail (state != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_ELEMENT_STATE (state), NULL);

  p = priv (state);

  if (p->label == NULL) {
    GuppiElementStateClass *klass;
    const gchar *name;
    gchar *buffer;

    klass = GUPPI_ELEMENT_STATE_CLASS (GTK_OBJECT (state)->klass);
    if (klass->name)
      name = klass->name;
    else
      name = gtk_type_name (GTK_OBJECT_TYPE (state));

    buffer = guppi_strdup_printf (_("Unnamed %s"), name);
    guppi_element_state_set_label (state, buffer);
    guppi_free (buffer);
  }

  return p->label;
}

void
guppi_element_state_set_label (GuppiElementState * state, const gchar * lab)
{
  GuppiElementStatePrivate *p;

  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_STATE (state));
  g_return_if_fail (lab != NULL);

  p = priv (state);
  
  guppi_free (p->label);
  p->label = guppi_strdup (lab);
}

/****************************************************************************/

GuppiElementView *
guppi_element_state_make_view (GuppiElementState * ges)
{
  GuppiElementStateClass *klass;
  GuppiElementView *view = NULL;
  GuppiElementViewClass *view_klass;

  g_return_val_if_fail (ges != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_ELEMENT_STATE (ges), NULL);

  klass = GUPPI_ELEMENT_STATE_CLASS (GTK_OBJECT (ges)->klass);

  if (klass->view_type && klass->make_view) {
    g_warning ("For %s, both a view type and a view constructor are defined.",
	       gtk_type_name (GTK_OBJECT_TYPE (ges)));
  }

  if (klass->view_type) {

    view = GUPPI_ELEMENT_VIEW (guppi_type_new (klass->view_type));

  } else if (klass->make_view) {

    view = klass->make_view (ges);

  }

  /* Tweak the insides of our view object. */
  if (view)
    guppi_element_view_set_state (view, ges);

  view_klass = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);
  if (view_klass->view_init)
    view_klass->view_init (view);

  return view;
}

/*****************************************************************************/

void
guppi_element_state_add_shared (GuppiElementState * state,
				const gchar * key, GuppiShared * sh)
{
  GuppiElementStatePrivate *p;

  g_return_if_fail (state != NULL && GUPPI_IS_ELEMENT_STATE (state));
  g_return_if_fail (key != NULL);
  g_return_if_fail (sh != NULL && GUPPI_IS_SHARED (sh));

  p = priv (state);

  guppi_shared_hash_add (p->hash, key, sh);

  /* Any shared object is considered to be part of our state,
     so its "changed" signals get propogated up and emitted as
     GuppiElementState "changed" signals. */
  guppi_element_state_connect_to_object (state, "changed", GTK_OBJECT (sh));
}

void
guppi_element_state_set_shared (GuppiElementState * state,
				const gchar * key, gpointer ptr)
{
  GuppiElementStatePrivate *p;

  g_return_if_fail (state != NULL && GUPPI_IS_ELEMENT_STATE (state));
  g_return_if_fail (key != NULL);
  g_return_if_fail (ptr == NULL || GTK_IS_OBJECT (ptr));

  p = priv (state);

  guppi_shared_hash_set (p->hash, key, ptr ? GTK_OBJECT (ptr) : NULL);
}

GtkObject *
guppi_element_state_get_shared (GuppiElementState * state, const gchar * key)
{
  GuppiElementStatePrivate *p;

  g_return_val_if_fail (state != NULL
			&& GUPPI_IS_ELEMENT_STATE (state), NULL);
  g_return_val_if_fail (key != NULL, NULL);

  p = priv (state);

  return guppi_shared_hash_get (p->hash, key);
}

/*****************************************************************************/

void
guppi_element_state_connect (GuppiElementState * s1, const gchar * key1,
			     GuppiElementState * s2, const gchar * key2)
{
  GuppiShared *sh1;
  GuppiShared *sh2;

  g_return_if_fail (s1 != NULL && GUPPI_IS_ELEMENT_STATE (s1));
  g_return_if_fail (key1 != NULL);
  g_return_if_fail (s2 != NULL && GUPPI_IS_ELEMENT_STATE (s2));
  g_return_if_fail (key2 != NULL);


  sh1 = guppi_shared_hash_extract (priv (s1)->hash, key1);
  g_return_if_fail (sh1 != NULL);

  sh2 = guppi_shared_hash_extract (priv (s2)->hash, key2);
  g_return_if_fail (sh2 != NULL);

  if (sh1 != sh2) {
    guppi_shared_connect (sh1, sh2);
  }
}

/*****************************************************************************/

void
guppi_element_state_changed (GuppiElementState * ges)
{
  GuppiElementStatePrivate *p;

  g_return_if_fail (ges != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_STATE (ges));

  p = priv (ges);

  if (p->pending_change) {
    gtk_idle_remove (p->pending_change);
    p->pending_change = 0;
  }

  gtk_signal_emit (GTK_OBJECT (ges), ges_signals[CHANGED]);
}

static gint
delayed_changer (gpointer foo)
{
  GuppiElementStatePrivate *p;
  GuppiElementState *ges;

  g_return_val_if_fail (foo != NULL && GUPPI_IS_ELEMENT_STATE (foo), FALSE);

  ges = GUPPI_ELEMENT_STATE (foo);
  p = priv (ges);

  p->pending_change = 0;
  guppi_element_state_changed (ges);

  return FALSE;
}

void
guppi_element_state_changed_delayed (GuppiElementState * ges)
{
  GuppiElementStatePrivate *p;

  g_return_if_fail (ges != NULL && GUPPI_IS_ELEMENT_STATE (ges));

  p = priv (ges);

  if (guppi_is_synchronous ()) {

    guppi_element_state_changed (ges);

  } else {

    if (p->pending_change == 0)
      p->pending_change = gtk_idle_add_priority (G_PRIORITY_HIGH_IDLE,
						 delayed_changer, ges);

  }
}

void
guppi_element_state_changed_size (GuppiElementState * ges)
{
  g_return_if_fail (ges != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_STATE (ges));

  gtk_signal_emit (GTK_OBJECT (ges), ges_signals[CHANGED_SIZE]);
}

void
guppi_element_state_connect_to_object (GuppiElementState * state,
				       const gchar * signal_name,
				       GtkObject * obj)
{
  g_return_if_fail (state != NULL && GUPPI_IS_ELEMENT_STATE (state));
  g_return_if_fail (signal_name != NULL);
  g_return_if_fail (obj != NULL && GTK_IS_OBJECT (obj));

  gtk_signal_connect_object_after (obj,
				   signal_name,
				   GTK_SIGNAL_FUNC
				   (guppi_element_state_changed),
				   GTK_OBJECT (state));
}

void
guppi_element_state_disconnect_from_object (GuppiElementState * state,
					    GtkObject * obj)
{
  g_return_if_fail (state != NULL && GUPPI_IS_ELEMENT_STATE (state));
  g_return_if_fail (obj != NULL && GTK_IS_OBJECT (obj));

  gtk_signal_disconnect_by_func (obj,
				 GTK_SIGNAL_FUNC
				 (guppi_element_state_changed), state);
}

void
guppi_element_state_write_xml (GuppiElementState *state,
			       GuppiOutputXML *out)
{
  GuppiElementStateClass *klass;
  GuppiElementStatePrivate *p;
  gchar uniq_str[17];

  g_return_if_fail (state && GUPPI_IS_ELEMENT_STATE (state));
  g_return_if_fail (out && GUPPI_IS_OUTPUT_XML (out));

  klass = GUPPI_ELEMENT_STATE_CLASS(GTK_OBJECT (state)->klass);
  p = priv (state);

  guppi_output_xml_tagf (out, "ElementState", "Type='%s'",
			 klass->plug_in_code ? klass->plug_in_code : "?");

  guppi_output_xml_tag (out, "Attributes");

  guppi_uniq2str (p->id, uniq_str);
  guppi_output_xml_attribute (out, "unique_id", uniq_str);

  if (klass->write_xml_attributes)
    klass->write_xml_attributes (state, out);

  guppi_output_xml_end_tag_check (out, "Attributes");

  if (klass->write_xml_other)
    klass->write_xml_attributes (state, out);

  guppi_output_xml_end_tag_check (out, "ElementState");
}

void
guppi_element_state_spew_xml (GuppiElementState *state)
{
  GuppiOutputXML *out;

  g_return_if_fail (state && GUPPI_IS_ELEMENT_STATE (state));
  
  out = guppi_output_xml_new ();

  guppi_element_state_write_xml (state, out);

  guppi_unref0 (out);
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

typedef struct _StateParse StateParse;
struct _StateParse {
  GuppiElementState *state;
  GuppiElementStateClass *klass;
  GuppiAttributesXML *ax;

  void (*handler) (GuppiElementState *, gpointer);
  gpointer user_data;
};

static StateParse *
sp_new (void)
{
  StateParse *sp = guppi_new0 (StateParse, 1);
  
  sp->ax = guppi_attributes_xml_new ();

  return sp;
}

static void
sp_free (gpointer ud)
{
  StateParse *sp = (StateParse *)sp;

  guppi_unref0 (sp->ax);

  guppi_free (sp);
}

static GuppiContextXML *
sp_begin_element (const CHAR *name, const CHAR **attrs, gpointer ud)
{
  /* blah blah blah */

  return NULL;
}

static void
sp_end_context (gpointer ud, gpointer outer_context_data)
{
  StateParse *sp = (StateParse *)sp;

  if (sp->handler)
    sp->handler (sp->state, sp->user_data);
}

GuppiContextXML *
guppi_element_state_read_xml_context (void (*fn)(GuppiElementState *,
						 gpointer),
				      gpointer user_data)
{
  GuppiContextXML *ctx;
  GuppiContextXML *attr_ctx;
  StateParse *sp;

  if (fn == NULL)
    g_warning ("No ElementState handler function specified.");

  sp = sp_new ();
  sp->handler = fn;
  sp->user_data = user_data;

  ctx = guppi_context_xml_new ();
  guppi_context_xml_set_begin_element_fn (ctx, sp_begin_element);
  guppi_context_xml_set_end_context_fn (ctx, sp_end_context);
  guppi_context_xml_set_user_data_destroy_fn (ctx, sp, sp_free);

  attr_ctx = guppi_attributes_xml_build_context (sp->ax);
  guppi_context_xml_add_element_context (ctx, "gpi:Attributes", attr_ctx);

  return ctx;
}

/* $Id: guppi-element-state.c,v 1.21 2001/01/16 23:36:26 trow Exp $ */
